/*
 *      File MOUI.H
 *      A Header for Mouse Interrupt Interface Library
 *      (Interrupt Based Mouse Functions)
 *      '91.11.10,13, 12.4
 *      '92.2.5
 *      Written by Lim, I.K.
 *
 *      Note: Use Microsoft Mouse Driver Version 6.25 or later, if possible.
 *            Other compatible mouse drivers which have
 *            version number 6.25 or later may be available, too.
 */


#ifndef _MOUI_H_
#define _MOUI_H_

#if !defined(__DOS_DEF_) || !defined(__DOS_H)
    #include <dos.h>
#endif

#ifndef _TYPEDEFS_H_
    #include "typedefs.h"
#endif


/** Macro constants ********************************************************/

#define MOUINT 0x33  /* Mouse interrupt */

#define MOUI_NOERROR 0   /* No mouse error */
#define MOUI_ERROR (-1)  /* Mouse error    */

/* Event masks */

#ifndef EVN_NONE
    #define EVN_NONE   0x00  /* No event occurred                */
    #define EVN_MOVE   0x01  /* Mouse moved                      */
    #define EVN_LDOWN  0x02  /* Left button pressed              */
    #define EVN_LUP    0x04  /* Left button released             */
    #define EVN_RDOWN  0x08  /* Right button pressed             */
    #define EVN_RUP    0x10  /* Right button released            */
    #define EVN_L      0x06  /* Left button pressed or released  */
    #define EVN_R      0x18  /* Right button pressed or released */
    #define EVN_DOWN   0x0a  /* Any button pressed               */
    #define EVN_UP     0x14  /* Any button pressed               */
    #define EVN_ALLBTN 0x1e  /* Any button pressed or released   */
    #define EVN_ALL    0x1f  /* All mouse events                 */
#endif

/* Button masks */

#ifndef BTN_NONE
    #define BTN_NONE 0x00  /* No button     */
    #define BTN_L    0x01  /* Left button   */
    #define BTN_R    0x02  /* Right button  */
    #define BTN_C    0x04  /* Center button */
    #define BTN_ALL  0x07  /* All buttons   */
#endif

/* The cursor type for moui_settextmouse */

#define MOUI_SOFTWARE 0
#define MOUI_HARDWARE 1


/** Type definitions *******************************************************/

/* Type of the mouse event handler */

#ifndef EVNHAND_T
    #define EVNHAND_T
    typedef void (far *evnhand_t)();
#endif


/** Function declarations **************************************************/

#ifdef __cplusplus
extern "C" {
#endif

/*
 *   << The Summary Of The Mouse Function Calls >>
 *  For Microsoft Mouse Driver Version 6.25 Or Later
 *   - By Microsoft Mouse Programmer's Reference -
 *
 *--------------------------------------------------------------------------
 *  No.   Mouse Function Calls (Total 35)
 *--------------------------------------------------------------------------
 *   0    Mouse Reset And Status
 *   1    Show Cursor
 *   2    Hide Cursor
 *   3    Get Button Status & Mouse Position
 *   4    Set Mouse Cursor Position
 *   5    Get Button Press Information
 *   6    Get Button Release Information
 *   7    Set Minimum And Maximum Horizontal Cursor Position
 *   8    Set Minimum And Maximum Vertical Cursor Position
 *   9    Set Graphics Cursor Block
 *  10    Set Text Cursor
 *  11    Read Mouse Motion Counters
 *  12   *Set Interrupt Subroutine Call Mask And Address (-> No.20 & No.24)
 *  13    Light Pen Emulation Mode On
 *  14    Light Pen Emulation Mode Off
 *  15    Set Mickey/Pixel Ratio
 *  16    Conditional Off
 *  17   *None
 *  18   *None
 *  19   *Set Double-Speed Threshold (-> No.26)
 *        --- Preceeding functions are old mouse functions ---
 *        --- Following functions are new mouse functions ----
 *  20    Swap Interrupt Subroutines
 *  21    Get Mouse Driver State Storage Requirements
 *  22    Save Mouse Driver State
 *  23    Restore Mouse Driver State
 *  24    Set Alternate Subroutine Call Mask And Address
 *  25    Get User Alternate Interrupt Address
 *  26    Set Mouse Sensitivity
 *  27    Get Mouse Sensitivity
 *  28   *Set Mouse Interrupt Rate (-> Only InPort Mouse)
 *  29    Set CRT Page Number
 *  30    Get CRT Page Number
 *  31    Diable Mouse Driver
 *  32    Enable Mouse Driver
 *  33    Software Reset
 *  34    Set Language For Messages
 *  35    Get Language Number
 *  36    Get Driver Version, Mouse Type, And IRQ Number
 *--------------------------------------------------------------------------
 *        Functions at which '*' is marked are not available.
 *        Because some functions are not supported
 *        and other functions have new alternative ones.
 *
 *
 *   - By Advanced MS-DOS Programming, 2nd Ed. -
 *
 *--------------------------------------------------------------------------
 *  No.   Mouse Function Calls (Total 35)
 *--------------------------------------------------------------------------
 *   0    Reset Mouse and Get Status
 *   1    Show Mouse Pointer
 *   2    Hide Mouse Pointer
 *   3    Get Mouse Position and Button Status
 *   4    Set Mouse Pointer Position
 *   5    Get Button Press Information
 *   6    Get Button Release Information
 *   7    Set Horizontal Limits for Pointer
 *   8    Set Vertical Limits for Pointer
 *   9    Set Graphics Pointer Shape
 *  10    Set Text Pointer Type
 *  11    Read Mouse Motion Counters
 *  12   *Set User-defined Mouse Event Handler (-> No.20 & No.24)
 *  13    Turn On Light Pen Emulation
 *  14    Turn Off Light Pen Emulation
 *  15    Set Mickeys to Pixels Ratio
 *  16    Set Mouse Pointer Exclusion Area
 *  17   *None
 *  18   *None
 *  19   *Set Double Speed Threshold (-> No.26)
 *        --- Preceeding functions are old mouse functions ---
 *        --- Following functions are new mouse functions ----
 *  20    Swap User-defined Mouse Event Handlers
 *  21    Get Mouse Save State Buffer Size
 *  22    Save Mouse Driver State
 *  23    Restore Mouse Driver State
 *  24    Set Alternate Mouse Event Handler
 *  25    Get Address of Alternate Mouse Event Handler
 *  26    Set Mouse Sensitivity
 *  27    Get Mouse Sensitivity
 *  28   *Set Mouse Interrupt Rate (-> Only InPort Mouse)
 *  29    Select Pointer Page
 *  30    Get Pointer Page
 *  31    Diable Mouse Driver
 *  32    Enable Mouse Driver
 *  33    Reset Mouse Driver
 *  34    Set Language for Mouse Driver Messages
 *  35    Get Language Number
 *  36    Get Mouse Information
 *--------------------------------------------------------------------------
 *
 *
 * << The Old List of The Mouse Function Calls >>
 *        For All Microsoft Mouse Drivers
 *
 *------------------------------------------------
 *  No.   Function Calls (Total 17)
 *------------------------------------------------
 *   0    Mouse Initialization
 *   1    Show Cursor
 *   2    Hide Cursor
 *   3    Get Button Status & Cursor Position
 *   4    Set Cursor Position
 *   5    Get Button Press Information
 *   6    Get Button Release Information
 *   7    Set Horizontal Cursor Range
 *   8    Set Vertical Cursor Range
 *   9    Define Graphics Cursor Block
 *  10    Define Text Cursor
 *  11    Get Mouse Movement Counters
 *  12    Define Event Handler
 *  13    Light Pen Emulation On
 *  14    Light Pen Emulation Off
 *  15    Set Mouse Movement/Pixel Ratio
 *  16    Conditional Off
 *  19    Set Double-Speed Threshold
 *------------------------------------------------
 */

/* Interrupt based mouse functions */

int  moui_init(void);
void moui_close(void);
void moui_show(void);
void moui_hide(void);
void moui_getbtnpxy(int *btn, int *px, int *py);
byte moui_getbtn(void);
int  moui_getpx(void);
int  moui_getpy(void);
void moui_moveto(int px, int py);
void moui_getbtnpressinfo(byte btn, byte *status, int *times,
                          int *lastx, int *lasty);
void moui_getpxylastpress(byte btn, int *lastx, int *lasty);
void moui_getbtnreleaseinfo(byte btn, byte *status, int *times,
                            int *lastx, int *lasty);
void moui_getpxylastrelease(int btn, int *lastx, int *lasty);
void moui_setwindow(int px1, int py1, int px2, int py2);
void moui_setgraphmouse(int hotspotx, int hotspoty, word *shape);
void moui_settextmouse(int type, word screenmask, word cursormask);
void moui_getmickey(int *mickeyx, int *mickeyy);
void moui_lightpenemul(bool on);
void moui_setspeed(int xspeed, int yspeed);
void moui_hideregion(int px1, int py1, int px2, int py2);
void moui_installevnhandler(word evn, evnhand_t handler);
void moui_uninstallevnhandler(void);
int  moui_getstatbufsize(void);
void moui_savestat(void *statbuf);
void moui_restorestat(const void *statbuf);
void moui_installaltevnhandler(word evn, evnhand_t handler);
void moui_uninstallaltevnhandler(void);
evnhand_t moui_getaltevnhandler(word evn);
void moui_setsens(int xsens, int ysens, int threshold);
int  moui_getxsens(void);
int  moui_getysens(void);
int  moui_getthres(void);
void moui_setpage(int page);
int  moui_getpage(void);
int  moui_disable(void);
void moui_enable(void);
int  moui_reset(void);
void moui_setlang(int lang);
int  moui_getlang(void);
word moui_getver(void);
int  moui_gettype(void);
int  moui_getirq(void);

#ifdef __cplusplus
}
#endif


/** Macro functions ********************************************************/

#define moui_isactive() _mouiactive

#define moui_int(regs) int86(MOUINT, &(regs), &(regs))
#define moui_intx(regs, segregs) int86x(MOUINT, &(regs), &(regs), &(segregs))


/** External variables *****************************************************/

extern const bool _mouiactive;       /* These variables are all READ-ONLY! */
extern const int  _mouihiddenlevel;
extern const int  _mouibtnnum;
extern const word _mouiversion;

extern interrupt (*_mouioldvector)();

extern const bool _evnhandlerinstalled;
extern const bool _altevnhandlerinstalled;

#endif
