/*
 *      File HANMOU.H
 *      A Header for Hangul Mouse Library
 *      (Event Handler Based Mouse Functions)
 *      '91.11.10,13-14, 12.1,3-4
 *      '92.1.6,9,12, 2.5, 4.24, 9.13
 *      Written by Lim, I.K.
 */


#ifndef _HANMOU_H_
#define _HANMOU_H_

#ifndef _CHKVER_H_
    #include "chkver.h"
#endif

#ifndef _TYPEDEFS_H_
    #include "typedefs.h"
#endif


/** Macro constants ********************************************************/

/* Arrow-type mouse cursors */

#define MOU_ARROW      (&_mouarrow)
#define MOU_ARROWMONO  (&_mouarrowmono)
#define MOU_ARROW16    (&_mouarrow16)
#define MOU_ARROW256   (&_mouarrow256)
#define MOU_ARROW32768 (&_mouarrow32768)

/* Event masks */

#ifndef EVN_NONE
    #define EVN_NONE   0x00  /* No event occurred                */
    #define EVN_MOVE   0x01  /* Mouse moved                      */
    #define EVN_LDOWN  0x02  /* Left button pressed              */
    #define EVN_LUP    0x04  /* Left button released             */
    #define EVN_RDOWN  0x08  /* Right button pressed             */
    #define EVN_RUP    0x10  /* Right button released            */
    #define EVN_LBTN   0x06  /* Left button pressed or released  */
    #define EVN_RBTN   0x18  /* Right button pressed or released */
    #define EVN_DOWN   0x0a  /* Any button pressed               */
    #define EVN_UP     0x14  /* Any button pressed               */
    #define EVN_ANYBTN 0x1e  /* Any button pressed or released   */
    #define EVN_ALL    0x1f  /* All mouse events                 */
#endif

/* Button masks */

#ifndef BTN_NONE
    #define BTN_NONE 0x00  /* No button     */
    #define BTN_L    0x01  /* Left button   */
    #define BTN_R    0x02  /* Right button  */
    #define BTN_C    0x04  /* Center button */
    #define BTN_ALL  0x07  /* All buttons   */
#endif

#define MOU_NOERROR 0   /* No mouse error */
#define MOU_ERROR (-1)  /* Mouse error    */


/** Type definitions *******************************************************/

#ifndef MOUCURSOR_T
    #define MOUCURSOR_T
    typedef struct moucursor__t {
        int width, height;
        int hotspotx, hotspoty;
        void far *shape;
        void far *bitmask;
        int maxcolor;
        struct moucursor__t *nextcursor;
    } moucursor_t;
#endif

#ifndef MOUEVN_T
    #define MOUEVN_T
    typedef word mouevn_t;
    typedef word moubtnstat_t;
#endif

#ifndef MOUHIDAREA_T
    #define MOUHIDAREA_T
    struct mouhidarea_t {
        bool installed;
        int px1, py1, px2, py2;
    };
#endif

typedef struct {
    int  left, top, right, bottom;
    int  width, height;
} mouwindow_t;


/** Function declarations **************************************************/

#ifdef __cplusplus
extern "C" {
#endif

/* Event handler based mouse functions */

int mou_init(moucursor_t *moucuror);
int mou_close(void);

void mou_show(void);
void mou_hide(void);

void mou_moveto(int px, int py);
void mou_gotoxy(int x, int y, int dx, int dy);

moucursor_t *mou_linkcursor(moucursor_t *firstcursor, ...);

int mou_changecursor(moucursor_t *moucursor);
int mou_changeshape(moucursor_t *moucursor);

int mou_setwindow(int left, int top, int right, int bottom);
int mou_setwindowpxy(int left, int top, int right, int bottom);

int  mou_sethidarea(int x1, int y1, int x2, int y2);
int  mou_sethidareapxy(int px1, int py1, int px2, int py2);
int  mou_removehidarea(int index);
bool mou_ishidarea(int index);

void mou_evnwait(mouevn_t waitevn);

byte moui_getbtn(void);

#ifdef __cplusplus
}
#endif


/** Macro functions ********************************************************/

#define mou_isactive() _mouactive

#define mou_isshown() (_mouhiddenlevel == 0)
#define mou_isdisplayed() _mouisdisplayed
#define mou_isinhidarea() _mouinhidarea

#ifndef mou_ismelted
    #define mou_ismelted() (_moufrozenlevel == 0)
    #define mou_isfrozen() (_moufrozenlevel != 0)
    #define mou_freeze()   ((*(int *)&_moufrozenlevel)--)
    #define mou_melt()     (mou_ismelted() ? 0 : (*(int *)&_moufrozenlevel)++)
#endif

#define mou_isnoevn()     (_moucurevn == EVN_NONE)
#define mou_ismovevn()    ((_moucurevn & EVN_MOVE)   != 0x00)
#define mou_isldownevn()  ((_moucurevn & EVN_LDOWN)  != 0x00)
#define mou_islupevn()    ((_moucurevn & EVN_LUP)    != 0x00)
#define mou_isrdownevn()  ((_moucurevn & EVN_RDOWN)  != 0x00)
#define mou_isrupevn()    ((_moucurevn & EVN_RUP)    != 0x00)
#define mou_islbtnevn()   ((_moucurevn & EVN_LBTN)   != 0x00)
#define mou_isrbtnevn()   ((_moucurevn & EVN_RBTN)   != 0x00)
#define mou_isdownevn()   ((_moucurevn & EVN_DOWN)   != 0x00)
#define mou_isupevn()     ((_moucurevn & EVN_UP)     != 0x00)
#define mou_isanybtnevn() ((_moucurevn & EVN_ANYBTN) != 0x00)
#define mou_isanyevn()    (_moucurevn != EVN_NONE)

#define mou_haveanyevn(evn)     ((_moucurevn & (evn))  != 0x00)
#define mou_haveotherevn(evn)   ((_moucurevn & ~(evn)) != 0x00)
#define mou_havenoevn(evn)      ((_moucurevn & (evn))  == 0x00)
#define mou_havenootherevn(evn) ((_moucurevn & ~(evn)) == 0x00)

#define mou_clrevn() (_moucurevn = EVN_NONE)

#define mou_getbtnstat() _moucurbtnstat

#define mou_isldown() ((_moucurbtnstat & BTN_L) != 0x00)
#define mou_islup()   ((_moucurbtnstat & BTN_L) == 0x00)
#define mou_isrdown() ((_moucurbtnstat & BTN_R) != 0x00)
#define mou_isrup()   ((_moucurbtnstat & BTN_R) == 0x00)
#define mou_iscdown() ((moui_getbtn() & BTN_C) != 0x00)
#define mou_iscup()   ((moui_getbtn() & BTN_C) == 0x00)

#define mou_getpx() apx2px(_moucurapx)
#define mou_getpy() apy2py(_moucurapy)
#define mou_getx()  apx2x(_moucurapx)
#define mou_gety()  apy2y(_moucurapy)

#define mou_getapx() _moucurapx
#define mou_getapy() _moucurapy
#define mou_getax()  apx2ax(_moucurapx)
#define mou_getay()  apy2ay(_moucurapy)

#define mou__isnoevn(evn)     ((evn) == EVN_NONE)
#define mou__ismovevn(evn)    (((evn) & EVN_MOVE)   != 0x00)
#define mou__isldownevn(evn)  (((evn) & EVN_LDOWN)  != 0x00)
#define mou__islupevn(evn)    (((evn) & EVN_LUP)    != 0x00)
#define mou__isrdownevn(evn)  (((evn) & EVN_RDOWN)  != 0x00)
#define mou__isrupevn(evn)    (((evn) & EVN_RUP)    != 0x00)
#define mou__islbtnevn(evn)   (((evn) & EVN_LBTN)   != 0x00)
#define mou__isrbtnevn(evn)   (((evn) & EVN_RBTN)   != 0x00)
#define mou__isdownevn(evn)   (((evn) & EVN_DOWN)   != 0x00)
#define mou__isupevn(evn)     (((evn) & EVN_UP)     != 0x00)
#define mou__isanybtnevn(evn) (((evn) & EVN_ANYBTN) != 0x00)
#define mou__isanyevn(evn)    ((evn) != EVN_NONE)

#define mou__clrevn(evn)    ((evn) = EVN_NONE)
#define mou__clrmovevn(evn) ((evn) &= ~EVN_MOVE)

#define mou__isldown(btnstat) (((btnstat) & BTN_L) != 0x00)
#define mou__islup(btnstat)   (((btnstat) & BTN_L) == 0x00)
#define mou__isrdown(btnstat) (((btnstat) & BTN_R) != 0x00)
#define mou__isrup(btnstat)   (((btnstat) & BTN_R) == 0x00)
#define mou__iscdown(btnstat) (((btnstat) & BTN_C) != 0x00)
#define mou__iscup(btnstat)   (((btnstat) & BTN_C) == 0x00)

#define mou_getbtnnum() _mouibtnnum
#define mou_getver()    _mouiversion


/** External variables *****************************************************/

extern const bool _mouactive;        /* These variables are READ-ONLY!       */
extern const int  _mouhiddenlevel;   /* Defined in the Hangul Output Library */
extern const int  _moufrozenlevel;
extern const bool _mouautohidden;

extern const bool _mouisdisplayed;
extern const bool _mouismoving;

extern mouwindow_t _mouwin;

extern volatile mouevn_t _moucurevn;          /* READ-ONLY! */
extern volatile moubtnstat_t _moucurbtnstat;
extern volatile int _moucurapx, _moucurapy;

extern int _moumaxhidarea;
extern volatile bool _mouinhidarea;  /* READ-ONLY! */

extern void (*_mouhookmove)(moubtnstat_t btnstat, int mouapx, int mouapy);
extern void (*_mouhooklbtn)(moubtnstat_t btnstat, int mouapx, int mouapy);
extern void (*_mouhookrbtn)(moubtnstat_t btnstat, int mouapx, int mouapy);

extern moucursor_t _mouarrow;
extern moucursor_t _mouarrowmono;
extern moucursor_t _mouarrow16;
extern moucursor_t _mouarrow256;
extern moucursor_t _mouarrow32768;

extern const int  _mouibtnnum;
extern const word _mouiversion;

#endif
