/*
 *      File HANMEQ.H
 *      A Header for Hangul Mouse Library
 *      (Event Queue Based Mouse Functions)
 *      '92.1.7-9, 2.5, 6.14
 *      '93.8.8
 *      Written by Lim, I.K.
 */


#ifndef _HANMEQ_H_
#define _HANMEQ_H_

#ifndef _CHKVER_H_
    #include "chkver.h"
#endif

#ifndef _TYPEDEFS_H_
    #include "typedefs.h"
#endif

#ifndef _HANCOORD_H_
    #include "hancoord.h"
#endif


/** Macro constants ********************************************************/

#define MEQ_NOERROR 0   /* No mouse event queue error */
#define MEQ_ERROR (-1)  /* Mouse event queue error    */

/* Event masks */

#ifndef EVN_NONE
    #define EVN_NONE   0x00  /* No event occurred                */
    #define EVN_MOVE   0x01  /* Mouse moved                      */
    #define EVN_LDOWN  0x02  /* Left button pressed              */
    #define EVN_LUP    0x04  /* Left button released             */
    #define EVN_RDOWN  0x08  /* Right button pressed             */
    #define EVN_RUP    0x10  /* Right button released            */
    #define EVN_LBTN   0x06  /* Left button pressed or released  */
    #define EVN_RBTN   0x18  /* Right button pressed or released */
    #define EVN_DOWN   0x0a  /* Any button pressed               */
    #define EVN_UP     0x14  /* Any button pressed               */
    #define EVN_ANYBTN 0x1e  /* Any button pressed or released   */
    #define EVN_ALL    0x1f  /* All mouse events                 */
#endif

/* Extended event masks */

#define XEVN_LDRAG     0x0100  /* Dragged(moved) with left button pressed    */
#define XEVN_RDRAG     0x0200  /* Dragged(moved) with right button pressed   */
#define XEVN_LDBLCLICK 0x0400  /* Left button double-clicked                 */
#define XEVN_RDBLCLICK 0x0800  /* Right button double-clicked                */
#define XEVN_LRDOWN    0x1000  /* Left & right button pressed simutaneosly   */
#define XEVN_LCDOWN    0x2000  /* Left & center button pressed simutaneosly  */
#define XEVN_RCDOWN    0x4000  /* Right & center button pressed simutaneosly */

/* Button masks */

#ifndef BTN_NONE
    #define BTN_NONE 0x00  /* No button     */
    #define BTN_L    0x01  /* Left button   */
    #define BTN_R    0x02  /* Right button  */
    #define BTN_C    0x04  /* Center button */
    #define BTN_ALL  0x07  /* All buttons   */
#endif


/** Type definitions *******************************************************/

#ifndef MOUCURSOR_T
    #define MOUCURSOR_T
    typedef struct moucursor__t {
        int width, height;
        int hotspotx, hotspoty;
        void far *shape;
        void far *bitmask;
        int maxcolor;
        struct moucursor__t *nextcursor;
    } moucursor_t;
#endif

#ifndef MOUEVN_T
    #define MOUEVN_T
    typedef word mouevn_t;
    typedef word moubtnstat_t;
#endif

#ifndef __CLOCK_T
    #define __CLOCK_T
    typedef long clock_t;
    #define CLK_TCK 18.2
#endif

typedef struct {
    mouevn_t evn;
    moubtnstat_t btnstat;
    int apx, apy;
    clock_t clock;
} mouevnque_t;


/** Function declarations **************************************************/

#ifdef __cplusplus
extern "C" {
#endif

/* Event queue based mouse functions */

int meq_init(moucursor_t *moucursor);
int meq_close(void);

mouevnque_t *meq_getevn(void);
mouevnque_t *meq_popevn(void);

void meq_clrallevn(void);

mouevnque_t *meq_scanevn(int offset);
int meq_getuprlimit(void);
int meq_getlwrlimit(void);

void meq_skipevn(mouevn_t evn);
void meq_skiptoevn(mouevn_t evn);
int  meq_findevn(mouevn_t evn);
int  meq_findnextevn(int offset, mouevn_t evn);

int meq_genevn(const mouevnque_t *mouevn, bool resetclock);

bool meq_isempty(void);
int  meq_getevnnum(void);

#ifdef __cplusplus
}
#endif


/** Macro functions ********************************************************/

#define meq_isactive() _meqactive

#define meq_getcurevn() _meqcurevn

#define meq_isnoevn(mouevn)     ((mouevn)->evn == EVN_NONE)
#define meq_ismovevn(mouevn)    (((mouevn)->evn & EVN_MOVE)   != 0x00)
#define meq_isldownevn(mouevn)  (((mouevn)->evn & EVN_LDOWN)  != 0x00)
#define meq_islupevn(mouevn)    (((mouevn)->evn & EVN_LUP)    != 0x00)
#define meq_isrdownevn(mouevn)  (((mouevn)->evn & EVN_RDOWN)  != 0x00)
#define meq_isrupevn(mouevn)    (((mouevn)->evn & EVN_RUP)    != 0x00)
#define meq_islbtnevn(mouevn)   (((mouevn)->evn & EVN_LBTN)   != 0x00)
#define meq_isrbtnevn(mouevn)   (((mouevn)->evn & EVN_RBTN)   != 0x00)
#define meq_isdownevn(mouevn)   (((mouevn)->evn & EVN_DOWN)   != 0x00)
#define meq_isupevn(mouevn)     (((mouevn)->evn & EVN_UP)     != 0x00)
#define meq_isanybtnevn(mouevn) (((mouevn)->evn & EVN_ANYBTN) != 0x00)
#define meq_isanyevn(mouevn)    ((mouevn)->evn != EVN_NONE)

#define meq_isldragevn(mouevn)     (((mouevn)->evn & XEVN_LDRAG)     != 0x00)
#define meq_isrdragevn(mouevn)     (((mouevn)->evn & XEVN_RDRAG)     != 0x00)
#define meq_isldblclickevn(mouevn) (((mouevn)->evn & XEVN_LDBLCLICK) != 0x00)
#define meq_isrdblclickevn(mouevn) (((mouevn)->evn & XEVN_RDBLCLICK) != 0x00)
#define meq_islrdownevn(mouevn)    (((mouevn)->evn & XEVN_LRDOWN)    != 0x00)
#define meq_islcdownevn(mouevn)    (((mouevn)->evn & XEVN_LCDOWN)    != 0x00)
#define meq_isrcdownevn(mouevn)    (((mouevn)->evn & XEVN_RCDOWN)    != 0x00)

#define meq_haveanyevn(mouevn, event)     (((mouevn)->evn & (event))  != 0x00)
#define meq_haveotherevn(mouevn, event)   (((mouevn)->evn & ~(event)) != 0x00)
#define meq_havenoevn(mouevn, event)      (((mouevn)->evn & (event))  == 0x00)
#define meq_havenootherevn(mouevn, event) (((mouevn)->evn & ~(event)) == 0x00)

#define meq_getbtnstat(mouevn) ((mouevn)->btnstat)

#define meq_isldown(mouevn) (((mouevn)->btnstat & BTN_L) != 0x00)
#define meq_islup(mouevn)   (((mouevn)->btnstat & BTN_L) == 0x00)
#define meq_isrdown(mouevn) (((mouevn)->btnstat & BTN_R) != 0x00)
#define meq_isrup(mouevn)   (((mouevn)->btnstat & BTN_R) == 0x00)
#define meq_iscdown(mouevn) (((mouevn)->btnstat & BTN_C) != 0x00)
#define meq_iscup(mouevn)   (((mouevn)->btnstat & BTN_C) == 0x00)

#define meq_getpx(mouevn) apx2px((mouevn)->apx)
#define meq_getpy(mouevn) apy2py((mouevn)->apy)
#define meq_getx(mouevn)  apx2x((mouevn)->apx)
#define meq_gety(mouevn)  apy2y((mouevn)->apy)

#define meq_getapx(mouevn) ((mouevn)->apx)
#define meq_getapy(mouevn) ((mouevn)->apy)
#define meq_getax(mouevn)  apx2ax((mouevn)->apx)
#define meq_getay(mouevn)  apy2ay((mouevn)->apy)

#define meq_getmsec(mouevn)  ((mouevn)->clock * 10000L / 182L)
#define meq_gettime(mouevn)  ((mouevn)->clock / CLK_TCK)
#define meq_getclock(mouevn) ((mouevn)->clock)

#define meq_isoverflowed() (_meqoverflowcnt > 0)


/** External variables *****************************************************/

extern const bool _meqactive;  /* READ-ONLY! */

extern volatile mouevn_t _meqcurevn;  /* READ-ONLY! */

/* The length(number of elements) of the mouse event queue */
/* Must set before calling of meq_init                     */
/* The default is 32                                       */

extern size_t _meqlen;

/* Event queue options                                      */
/* The defaults are true, false, false, false, respectively */

extern bool _meqstoredownlbtnmovevn;
extern bool _meqstoreuplbtnmovevn;
extern bool _meqstoredownrbtnmovevn;
extern bool _meqstoreuprbtnmovevn;

/* A double click threshold, the unit is a clock tick */
/* The default is 9 clock tick == about 0.495 sec     */

extern unsigned _meqdblclickthres;

/* A hook vector, called when the mouse event queue is overflowed        */
/* If the return value is false, the very old event will be ignored,     */
/* otherwise, the occurred event will be not stored                      */
/* The default is NULL, in this case, the occurred event will be ignored */

extern bool (*_meqhookoverflow)(mouevn_t evn, moubtnstat_t btnstat,
                                int mouapx, int mouapy);

/* A current overflow counter of the mouse event queue */

extern volatile int _meqoverflowcnt;

/* A maximum value of the overflow counter of the mouse event queue */

extern volatile int _meqmaxoverflowcnt;

#endif
