/*
 *      File HANIN.H
 *      A Header for Hangul Input Library
 *      '90.8.10
 *      '91.1.25, 3.25-28, 4.7-8, 6.19-22, 8.15-18, 9.15, 10.17
 *      '92.1.6,12,18,26, 2.5, 4.25
 *      '93.3.25, 7.13, 8.7
 *      Written by Lim, I.K.
 */


#ifndef _HANIN_H_
#define _HANIN_H_

#ifndef _CHKVER_H_
    #include "chkver.h"
#endif

#ifndef _TYPEDEFS_H_
    #include "typedefs.h"
#endif


/** Macro constants ********************************************************/

/* Macro constants for iswhat */

#define IS_CTRL    0x01
#define IS_KOREAN  0x02
#define IS_ALPHA   0x04
#define IS_DIGIT   0x08
#define IS_SPECIAL 0x10
#define IS_SIGN    0x20
#define IS_SPACE   0x40
#define IS_USER    0x80

/* Cursor modes */

#define CM_BUFFERED    0
#define CM_DIRECTUNDER 1
#define CM_DIRECTXOR   2  /* Not supported */

/* Macros for hsetbufcursor */

#define BOLDCURSOR    hsetbold
#define OUTLINECURSOR hsetoutline
#define THREEDCURSOR  hsetthreed
#define FAINTCURSOR   hsetfaint
#define SHADOWCURSOR  hsetshadow
#define UNDERCURSOR   hsetunder
#define FRAMECURSOR   hsetframe
#define BLOCKCURSOR   hsetblock
#define REVERSECURSOR hsetreverse

#define DEFAULTCURSOR REVERSECURSOR

#define DEFAULTCOLOR (-1)

#define MAXLONG 0x7fffffffL
#define MINLONG ((long)0x80000000L)

#define MAXHANKEY 5

#define HANKEY_NEW 0
#define HANKEY_ADD 1

#ifndef NUL
    #define NUL '\x00'
#endif


/** Type definitions *******************************************************/

#ifndef ONOFF_T
    #define ONOFF_T
    typedef enum {OFF, ON} onoff_t;
#endif

enum shiftstatus_t {
    NO_SHIFT    = 0x00,
    RIGHT_SHIFT = 0x01,
    LEFT_SHIFT  = 0x02,
    SHIFT       = 0x03,
    CTRL        = 0x04,
    ALT         = 0x08,
    SCROLL_LOCK = 0x10,
    NUM_LOCK    = 0x20,
    CAPS_LOCK   = 0x40,
    INSERT      = 0x80,
    SHIFTSTATUS = 0xff,
};

typedef void (*cursor_t)(bool flag);

struct key_t {      /* Refer to _key              */
    int enter;      /* The default is CR          */
    int cancel;     /* The default is ESC         */
    int leftchar;   /* The default is LEFTARROW   */
    int rightchar;  /* The default is RIGHTARROW  */
    int leftword;   /* The default is CTRL_LEFT   */
    int rightword;  /* The default is CTRL_RIGHT  */
    int linebegin;  /* The default is HOMEKEY     */
    int lineend;    /* The default is ENDKEY      */
    int delleft;    /* The default is BS          */
    int delchar;    /* The default is DEL         */
    int delword;    /* The default is CTRL_T      */
    int delafter;   /* The default is not defined */
    int delline;    /* The default is CTRL_Y      */
};

struct tpadkey_t {
    int quit;      /* The default is ALT_X      */
    int upline;    /* The default is UPARROW    */
    int downline;  /* The default is DOWNARROW  */
    int padhome;   /* The default is CTRL_HOME  */
    int padend;    /* The default is CTRL_END   */
};

typedef struct {
    byte *buffer;
    bool refresh;
    int x1, y1;
    int width, height;
    int curx, cury;
} tpadinfo_t;

enum charkind_t {
    ASCII, HANGUL1ST, HANGUL2ND, HANCOMBI1ST, HANCOMBI, HANNULL
};

struct hankey_t {
    int key;
    word status;
};

enum hkbhitmode_t {
    HKBHIT_DEFAULT, HKBHIT_BIOSKEY, HKBHIT_UNGETXCH,
    HKBHIT_GETXCH, HKBHIT_HGETCH, HKBHIT_ALL
};


/** Function declarations **************************************************/

#ifdef __cplusplus
extern "C" {
#endif

/* Low level character input */

int getxch(void);
int getkey(void);
int ungetxch(int c);

bool keypressed(void);
bool shiftstatus(int shiftkey);
int  iswhat(int keycode, const char *user);

/* Hangul-English toggle key */

bool checkhankey(int key, word status);
int  hsethankey(int key, word status, int whence);

/* High level character input */

bool hkbhit(int mode);
int  hgetch(void);
int  hgetxch(void);
void hclearkeybuf(void);

int  hwaitkey(int x, int y, const byte *s, int curp);

/* Line input */

byte *hgets(byte *string);
byte *hgetls(byte *string);
int  hgetln(byte *string, size_t n);
int  hgetline(byte *string, size_t n);
int  hgetlongline(byte *string, size_t maxlen, size_t fieldlen);
int  hscanf(const byte *format, ...);

/* Number input */

long   hgetlval(byte *string, size_t n, const byte *prefix);
double hgetdval(byte *string, size_t n, const byte *prefix);

/* Line input for database */

int hgetdata(byte *string, const byte *picture);
int hgetnumdata(char *string, const char *picture);
int hchecknumpicture(const char *string, const char *picture);

/* Screen input (tinypad) */

int  htinypad(tpadinfo_t *tpadinfo);
int  hgettinypadsize(int x1, int y1, int x2, int y2);
int  hsettinypad(tpadinfo_t *tpadinfo, byte *buffer, bool init, bool refresh,
                 int x1, int y1, int x2, int y2);
byte *hgettpadln(tpadinfo_t *tpadinfo, int padln);

/* String manipulation */

char *strnins(char *dest, int pos, const char *src, size_t n);
char *strins(char *dest, int pos, const char *src);
char *strndel(char *s, int pos, size_t n);
char *strcuttail(char *s);
char *strskip(const char *str, const char *substr);

int    hstrnumtopic(char *dest, const char *src, const char *picture);
long   hstrtol(char *s, char **endptr, int radix, const char *prefix);
double hstrtod(char *s, char **endptr, const char *prefix);

bool strhins(byte *s, int pos, size_t n);
bool streins(byte *s, int pos, size_t n);
bool strhdel(byte *s, int pos);

/* Hangul code checking */

bool ishangul1st(byte *s, int pos);
bool ishangul2nd(byte *s, int pos);

/* Cursor control */

int hsetcursormode(int cursormode);
int hgetcursormode(void);

void hsetbufcursor(cursor_t normalcursor, cursor_t blankcursor, int color);
void hsetcursorbuf(const byte *cursorbuf);

void hsetundercursor(int start, int end, int color);

/* Miscellaneous */

int alpha2han(int c);

void hsetbold(bool flag);
void hsetoutline(bool flag);
void hsetthreed(bool flag);
void hsetfaint(bool flag);
void hsetshadow(bool flag);
void hsetunder(bool flag);
void hsetframe(bool flag);
void hsetblock(bool flag);
void hsetreverse(bool flag);

int hputch(int c);

#ifdef __cplusplus
}
#endif


/** Macro functions ********************************************************/

/* Handle low level key input */

#define getchext()    getxch()
#define ungetchext(c) ungetxch(c)
#define pause()       ((void)getxch())

/* Get a character with echo */

#define hgetche() hputch(hgetch())

/* Get a character after clearing internal buffers */

#define hgetch1() ((hputch(-1), hclearkeybuf(), hgetch()))
#define hgetche1() hputch((hputch(-1), hclearkeybuf(), hgetch()))

/* Set the Hangul keyboard */

#define hsethan2board(split) (                     \
    _alpha2hantbl = _alpha2hantbl2,                \
    _initial2final = true,                         \
    _issplitmode = (split),                        \
    _splitinitialdouble = (split) ? true  : false, \
    _splitmedialcomplex = (split) ? true  : false, \
    _splitfinaldouble   = (split) ? false : false, \
    _splitfinalcomplex  = (split) ? true  : true   \
)

#define hsethan389board(split) (                  \
    _alpha2hantbl = _alpha2hantbl389,             \
    _initial2final = false,                       \
    _issplitmode = (split),                       \
    _splitinitialdouble = (split) ? true : true,  \
    _splitmedialcomplex = (split) ? true : false, \
    _splitfinaldouble   = (split) ? true : false, \
    _splitfinalcomplex  = (split) ? true : false  \
)

#define hsethan390board(split) (                  \
    _alpha2hantbl = _alpha2hantbl390,             \
    _initial2final = false,                       \
    _issplitmode = (split),                       \
    _splitinitialdouble = (split) ? true : true,  \
    _splitmedialcomplex = (split) ? true : false, \
    _splitfinaldouble   = (split) ? true : false, \
    _splitfinalcomplex  = (split) ? true : false  \
)

#define ishan2board()   (_alpha2hantbl == _alpha2hantbl2)
#define ishan389board() (_alpha2hantbl == _alpha2hantbl389)
#define ishan390board() (_alpha2hantbl == _alpha2hantbl390)
#define issplitmode()   _issplitmode


/** External variables *****************************************************/

/* Input modes for all Hangul input functions */

extern bool _hangulmode;  /* The default is true */
extern bool _insertmode;  /* The default is true */

/* Options of hget... functions whice have editing keys */

extern bool _allowfirstdelline;   /* The default is true */
extern bool _allowcanceldelline;  /* The default is true */
extern bool _restoreline;         /* The default is true */
extern bool _confirm;             /* The default is true - not supported now */

/* Options of hgetlongline                                */
/* The defaults of _leftmark and _rightmark are           */
/* '\x11' and '\x10' respectively                         */
/* The defaults of _markcolor and _markbkcolor are all -1 */
/* ie. the current color and the current background color */

extern byte _leftmark, _rightmark;
extern int _markcolor, _markbkcolor;

/* An option of hgetdata and hgetnumdata functions            */
/* If true, trailing blanks will be truncated when returned   */
/* otherwise, will be left                                    */
/* For hgetnumdata function, if _numdataleftjustify == false, */
/* this option will have no effect                            */
/* The default is true                                        */

extern bool _cuttail;

/* An option of hgetnumdata function                                       */
/* If false, number data will be right-justified when stored to the buffer */
/* otherwise will be left-justified                                        */
/* The default is false                                                    */

extern bool _numdataleftjustify;

/* An option of hgetnumdata, hputnumdata and hstrnumtopic functions */
/* If true, blanks after point '.' will be filled with zero '0'     */
/* otherwise will be left                                           */
/* The default is true                                              */

extern bool _filltailblankzero;

/* An option of hgetnumdata, hputnumdata and hcheckpicture functions    */
/* If false, the parameter picture will not be checked anyway           */
/* Otherwise picture will fully be checked for safety                   */
/* Normally, if picture is a (valid) string constant                    */
/* this option may be set false to speed up, especially for hputnumdata */
/* The default is true                                                  */

extern bool _checknumpicture;

/* Options of hgetlval and hgetdval functions */

extern bool _ignoreafterlval;  /* The default is false */
extern char _lvalpicture;      /* The default is 'X'   */
extern bool _ignoreafterdval;  /* The default is false */
extern char _dvalpicture;      /* The default is 'X'   */

/* Set by the hgetlval and hgetdval functions */

extern byte *_endptr;

/* The base radix for hgetlval function        */
/* The valid range of _inputradix is 2 to 36   */
/* If _inputradix is 0,                        */
/* the base radix is automatically determined  */
/* by the prefix of the input number           */
/* Refer to strtol function                    */
/* The default is 10                           */

extern int _inputradix;

/* The last key input during hgetline and hgetdata           */
/* Normally CR or ESC, if the vector _hookhget... not hooked */

extern int _lastkey;

/* The final key input during htinypad (like _lastkey)        */
/* Normally CR or ESC, if the vector _hookhtinypad not hooked */

extern int _finalkey;

/* The kind of the character hgetch/hgetxch returned             */
/* ASCII, HANGUL1ST, HANGUL2ND, HANCOMBI1ST, HANCOMBI or HANNULL */

extern enum charkind_t _charkind;

/* The cursor delay time                                      */
/* The default is 6, about 0.33 sec (exactly 6 / CLK_TCK sec) */

extern int _cursordelaytime;

/* If false, the cursor will be not shown */
/* The default is true                    */

extern bool _showcursor;

/* The cursor color            */
/* The default is DEFAULTCOLOR */

extern int _cursorcolor;

/* Cursor control */

extern void (*hremovecursor)(void);   /* Call ONLY in the function */
extern void (*hrestartcursor)(void);  /* pointed by _hookhwaitkey  */

/* Define codes of all editing keys used by Hangul input functions */

extern struct key_t _key;
extern struct tpadkey_t _tpadkey;

/* Hook vectors called by hget... functions */

extern void (*_hookhangulmodetoggle)(bool hangulmode);
extern void (*_hookinsertmodetoggle)(bool insertmode);
extern void (*_hookmistyping)(void);

extern void (*_hookhwaitkey)(void);
extern void (*_hookhgets)(int *keycodep);
extern void (*_hookhgetls)(int *keycodep);
extern void (*_hookhgetln)(int *keycodep);
extern void (*_hookhtinypad)(int *keycodep, tpadinfo_t *tpadinfo);

#ifndef __cplusplus
    extern bool (*_hookhgetline)(/*int *keycodep, byte *string,
                                 int *curpp, int *lastpp, size_t n*/);
    extern bool (*_hookhgetdata)(/*int *keycodep, byte *string,
                                 int *curpp, int *lastpp, size_t n*/);
    extern bool (*_hooktemplate)(byte c, int *keycodep);
    extern bool (*_hookhgetnumdata)(/*int *keycodep, char *string,
                                    int *curpp, int *lastpp, size_t n*/);
#else
    extern bool (*_hookhgetline)(int *keycodep, byte *string,
                                 int *curpp, int *lastpp, size_t n);
    extern bool (*_hookhgetdata)(int *keycodep, byte *string,
                                 int *curpp, int *lastpp, size_t n);
    extern bool (*_hooktemplate)(byte c, int *keycodep);
    extern bool (*_hookhgetnumdata)(int *keycodep, char *string,
                                    int *curpp, int *lastpp, size_t n);
#endif

/* Allow/prohibit mistypings */
/* The default is true       */

extern bool _allowmistyping;

/* Options to control the Hangul automata, set by sethan...board functions */

extern bool _issplitmode;         /* The default is true */
extern bool _splitinitialdouble;  /* The default is true */
extern bool _splitmedialcomplex;  /* The default is true */
extern bool _splitfinaldouble;    /* The default is true */
extern bool _splitfinalcomplex;   /* The default is true */
/* Enable/disable split-write mode by phoneme               */
/* If 2-set type keyboard, _splitfinalcomplex must be true  */
/* If 3-set type keyboard, _splitinitialdouble must be true */

extern bool _initial2final;
/* Control initial-to-final transition     */
/* If 2-set type keyboard, it must be true */
/* The default is true                     */

/* A table to convert internal Hangul indexes to Hangul code indexes */
/* The default is a pointer to the common combined Hangul code table */

extern const byte (*_index2code)[27 + 1];

/* Pointers to tables to convert alphabets to Hangul indexes        */
/* The default is _alpha2hantbl2 for the 2-set type Hangul keyboard */

extern const byte *_alpha2hantbl;

extern const byte _alpha2hantbl2[0x60];
extern const byte _alpha2hantbl389[0x60];  /* IBM-3-89 */
extern const byte _alpha2hantbl390[0x60];  /* IBM-3-90 */

/* A table of Hangul-English toggle keys */
/* Maximum number of keys is MAXHANKEY   */

extern struct hankey_t _hankey[];

#endif
