/*
 *      File TEXTTERM.C
 *      Simple Text Terminal Emulator
 *      '91.10.19-20
 *      Written by Lim, I.K.
 */


#include <bios.h>
#include <conio.h>
#include <dos.h>
#include <stdio.h>


typedef enum {false, true} bool;


#define BUFSIZE 0x4000  /* Size of comm buffer */

#define CHARDELAY 50  /* Character delay time */

/* RS-232C serial ports */

#define COM1 0
#define COM2 1

#define COMPORT COM2  /* Alterable by user */

/* Base addresses of serial ports */

#define COM1BASE 0x03f8
#define COM2BASE 0x02f8

#define COMBASE (COMPORT ? COM2BASE : COM1BASE)

/* Registers */

#define THR (COMBASE + 0)  /* Transmit Holding Register         */
#define RBR (COMBASE + 0)  /* Receive Buffer Register           */
#define IER (COMBASE + 1)  /* Interrupt Enable Register         */
#define IIR (COMBASE + 2)  /* Interrupt Identification Register */
#define LCR (COMBASE + 3)  /* Line Control Register             */
#define MCR (COMBASE + 4)  /* Modem Control Register            */
#define LSR (COMBASE + 5)  /* Line Status Register              */
#define MSR (COMBASE + 6)  /* Modem Status Register             */

/* Parameters to bioscom function */

#define DATABIT7   0x02  /* Data bit */
#define DATABIT8   0x03

#define STOPBIT1   0x00  /* Stop bit */
#define STOPBIT2   0x04

#define NOPARITY   0x00  /* Parity */
#define ODDPARITY  0x08
#define EVENPARITY 0x18

#define BAUD1200   0x80   /* Baud rate */
#define BAUD2400   0xa0
#define BAUD4800   0xc0
#define BAUD9600   0xe0

/* 8259 PIC(Programmable Interrupt Controller) */

#define IMR  0x21  /* I/O address of OCW1(IMR) of 8259 PIC  */
#define OCW2 0x20  /* I/O address of OCW2 of 8259 PIC       */
                   /* OCW: Operation Command Word           */
                   /* IMR: Interrupt Mask Register          */

#define MASKON  0xe7  /* Mask IRQ3/IRQ4 on  -> IMR */
#define MASKOFF 0x18  /* Mask IRQ3/IRQ4 off -> IMR */

#define EOI 0x20  /* Non-specific End of Interrupt command -> OCW2 */

/* Interrupt Request Numbers */

#define IRQ3 0x0b  /* of COM2 */
#define IRQ4 0x0c  /* of COM1 */

#define IRQNUM (COMPORT ? IRQ3 : IRQ4)


char commbuf[BUFSIZE];  /* Comm buffer */
int bufptr0 = 0, bufptr1 = 0;

void interrupt (*oldvect)();  /* Old interrupt vector */


void interrupt commisr()
{
    bufptr1 %= BUFSIZE;
    commbuf[bufptr1++] = inportb(RBR);
    outportb(OCW2, EOI);
}

void initcomm(void)
{
    bioscom(0, DATABIT8 | STOPBIT1 | NOPARITY | BAUD2400, COMPORT);

    outportb(MCR, 0x0b);
    outportb(IER, 0x01);

    oldvect = getvect(IRQNUM);
    setvect(IRQNUM, commisr);

    outportb(IMR, inportb(IMR) & MASKON);
}

void closecomm(void)
{
    outportb(MCR, 0x00);
    outportb(IER, 0x00);

    outportb(IMR, inportb(IMR) | MASKOFF);
    setvect(IRQNUM, oldvect);
}

int c_getc(void)
{
    if (bufptr0 == bufptr1) return (-1);
    bufptr0 %= BUFSIZE;
    return commbuf[bufptr0++];
}

void c_putc(int c)
{
    while (!(inportb(LSR) & 0x20)) ;
    outportb(THR, (char)c);
}

void c_puts(char *s)
{
    while (*s) c_putc(*s++);
}

void main(void)
{
    char c;

    printf("Simple Text Terminal Emulator(<ESC> to quit)\n");
    printf("COM%d-2400N81\n\n", COMPORT + 1);

    initcomm();

    c_puts("ATV1X4\r");

    for (; ; ) {
        if ((c = c_getc()) != (-1)) putch(c);
        if (kbhit())
            if ((c = getch()) == '\x1b') break;
            else c_putc(c);
    }

    closecomm();
}
