/*
 *      File HANVT100.C
 *      Simple Hangul VT-100 Terminal Emulator
 *      '91.10.20
 *      '92.5.2-6, 6.14
 *      '93.7.13, 10.17,21
 *      Written by Lim, I.K.
 */


#include <bios.h>
#include <ctype.h>
#include <dos.h>
#include <mem.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>

#include "ascii.h"
#include "extkey.h"
#include "hanin.h"
#include "hanlib.h"
#include "hanmeq.h"
#include "hanmou.h"
#include "hanwcord.h"
#include "hanwout.h"


/** Macro constants ********************************************************/

/* RS-232C serial ports */

#define COM1 0
#define COM2 1

/* Base addresses of serial ports */

#define COM1BASE 0x03f8
#define COM2BASE 0x02f8

#define COMBASE ((comport == COM1) ? COM1BASE : COM2BASE)

/* Registers */

#define THR (COMBASE + 0)  /* Transmit Holding Register         */
#define RBR (COMBASE + 0)  /* Receive Buffer Register           */
#define IER (COMBASE + 1)  /* Interrupt Enable Register         */
#define IIR (COMBASE + 2)  /* Interrupt Identification Register */
#define LCR (COMBASE + 3)  /* Line Control Register             */
#define MCR (COMBASE + 4)  /* Modem Control Register            */
#define LSR (COMBASE + 5)  /* Line Status Register              */
#define MSR (COMBASE + 6)  /* Modem Status Register             */

/* Parameters to bioscom function */

#define DATABIT7   0x02  /* Data bit */
#define DATABIT8   0x03

#define STOPBIT1   0x00  /* Stop bit */
#define STOPBIT2   0x04

#define NOPARITY   0x00  /* Parity */
#define ODDPARITY  0x08
#define EVENPARITY 0x18

#define BAUD1200   0x80   /* Baud rate */
#define BAUD2400   0xa0
#define BAUD4800   0xc0
#define BAUD9600   0xe0

/* 8259 PIC(Programmable Interrupt Controller) */

#define IMR  0x21  /* I/O address of OCW1(IMR) of 8259 PIC  */
#define OCW2 0x20  /* I/O address of OCW2 of 8259 PIC       */
                   /* OCW: Operation Command Word           */
                   /* IMR: Interrupt Mask Register          */

#define MASKON  0xe7  /* Mask IRQ3/IRQ4 on  -> IMR */
#define MASKOFF 0x18  /* Mask IRQ3/IRQ4 off -> IMR */

#define EOI 0x20  /* Non-specific End of Interrupt command -> OCW2 */

/* Interrupt Request Numbers */

#define IRQ3 0x0b  /* of COM2 */
#define IRQ4 0x0c  /* of COM1 */

#define IRQNUM ((comport == COM1) ? IRQ4 : IRQ3)

/* Miscellaneos */

#define BUFSIZE 0x4000  /* Size of comm buffer */

#define BUFFEREMPTY (-1)  /* Buffer empty */

#define MOUKEY 0x7fff  /* Virtual key code to indicate mouse button pressed */

/* Attribute-independent output windows */

#define SCRBUFWIN (&scrbufwin)
#define CODEWIN   (&codewin)
#define TEXTWIN   (&textwin)
#define STATWIN   (&statwin)


/** External variables *****************************************************/

int comport = COM2;  /* Default COM port, alterable by user */

void interrupt (*oldvect)();   /* Old interrupt vector */
byte commbuf[BUFSIZE];         /* Comm buffer */
int bufptr0 = 0, bufptr1 = 0;  /* Pointers to comm buffer */

window_t scrbufwin, codewin, textwin, statwin;  /* Output windows */

byte scrbuf[30 + 1][80 + 1];  /* Screen buffer for 80x30 size */
int escstep = 0;
int moupx, moupy;

bool hgconvert = true;
bool debugmode = false;

int htrans(byte s, byte d, byte *hg);  /* Hangul code converter */


/** Communication subroutines **********************************************/

void interrupt com_isr()
{
    bufptr1 %= BUFSIZE;
    commbuf[bufptr1++] = inportb(RBR);
    outportb(OCW2, EOI);
}

void com_init(int comport)
{
    bioscom(0, DATABIT8 | STOPBIT1 | NOPARITY | BAUD2400, comport);

    outportb(MCR, 0x0b);
    outportb(IER, 0x01);

    oldvect = getvect(IRQNUM);
    setvect(IRQNUM, com_isr);

    outportb(IMR, inportb(IMR) & MASKON);
}

void com_close(void)
{
    outportb(MCR, 0x00);
    outportb(IER, 0x00);

    outportb(IMR, inportb(IMR) | MASKOFF);
    setvect(IRQNUM, oldvect);
}

int com_getc(void)
{
    if (bufptr0 == bufptr1) return BUFFEREMPTY;
    bufptr0 %= BUFSIZE;

    return commbuf[bufptr0++];
}

void com_putc(int c)
{
    while (!(inportb(LSR) & 0x20)) ;
    outportb(THR, (byte)c);
}

void com_puts(char *s)
{
    while (*s) com_putc(*s++);
}

int com_printf(const byte *format, ...)
{
    byte buffer[1024];
    int cnt;
    va_list arglist;

    va_start(arglist, format);
    cnt = vsprintf(buffer, format, arglist);
    com_puts(buffer);
    va_end(arglist);

    return cnt;
}

/** Subroutines ************************************************************/

void hookhangulmodetoggle(bool hangulmode)
{
    if (hangulmode) {
        hsetundercursor(-1, -1, (hgetmaxcolor() == 1) ? HWHITE : HYELLOW);
        wputsxy(STATWIN, 2, 1, "ei");
    } else {
        hsetundercursor(-1, -1, (hgetmaxcolor() == 1) ? HWHITE : HLIGHTCYAN);
        wputsxy(STATWIN, 2, 1, "w");
    }
}

void hookputch(window_t *win, byte c, int apx, int apy)
{
    /* To update screen buffer only when the default output window */
    if (win != DEFWIN) return;

    scrbuf[apy2y(apy)][px2x(apx)] = c;
}

void hookscroll(wincoord_t *wc, window_t *win)
{
    /* To scroll only the default output window */
    if (wc != (wincoord_t *)win) return;
    if (win != DEFWIN) return;

    memcpy(scrbuf[0], scrbuf[1], sizeof(scrbuf) - sizeof(scrbuf[0]));
    memset(scrbuf[wgetmaxy(win)], ' ', sizeof(scrbuf[0]));
}

void execute_vt100(int cmd, int param1, int param2)
{
    int x, y, attr;

    switch (cmd) {
    case 'A':
        hgotoxy(hwherex(), hwherey() - 1);
        break;
    case 'B':
        hgotoxy(hwherex(), hwherey() + 1);
        break;
    case 'C':
        hgotoxy(hwherex() + 1, hwherey());
        break;
    case 'D':
        hgotoxy(hwherex() - 1, hwherey());
        break;
    case 'H':
        x = (param2 < 0) ? 1 : param2;
        y = (param1 < 0) ? 1 : param1;
        hgotoxy(x, y);
        break;
    case 'J':
        switch (param1) {
        case 2:
            hclrscr();
            memset(scrbuf, ' ', sizeof(scrbuf));
            break;
        }
        break;
    case 'm':
        attr = (param1 < 0) ? 0 : param1;
        switch (attr) {
        default:
        case 0:
            hnormattr();
            break;
        case 1:
            hsetbold(true);
            break;
        case 4:
            hsetunder(true);
            break;
        case 7:
            hsetreverse(true);
            break;
        }
    }
}

void process_vt100(int rc)
{
    static char param1[3 + 1], param2[3 + 1];
    static int param1ptr = 0, param2ptr = 0;

    switch (escstep) {
    case 0:
        if (rc == ESC) escstep++;
        break;
    case 1:
        if (rc == '[') escstep++;
        else escstep = 0;
        param2ptr = param1ptr = 0;
        break;
    case 2:
        if (isdigit(rc)) {
            if (param1ptr < 3) param1[param1ptr++] = rc;
            else escstep = 0;
            break;
        }
        escstep++;
        param1[param1ptr] = '\0';
    case 3:
        if (rc == ';') {
            escstep++;
            break;
        }
        escstep++;
    case 4:
        if (isdigit(rc)) {
            if (param2ptr < 3) param2[param2ptr++] = rc;
            else escstep = 0;
            break;
        }
        escstep++;
        param2[param2ptr] = '\0';
    case 5:
        escstep = 0;
        if (!isalpha(rc)) break;
        execute_vt100(rc, (param1ptr == 0) ? -1 : atoi(param1),
                      (param2ptr == 0) ? -1 : atoi(param2));
    }
}

void config_screen(bool debugmode)
{
    int x;

    /* Set output windows and a mouse window */

    hallowautoscroll(true);

    if (debugmode) {
        hwindow(1, 1, 80, hgetmaxay() - 6);
        mou_setwindow(1, 1, 80, hgetmaxay() - 6);
        hsetwindow(SCRBUFWIN,  1, hgetmaxay() - 5, 80, hgetmaxay() - 5,
                   HWHITE, HBLACK, DEFWIN);
        wclrscr(SCRBUFWIN);
        wallowautoscroll(SCRBUFWIN, false);
        hsetwindow(CODEWIN, 1, hgetmaxay() - 4, 80, hgetmaxay() - 3,
                   HWHITE, HDARKGRAY, DEFWIN);
        wclrscr(CODEWIN);
        hsetwindow(TEXTWIN, 1, hgetmaxay() - 2, 80, hgetmaxay() - 1,
                   HWHITE, HLIGHTGRAY, DEFWIN);
        wclrscr(TEXTWIN);
    } else {
        hwindow(1, 1, 80, hgetmaxay() - 1);
        mou_setwindow(1, 1, 80, hgetmaxay() - 1);
    }
    hsetwindow(STATWIN, 1, hgetmaxay(), 80, hgetmaxay(), HMAGENTA, HWHITE,
               DEFWIN);
    wsetreverse(STATWIN, true);
    wregisterhanfont(STATWIN, HAN8GD1);
    wallowautoscroll(STATWIN, false);
    for (x = 1; x <= hgetmaxx(); x++) wputch(STATWIN, ' ');

    hsetbkcolor(HBLUE);
    hclrscr();
    memset(scrbuf, ' ', sizeof(scrbuf));

    /* Print status */

    hookhangulmodetoggle(_hangulmode);
    wprintfxy(STATWIN, 8, 1, hgconvert ? "Ŭw" : "sw");
    wprintfxy(STATWIN, 16, 1, "COM%d-2400N81", comport + 1);
    wprintfxy(STATWIN, 30, 1, "BS %s", _clearblank ? "ON " : "OFF");
    wprintfxy(STATWIN, 71, 1, "DEBUG %s", debugmode ? "ON " : "OFF");
}

void debug_code(int rc)
{
    /* Window for screen buffer */

    _noctrlcode = true;
    wputsnxy(SCRBUFWIN, 1, 1, 80, &scrbuf[hwherey()][1]);
    _noctrlcode = false;

    /* Window for hex codes */

    wprintf(CODEWIN, "%02X ", rc);

    /* Window for character codes */

    _noctrlcode = true;
    weputch(TEXTWIN, rc);
    _noctrlcode = false;
}

void process_ctrlkey(int c)
{
    switch (c) {
    case ALT_B:
        _clearblank = !_clearblank;
        wprintfxy(STATWIN, 30, 1, "BS %s", _clearblank ? "ON " : "OFF");
        break;
    case ALT_C:
        comport = (comport == COM1) ? COM2 : COM1;
        com_close();
        com_init(comport);
        com_puts("ATV1X4\r");
        wprintfxy(STATWIN, 16, 1, "COM%d", comport + 1);
        break;
    case ALT_K:
        hgconvert = !hgconvert;
        wprintfxy(STATWIN, 8, 1, hgconvert ? "Ŭw" : "sw");
        break;
    case SHIFT_TAB:
        debugmode = !debugmode;
        config_screen(debugmode);
        break;
    case UPARROW:
        com_puts("\x1b[A");
        break;
    case DOWNARROW:
        com_puts("\x1b[B");
        break;
    case RIGHTARROW:
        com_puts("\x1b[C");
        break;
    case LEFTARROW:
        com_puts("\x1b[D");
        break;
    case MOUKEY:
        com_printf("\x1b[%d;%dH", py2y(moupy), px2x(moupx));
        break;
    }
}

void hookhwaitkey(void)
{
    static bool han1st = false;
    static byte hg[3];
    int rc;
    mouevnque_t *mouevn;

    while ((rc = com_getc()) != BUFFEREMPTY)
        if (han1st) {  /* If 2nd byte of Hangul code */
            hg[1] = rc;
            han1st = false;

            if (hgconvert) htrans('F', '1', hg);  /* Convert Hangul code */

            hremovecursor();
            if (hwherex() == hgetmaxx()) hputch(' ');
            hputch(hg[0]);
            hputch(hg[1]);
            hrestartcursor();

            if (debugmode) debug_code(hg[0]);
            if (debugmode) debug_code(hg[1]);
        } else if (rc >= 0x80) {  /* If 1st byte of Hangul code */
            hg[0] = rc;
            han1st = true;
        } else {
            hremovecursor();
            if (rc == BS && hwherex() == 1) ;  /* For line break */
            else if (rc == ESC || escstep > 0) process_vt100(rc);
            else hputch(rc);
            hrestartcursor();

            if (debugmode) debug_code(rc);
        }

    if (meq_findevn(EVN_LDOWN) >= 0) {
        meq_skiptoevn(EVN_LDOWN);
        mouevn = meq_getevn();
        if (meq_isldownevn(mouevn)) {
            moupx = meq_getpx(mouevn);
            moupy = meq_getpy(mouevn);
            ungetxch(MOUKEY);
        }
    }
}

/** Main routine ***********************************************************/

void main_loop(void)
{
    static char hg[3];
    int c;

    hclearkeybuf();  /* Clear internal key buffer of hgetch if any */

    while ((c = hgetch()) != ALT_X)
        if (c >= 0x100) process_ctrlkey(c);  /* For control key */
        else if (c >= 0x80) {  /* For Hangul */
            hg[0] = c;
            hg[1] = hgetch();
            if (hgconvert) htrans('1', 'F', hg);  /* Convert Hangul code */
            com_puts(hg);
        } else if (c == BS && hwherex() == 1) ;  /* For line break */
        else {
            com_putc(c);  /* For ASCII */
            if (c == BS && ishangul1st(&scrbuf[hwherey()][0], hwherex() - 2))
                com_putc(BS);  /* For backspace */
        }
}

int main(void)
{
    int i;

    /* Initialize the library, Standard Mode, Autodetecting */

    inithanlib(STDMODE, HANDETECT, DEFHANFONT, DEFENGFONT);
    if (hgraphresult() < NOERROR)
        puterr("This program does not run in MDA or CGA!", 1);

    /* Register and load fonts */

    registeregrfont(EGR1);
    registerkssfont(KSS1);

    if (loadhanjafont("hanja.fnt") < NOERROR) {
        hprintf("ea aw sa.\n");
        hprintf("e BɡA hanja.fnta ᴡ sa.\n");
        hprintf("a iAa a.\n");
        pause();
    }

    /* Color mapping */

    for (i = 0; i <= MAXCOLORS; i++)
        hsetcolormap(i, (hgetmaxcolor() == MAXCOLORS) ? i : i / 15);

    /* Initialize the Mouse Event Queue */

    meq_init(MOU_ARROW);

    /* Set options */

    _hangulmode = false;
    /* English mode at start */
    _islinefeednewline = false;
    /* LF is only LF, not converted to CR/LF pair */
    _clearblank = false;
    /* Let Ctrl-H(BS) not delete left char, VT100 default */

    /* Set hook vectors */

    _hookhangulmodetoggle = hookhangulmodetoggle;
    _hookhwaitkey = hookhwaitkey;

    _hookputch = hookputch;    /* Used to update screen buffer */
    _hookscroll = hookscroll;

    /* Set the cursor mode to 'direct underline' and its color to yellow */

    hsetcursormode(CM_DIRECTUNDER);
    hsetundercursor(-1, -1, HYELLOW);

    /* Configure screen */

    config_screen(debugmode);

    /* Show the mouse cursor */

    mou_show();

    /* Print title and key usage */

    registerhanfont(HAN8GD1);
    hprintf(">>> e ei VT-100 ei AIA <<<\r\n\n");
    hprintf("ia iA  aa e:  \r\n");
    hprintf("<Alt-X>: a\r\n");
    hprintf("<Alt-C>: ɷ ͡a (COM1/COM2)\r\n");
    hprintf("<Alt-K>: ei ša (sw/Ŭw)\r\n");
    hprintf("<Shift-Tab>: a a \r\n\n");
    registerhanfont(DEFHANFONT);

    /* Initialize the serial port and the modem */

    com_init(comport);
    com_puts("ATV1X4\r");

    /* Call the main loop */

    main_loop();

    /* Close the serial port */

    com_close();

    return EXIT_SUCCESS;
}
