/*
 *      File HANTERM.C
 *      Simple Hangul Terminal Emulator
 *      '93.7.17
 *      Written by Lim, I.K.
 */


#include <bios.h>
#include <dos.h>
#include <stdlib.h>

#include "ascii.h"
#include "hanin.h"
#include "hanlib.h"


/* RS-232C serial ports */

#define COM1 0
#define COM2 1

/* Base addresses of serial ports */

#define COM1BASE 0x03f8
#define COM2BASE 0x02f8

#define COMBASE ((comport == COM1) ? COM1BASE : COM2BASE)

/* Registers */

#define THR (COMBASE + 0)  /* Transmit Holding Register         */
#define RBR (COMBASE + 0)  /* Receive Buffer Register           */
#define IER (COMBASE + 1)  /* Interrupt Enable Register         */
#define IIR (COMBASE + 2)  /* Interrupt Identification Register */
#define LCR (COMBASE + 3)  /* Line Control Register             */
#define MCR (COMBASE + 4)  /* Modem Control Register            */
#define LSR (COMBASE + 5)  /* Line Status Register              */
#define MSR (COMBASE + 6)  /* Modem Status Register             */

/* Parameters to bioscom function */

#define DATABIT7   0x02  /* Data bit */
#define DATABIT8   0x03

#define STOPBIT1   0x00  /* Stop bit */
#define STOPBIT2   0x04

#define NOPARITY   0x00  /* Parity */
#define ODDPARITY  0x08
#define EVENPARITY 0x18

#define BAUD1200   0x80   /* Baud rate */
#define BAUD2400   0xa0
#define BAUD4800   0xc0
#define BAUD9600   0xe0

/* 8259 PIC(Programmable Interrupt Controller) */

#define IMR  0x21  /* I/O address of OCW1(IMR) of 8259 PIC  */
#define OCW2 0x20  /* I/O address of OCW2 of 8259 PIC       */
                   /* OCW: Operation Command Word           */
                   /* IMR: Interrupt Mask Register          */

#define MASKON  0xe7  /* Mask IRQ3/IRQ4 on  -> IMR */
#define MASKOFF 0x18  /* Mask IRQ3/IRQ4 off -> IMR */

#define EOI 0x20  /* Non-specific End of Interrupt command -> OCW2 */

/* Interrupt Request Numbers */

#define IRQ3 0x0b  /* of COM2 */
#define IRQ4 0x0c  /* of COM1 */

#define IRQNUM ((comport == COM1) ? IRQ4 : IRQ3)

/* Miscellaneos */

#define BUFSIZE 0x4000  /* Size of comm buffer */

#define BUFFEREMPTY (-1)  /* Buffer empty */


int comport = COM2;  /* Default COM port, alterable by user */

void interrupt (*oldvect)();   /* Old interrupt vector */
byte commbuf[BUFSIZE];         /* Comm buffer */
int bufptr0 = 0, bufptr1 = 0;  /* Pointers to comm buffer */


void interrupt com_isr()
{
    bufptr1 %= BUFSIZE;
    commbuf[bufptr1++] = inportb(RBR);
    outportb(OCW2, EOI);
}

void com_init(int comport)
{
    bioscom(0, DATABIT8 | STOPBIT1 | NOPARITY | BAUD2400, comport);

    outportb(MCR, 0x0b);
    outportb(IER, 0x01);

    oldvect = getvect(IRQNUM);
    setvect(IRQNUM, com_isr);

    outportb(IMR, inportb(IMR) & MASKON);
}

void com_close(void)
{
    outportb(MCR, 0x00);
    outportb(IER, 0x00);

    outportb(IMR, inportb(IMR) | MASKOFF);
    setvect(IRQNUM, oldvect);
}

int com_getc(void)
{
    if (bufptr0 == bufptr1) return BUFFEREMPTY;
    bufptr0 %= BUFSIZE;

    return commbuf[bufptr0++];
}

void com_putc(int c)
{
    while (!(inportb(LSR) & 0x20)) ;
    outportb(THR, (byte)c);
}

void com_puts(char *s)
{
    while (*s) com_putc(*s++);
}

void hookhwaitkey(void)
{
    int c;

    while ((c = com_getc()) != BUFFEREMPTY) {
        hremovecursor();
        hputch(c);
        hrestartcursor();
    }
}

int main(void)
{
    int c;

    inithanlib(FIXRESMODE, HANDETECT, DEFHANFONT, DEFENGFONT);
    if (hgraphresult() < NOERROR)
        puterr("This program does not run in MDA or CGA!", 1);

    /* Set options */

    hallowautoscroll(true);
    _hangulmode = false;         /* English mode at start               */
    _islinefeednewline = false;  /* '\n' is not converted to CR/LF pair */
    _hookhwaitkey = hookhwaitkey;
    hsetcursormode(CM_DIRECTUNDER);

    /* Print title */

    hclrscr();
    hprintf("*** e ei ei AIA(<ESC> a) ***\r\n\n");
    hprintf("COM%d-2400N81\r\n\n", comport + 1);

    /* Initialize the serial port and the modem */

    com_init(comport);
    com_puts("ATV1X4\r");

    /* Main loop */

    while ((c = hgetch()) != ESC) {
        if (c >= 0x100) continue;  /* Ignore control keys */
        if (c == BS && hwherex() == 1) continue;  /* For line break */
        com_putc(c);
    }

    /* Close the serial port */

    com_close();

    return EXIT_SUCCESS;
}
