/* File EX_TIMER.C: An example of the timer interrupt */

#include <dos.h>
#include <stdio.h>
#include <time.h>

#include "ascii.h"
#include "hanin.h"
#include "hanlib.h"
#include "hanmeq.h"
#include "hanmou.h"
#include "hansw.h"
#include "hanwout.h"

#define TIMER_ISR 0x08

window_t headerwin, timerwin;

void interrupt (*oldvect)(void);

/* User timer interrupt serveice routine, alterable by user */

void user_timer_isr(void)
{
    /* Do not allocate lots of stack(ie. automatic variables) */
    /* otherwise it causes system down                        */
    static char buffer[100];  /* buffer MUST BE static, not automatic! */
    static unsigned cnt = 0;
    static struct time timeinfo;

    /* DO NOT USE ...printf... functions directly!                      */
    /* Because the DOS internal stack is too small to call ...printf... */
    /* So USE sprintf and ...puts... functions together.                */
    /* If ...printf... functions are used directly,                     */
    /* the program may be down sometimes                                */
    /* with a internal stack overflow error of DOS                      */
    /* when the timer interrupt occurred in DOS. (ex. key input)        */

    gettime(&timeinfo);
    sprintf(buffer, " e b: %2d:%02d:%02d",
            timeinfo.ti_hour, timeinfo.ti_min, timeinfo.ti_sec);
    wputsxy(&timerwin, 2, 1, buffer);

    sprintf(buffer, "a ѡ U: %d", cnt++);
    wputsxy(&timerwin, 30, 1, buffer);

    sprintf(buffer, "aȂ: %04X:%04X", _SS, _SP);
    wputsxy(&timerwin, 60, 1, buffer);
}

/* Timer interrupt serveice routine, NOT ALTERABLE by user */

void interrupt timer_isr(void)
{
    static bool inflag = false;

    /* Process the hardware interrupt */

    if (oldvect != NULL) /* Call NOT LAST BUT FIRST!                 */
        oldvect();       /* Original timer interrupt service routine */

    /* Check if valid entry */

    if (_mouautohidden) return;  /* If mouse is autohidden */
    if (_mouismoving) return;    /* If mouse is moving     */
    if (inflag) return;          /* If in service          */

    /* Entry code */

    inflag = true;  /* In-service flag */

    mou_freeze();  /* Freeze mouse not to move */

    _hsaveregs();  /* Save and initialize registers of the video adapter */
                   /* This function is non-reentrant                     */
                   /* so _mouismoving must be checked                    */

    /* Call the user timer interrupt service routine */

    user_timer_isr();

    /* Exit code */

    _hloadregs();  /* Load(restore) the registers of the video adapter */

    mou_melt();  /* Melt the mouse cursor so that it can move */

    inflag = false;
}

void main(void)
{
    int i, c;

    /* Do NOT use STDMODE or BGIMODE with HANDETECT,        */
    /* because this program does not run in EGA             */
    /* due to _hsaveregs function which does not run in EGA */

    inithanlib(FIXRESMODE, HANDETECT, DEFHANFONT, DEFENGFONT);
    if (hgraphresult() < NOERROR)
        puterr("This program does not run in MDA or CGA!", 1);

    /* Color mapping */

    for (i = 0; i <= MAXCOLORS; i++)
        hsetcolormap(i, (hgetmaxcolor() == MAXCOLORS) ? i : i / 15);

    /* Allow the autoscroll feature */

    hallowautoscroll(true);

    /* Initialize the Mouse Event Handler */

    mou_init(MOU_ARROW);

    /* Set output windows */

    hwindow(1, 3, 80, 24);
    hsetbkcolor(HMAGENTA);

    hsetwindow(&headerwin, 1, 1, 80, 2, HWHITE, HDARKGRAY, DEFWIN);
    wregisterhanfont(&headerwin, HAN212GD1);
    wregisterengfont(&headerwin, ENGMD3);

    hsetwindow(&timerwin, 1, 25, 80, 25, HYELLOW, HBLUE, DEFWIN);
    wregisterhanfont(&timerwin, HAN8GD1);

    /* Configure the screen */

    hclrscr();
    wclrscr(&headerwin);
    wclrscr(&timerwin);

    wprintf(&headerwin, "ae ESC...\n");
    wprintf(&headerwin, "aȂ : %04X:%04X", _SS, _SP);

    /* Set the timer ISR */

    oldvect = getvect(TIMER_ISR);
    setvect(TIMER_ISR, timer_isr);

    /* Show the mouse cursor */

    mou_show();

    /* Set attribute of the cursor */

    hsetbufcursor(FRAMECURSOR, FRAMECURSOR, DEFAULTCOLOR);

    /* Get key input with the hgetch fuction */

    while ((c = hgetch()) != ESC)
        if (c == '\r') hputch('\n');
        else hputch(c);

    /* Uninstall the timer ISR */

    setvect(TIMER_ISR, oldvect);
}
