static char rcsid[] = "$Id: scan.c,v 1.4 1994/08/04 23:28:19 sls Exp $";

/*
 * This software is copyright (C) 1994 by the Lawrence Berkeley Laboratory.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include "tcl.h"
#include "narray.h"
#include "narrayInt.h"
#include "parse.h"
#include <setjmp.h>
#include <ctype.h>
#include <assert.h>
#include <string.h>
#include <math.h>

extern Code* yylval;

static Tcl_HashTable keyword_table;
#define ADD_KEYWORD(n, k) \
    entry = Tcl_CreateHashEntry(&keyword_table, n, &new_flag); \
    Tcl_SetHashValue(entry, (ClientData) k)

static jmp_buf scanner_toplevel;
static char* scanner_errmsg;
#define TRY \
    if (setjmp(scanner_toplevel) == 0)
#define CATCH \
    else
static void throw(char* msg)
{
    scanner_errmsg = msg;
    longjmp(scanner_toplevel, 1);
}

static int char_predicate_table[256];
#define IS_OCTAL_DIGIT   1
#define IS_NONZERO_DIGIT 2
#define IS_DELIMITER     4
#define isoctaldigit(c) (IS_OCTAL_DIGIT & char_predicate_table[c])
#define isnonzerodigit(c) (IS_NONZERO_DIGIT & char_predicate_table[c])
#define isdelimiter(c) (IS_DELIMITER & char_predicate_table[c])

static Tcl_DString string_buf;

static void init_scanner(void)
{
    int c, new_flag;
    Tcl_HashEntry* entry;
    for (c = 0; c < 256; c++) {
	char_predicate_table[c] = 0;
	if (c >= '0' && c <= '7')
	    char_predicate_table[c] |= IS_OCTAL_DIGIT;
	if (c >= '1' && c <= '9')
	    char_predicate_table[c] |= IS_NONZERO_DIGIT;
	if (c == ',' || c == '(' || c == ')' || c == '+' || c == '-'
	    || c == '*' || c == '/' || c == ';' || c == '[' || c == ']'
	    || c == '{' || c == '}' || c == '=' || c == '!' || c == '|'
	    || c == '^' || c == '&' || c == '?' || c == ':' || c == '#'
	    || c == '.' || c == '>' || c == '<' || c == '@' || c == '$')
	    char_predicate_table[c] |= IS_DELIMITER;
    }
    Tcl_InitHashTable(&keyword_table, TCL_STRING_KEYS);
    ADD_KEYWORD("for", FOR);
    ADD_KEYWORD("if", IF);
    ADD_KEYWORD("while", WHILE);
    Tcl_DStringInit(&string_buf);
}

#define GET_CHAR_BUFSIZE 16
static int get_char(int get, char c)
{
    static char buf[GET_CHAR_BUFSIZE];
    int result;
    static int unget = 0;
    if (get) {
	if (unget > 0) {
	    return buf[--unget];
	}
	YY_INPUT(buf, result, 1);
	if (result != 1)
	    throw("unexpected end of file");
	return buf[0];
    } else {
	assert(unget < GET_CHAR_BUFSIZE);
	buf[unget++] = c;
	return c;
    }
}

#define get_next_char() get_char(1, 0)
#define unget_char(c) get_char(0, c)

static Code* get_hex_integer(void)
{
    int c, digit;
    int val = 0;
    while (isxdigit(c = get_next_char())) {
	if (isdigit(c))
	    digit = c - '0';
	else
	    digit = c - 'a' + 10;
	val = 16 * val + digit;
    }
    unget_char(c);
    if (val >= OP_PUSH_MAXINT)
	throw("hex value too large -- use floating point");
    return NArray_VarMakeCode(val | OP_PUSH, OP_END);
}

static Code* get_octal_integer(void)
{
    int c, val;
    val = 0;
    while (isoctaldigit(c = get_next_char())) {
	val = 8 * val + c - '0';
    }
    unget_char(c);
    if (val >= OP_PUSH_MAXINT)
	throw("octal value too large -- use floating point");
    return NArray_VarMakeCode(val | OP_PUSH, OP_END);
}

static Code* get_number(int* type_out)
{
    int c, neg, is_float;
    int i_val;
    double d_val, f;
    Code* result;
    result = NArray_VarMakeCode(OP_END);
    d_val = 0.0;
    is_float = 0;
    i_val = 0;
    while (isdigit(c = get_next_char())) {
	i_val = 10 * i_val + c - '0';
    }
    if (c == '.') {
	d_val = (double) i_val;
	f = 0.1;
	while (isdigit(c = get_next_char())) {
	    d_val += (c - '0') * f;
	    f *= 0.01;
	}
	if (c == 'e') {
	    c = get_next_char();
	    if (c == '-') {
		neg = 1;
	    } else {
		neg = 0;
		unget_char(c);
	    }
	    i_val = 0;
	    while (isdigit(c = get_next_char())) {
		i_val = 10 * i_val + c - '0';
	    }
	    if (neg)
		d_val /= pow(10.0, (double) i_val);
	    else
		d_val *= pow(10.0, (double) i_val);
	}
	is_float = 1;
    }
    unget_char(c);
    if (!is_float && i_val >= OP_PUSH_MAXINT) {
	d_val = (double) i_val;
	is_float = 1;
    }
    if (is_float) {
	*type_out = FLOAT;
	result = NArray_AppendDouble(result, d_val);
    } else {
	*type_out = INT;
	result = NArray_AppendOps(result, OP_PUSH | (i_val & OPERAND_MASK),
				  OP_END);
    }
    return result;
}

#define handle_two_char_token(first, second, ttype) \
    if (c == first && (((next_c = get_next_char()) == second) \
		       || ((unget_char(next_c), 0), 0))) { \
        *type_out = ttype; \
        result = 0; \
	goto finish; \
    }

static Code* get_next_token(int* type_out)
{
    enum {
	S_IDLE = 0, S_COMMENT, S_STRING, S_IDENTIFIER,
    } state;
    int c, next_c;
    Code* result;
    Tcl_HashEntry* entry;
    char buf[2];

    state = S_IDLE;

    while (1) {
	c = get_next_char();
	switch (state) {
	case S_IDLE:
	    if (c == '\n') {
		*type_out = ';';
		result = 0;
		goto finish;
	    } else if (isspace(c)) {
		/* do nothing */
	    } else if (c == '"') {
		state = S_STRING;
		*type_out = STRING;
		Tcl_DStringTrunc(&string_buf, 0);
	    } else if (c == '#') {
		state = S_COMMENT;
	    } else if (c == '0') {
		next_c = get_next_char();
		*type_out = INT;
		if (next_c == 'x' || next_c == 'X') {
		    result = get_hex_integer();
		    goto finish;
		} else if (isnonzerodigit(next_c)) {
		    result = get_octal_integer();
		    goto finish;
		} else {
		    unget_char(next_c);
		    unget_char('0');
		    result = get_number(type_out);
		    goto finish;
		}
	    } else if (isdigit(c)) {
		unget_char(c);
		result = get_number(type_out);
		goto finish;
	    } else if (c == '.') {
		next_c = get_next_char();
		unget_char(next_c);
		if (isdigit(next_c)) {
		    unget_char(c);
		    result = get_number(type_out);
		    goto finish;
		} else {
		    result = 0;
		    *type_out = c;
		    goto finish;
		}
	    }
	    else handle_two_char_token('=', '=', EQ)
	    else handle_two_char_token('!', '=', NEQ)
	    else handle_two_char_token('>', '=', GE)
	    else handle_two_char_token('<', '=', LE)
	    else handle_two_char_token('|', '|', OR)
	    else handle_two_char_token('&', '&', AND)
	    else handle_two_char_token('+', '=', PLUS_EQ)
	    else handle_two_char_token('-', '=', MINUS_EQ)
	    else handle_two_char_token('*', '=', MUL_EQ)
	    else handle_two_char_token('/', '=', DIV_EQ)
	    else if (isdelimiter(c)) {
		*type_out = c;
		result = 0;
		goto finish;
	    } else {
		unget_char(c);
		state = S_IDENTIFIER;
		Tcl_DStringTrunc(&string_buf, 0);
	    }
	    break;
	case S_COMMENT:
	    if (c == '\n')
		state = S_IDLE;
	    break;
	case S_STRING:
	    if (c == '\\') {
                c = get_next_char();
                switch (c) {
                case 'b': c = '\b'; break;
                case 'f': c = '\f'; break;
                case 'n': c = '\n'; break;
                case 'r': c = '\r'; break;
                case 't': c = '\t'; break;
                case 'v': c = '\v'; break;
                case '0': c = '\0'; break;
                }
            } else if (c == '"') {
		result = NArray_VarMakeCode(OP_END);
		result = NArray_AppendString(result,
					     Tcl_DStringValue(&string_buf));
		goto finish;
	    }
	    buf[0] = c;
	    buf[1] = '\0';
	    Tcl_DStringAppend(&string_buf, buf, 1);
	    break;
	case S_IDENTIFIER:
	    if (isspace(c) || isdelimiter(c)) {
		unget_char(c);
		entry = Tcl_FindHashEntry(&keyword_table,
					  Tcl_DStringValue(&string_buf));
		if (entry != 0) {
		    *type_out = (int) Tcl_GetHashValue(entry);
		    result = 0;
		} else {
		    *type_out = ID;
		    result = NArray_VarMakeCode(OP_END);
		    result = NArray_AppendId(result,
					     Tcl_DStringValue(&string_buf));
		}
		goto finish;
	    } else {
		buf[0] = c;
		buf[1] = '\0';
		Tcl_DStringAppend(&string_buf, buf, 1);
	    }
	    break;
	default:
	    assert(0);
	}
    }
 finish:
    return result;
}

int yylex(void)
{
    int t;
    static int initialized = 0;
    if (!initialized) {
	init_scanner();
	initialized = 1;
    }
    TRY {
	yylval = get_next_token(&t);
    } CATCH {
	yylval = NULL;
	t = -1;
    }
    return t;
}
