# copyright 1994 Applied Reasoning Systems
# by William R. Burdick Jr.
# You may hack away, as long as you give me credit if you distribute some of this code

# Here's a chunk of my .stkwmrc

# rooms are grouped into bars.  each room and each bar has a name.

# the room() entries are used (by the choose_room function) to classify
# new window into rooms.

# the syntax of the room() entries is:
# entry: room(class:name)	{ spec ... }
# entry: room(class)		{ spec ... }
# entry: room('default')	{ spec ... }
# spec: {regexp {barName roomName}}

# room(class:name) is compared first
# room(class) is compared next
# room(default) is used if none of the above entries are found.

# when an entry is chosen, specs are scanned left to right.  The
# first spec that has a regexp matching the window's name is chosen
# if none match, the last one is used.

# the rooms parse command takes a list of {bar {room ...}} entries.
# the rooms place command is run after parsing.  it aligns the rooms
# across the top of the screen.  rooms are placed left to right.

# defvar {
# 	room(default)	{ {.* {unix misc}} }
# 	room(XTerm)	{ {.* {unix xterm}} }
# 	room(Emacs)	{ {.* {unix emacs}} }
# 	room(Clock)	{ {.* {}}}
# 	room(St80)	{
# 		{Graph|Concept {smalltalk consolidate}}
# 		{Browser|Implementors|References {smalltalk browsers}}
# 		{{Directory|File|System Transcript|Interface} {smalltalk {file stuff}}}
# 		{orks|TODO|Inspector {smalltalk workspaces}}
# 		{Launcher|ARS|Commands|UPDATE|Socket|Sorel|CONSOLIDATE {smalltalk commands}}
# 		{.* {smalltalk misc}}
# 	}
# }
# 
# rooms parse {
# {unix {xterm emacs misc}}
# {smalltalk {commands browsers workspaces consolidate {file stuff} misc}}
# }


proc defvar {defs} {
    for {set i 0 ; set len [llength $defs]} {$i < $len} {incr i} {
	set varName [lindex $defs $i]
	set value [lindex $defs [incr i]]
	if {![uplevel "info exists $varName"]} {
	    uplevel "set $varName {$value}"
	}
    }
}

defvar {
    rooms(roomNum) 0
    rooms(barNum) 0
}

proc rooms {cmd args} {
    if {[info command rooms::$cmd] == {}} {
	tkerror "unknown rooms option: $cmd.  Must be one of: [rooms commands]"
	return
    }
    uplevel [concat rooms::$cmd $args]
}

proc rooms::commands {} {
    regsub -all rooms:: [info procs rooms::*] {} commands
    return [lsort $commands]
}

proc rooms::load {} {
    global rooms
    
    catch {unset rooms}
    source ~/tcl/rooms.tcl
}

proc rooms::place {} {
    set pos 0
    set top 0
    foreach bar [rooms bars] {
	wm geometry $bar +$pos+$top
	set pos [expr {$pos + [winfo width $bar]}]
    }
}

proc rooms::addbar {name} {
    global rooms
    
    if {![catch {rooms bar $name}]} {
	tkerror "Bar: $name already exists."
	return
    }
    set top .tkwin[incr rooms(barNum)]
    set win [create_iwindow $top roomBar RoomBar]
    register roomType bar $top
    register barName $name $top
    register innerWidget $win $top
#    wm geometry $top -0+0
    button $win.button -text "  "
    pack $win.button -side left -expand 1 -fill both
    float up $top
    $top.roomBar map
    bind $win.button <ButtonPress-1> "rooms raisebar $top"
    bind $win.button <ButtonPress-2> "rooms togglebar $top"
    bind $win.button <ButtonPress-3> "rooms lowerbar $top"
    foreach binding [bind $top] {
	if {[string match *Button* $binding]} {
	    bind $top $binding {}
	}
    }
}

proc rooms::addroom {barName roomName args} {
    global rooms
    
    if {$args == {}} {
	set args "-text {$roomName}"
    }
    set bar [rooms bar $barName]
    if {![catch {rooms room $barName $roomName}]} {
	tkerror "Room: $roomName already exists in bar: $barName."
	return
    }
    set widget [registration $bar innerWidget].room[incr rooms(roomNum)]
    register bar $bar $widget
    register roomName $roomName $widget
    register roomType room $widget
    eval [concat button [list $widget] $args]
    pack $widget -side left
    bind $widget <Destroy> "+rooms vacate $widget ; deregister $widget"
    bind $widget <ButtonPress-1> "rooms process $widget {rooms raise $widget ; rooms open $widget}"
    bind $widget <ButtonPress-2> "rooms process $widget {rooms toggle $widget}"
    bind $widget <ButtonPress-3> "rooms process $widget {rooms lower $widget ; rooms open $widget}"
    bind $widget <Enter> "$widget configure -state active ; rooms process $widget {rooms setFocus $widget}"
    rooms setstate $widget disabled
    global tkwm_registry
    trace variable tkwm_registry(room,$widget) w "rooms update $widget"
}

proc rooms::bar {barName} {
    set widget [registered barName $barName]
    if {$widget == {}} {
	error "No bar: $barName."
	return
    }
    return $widget
}

proc rooms::room {barName roomName} {
    foreach room [rooms rooms [rooms bar $barName]] {
	if {[registration $room roomName] == $roomName} {
	    return $room
	}
    }
    error "No room: $roomName in bar: $barName."
    return
}

proc rooms::bars {} {
    return [registered roomType bar]
}

proc rooms::allrooms {} {
    return [registered roomType room]
}

proc rooms::rooms {bar} {
    return [registered bar $bar]
}

proc rooms::windows {room} {
    return [registered room $room]
}

proc rooms::bar? {win} {
    return [expr {[registration $win roomType] == "bar"}]
}

proc rooms::room? {win} {
    return [expr {[registration $win roomType] == "room"}]
}

proc rooms::state {room} {
    return [registration $room state]
}

proc rooms::process {room cmds} {
    if {[rooms state $room] != "disabled"} {
	uplevel 1 $cmds
    }
}

proc rooms::setstate {room state} {
    set bar [registration $room bar]
    register state $state $room
    switch -exact $state {
	open {
	    $room configure -relief raised
	    rooms baropen $bar
	}
	closed {
	    $room configure -relief sunken
	    foreach room [rooms rooms $bar] {
		if {[rooms state $room] == "open"} {
		    return
		}
	    }
	    rooms barclosed $bar
	}
	disabled {
	    $room configure -relief ridge
	}
    }
}

proc rooms::forwindows {room winVar block} {
    upvar $winVar win
    foreach win [rooms windows $room] {
	if {[winfo exists $win]} {
	    uplevel $block
	}
    }
}

proc rooms::raise {room} {
    rooms forwindows $room win {
	switch -exact [uplevel #0 "set ${win}(state)"] {
	    w_iconified -
	    iconified {
		restack [uplevel #0 "set ${win}(icon)"] Above None
	    }
	    w_normal -
	    normal -
	    default {
		restack $win Above None
	    }
	}
    }
}

proc rooms::lower {room} {
    rooms forwindows $room win {
	switch -exact [uplevel #0 "set ${win}(state)"] {
	    normal {
		lower $win
	    }
	    iconified {
		lower [uplevel #0 "set ${win}(icon)"]
	    }
	}
    }
}

proc rooms::open {room} {
    rooms setstate $room open
    rooms forwindows $room win {
	wset_restore $win
    }
}

proc rooms::close {room} {
    rooms setstate $room closed
    wset_withdraw withdrawn [rooms windows $room]
}

proc rooms::vacate {room} {
    rooms forwindows $room win {
	catch {rooms exit $win}
    }
}

proc rooms::enter {room win} {
    if {![rooms room? $room]} {
	tkerror "$room is not a room."
	return
    }
    if {![winfo exists $win]} {
	tkerror "$win is not a window."
	return
    }
    set win [winfo toplevel $win]
    register room $room $win
    rooms setstate $room open
    return $room
}

proc rooms::update {room var index op} {
    upvar #0 ${var}($index) list
    if {![info exists list] || $list == {}} {
	rooms setstate $room disabled
    } {
	if {[rooms state $room] == "disabled"} {
	    rooms setstate $room open
	}
    }
}

proc rooms::exit {win} {
    set room [registration $win room]
    if {[winfo exists $win]} {
	wset_restore $win
	register room {} $win
    }
    if {[rooms windows $room] == {}} {
	rooms setstate $room disabled
    }
}

proc rooms::toggle {room} {
    switch -exact [rooms state $room] {
	closed {
	    rooms open $room
	}
	open -
	default {
	    rooms close $room
	}
    }
}

proc rooms::baropen {bar} {
    [registration $bar innerWidget].button configure -relief raised
}

proc rooms::barclosed {bar} {
    [registration $bar innerWidget].button configure -relief sunken
}

proc rooms::togglebar {bar} {
    set roomlist [rooms rooms $bar]
    set open 0
    set closed {}
    foreach room $roomlist {
	if {[rooms state $room] == "open"} {
	    set open 1
	    rooms process $room {rooms close $room}
	}
    }
    if {!$open} {
	foreach room $roomlist {
	    rooms process $room {rooms open $room}
	}
    }
}

proc rooms::raisebar {bar} {
    foreach room [registered bar $bar] {
	rooms process $room {rooms raise $room}
    }
}

proc rooms::lowerbar {bar} {
    foreach room [registered bar $bar] {
	rooms process $room {rooms lower $room}
    }
}

proc rooms::parse {bars} {
    foreach bar $bars {
	set barName [lindex $bar 0]
	rooms addbar $barName
	foreach room [lindex $bar 1] {
	    eval [concat rooms addroom [list $barName] $room]
	}
    }
    update
#    rooms place
}


# arrange for focus to be set to a particular room member
# when the cursor is on the room button
# setFocus does the actual focussing
# focus registers the window so that setFocus will choose it

proc rooms::setFocus {room} {
    set win [registered focus $room]
    if {$win != {}} {
	change_focus [lindex $win 0]
    }
}


proc rooms::focus {win} {
    set room [registration $win room]
    foreach old [registered focus $room] {
	register focus {} $old
    }
    register focus $room $win
}


proc final_room_choice {widget name list} {
    foreach choice $list {
	set exp [lindex $choice 0]
	set room [lindex $choice 1]
	if {[regexp $exp $name]} {
	    if {$room == {}} {
		return global
	    }
	    return [rooms enter [rooms room [lindex $room 0] [lindex $room 1]] $widget]
	}
    }
    return [rooms enter [rooms room [lindex $room 0] [lindex $room 1]] $widget]
}


proc choose_room {plugData} {
    global room
    
    if {[regexp {^.*\.plug$} $plugData]} {
	set c [$plugData info res_class]
	set r [$plugData info res_name]
	set n [$plugData info name]
	set w $plugData
    } {
	upvar $plugData data
	set c $data(class_name)
	set r $data(resource_name)
	set n $data(name)
	set w $data(widget)
    }
    if {[rooms room? $w]} {
	return global
    }
    if {[info exists room($c:$r)]} {
	return [final_room_choice $w $n $room($c:$r)]
    }
    if {[info exists room($c)]} {
	return [final_room_choice $w $n $room($c)]
    }
    if {[info exists room(default)]} {
	return [final_room_choice $w $n $room(default)]
    }
    return global
}
