########################################################################
#
# Copyright (c) 1994 John F. Sefler.
# All rights reserved.
#
########################################################################
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose, without fee, and without written
# agreement is hereby granted, provided the above copyright notice and
# the following two paragraphs appear in all copies of this software.
#
# IN NO EVENT SHALL JOHN F. SEFLER BE LIABLE TO ANY PARTY FOR DIRECT,
# INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
# OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF HE HAS
# BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# JOHN F. SEFLER SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING, BUT
# NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
# FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
# "AS IS" BASIS, AND JOHN F. SEFLER HAS NO OBLIGATION TO PROVIDE
# MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
#
########################################################################

########################################################################
#
# TITLE:	Tcl/Tk Interface Script for mazeman
#
# DATE:		Fall 1994
# AUTHOR:	John F. Sefler
#
########################################################################

# Copyright
set gl(application)		mazeman
set gl(version)			1.0
puts "$gl(application) version $gl(version)"
puts "(C) Copyright 1994  John F. Sefler"
puts "All Rights Reserved"

# where is this game installed?
set gl_install_dir $env(MAZEMANHOME)

# add to the auto path a path for finding unknown commands
# that have been indexed using auto_mkindex in a tclsh
# (set it to this directory)
set auto_path [linsert $auto_path 0 $gl_install_dir/scripts]

# global variable array for the maze
set gl_maze(width)		550
set gl_maze(height)		550
set gl_maze(width_offset)	10
set gl_maze(height_offset)	10
set gl_maze(background)		white
set gl_maze(id)			0
set gl_maze(size)		5
set gl_maze(size_min)		5
set gl_maze(size_max)		10
set gl_maze(cur_level)		1
set gl_maze(num_levels)		4
set gl_maze(num_levels_min)	1
set gl_maze(num_levels_max)	10
set gl_maze(num_elevators)	3
set gl_maze(num_elevators_min)	1
set gl_maze(num_elevators_max)	10
set gl_maze(num_loops)		0
set gl_maze(num_loops_min)	0
set gl_maze(num_loops_max)	5
set gl_maze(num_ghosts)		5
set gl_maze(num_ghosts_min)	0
set gl_maze(num_ghosts_max)	10 
set gl_maze(ghost_speed)	7
set gl_maze(ghost_speed_min)	1
set gl_maze(ghost_speed_max)	10
set gl_maze(wall_color)		black
set gl_maze(wall_width)		2

# global variable array for the game options (0 = off, 1 = on)
set gl_options(show_solution)		1
set gl_options(leave_trail)		0
set gl_options(snap_motion)		1
set gl_options(snap_motion_x)		0
set gl_options(snap_motion_y)		0
set gl_options(leave_trail_color)	Blue
set gl_options(show_solution_color)	Snow2
#set gl_options(show_solution_color)	LightGrey

# global variable array for the competition mode
set gl_competition(cur_stage)	0
set gl_competition(num_stages)	0

# global variable array for the top 10 maze men statistics
set gl_top10(names_list)	{}
set gl_top10(scores_list)	{}

# default key bindings and lists
set gl_key(move_up)		Up
set gl_key(move_down)		Down
set gl_key(move_left)		Left
set gl_key(move_right)		Right
set gl_key(move_up_elevator)	q
set gl_key(move_down_elevator)	z
set gl_key(view_up_level)	w
set gl_key(view_down_level)	x
set gl_key(start_stop)		Return

set gl_key_names_list [list \
  {Move man up} \
  {Move man down} \
  {Move man left} \
  {Move man right} \
  {Take elevator up} \
  {Take elevator down} \
  {View level above} \
  {View level below} \
  {Press Start/Pause}]

set gl_key_variables_list [list \
  gl_key(move_up) \
  gl_key(move_down) \
  gl_key(move_left) \
  gl_key(move_right) \
  gl_key(move_up_elevator) \
  gl_key(move_down_elevator) \
  gl_key(view_up_level) \
  gl_key(view_down_level) \
  gl_key(start_stop)]
  
# various global variables stored in a gl array
set gl(elevator_scale_factor)	0.80
set gl(ghost_scale_factor)	0.80
set gl(man_scale_factor)	0.65
set gl(game_over)		1
set gl(game_paused)		0
set gl(game_speed)		5
set gl(game_rest_delay)		60000
set gl(man_ghost_collision)			0
set gl(man_ghost_collision_delay)		500
set gl(num_man_ghost_collision_frames)	7
set gl(new_maze_initialized)	0
set gl(score)			0
set gl(bonus)			0
set gl(help_file) $gl_install_dir/scripts/help.tcl
set gl(picture) $gl_install_dir/bitmaps/JohnSefler.btm
set gl(about_author_msg) \
	"John F. Sefler \
	\nGraduate Student \
	\nFall 1994 \
	\n\n\
        \nComputer Science Department \
	\nUniversity of California, Berkeley \
        \n\
        \nsefler@eecs.Berkeley.EDU"


########################################################################
proc InitializeMazeMan { } {
  global gl
  global gl_install_dir
  global gl_level
  global gl_maze
  global gl_options
  global gl_competition

  # Main Window Setup
  wm title . "Maze Man $gl(version)"

  # Main Window Menubar Setup
  frame .menu -borderwidth 1 -relief raised
  pack  .menu -side top -anchor w -fill x
  menubutton .menu.game -text "Game" -menu .menu.game.m 
  menubutton .menu.opt  -text "Options"     -menu .menu.opt.m
  menubutton .menu.help -text "Help"        -menu .menu.help.m
  pack .menu.game	-side left
  pack .menu.opt	-side left -padx 2m
  pack .menu.help	-side right


  # Main Window Menubar Pull Down Setup
  menu .menu.game.m
  .menu.game.m add command -label "Practice..." -command {MazeOptions}
  .menu.game.m add command -label "Competition" -command {CompetitionMaze}
  .menu.game.m add separator
  .menu.game.m add command -label "Quit"     -command {QuitMaze}
   
  menu .menu.opt.m
  .menu.opt.m add check -label "Show Solution Paths" \
                 -variable gl_options(show_solution) \
                 -command {OptionsShowSolution $gl_options(show_solution)}
  .menu.opt.m add check -label "Leave A Trail" \
                 -variable gl_options(leave_trail) \
                 -command {OptionsLeaveTrail   $gl_options(leave_trail)}
  .menu.opt.m add check -label "Snap Mouse Motion" \
                 -variable gl_options(snap_motion) \
                 -command {OptionsSnapMotion   $gl_options(snap_motion)}
  .menu.opt.m add separator
  .menu.opt.m add command -label "Key Bindings..." \
 	  -command {ChangeKeyBindings}
  .menu.opt.m add command -label "Game Speed..." \
		  -command {ChangeGameSpeed}

  menu .menu.help.m
  .menu.help.m add command -label "How To Play..." \
                  -command {HelpDialog . .helpdialog "How To Play" \
                            $gl(help_file) 0 { } {Done} 0 }

  .menu.help.m add command -label "Top 10 Maze Men..." \
		  -command {ShowTop10Display}
  .menu.help.m add separator
  .menu.help.m add command -label "About Author..." \
		  -command {MessageDialog . .about_author "About Author" \
                            $gl(about_author_msg) 0 \
                            "@$gl(picture) black white" {Done} 0}
   

  # Main Window Side Bar Setup
  frame .sb -borderwidth 1 -relief sunken

  frame .score_f -relief sunken -borderwidth 1
# pack .score_f -in .sb -side top -fill x

  frame .score_f.s_f -relief flat -borderwidth 0
  label .score_f.s_f.s_l -text "Score" -width 5
  label .score -relief sunken -borderwidth 2 -width 5 -textvariable gl(score)
  pack .score_f.s_f -side top -fill x
  pack .score_f.s_f.s_l -side left
  pack .score -in .score_f.s_f -side right -expand yes -fill x

  frame .score_f.b_f -relief flat -borderwidth 0
  label .score_f.b_f.b_l -text "Bonus" -width 5
  label .bonus -relief sunken -borderwidth 2 -width 5 -textvariable gl(bonus)

  pack .score_f.b_f -side top -fill x
  pack .score_f.b_f.b_l -side left
  pack .bonus -in .score_f.b_f -side right -expand yes -fill x

  frame .score_f.st_f -relief flat -borderwidth 0
  label .score_f.st_f.st_l -text "Stage" -width 5
  label .stage_cur -relief flat -borderwidth 0 -textvariable gl_competition(cur_stage)
  label .score_f.st_f.st_of_l -text "of" -width 2
  label .stage_num -relief flat -borderwidth 0 -textvariable gl_competition(num_stages)

  pack .score_f.st_f -side top -fill x
  pack .score_f.st_f.st_l -side left
  pack .stage_cur -in .score_f.st_f -side left -expand yes -fill x
  pack .score_f.st_f.st_of_l -side left 
  pack .stage_num -in .score_f.st_f -side left -expand yes -fill x

  scale .cur_level_s -label "Level" \
               -state disabled -orient vertical -borderwidth 1 \
               -relief sunken \
	       -from $gl_maze(num_levels) -to 1 -tickinterval -1 \
               -showvalue no \
	       -command {ShowLevel}

  frame .start_stop_f -relief sunken -borderwidth 2
  button .start_stop_b -text "Start" -state disabled -borderwidth 2
  pack .start_stop_b -in .start_stop_f -fill both -padx 2m -pady 1m -ipady 1m

# pack .start_stop_b -side bottom -in .sb -fill x
# pack .cur_level_s -in .sb -side top -expand yes -fill y
  pack .sb -side left -fill y

  # Create a frame for all the Maze canvases
  set canvas_f_width [expr $gl_maze(width) + \
                ($gl_maze(num_levels_max) + 1) * $gl_maze(width_offset)]
  set canvas_f_height [expr $gl_maze(height) + \
                ($gl_maze(num_levels_max) + 1) * $gl_maze(height_offset)]
  frame  .canvas_f  -borderwidth 1 -relief sunken \
                    -width $canvas_f_width \
                    -height $canvas_f_height
  pack  .canvas_f -side right -expand no

  # add a large title
  set font [GetNearestFontFromPixels helvetica bold o 70]
  label .canvas_f.title_horiz -text "Maze Man" -font $font
  set x_position [expr $canvas_f_width/2.0]
  set y_position 0
  place .canvas_f.title_horiz -anchor n -x $x_position -y $y_position

  # Create an array of maze canvases
  # gl_level(1,2,...,gl_maze(num_levels_max))
  for {set level 1} {$level <= $gl_maze(num_levels_max)} {incr level} {
    set     gl_level($level) .level_$level
    canvas $gl_level($level) \
                 -background $gl_maze(background) \
                 -relief ridge -borderwidth 4 \
                 -width  $gl_maze(width) \
                 -height $gl_maze(height)
  }

  # load the last setting of the game speed from file
  LoadGameSpeed
}


########################################################################
proc InitializeIntroductionControls {num_levels} {
  global gl_key

  # unmanage unneeded controls
  pack forget .score_f
  pack forget .start_stop_f
  bind . <$gl_key(start_stop)> {}

  # manage and reconfigure the current level scale
  pack .cur_level_s -in .sb -side top -expand yes -fill both
  .cur_level_s set 1
  if {$num_levels == 1 || $num_levels == 0} {
    .cur_level_s configure -from $num_levels -tickinterval 0
  } else {
    .cur_level_s configure -from $num_levels -tickinterval -1
  }
  LevelBind

}

########################################################################
proc InitializePracticeControls {num_levels} {
  global gl_key

  # initialize the same controls from the Introduction
  InitializeIntroductionControls $num_levels

  # unmanage unneeded controls
  pack forget .score_f

  # disable the current level scale 
  LevelUnBind

  # reconfigure the start_stop button
  .start_stop_b configure -text "Start" -state normal \
                        -background green \
                        -activebackground green \
                        -command {Start}
  pack .start_stop_f -side bottom -in .sb -fill x

  bind . <$gl_key(start_stop)> {.start_stop_b invoke}
}

########################################################################
proc InitializeCompetitionControls {num_levels} {

  # initialize the same controls from a practice maze
  InitializePracticeControls $num_levels

  # reset the bonus (do not reset the score)
  ResetBonus

  # disable the current level scale 
  LevelUnBind

  pack .score_f -in .sb -before .cur_level_s -side top -fill x

}

########################################################################
proc ResetBonus { } {
  global gl
  global gl_maze

  # set the stage bonus equal to the number of squares in the maze
  set gl(bonus) [expr $gl_maze(size) * $gl_maze(size) * $gl_maze(num_levels)]
}

########################################################################
proc ResetScoreTo {score} {
  global gl

  set gl(score) $score
}

########################################################################
proc AddBonusToScore { } {
  global gl

# set gl(score) [expr $gl(score) + $gl(bonus)]
  while {$gl(bonus) > 0} {
    incr gl(bonus) -1
    incr gl(score) +1
    after 25
    update
  }
}

########################################################################
proc IncrementScoreBy {amount} {
  global gl

  incr gl(score) $amount
}

########################################################################
proc DecrementBonus { } {
  global gl

  if {!$gl(game_over) && !$gl(game_paused) && ($gl(bonus) > 0)} {
    incr gl(bonus) -1
    after 1000 DecrementBonus
  }
}


########################################################################
proc CreateIntroductionMaze {maze_id} {
  global gl
  global gl_level
  global gl_maze
  global gl_options

  # is it okay to create an introductory maze?
  if {!$gl(game_over)}          return;
  if {$gl_maze(id) != $maze_id} return;
  if {[winfo exists .who]} {
    after $gl(game_rest_delay) CreateIntroductionMaze $gl_maze(id)
    return
  }

  # start an introductory game
  set gl(game_over) 0

  # initialize the controls for the introduction
  InitializeIntroductionControls $gl_maze(num_levels)

  # Create a new Maze
  NewMaze

  # Remove the man from the maze
  set level [GetManLevel]
  if {$level} { $gl_level($level) delete man }

  # display the finish level
  update idletasks
  LevelBind
  ShowLevel $gl_maze(num_levels)
# LevelUnBind

  OptionsShowSolution $gl_options(show_solution)

  # Set the Ghosts to motion
  AdvanceGhostsForever
}


########################################################################
proc ShowLevel {cur_level} {
  global gl
  global gl_maze
  global gl_level

  # make sure the level request is valid
  if {$cur_level < 0 || $cur_level > $gl_maze(num_levels)} return;

  # make sure the current level scale shows the current level
  .cur_level_s set $cur_level

  # reset the global variable for the current level
  set gl_maze(cur_level) $cur_level

  # forget the current canvases
  set next_level [expr $cur_level + 1]
  for {set level $gl_maze(num_levels_max)} {$level > $cur_level} {incr level -1} {
    place forget $gl_level($level)
    update
  }

  set canvas_f_height [expr $gl_maze(height) + \
              ($gl_maze(num_levels_max) + 1) * $gl_maze(height_offset)]
  for {set level 1} {$level <= $cur_level} {incr level} {
    set x_position [expr $level * $gl_maze(width_offset)]
    set y_position [expr $canvas_f_height - ($level * $gl_maze(width_offset))]
    place $gl_level($level) -in .canvas_f -anchor sw \
                                  -x $x_position -y $y_position
    update
                
  }

}

########################################################################
proc Start { } {
  global gl
  global gl_competition

  # reconfigure widgets on the main display
  .start_stop_b configure -text "Pause" -state normal \
                        -background red \
                        -activebackground red \
                        -command {Stop}

  # attach mouse and key bindings to the man, elevators, and levels
  ManBind
  LevelBind

  # start a new game
  set gl(game_over) 0
  
  # start decrementing the bonus
  if {$gl_competition(cur_stage)} DecrementBonus

  # Set all the Ghosts to motion in an endless loop
  AdvanceGhostsForever
}

########################################################################
proc Stop { } {
  global gl
  global gl_competition

  .start_stop_b configure -relief sunken
  set gl(game_paused) 1
  update idletasks

  MessageDialog . .stop_dlg "Pause" \
         "Maze Man has been paused." \
         1 {warning red white} {Continue} 1

  set gl(game_paused) 0
  .start_stop_b configure -relief raised

  # restart the ghosts
  AdvanceGhostsForever

  # continue the competition
  if {$gl_competition(cur_stage)} DecrementBonus
}

########################################################################
proc ChangeKeyBindings { } {
  global gl
  global gl_key
  global gl_key_names_list
  global gl_key_variables_list

  # eliminate the current key bindings
  ManKeyUnBind
  LevelKeyUnBind
  bind . <$gl_key(start_stop) {}

  # get the new key bindings
  KeyBindingDialog . $gl_key_names_list $gl_key_variables_list

  # should the new man bindings be set?
  if {!$gl(game_over) && !$gl(man_ghost_collision)} {
    ManKeyBind
  }
    LevelKeyBind
    bind . <$gl_key(start_stop)> {.start_stop_b invoke}
}

########################################################################
proc ManGrab {x y} {
  global gl
  global gl_maze
  global gl_options
  global gl_level

  set c $gl_level($gl_maze(cur_level))
  set current_tags [$c gettags current]
  if {[lsearch $current_tags man] != -1} {
    # the catch avoids an error display when no elevator exists
    # in the case when there is only one level
    catch {$c lower man elevator} err_msg
    ItemStartDrag $c $x $y \
           $gl_options(snap_motion_x) $gl_options(snap_motion_y) man
  }
}

########################################################################
proc ManMove {x y} {
  global gl
  global gl_level
  global gl_maze
  global gl_options

  set c $gl_level($gl_maze(cur_level))
  ItemBoundedDrag $c $x $y \
       $gl_options(snap_motion_x) $gl_options(snap_motion_y) man wall

  # determine if the man ran into a ghost
  CheckManCollideWithGhostOrFinish

  OptionsLeaveTrail $gl_options(leave_trail)
  update idletasks
}

########################################################################
proc ManKeyMove {dir} {
  global gl
  global gl_level
  global gl_maze
  global gl_options

  ManKeyUnBind; update

  set c $gl_level($gl_maze(cur_level))
  set man_coords_list [GetCenterCoordsOfTag $gl_maze(cur_level) man]
  if {[llength $man_coords_list] == 0} return
  set xc [lindex $man_coords_list 0]
  set yc [lindex $man_coords_list 1]
  set x_snap [expr double($gl(square_width))/2.0]
  set y_snap [expr double($gl(square_height))/2.0]

  switch $dir {
    up    {set next_yc [expr $yc - $y_snap]; set next_xc $xc}
    down  {set next_yc [expr $yc + $y_snap]; set next_xc $xc}
    left  {set next_xc [expr $xc - $x_snap]; set next_yc $yc}
    right {set next_xc [expr $xc + $x_snap]; set next_yc $yc}
  }

  ItemStartDrag   $c $xc $yc $x_snap $y_snap man
  ItemBoundedDrag $c $next_xc $next_yc $x_snap $y_snap man wall

  # lower the man below the elevators
  catch {$c lower man elevator} err_msg

  OptionsLeaveTrail $gl_options(leave_trail)

  # determine if the man ran into a ghost
  if {![CheckManCollideWithGhostOrFinish]} {
    ManKeyBind; update
  }
}

########################################################################
proc ManBoardElevator {x y dir} {
  global gl
  global gl_level
  global gl_maze

  set c $gl_level($gl_maze(cur_level))
  set overlappping_list [GetOverlappingListOfElevatorAt $x $y]

  # is the man positioned over or under an up or down elevator?
  set man_found 0
  set dir_found 0
  foreach item $overlappping_list {
    if {[lsearch [$c gettags $item] man] != -1} {set man_found 1}
    if {[lsearch [$c gettags $item] $dir] != -1} {
      set dir_found 1
      set item_coords_list [GetCenterCoordsOfTag $gl_maze(cur_level) $item]
      set xc [lindex $item_coords_list 0]
      set yc [lindex $item_coords_list 1]
      set square_coords_list [GetCenterCoordsOfSquareAt $xc $yc]
      set xc [lindex $square_coords_list 0]
      set yc [lindex $square_coords_list 1]
    }
    if {$man_found && $dir_found} {break}
  }

  # take the elevator to the next level
  if {$man_found && $dir_found} {
      switch $dir {
        up   {GoUpElevatorAt   $xc $yc}
        down {GoDownElevatorAt $xc $yc}
      }
  }
}

########################################################################
proc ManKeyBoardElevator {dir} {
  global gl
  global gl_level
  global gl_maze

  set c $gl_level($gl_maze(cur_level))
  set overlappping_list [GetOverlappingListOfMan]

  # is the man positioned over an up or down elevator?
  set dir_found 0
  foreach item $overlappping_list {
    if {[lsearch [$c gettags $item] $dir] != -1} {
      set dir_found 1
      set item_coords_list [GetCenterCoordsOfTag $gl_maze(cur_level) $item]
      set xc [lindex $item_coords_list 0]
      set yc [lindex $item_coords_list 1]
      set square_coords_list [GetCenterCoordsOfSquareAt $xc $yc]
      set xc [lindex $square_coords_list 0]
      set yc [lindex $square_coords_list 1]
      break
    }
  }

  # take the elevator to the next level
  if {$dir_found} {
    switch $dir {
      up   {GoUpElevatorAt   $xc $yc}
      down {GoDownElevatorAt $xc $yc}
    }
  }
}

########################################################################
proc ManBind { } {
  global gl_maze
  global gl_level

  # create man bindings for all levels
  for {set level 1} {$level <= $gl_maze(num_levels)} {incr level} {
    set c $gl_level($level)

    # grab only the man
    bind $c <ButtonPress-1>  {ManGrab %x %y}

    # drag the man
    bind $c <B1-Motion>      {ManMove %x %y}

    # go Up an elevator
    bind $c <ButtonPress-2>  {ManBoardElevator %x %y up}

    # go Down an elevator
    bind $c <ButtonPress-3>  {ManBoardElevator %x %y down}
  }

  # also bind the man with the keyboard
  ManKeyBind
}

########################################################################
proc ManUnBind { } {
  global gl
  global gl_maze
  global gl_level

  # destroy man bindings for all levels
  for {set level 1} {$level <= $gl_maze(num_levels)} {incr level} {
    set c $gl_level($level)

    bind $c <ButtonPress-1>  {}
    bind $c <B1-Motion>      {}
    bind $c <ButtonPress-2>  {}
    bind $c <ButtonPress-3>  {}
  }

  # also unbind the man from the keyboard
  ManKeyUnBind
}

########################################################################
proc ManKeyBind {} {
  global gl_key

  # move the man in 2D
  bind . <KeyPress-$gl_key(move_up)>    {ManKeyMove up}
  bind . <KeyPress-$gl_key(move_down)>  {ManKeyMove down}
  bind . <KeyPress-$gl_key(move_left)>  {ManKeyMove left}
  bind . <KeyPress-$gl_key(move_right)> {ManKeyMove right}

  # move the man in 3D
  bind . <KeyPress-$gl_key(move_up_elevator)>  \
         {ManKeyBoardElevator up}
  bind . <KeyPress-$gl_key(move_down_elevator)> \
         {ManKeyBoardElevator down}
}

########################################################################
proc ManKeyUnBind { } {
  global gl_key

  # unbind the 2D motion
  bind . <KeyPress-$gl_key(move_up)>    {}
  bind . <KeyPress-$gl_key(move_down)>  {}
  bind . <KeyPress-$gl_key(move_left)>  {}
  bind . <KeyPress-$gl_key(move_right)> {}

  # unbind the 3D motion
  bind . <KeyPress-$gl_key(move_up_elevator)>   {}
  bind . <KeyPress-$gl_key(move_down_elevator)> {}
}

########################################################################
proc LevelBind { } {

  # also bind the level to the keys
  LevelKeyBind
}

########################################################################
proc LevelKeyBind { } {
  global gl_key
  global gl_maze

  .cur_level_s configure -state normal

  # use the w x keys to view the level above(w) or below(x) the current
  bind . <KeyPress-$gl_key(view_up_level)> \
         {ShowLevel [expr $gl_maze(cur_level) + 1]}
  bind . <KeyPress-$gl_key(view_down_level)> \
         {if {$gl_maze(cur_level) > 1} \
         {ShowLevel [expr $gl_maze(cur_level) - 1]}}
}

########################################################################
proc LevelUnBind { } {

  # also unbind the levels from the keyboard
  LevelKeyUnBind
}

########################################################################
proc LevelKeyUnBind { } {
  global gl_key

  .cur_level_s configure -state disabled

  bind . <KeyPress-$gl_key(view_up_level)>     {}
  bind . <KeyPress-$gl_key(view_down_level)>   {}
}

########################################################################
proc ItemStartDrag {c x y x_grid y_grid drag_tag} {
  global gl
  global gl_maze

  # if applicable, snap the drag item to the grid
  scan $x_grid %d x_spacing
  scan $y_grid %d y_spacing
  if {$x_spacing > 0 || $y_spacing > 0} {
    set tag_coords_list [GetCenterCoordsOfTag $gl_maze(cur_level) $drag_tag]
    set xc [lindex $tag_coords_list 0]
    set yc [lindex $tag_coords_list 1]

    set delta_x [expr [$c canvasx $xc $x_grid] - $xc]
    set delta_y [expr [$c canvasy $yc $y_grid] - $yc]
    $c move $drag_tag $delta_x $delta_y
  }

  set gl(lastX) [$c canvasx $x $x_grid]
  set gl(lastY) [$c canvasy $y $y_grid]
}

########################################################################
proc ItemBoundedDrag {c x y x_grid y_grid drag_tag boundary_tag} {
  global gl

  # only drag the $drag_tag
  if {[lsearch [$c gettags $drag_tag] $drag_tag] == -1} {return}

  set x [$c canvasx $x $x_grid]
  set y [$c canvasy $y $y_grid]

  # current bounding box of $drag_tag
  set bbox_list [$c bbox $drag_tag]
  set x1 [lindex $bbox_list 0]
  set y1 [lindex $bbox_list 1]
  set x2 [lindex $bbox_list 2]
  set y2 [lindex $bbox_list 3]

  set delta_x [expr $x-$gl(lastX)]
  set delta_y [expr $y-$gl(lastY)]

  # these flags are used for rubber banding the move
  set move_in_x 0
  if {$delta_x > 0 && $x > $gl(lastX)} {set move_in_x 1}
  if {$delta_x < 0 && $x < $gl(lastX)} {set move_in_x 1}
  set move_in_y 0
  if {$delta_y > 0 && $y > $gl(lastY)} {set move_in_y 1}
  if {$delta_y < 0 && $y < $gl(lastY)} {set move_in_y 1}

  # assuming the $drag_tag is not touching a $boundary_tag
  if {$delta_x > 0} {
    set req_x1 $x1
    set req_x2 [expr $x2 + $delta_x]
  } else {
    set req_x1 [expr $x1 + $delta_x]
    set req_x2 $x2
  }
  if {$delta_y > 0} {
    set req_y1 $y1
    set req_y2 [expr $y2 + $delta_y]
  } else {
    set req_y1 [expr $y1 + $delta_y]
    set req_y2 $y2
  }
  set move_list [$c find overlapping $req_x1 $req_y1 $req_x2 $req_y2]

  set not_touching 1
  foreach item $move_list {
    set tags_list [$c gettags $item]
    if {[lsearch $tags_list $boundary_tag] != -1} {
      set not_touching 0
      break
    }
  }
  
  # when the $drag_tag is NOT touching a $boundary_tag
  if {$not_touching && $move_in_x && $move_in_y} {
    set delta_x [expr $x-$gl(lastX)]
    set delta_y [expr $y-$gl(lastY)]

    $c move $drag_tag $delta_x $delta_y
    set gl(lastX) $x
    set gl(lastY) $y

    # avoid the ability to rubber band past a ghost
    foreach item $move_list {
      set tags_list [$c gettags $item]
      if {[lsearch $tags_list ghost] != -1} {
        ManGhostCollision
        break
      }
    }
    return
  }

  # assuming the $drag_tag is touching a horizontal $boundary_tag
  if {$delta_x > 0} {
    set req_x1 $x1
    set req_x2 [expr $x2 + $delta_x]
  } else {
    set req_x1 [expr $x1 + $delta_x]
    set req_x2 $x2
  }
  set req_y1 $y1
  set req_y2 $y2
  set move_list [$c find overlapping $req_x1 $req_y1 $req_x2 $req_y2]

  set not_touching 1
  foreach item $move_list {
    set tags_list [$c gettags $item]
    if {[lsearch $tags_list $boundary_tag] != -1} {
      set not_touching 0
      break
    }
  }
  
  # when the $drag_tag is touching only a horizontal $boundary_tag
  if {$not_touching && $move_in_x} {
    set delta_x [expr $x-$gl(lastX)]
    set delta_y 0

    $c move $drag_tag $delta_x $delta_y
    set gl(lastX) [expr $gl(lastX) + $delta_x]

    # avoid the ability to rubber band past a ghost
    foreach item $move_list {
      set tags_list [$c gettags $item]
      if {[lsearch $tags_list ghost] != -1} {
        ManGhostCollision
        break
      }
    }
    return
  }

  # assuming the $drag_tag is touching a vertical $boundary_tag
  set req_x1 $x1
  set req_x2 $x2
  if {$delta_y > 0} {
    set req_y1 $y1
    set req_y2 [expr $y2 + $delta_y]
  } else {
    set req_y1 [expr $y1 + $delta_y]
    set req_y2 $y2
  }
  set move_list [$c find overlapping $req_x1 $req_y1 $req_x2 $req_y2]
  
  set not_touching 1
  foreach item $move_list {
    set tags_list [$c gettags $item]
    if {[lsearch $tags_list $boundary_tag] != -1} {
      set not_touching 0
      break
    }
  }
  
  # when the $drag_tag is touching only a vertical $boundary_tag
  if {$not_touching && $move_in_y} {
    set delta_x 0
    set delta_y [expr $y-$gl(lastY)]

    $c move $drag_tag $delta_x $delta_y
    set gl(lastY) [expr $gl(lastY) + $delta_y]

    # avoid the ability to rubber band past a ghost
    foreach item $move_list {
      set tags_list [$c gettags $item]
      if {[lsearch $tags_list ghost] != -1} {
        ManGhostCollision
        break
      }
    }
    return
  }

}

########################################################################
proc TakeExpressElevatorAt {dir x y} {
  global gl
  global gl_level
  global gl_maze

  # delete the man from the current level
  $gl_level($gl_maze(cur_level)) delete man

  # show the next level
  ShowLevel [expr $gl_maze(cur_level) + $dir]

  # create the man on the next level
  CreateManAt $gl_maze(cur_level) $x $y
}

########################################################################
proc GoUpElevatorAt {x y} {
  TakeExpressElevatorAt 1 $x $y
}

########################################################################
proc GoDownElevatorAt {x y} {
  TakeExpressElevatorAt -1 $x $y
}

########################################################################
proc CreateManAt {level x y {id 0} {lower 0}} {
  global gl
  global gl_level
  global gl_install_dir

  # get the maze level
  set c $gl_level($level)

  # create the man
  source $gl_install_dir/man/man$id.draw

  # scale the man to the appropriate size
  set bbox_list [$c bbox man]
  set x1 [lindex $bbox_list 0]
  set y1 [lindex $bbox_list 1]
  set x2 [lindex $bbox_list 2]
  set y2 [lindex $bbox_list 3]

  set x_scale [expr $gl(man_scale_factor) * $gl(square_width)/($x2 - $x1)]
  set y_scale [expr $gl(man_scale_factor) * $gl(square_height)/($y2 - $y1)]
  if {$x_scale <= $y_scale} {set xy_scale $x_scale}
  if {$x_scale >  $y_scale} {set xy_scale $y_scale}

  $c scale man 0 0 $xy_scale $xy_scale

  # move the man to x y
  set man_coords_list [GetCenterCoordsOfTag $level man]
  set xc [lindex $man_coords_list 0]
  set yc [lindex $man_coords_list 1]
  $c move man [expr $x - $xc] [expr $y - $yc]

  # display the man below elevators
  if {$lower} {catch {$c lower man elevator} err_msg}
}

########################################################################
proc CreateWallAt {level x1 y1 x2 y2} {
  global gl
  global gl_maze
  global gl_level

  # get the maze level
  set c $gl_level($level)

  # create the wall
  $c create line $x1 $y1 $x2 $y2 -tag wall \
                                 -width $gl_maze(wall_width) \
                                 -fill  $gl_maze(wall_color)
}

########################################################################
proc CreateSolutionSquareAt {level x y} {
  global gl
  global gl_maze
  global gl_level

  # get the maze level
  set c $gl_level($level)

  # what are the dimensions of the solution square
  set x1 [expr $x - $gl(square_width)/2.0]
  set y1 [expr $y - $gl(square_height)/2.0]
  set x2 [expr $x + $gl(square_width)/2.0]
  set y2 [expr $y + $gl(square_height)/2.0]

  # create the solution square
  $c create rect $x1 $y1 $x2 $y2 -tag solution -width 0 \
                                 -outline $gl_maze(background) \
                                 -fill $gl_maze(background)
}

########################################################################
proc CreateTextAt {level x y text tags justify} {
  global gl
  global gl_maze
  global gl_level

  # get the maze level
  set c $gl_level($level)

  # create the text
  $c create text $x $y -text $text -tag $tags -justify $justify
}

########################################################################
proc CreateElevatorAt {level x y elev_tag type} {
  global gl
  global gl_level
  global gl_install_dir

  # get the maze level
  set c $gl_level($level)

  # decide what type of elevator to create
  if {$type == "up_down"} {
    source $gl_install_dir/elevators/elev_up_down.draw
  } elseif {$type == "up"} {
    source $gl_install_dir/elevators/elev_up.draw
  } elseif {$type == "down"} {
    source $gl_install_dir/elevators/elev_down.draw
  } else {
    puts "Don't know elevator type $type"
    return
  }

  # scale the elevator to the appropriate size
  set bbox_list [$c bbox $elev_tag]
  set x1 [lindex $bbox_list 0]
  set y1 [lindex $bbox_list 1]
  set x2 [lindex $bbox_list 2]
  set y2 [lindex $bbox_list 3]

  set x_scale [expr $gl(elevator_scale_factor) * $gl(square_width)/($x2 - $x1)]
  set y_scale [expr $gl(elevator_scale_factor) * $gl(square_height)/($y2 - $y1)]
  if {$x_scale <= $y_scale} {set xy_scale $x_scale}
  if {$x_scale >  $y_scale} {set xy_scale $y_scale}

  $c scale $elev_tag 0 0 $xy_scale $xy_scale

  # move the elevator to x y
  set elev_coords_list [GetCenterCoordsOfTag $level $elev_tag]
  set xc [lindex $elev_coords_list 0]
  set yc [lindex $elev_coords_list 1]
  $c move $elev_tag [expr $x - $xc] [expr $y - $yc]

}

########################################################################
proc CreateUpDownElevatorAt {level x y elev_tag} {
  CreateElevatorAt $level $x $y $elev_tag up_down
}

########################################################################
proc CreateUpElevatorAt {level x y elev_tag} {
  CreateElevatorAt $level $x $y $elev_tag up
}

########################################################################
proc CreateDownElevatorAt {level x y elev_tag} {
  CreateElevatorAt $level $x $y $elev_tag down
}


########################################################################
proc CreateGhostAt {level x y id} {
  global gl
  global gl_level
  global gl_maze
  global gl_install_dir

  # get the maze level
  set c $gl_level($level)

  # create the ghost and give it a unique tag
  set ghost_tag g$id
  source $gl_install_dir/ghosts/ghost$id.draw

  # scale the ghost to the appropriate size
  set bbox_list [$c bbox $ghost_tag]
  set x1 [lindex $bbox_list 0]
  set y1 [lindex $bbox_list 1]
  set x2 [lindex $bbox_list 2]
  set y2 [lindex $bbox_list 3]

  set x_scale [expr $gl(ghost_scale_factor) * $gl(square_width)/($x2 - $x1)]
  set y_scale [expr $gl(ghost_scale_factor) * $gl(square_height)/($y2 - $y1)]
  if {$x_scale <= $y_scale} {set xy_scale $x_scale}
  if {$x_scale >  $y_scale} {set xy_scale $y_scale}

  $c scale $ghost_tag 0 0 $xy_scale $xy_scale

  # position the ghost at x y
  set ghost_coords_list [GetCenterCoordsOfTag $level $ghost_tag]
  set xc [lindex $ghost_coords_list 0]
  set yc [lindex $ghost_coords_list 1]
  $c move $ghost_tag [expr $x - $xc] [expr $y - $yc]

# catch {$c lower $ghost_tag elevator} err_msg
  catch {$c raise $ghost_tag man} err_msg
  catch {$c lower $ghost_tag title} err_msg
}

########################################################################
proc GhostBoardElevator {level id} {
  global gl
  global gl_level

  set ghost_tag g$id
  set c $gl_level($level)

  catch {$c lower $ghost_tag elevator} err_msg
}

########################################################################
proc MoveGhostTo {level x y id} {
  global gl
  global gl_level

  set ghost_tag g$id
  set c $gl_level($level)

  set ghost_coords_list [GetCenterCoordsOfTag $level $ghost_tag]
  set xc [lindex $ghost_coords_list 0]
  set yc [lindex $ghost_coords_list 1]

  $c move $ghost_tag [expr $x - $xc] [expr $y - $yc]

  CheckGhostCollideWithManOrTrail $level $ghost_tag
}

########################################################################
proc DeleteGhost {level id} {
  global gl
  global gl_level

  set ghost_tag g$id
  set c $gl_level($level)
  $c delete $ghost_tag
}

########################################################################
proc AdvanceGhostsForever { } {
  global gl

  if {!$gl(game_over) && !$gl(game_paused)} {
    AdvanceGhostsOneStep_CProc
    set delay [expr $gl(game_speed) * 100]
    after $delay AdvanceGhostsForever
  }
}

########################################################################
proc PracticeMaze { } {
  global gl
  global gl_level
  global gl_maze
  global gl_options
  global gl_competition

  # if there is a current game, end it
  set gl_competition(cur_stage) 0
  set gl(game_over) 1

  # remove levels from site
  ShowLevel 0
  LevelUnBind

  # enable the "Show Solution Paths" option to appear
  .menu.opt.m enable "Show Solution Paths"

  # manage the controls for the practice maze
  InitializePracticeControls $gl_maze(num_levels)

  # remove bindings
  ManUnBind
  LevelUnBind

  # create a new maze
  NewMaze

  # begin a new maze on level 1
  ShowLevel 1

  # make sure the man start dragging from the start
  set man_coords_list [GetCenterCoordsOfTag $gl_maze(cur_level) man]
  set xc [lindex $man_coords_list 0]
  set yc [lindex $man_coords_list 1]
  ItemStartDrag $gl_level($gl_maze(cur_level)) $xc $yc 0 0 man

  OptionsLeaveTrail   $gl_options(leave_trail)
  OptionsShowSolution $gl_options(show_solution)
  OptionsSnapMotion   $gl_options(snap_motion)

  # Show all the maze options
# MazeOptions
}

########################################################################
proc CompetitionMaze { } {
  global gl
  global gl_competition
  global gl_install_dir
  global gl_options

  # if there is a current game, end it
  ResetBonus; ResetScoreTo 0
  set gl_competition(cur_stage) 0
  set gl(game_over) 1
  if {[winfo exists .maze_opt]} {destroy .maze_opt}

  # disable the "Show Solution Paths" option to appear
  set gl_options(show_solution) 0
  .menu.opt.m disable "Show Solution Paths"

  # remove bindings
  ManUnBind
  LevelUnBind

  # load the competition course
  source $gl_install_dir/scripts/competition.tcl

  # how many stages are in the competition table?
  set gl_competition(num_stages) [llength $gl_competition(table)]

  # Start the competition
  NextCompetitionMaze
}

########################################################################
proc NextCompetitionMaze { } {
  global gl
  global gl_competition
  global gl_maze
  global gl_options

  # is Competition is over?
  if {$gl_competition(cur_stage) == $gl_competition(num_stages)} {
    CompetitionOver
    return
  }

  # load the new global maze parameters
  # Note: gl_competition(cur_stage) is automatically updated from the table
  set column 0
  set stage_paramters [lindex $gl_competition(table) $gl_competition(cur_stage)]
  foreach gl_parameter $gl_competition(columns) {
    set paramter [lindex $stage_paramters $column]
    set $gl_parameter $paramter
    incr column
  }

  # set up the competition constrols
  InitializeCompetitionControls $gl_maze(num_levels)

  # generate a new maze
  NewMaze

  # begin a new maze on level 1
  ShowLevel 1

  OptionsLeaveTrail   $gl_options(leave_trail)
# OptionsShowSolution $gl_options(show_solution)
  OptionsSnapMotion   $gl_options(snap_motion)

}

########################################################################
proc ShowTop10Display { } {

  # if the .top10 already exists, do nothing
  set w .top10
  if [winfo exists $w] {
      wm deiconify $w
      raise $w
      return
  }

  # create a toplevel
  toplevel $w
  wm title $w "Top 10 Maze Men"
  wm withdraw $w

  # create the frame widgets
  frame $w.top    -borderwidth 1 -relief raised
  frame $w.bottom -borderwidth 1 -relief raised
  pack  $w.top    -side top -fill both -expand yes
  pack  $w.bottom -side bottom -fill x

  frame $w.top.title_f
  frame $w.top.name_score_f
  frame $w.top.name_score_f.name_f
  frame $w.top.name_score_f.score_f
  pack $w.top.title_f -side top -fill x
  pack $w.top.name_score_f -side top -expand yes -fill both
  pack $w.top.name_score_f.name_f -side left -fill both -padx 3m -ipady 1m
  pack $w.top.name_score_f.score_f -side right -fill both -padx 3m -ipady 1m

  # create the fun widgets
  set font [GetNearestFontFromPixels helvetica bold o 20]
  label $w.title	-text "Top 10 Maze Men" \
			-relief ridge -borderwidth  4 -font $font

  label $w.name		-text "Names:" -relief flat
  label $w.score	-text "Score:" -relief flat

  for {set num 0} {$num < 10} {incr num} {
    label $w.name_$num	-text $num -relief flat -borderwidth  2
    label $w.score_$num	-text $num -relief flat -borderwidth  2
  }
  
  button $w.ok		-text "Done" \
			-relief raised -borderwidth 2 \
			-command "destroy $w"

  # pack the widgets
  pack $w.title		-in $w.top.title_f -side top -fill x \
			-padx 3m -pady 3m \
			-ipadx 2m -ipady 1m
  pack $w.name		-in $w.top.name_score_f.name_f -side top -anchor w
  pack $w.score		-in $w.top.name_score_f.score_f -side top -anchor w
  for {set num 0} {$num < 10} {incr num} {
    pack $w.name_$num	-in $w.top.name_score_f.name_f -side top -anchor w
    pack $w.score_$num	-in $w.top.name_score_f.score_f -side top -anchor w
  }
  pack $w.ok		-in $w.bottom \
			-padx 3m -pady 3m \
			-ipadx 2m -ipady 1m
  CenterWindow $w .
  UpdateTop10Display
}

########################################################################
proc UpdateTop10Display { } {
  global gl
  global gl_top10

  # if the .top10 does not exist, do nothing
  set w .top10
  if {![winfo exists $w]} return

  # read the names and scores lists
  ReadTop10Lists

  # set the names and scores from the top 10 lists
  for {set num 0} {$num < 10} {incr num} {
    set name  [lindex $gl_top10(names_list) $num]
    set score [lindex $gl_top10(scores_list) $num]

    $w.name_$num  configure -text "[expr $num + 1].  $name"
    $w.score_$num configure -text $score
  }

  # bring the window to the top to see the new top 10
  raise $w
}

########################################################################
proc CompetitionOver { } {
  global gl
  global gl_maze
  global gl_level

  # show a Competition Over on each level
  set font [GetNearestFontFromPixels helvetica bold o 36]
  for {set level 1} {$level <= $gl_maze(num_levels)} {incr level} {
    set c $gl_level($level)
    $c create text [expr $gl_maze(width) / 2] [expr $gl_maze(height) / 2] \
         -text "Competition Over" -tags title -font $font
  }

  # check for a top 10 score
  CheckTop10 $gl(score)
}

########################################################################
proc CheckTop10 {new_score} {
  global gl
  global gl_top10

  # who are in the top 10?
  ReadTop10Lists

  # is the current score in the top 10? if so where?
  set num 0
  foreach score $gl_top10(scores_list) {
    if {$new_score > $score} break
    incr num
  }

  # not a top 10 score?
  if {$num >= 10} return

  set msg   "Congratulations, you have scored \
           \namong the Top 10 Maze Men! \
         \n\nWhat is your name?"

  # get the new top 10 name
  set new_name [EntryDialog . .who "New Top 10 Name" \
               $msg {} 1 {question red white} {Ok} 1]
  set new_name [string trim $new_name]
  if {[string length $new_name] == 0} return

  # insert the new top 10 name and score into the lists and write them
  set gl_top10(names_list)  [linsert $gl_top10(names_list)  $num $new_name]
  set gl_top10(scores_list) [linsert $gl_top10(scores_list) $num $new_score]
  set gl_top10(names_list)  [lrange $gl_top10(names_list)  0 9]
  set gl_top10(scores_list) [lrange $gl_top10(scores_list) 0 9]

  # save the top 10 to file
  WriteTop10Lists

  # reflect the new top 10 in the top 10 display abd show it
  UpdateTop10Display
  ShowTop10Display
}

########################################################################
proc ReadTop10Lists { } {
  global gl
  global gl_top10
  global gl_install_dir

  # if the Top 10 files don't exist, don't try to open them
  if {![file exists $gl_install_dir/scripts/top10names] ||
      ![file exists $gl_install_dir/scripts/top10scores]} {
    for {set num 0} {$num < 10} {incr num} {
      lappend gl_top10(names_list)  {}
      lappend gl_top10(scores_list) {}
    }
    return
  }

  # open the Top 10 files
  set names_file_id  [open $gl_install_dir/scripts/top10names  r]
  set scores_file_id [open $gl_install_dir/scripts/top10scores r]

  # load the gl_top10(names_list) and gl_top10(scores_list)
  gets $names_file_id  comment_line
  gets $scores_file_id comment_line
  set gl_top10(names_list)  {}
  set gl_top10(scores_list) {}
  set num 0
  while {[gets $names_file_id name] != -1  &&  $num < 10} {
    gets $scores_file_id score

    lappend gl_top10(names_list) $name
    lappend gl_top10(scores_list) $score
    incr num
  }
  while {$num < 10} {
    lappend gl_top10(names_list)  {}
    lappend gl_top10(scores_list) {}
    incr num
  }

  # close the Top 10 files
  close $names_file_id
  close $scores_file_id
}

########################################################################
proc WriteTop10Lists { } {
  global gl
  global gl_top10
  global gl_install_dir

  # open the Top 10 files
  set names_file_id  [open $gl_install_dir/scripts/top10names  w]
  set scores_file_id [open $gl_install_dir/scripts/top10scores w]

  # write the gl_top10(names_list) and gl_top10(scores_list)
  puts $names_file_id  "# Top 10 Maze Men"
  puts $scores_file_id "# Top 10 Maze Scores"
  foreach name  $gl_top10(names_list)  {puts $names_file_id $name}
  foreach score $gl_top10(scores_list) {puts $scores_file_id $score}

  # close the Top 10 files
  close $names_file_id
  close $scores_file_id

  # make the top 10 files totally readable and writable
  if {[file owned $gl_install_dir/scripts/top10names]} {
    exec chmod a+rw $gl_install_dir/scripts/top10names
  }
  if {[file owned $gl_install_dir/scripts/top10scores]} {
    exec chmod a+rw $gl_install_dir/scripts/top10scores
  }
}


########################################################################
proc NewMaze { } {
  global gl
  global gl_level
  global gl_maze

  # show a busy cursor
  ShowBusyCursor

  # before creating a new maze, pause while the man is being shocked
  if {$gl(man_ghost_collision)} {
     update idletasks
     set delay [expr (1 + $gl(num_man_ghost_collision_frames)) * \
                          $gl(man_ghost_collision_delay)]
     after $delay
  }

  # remove all the levels
  if {$gl_maze(id) != 0} {
    LevelBind
    ShowLevel 0
    LevelUnBind
  }

  # erase the level canvases
  for {set level 1} {$level <= $gl_maze(num_levels_max)} {incr level} {
    set item_list [$gl_level($level) find all]
    foreach item $item_list {$gl_level($level) delete $item}
  }

  # what are the square dimensions for this maze
  set gl(new_maze_initialized) 1
  set gl(square_width)  [expr double($gl_maze(width))/double($gl_maze(size))]
  set gl(square_height) [expr double($gl_maze(height))/double($gl_maze(size))]

  # External C procedure call to create a new maze
  NewMaze_CProc $gl_maze(width) $gl_maze(height) $gl_maze(num_levels) \
                $gl_maze(size) $gl_maze(size) \
                $gl_maze(num_elevators) $gl_maze(num_loops)

  # External C procedure call to create add ghosts to a maze
  for {set gho 1} {$gho <= $gl_maze(num_ghosts)} {incr gho} {

    # linearly distrubute the speed of the ghosts
    set ghost_speed \
      [expr $gl_maze(ghost_speed_max) + $gl_maze(ghost_speed_min) - \
            $gl_maze(ghost_speed)]
    set speed_inc \
      [expr double($gl_maze(ghost_speed_max) - $ghost_speed)/ \
            double($gl_maze(num_ghosts))]
    set ghost_speed [expr $gl_maze(ghost_speed_max) - round($gho * $speed_inc)]
    AddGhost_CProc $ghost_speed
  }

  # increment the current maze id
  incr gl_maze(id)

  # remove busy cursor
  RemoveBusyCursor
}

########################################################################
proc MazeOptions { } {
  global gl
  global gl_maze

  # if the .maze_opt already exists, start a practice game
  set w .maze_opt
  if [winfo exists $w] {
      wm deiconify $w
      raise $w
      return
  }

  # create a toplevel
  toplevel $w
  wm title $w "Maze Options"
  wm withdraw $w

  # create the frame widgets
  frame $w.top    -borderwidth 1 -relief raised
  frame $w.bottom -borderwidth 1 -relief raised
  pack  $w.top    -side top -fill both -expand yes
  pack  $w.bottom -side bottom -fill x

  # create the fun widgets
  set font [GetNearestFontFromPixels helvetica bold o 20]
  label $w.title	 -text "Maze Options" \
			 -relief ridge -borderwidth  4 -font $font
  scale $w.size   -label "Maze Dimensions (n by n)" \
			 -orient horizontal \
			 -relief sunken -borderwidth  2 \
			 -from $gl_maze(size_min) \
			 -to $gl_maze(size_max) \
			 -tickinterval 1 \
			 -showvalue no \
		         -command "uplevel #0 set gl_maze(size)"
        $w.size set $gl_maze(size)
  scale $w.levels -label "Number of Levels" \
			 -orient horizontal \
			 -relief sunken -borderwidth 2 \
			 -from $gl_maze(num_levels_min) \
			 -to $gl_maze(num_levels_max) \
			 -tickinterval 1 \
			 -showvalue no \
		         -command "uplevel #0 set gl_maze(num_levels)"
        $w.levels set $gl_maze(num_levels)
  scale $w.elevators \
			 -label "Maximum Number of Elevators between Levels" \
			 -orient horizontal \
			 -relief sunken -borderwidth 2 \
			 -from $gl_maze(num_elevators_min) \
			 -to $gl_maze(num_elevators_max) \
			 -tickinterval 1 \
			 -showvalue no \
		         -command "uplevel #0 set gl_maze(num_elevators)"
        $w.elevators set $gl_maze(num_elevators)
  scale $w.loops \
			 -label "Number of Loops per Level" \
			 -orient horizontal \
			 -relief sunken -borderwidth 2 \
			 -from $gl_maze(num_loops_min) \
			 -to $gl_maze(num_loops_max) \
			 -tickinterval 1 \
			 -showvalue no \
		         -command "uplevel #0 set gl_maze(num_loops)"
        $w.loops set $gl_maze(num_loops)
  scale $w.ghosts -label "Number of Ghosts" \
			 -orient horizontal \
			 -relief sunken -borderwidth  2 \
			 -from $gl_maze(num_ghosts_min) \
			 -to $gl_maze(num_ghosts_max) \
			 -tickinterval 1 \
			 -showvalue no \
		         -command "uplevel #0 set gl_maze(num_ghosts)"
        $w.ghosts set $gl_maze(num_ghosts)
  scale $w.speed  -label "Maximum Ghost Speed" \
			 -orient horizontal \
			 -relief sunken -borderwidth 2 \
			 -from $gl_maze(ghost_speed_min) \
			 -to $gl_maze(ghost_speed_max) \
			 -tickinterval 1 \
			 -showvalue no \
		         -command "uplevel #0 set gl_maze(ghost_speed)"
        $w.speed set $gl_maze(ghost_speed)



  message $w.msg  -width 100m \
                         -text "Note: Changes to the maze options will not\
				take affect until a \"Practice\" game begins."
  

  button   $w.ok  -text "Practice" -relief raised -borderwidth 2 \
                           -command "raise .; PracticeMaze"

  # pack the widgets
  pack $w.title	    -in $w.top -side top -fill x \
                    -padx 3m -pady 3m -ipadx 2m -ipady 1m
  pack $w.size      -in $w.top -side top -fill x
  pack $w.levels    -in $w.top -side top -fill x
  pack $w.elevators -in $w.top -side top -fill x
  pack $w.loops     -in $w.top -side top -fill x
  pack $w.ghosts    -in $w.top -side top -fill x
  pack $w.speed     -in $w.top -side top -fill x
  pack $w.msg       -in $w.top -side top -fill x -ipadx 2m -ipady 1m
  pack $w.ok        -in $w.bottom \
                    -padx 3m -pady 3m -ipadx 2m -ipady 1m

  CenterWindow $w .
}

########################################################################
proc ChangeGameSpeed { } {
  global gl_install_dir
  global gl
  global gl_maze

  # if the .game_speed already exists, do nothing
  set w .game_speed
  if [winfo exists $w] {
      wm deiconify $w
      raise $w
      return
  }

  # create a toplevel
  toplevel $w
  wm title $w "Game Speed"
  wm withdraw $w

  # create the frame widgets
  frame $w.top    -borderwidth 1 -relief raised
  frame $w.bottom -borderwidth 1 -relief raised
  pack  $w.top    -side top -fill both -expand yes
  pack  $w.bottom -side bottom -fill x

  # create the fun widgets
  set font [GetNearestFontFromPixels helvetica bold o 20]
  label $w.title	-text "Game Speed" \
			-relief ridge -borderwidth  4 -font $font
  frame $w.top.scale_f    -borderwidth 0 -relief flat
  set font [GetNearestFontFromPixels helvetica medium r 20]
  label $w.left_arrow	-bitmap @$gl_install_dir/bitmaps/left_arrow.btm \
			-relief flat
  label $w.faster	-text "Faster" -relief flat -font $font
  label $w.right_arrow	-bitmap @$gl_install_dir/bitmaps/right_arrow.btm \
			-relief flat
  label $w.slower	-text "Slower" -relief flat -font $font
  scale $w.speed  	-orient horizontal \
			-relief flat -borderwidth 2 \
			-from 1 \
			-to 10 \
			-showvalue no \
		        -command {SetGameSpeed}
        $w.speed set $gl(game_speed)



  message $w.msg  -width 100m \
    -text "Note: Game speed is actually a measure of the time delay\
           between ghost steps.  Unfortunately, the delay also affects\
           how fast you can move maze man.  The optimal game speed\
           depends on your computer."
  

  button   $w.ok  -text "Done" -relief raised -borderwidth 2 \
		    -command "destroy $w"

  # pack the widgets
  pack $w.title	    -in $w.top -side top -fill x \
                    -padx 3m -pady 3m -ipadx 2m -ipady 1m
  pack $w.top.scale_f    -in $w.top -side top -fill x -padx 3m
  pack $w.left_arrow     -in $w.top.scale_f -side left
  pack $w.faster    -in $w.top.scale_f -side left
  pack $w.speed     -in $w.top.scale_f -side left -fill x -expand yes
  pack $w.slower    -in $w.top.scale_f -side left
  pack $w.right_arrow    -in $w.top.scale_f -side left
  pack $w.msg       -in $w.top -side top -fill x -ipadx 2m -ipady 1m
  pack $w.ok        -in $w.bottom \
                    -padx 3m -pady 3m \
                    -ipadx 2m -ipady 1m

  CenterWindow $w .
}

########################################################################
proc SetGameSpeed {speed} {
  global gl

  # set it immediately
  uplevel #0 set gl(game_speed) $speed

  # also save it to file
  SaveGameSpeed $speed
}

########################################################################
proc LoadGameSpeed { } {
  global gl_install_dir
  global gl

  if {[file exists $gl_install_dir/scripts/gamespeed.tcl]} {
    source $gl_install_dir/scripts/gamespeed.tcl
  }
}

########################################################################
proc SaveGameSpeed {speed} {
  global gl_install_dir

  # open the gamespeed file
  set speed_file_id [open $gl_install_dir/scripts/gamespeed.tcl w]

  # write the current game speed
  puts $speed_file_id  "# Maze Man game speed"
  puts $speed_file_id  "set gl(game_speed) $speed"

  # close the game speed file
  close $speed_file_id

  # make the gamespeed file totally readable and writable
  if {[file owned $gl_install_dir/scripts/gamespeed.tcl]} {
    exec chmod a+rw $gl_install_dir/scripts/gamespeed.tcl
  }
}


########################################################################
proc GetCenterCoordsOfTag {level tag} {
  global gl
  global gl_level

  # current bounding box of $tag
  set bbox_list [$gl_level($level) bbox $tag]
  if {[llength $bbox_list] == 0} return
  set x1 [lindex $bbox_list 0]
  set y1 [lindex $bbox_list 1]
  set x2 [lindex $bbox_list 2]
  set y2 [lindex $bbox_list 3]

  # coordinate list {x y} of the $tag
  lappend coord_list [expr ($x1 + $x2)/2] [expr ($y1 + $y2)/2]

  return $coord_list
}

########################################################################
proc GetCenterCoordsOfSquareAt {x y} {
  global gl

  # find the center of the square
  set xc [expr $gl(square_width)  * (int($x / $gl(square_width))  + 0.5)]
  set yc [expr $gl(square_height) * (int($y / $gl(square_height)) + 0.5)]

  # coordinate list {x y} of the center of the square
  lappend coord_list $xc $yc

  return $coord_list
}

########################################################################
proc GetCoordsOfSquareAt {x y} {
  global gl

  # find the center of the square
  set x1 [expr $gl(square_width)  * int($x / $gl(square_width))]
  set y1 [expr $gl(square_height) * int($y / $gl(square_height))]
  set x2 [expr $gl(square_width)  * (1 + int($x / $gl(square_width)))]
  set y2 [expr $gl(square_height) * (1 + int($y / $gl(square_height)))]

  # coordinate list {x1 y1 x2 y2} of the square
  lappend coord_list $x1 $y1 $x2 $y2

  return $coord_list
}

########################################################################
proc GetOverlappingListOfSquareAt {x y} {
  global gl
  global gl_level

  set c $gl_level($gl_maze(cur_level))

  # get the coordinates of the current square
  set square_coords_list [GetCoordsOfSquareAt $x $y]
  set x1 [lindex $square_coords_list 0]
  set y1 [lindex $square_coords_list 1]
  set x2 [lindex $square_coords_list 2]
  set y2 [lindex $square_coords_list 3]

  # get a list of the items overlapping this square
  set overlappping_list [$c find overlapping $x1 $y1 $x2 $y2]

  return $overlappping_list
}

########################################################################
proc GetOverlappingListOfElevatorAt {x y} {
  global gl
  global gl_level
  global gl_maze

  set c $gl_level($gl_maze(cur_level))

  set elevator_tag [GetElevatorTagAt $gl_maze(cur_level) $x $y]

  # get the bounding box of the elevator
  set bbox_list [$c bbox $elevator_tag]
  if {[llength $bbox_list] == 0} return 
  set x1 [lindex $bbox_list 0]
  set y1 [lindex $bbox_list 1]
  set x2 [lindex $bbox_list 2]
  set y2 [lindex $bbox_list 3]

  # get a list of the items overlapping this elevator
  set overlappping_list [$c find overlapping $x1 $y1 $x2 $y2]

  return $overlappping_list
}

########################################################################
proc GetOverlappingListOfTag {level tag} {
  global gl
  global gl_level

  set c $gl_level($level)

  # get the bounding box of the tag
  set bbox_list [$c bbox $tag]
  if {[llength $bbox_list] == 0} return 
  set x1 [lindex $bbox_list 0]
  set y1 [lindex $bbox_list 1]
  set x2 [lindex $bbox_list 2]
  set y2 [lindex $bbox_list 3]

  # get a list of the items overlapping the tag
  set overlappping_list [$c find overlapping $x1 $y1 $x2 $y2]

  return $overlappping_list
}

########################################################################
proc GetOverlappingListOfMan { } {
  global gl
  global gl_maze

  return [GetOverlappingListOfTag $gl_maze(cur_level) man]
}

########################################################################
proc GetColumnAt {x} {
  global gl

  return [expr (1 + int($x / $gl(square_width)))]
}

########################################################################
proc GetRowAt {y} {
  global gl

  return [expr (1 + int($y / $gl(square_height)))]
}

########################################################################
proc GetElevatorTagAt {level x y} {
  append tag l $level _ r [GetRowAt $y] _ c [GetColumnAt $x]

  return $tag
}

########################################################################
proc CheckGhostCollideWithManOrTrail {level ghost_tag} {
  global gl gl_level

  set c $gl_level($level)
  set overlappping_list [GetOverlappingListOfTag $level $ghost_tag]

  # is the ghost positioned over the man?
  foreach item $overlappping_list {
    if {[lsearch [$c gettags $item] man] != -1} {
      ManGhostCollision
      break
    }
    if {[lsearch [$c gettags $item] trail] != -1} {
      $c delete $item
    }
  }
}

########################################################################
proc CheckManCollideWithGhostOrFinish { } {
  global gl gl_level gl_maze

  set c $gl_level($gl_maze(cur_level))
  set overlappping_list [GetOverlappingListOfMan]

  foreach item $overlappping_list {
    set tags_list [$c gettags $item]
    if {[lsearch $tags_list ghost] != -1} {
      ManGhostCollision
      return 1
    }
    if {[lsearch $tags_list finish] != -1} {
      ManFinished
      return 1
    }
  }

  # no collision or finish
  return 0
}



########################################################################
proc ManGhostCollision { } {
  global gl gl_level
  
  # put a lock on this proceedure
  if {$gl(man_ghost_collision)} return
  set  gl(man_ghost_collision) 1

  ManUnBind

  set level [GetManLevel]
  set man_coords_list [GetCenterCoordsOfTag $level man]
  set xc [lindex $man_coords_list 0]
  set yc [lindex $man_coords_list 1]

  for {set inx 1} {$inx <= $gl(num_man_ghost_collision_frames)} {incr inx} {
    after [expr ($inx - 1) * $gl(man_ghost_collision_delay)] " \
      IncrementScoreBy -1; \
      $gl_level($level) delete man; \
      CreateManAt $level $xc $yc $inx 0"
  }

  after [expr $inx * $gl(man_ghost_collision_delay)] " \
    $gl_level($level) delete man; \
    CreateManAt $level $xc $yc 0 0; \
    set gl(man_ghost_collision) 0; \
    ManBind"

}


########################################################################
proc GetManLevel { } {
  global gl gl_maze gl_level

  for {set level 1} {$level <= $gl_maze(num_levels)} {incr level} {
    set c $gl_level($level)
    set item_list [$c find all]
    foreach item $item_list {
      if {[lsearch [$c gettags $item] man] != -1} {return $level}
    }
  }

  return 0
}

########################################################################
proc ManFinished { } {
  global gl
  global gl_level
  global gl_maze
  global gl_competition

  # end the game
  set gl(game_over) 1
  ManUnBind
  LevelUnBind
  .start_stop_b configure -state disabled
  after $gl(game_rest_delay) CreateIntroductionMaze $gl_maze(id)

  # replace the current man with a finish man
  set level $gl_maze(cur_level)
  set finish_coords_list [GetCenterCoordsOfTag $level finish]
  set xc [lindex $finish_coords_list 0]
  set yc [lindex $finish_coords_list 1]
  $gl_level($level) delete man
  CreateManAt $level $xc $yc 10

  # are we in the middle of a competition?
  if {$gl_competition(cur_stage)} {
     AddBonusToScore
     update idletasks
     set delay [expr (1 + $gl(num_man_ghost_collision_frames)) * \
                          $gl(man_ghost_collision_delay)]
     after $delay
     NextCompetitionMaze
     ManUnBind
     LevelUnBind
  } else {
    # show a Practice Over on each level
    set font [GetNearestFontFromPixels helvetica bold o 36]
    for {set level 1} {$level <= $gl_maze(num_levels)} {incr level} {
      set c $gl_level($level)
      $c create text [expr $gl_maze(width)/2] [expr $gl_maze(height)/2] \
         -text "Practice Over" -tags title -font $font
    }
  }
}

########################################################################
proc OptionsShowSolution {bool} {
  global gl
  global gl_maze
  global gl_options
  global gl_level

  # for each of the maze levels...
  for {set level 1} {$level <= $gl_maze(num_levels)} {incr level} {
    if {$bool} {
      # show the solution squares
      $gl_level($level) itemconfigure solution \
			-outline $gl_options(show_solution_color) \
			-fill $gl_options(show_solution_color)
    } else {
      # hide the solution squares
      $gl_level($level) itemconfigure solution \
			-outline $gl_maze(background) \
			-fill $gl_maze(background)
    }
  }
}

########################################################################
proc OptionsSnapMotion {bool} {
  global gl
  global gl_maze
  global gl_options

  # do not proceed unles a new maze has been initialized
  if {$gl(new_maze_initialized) != 1} {return}

  # for each of the maze levels...
  for {set level 1} {$level <= $gl_maze(num_levels)} {incr level} {
    if {$bool} {
      set gl_options(snap_motion_x) [expr double($gl(square_width))/2.0]
      set gl_options(snap_motion_y) [expr double($gl(square_height))/2.0]
    } else {
      set gl_options(snap_motion_x) 0
      set gl_options(snap_motion_y) 0
    }
  }
}

########################################################################
proc OptionsLeaveTrail {bool} {
  global gl
  global gl_options
  global gl_maze
  global gl_level

  if {$bool} {
    set c $gl_level($gl_maze(cur_level))
    set man_coords_list [GetCenterCoordsOfTag $gl_maze(cur_level) man]
    if {[llength $man_coords_list] == 0} return

    set xc [lindex $man_coords_list 0]
    set yc [lindex $man_coords_list 1]

    set x1 [expr $xc - 2]
    set y1 [expr $yc - 2]
    set x2 [expr $xc + 2]
    set y2 [expr $yc + 2]

    set id [$c create rect $x1 $y1 $x2 $y2 \
		 -tag trail -fill $gl_options(leave_trail_color)]
    $c lower $id man
#   $c lower $id ghost
  }
}

########################################################################
proc QuitMaze { } {

  QuitDialog
}


# initialize the game and show an introductory maze
InitializeMazeMan
CreateIntroductionMaze $gl_maze(id)
