/*
 * flash.c
 *
 * Some utility functions to implement flashing icons.
 *
 * Copyright (c) 1994
 *
 * J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "tkined.h"

typedef struct fli{
    char *id;
    struct fli *next;
} fli;

static fli *flashlist = NULL;

/*
 * This callback flashes every object in the flashlist and calls
 * it again if there are any jobs left.
 */

static void
doflash (clientData)
    ClientData clientData;
{
    Tcl_Interp *interp = (Tcl_Interp *) clientData;
    tkined_object *object;
    char *color;
    fli *p;

    int max = 0;

    for (p = flashlist; p != NULL; p = p->next) {

	if (p->id == NULL) continue;

	object = id_to_object (p->id);
	if (object == NULL) continue;

	if (! object->editor->color) {
	    if ((object->flash) % 2) {
		color = "black";
	    } else {
		color = "white";
	    }
	} else {
	    color = object->color;
	    if ((object->flash) % 2) {
		if (strcasecmp(color, "white") == 0) color = "black";
	    } else {
		color = "white";
	    }
	}

	Tcl_VarEval (interp, type_to_string (object->type),
		     "::color ", object->id, " ", color, 
		     (char *) NULL);

	object->flash -= 1;

	if (object->flash == 0) {
	    notrace (m_color, interp, object, 1, &object->color);
	    free (p->id);
	    p->id = NULL;
	}

	max = ( object->flash > max ) ? object->flash : max;
    }

    if (max <= 0) {      /* everything is done - remove the flashlist */

	fli *q;

        for (p = flashlist; p != NULL; p = q) {
            q = p->next;
            if (p->id != NULL) free (p->id);
            free (p);
        }

        flashlist = NULL;
    }

    Tcl_Eval (interp, "update");

    if (max > 0) Tk_CreateTimerHandler (500, doflash, (ClientData) interp);
}

/*
 * Add a new flash node to our list of flashing objects.
 * Start flash callback if we create the first entry.
 */

void
flash (interp, object)
    Tcl_Interp *interp;
    tkined_object *object;
{
    fli *p;

    if (flashlist == NULL) {
	
        flashlist = (fli *) xmalloc (sizeof(fli));
	p = flashlist;
	p->id = xstrdup(object->id);
	p->next = NULL;
	Tk_CreateTimerHandler (500, doflash, (ClientData) interp);

    } else {

	/* move to the end of the list and check if it exists already */

	for (p = flashlist; p->next != NULL; p = p->next) {
	    if (p->id && strcmp (p->id, object->id) == 0) return;
	}

	if (p->id && strcmp (p->id, object->id) == 0) return;

        p->next = (fli *) xmalloc (sizeof(fli));
	p = p->next;
	p->id = xstrdup(object->id);
	p->next = NULL;

    }
}
