# -*- Mode: Tcl -*-
# TK/Emacs key bindings version: release 0.1
# Copyright 1994 by GTE Laboratories Incorporated.
#
# Portions of this work are in the public domain.  Permission to use,
# copy, modify, and distribute this software and its documentation for
# any purpose and without fee is hereby granted, provided that the above
# copyright notice appear in all copies and that both the copyright
# notice and warranty disclaimer appear in supporting documentation, and
# that the names of GTE Laboratories or any of their entities not be
# used in advertising or publicity pertaining to distribution of the
# software without specific, written prior permission.
#
# GTE disclaims all warranties with regard to this software, including
# all implied warranties of merchantability and fitness for a particular
# purpose, even if GTE Laboratories Incorporated knows about the
# purpose.  In no event shall GTE be liable for any special, indirect or
# consequential damages or any damages whatsoever resulting from loss of
# use, data or profits, whether in an action of contract, negligence or
# other tortuous action, arising out of or in connection with the use or
# performance of this software.

proc abs {num} { return [expr { ($num < 0) ? -1 * $num : $num }] } 

catch { unset emSyntaxTable }
catch { unset em<Control-x>SyntaxTable }
catch { unset em<Control-q>SyntaxTable }
proc emDefineKey {keychord args} {
  ## keychord is a concatenation of key binding specs
  ## args is handled something like the last arg of bind
  ##   if args == {} (no args), return binding
  ##   else if args == {{}} (empty list as arg) , remove binding
  ##   else [lindex args 0] is the binding
  ##puts stderr "emDefineKey $keychord $args"
  regexp {^(.*)<[^>]+>$} $keychord ignore rootkey
  regexp {^.*(<[^>]+>)$} $keychord ignore localkey
  set syntaxTable "em${rootkey}SyntaxTable"
  global $syntaxTable
  set binding {}
  case $args {
    {} { catch { set binding [set ${syntaxTable}($localkey)] } }
    {{}} { 
      set binding {}
      catch {unset ${syntaxTable}($localkey)}
      case $syntaxTable {
	emSyntaxTable { 
	  bind Text  $localkey $binding
	  bind Entry $localkey $binding 
	}
      }
    }
    * {
      set binding [lindex $args 0]
      if {"[lindex $binding 0]"!="emSetSyntaxTable"} { set binding "emCallInteractively {$binding}" }
      set ${syntaxTable}($localkey) $binding
      case $syntaxTable {
	emSyntaxTable { 
	  bind Text  $localkey $binding
	  bind Entry $localkey $binding
	}
	* { emDefineKey $rootkey "emSetSyntaxTable $rootkey" }
      }
    }
  }
  return $binding
}
emDefineKey <Any-Key> {emSelfInsertCommand {} %A}
emDefineKey <Return>  emNewline
emDefineKey <Control-space> {emSetMark;emMarkRegion}
emDefineKey <Control-a> emBeginningOfLine
emDefineKey <Control-b> emBackwardChar
emDefineKey <Control-d> emDeleteChar
emDefineKey <Control-e> emEndOfLine
emDefineKey <Control-f> emForwardChar
emDefineKey <Control-g> emAbort
emDefineKey <Control-h> emDeleteBackwardChar
emDefineKey <BackSpace> [emDefineKey <Control-h>]
emDefineKey <Delete> [emDefineKey <Control-h>]
emDefineKey <Control-i> {emSelfInsertCommand {} "	"}
emDefineKey <Control-j> emNewline
emDefineKey <Control-k> emKillLine
emDefineKey <Control-l> emRecenter
emDefineKey <Control-m> emNewline
emDefineKey <Control-n> emNextLine
emDefineKey <Control-o> emOpenLine
emDefineKey <Control-p> emPreviousLine
emDefineKey <Control-q><Any-Key> {emInsert {} %A}
emDefineKey <Control-r> bell
emDefineKey <Control-s> bell
emDefineKey <Control-t> emTransposeChars
emDefineKey <Control-v> emScrollUp
emDefineKey <Control-w> emKillRegion
emDefineKey <Control-y> emYank
emDefineKey <Control-x><Control-x> emExchangePointAndMark
emDefineKey <Control-x><u> emUndo
emDefineKey <Control-x><parenleft> emBeginMacro
emDefineKey <Control-x><parenright> emEndMacro
emDefineKey <Control-x><e> emExecuteMacro
emDefineKey <Control-z> bell

emDefineKey <Meta-b> emBackwardWord
emDefineKey <Escape><b> [emDefineKey <Meta-b>]
emDefineKey <Meta-c> emCapitalizeWord
emDefineKey <Escape><c> [emDefineKey <Meta-c>]
emDefineKey <Meta-d> emKillWord
emDefineKey <Escape><d> [emDefineKey <Meta-d>]
emDefineKey <Meta-f> emForwardWord
emDefineKey <Escape><f> [emDefineKey <Meta-f>]
emDefineKey <Meta-l> emDowncaseWord
emDefineKey <Escape><l> [emDefineKey <Meta-l>]
emDefineKey <Meta-m> emBackToIndentation
emDefineKey <Escape><m> [emDefineKey <Meta-m>]
emDefineKey <Meta-t> emTransposeWords
emDefineKey <Escape><t> [emDefineKey <Meta-t>]
emDefineKey <Meta-u> emUpcaseWord
emDefineKey <Escape><u> [emDefineKey <Meta-u>]
emDefineKey <Meta-v> emScrollDown
emDefineKey <Escape><v> [emDefineKey <Meta-v>]
emDefineKey <Meta-y> emYankPop
emDefineKey <Escape><y> [emDefineKey <Meta-y>]

emDefineKey <Meta-less> {emSetPoint %W [emPointMin %W]}
emDefineKey <Escape><less> [emDefineKey <Meta-less>]
emDefineKey <Meta-greater> {emSetPoint %W [emPointMax %W]}
emDefineKey <Escape><greater> [emDefineKey <Meta-greater>]

# emDefineKey <Meta-less> "%W yview 0"
# emDefineKey <Escape><less> [emDefineKey <Meta-less>]
# emDefineKey <Meta-greater> "%W yview -pickplace end"
# emDefineKey <Escape><greater> [emDefineKey <Meta-greater>]

# emBackwardSentence is just a stub
emDefineKey  <Meta-a> emBackwardSentence
# emForwardSentence is just a stub
emDefineKey  <Meta-e> emForwardSentence

emDefineKey <Button-1> {
  set tk_priv(selectMode) char
  emSetPoint %W @%x,%y
  emSetMark %W {}
  if {[emConfiguration %W -state] == "normal"} {focus %W}
}
emDefineKey <B1-Motion> {
  if {"[emMark %W]"=={} && "[emPoint %W @%x,%y]"!="[emPoint %W]"} { emSetMark %W [emPoint %W] }
  emSetPoint %W @%x,%y
}
emDefineKey <ButtonRelease-1> {
  if {"[emPoint %W @%x,%y]"=="[emMark %W]"} { emSetMark %W {} }
}
emDefineKey <Double-Button-1> {
  set tk_priv(selectMode) char
  emSetPoint %W @%x,%y
  emForwardChar %W
  emBackwardWord %W
  emSetMark %W
  emForwardWord %W
  if {[emConfiguration %W -state] == "normal"} {focus %W}
}
emDefineKey <Triple-Button-1> {
  set tk_priv(selectMode) char
  emSetPoint %W @%x,%y
  emBeginningOfLine %W
  emSetMark %W
  emEndOfLine %W
  if {[emConfiguration %W -state] == "normal"} {focus %W}
}

# moved drag entry from middle to control-middle
# emDefineKey <Control-2> {emDrag %W %x 1}
# emDefineKey <Control-B2-Motion> {emDrag %W %x} 
# emDefineKey <B2-Motion> {}
# button <2> inserts selection, like usual in X
# emDefineKey <2> {emInsertSelection %W [emPoint %W @%x,%y]}

# move between entry and text fields with tab
emDefineKey <Tab>       {focus [emNextEntry]}
emDefineKey <Shift-Tab>        [emDefineKey <Tab>]
emDefineKey <Control-x><Key-o> [emDefineKey <Tab>]

emDefineKey <Control-Tab>  {focus [emPreviousEntry]}
emDefineKey <Escape><Tab>  [emDefineKey <Control-Tab>]
emDefineKey <Control-x><Key-p> [emDefineKey <Control-Tab>]

# Setup count bindings
emDefineKey <Meta-Key->    {emAdjustCount -1 0 0}
emDefineKey <Escape><Key-> [emDefineKey <Meta-Key->]
emDefineKey <Control-Key-> [emDefineKey <Meta-Key->]
emDefineKey <Meta-Key-0>    {emAdjustCount 10 0 0}
emDefineKey <Control-Key-0> [emDefineKey <Meta-Key-0>]
emDefineKey <Escape><Key-0> [emDefineKey <Meta-Key-0>]
emDefineKey <Meta-Key-1>    {emAdjustCount 10 1 0}
emDefineKey <Control-Key-1> [emDefineKey <Meta-Key-1>]
emDefineKey <Escape><Key-1> [emDefineKey <Meta-Key-1>]
emDefineKey <Meta-Key-2>    {emAdjustCount 10 2 0}
emDefineKey <Control-Key-2> [emDefineKey <Meta-Key-2>]
emDefineKey <Escape><Key-2> [emDefineKey <Meta-Key-2>]
emDefineKey <Meta-Key-3>    {emAdjustCount 10 3 0}
emDefineKey <Control-Key-3> [emDefineKey <Meta-Key-3>]
emDefineKey <Escape><Key-3> [emDefineKey <Meta-Key-3>]
emDefineKey <Meta-Key-4>    {emAdjustCount 10 4 0}
emDefineKey <Control-Key-4> [emDefineKey <Meta-Key-4>]
emDefineKey <Escape><Key-4> [emDefineKey <Meta-Key-4>]
emDefineKey <Meta-Key-5>    {emAdjustCount 10 5 0}
emDefineKey <Control-Key-5> [emDefineKey <Meta-Key-5>]
emDefineKey <Escape><Key-5> [emDefineKey <Meta-Key-5>]
emDefineKey <Meta-Key-6>    {emAdjustCount 10 6 0}
emDefineKey <Control-Key-6> [emDefineKey <Meta-Key-6>]
emDefineKey <Escape><Key-6> [emDefineKey <Meta-Key-6>]
emDefineKey <Meta-Key-7>    {emAdjustCount 10 7 0}
emDefineKey <Control-Key-7> [emDefineKey <Meta-Key-7>]
emDefineKey <Escape><Key-7> [emDefineKey <Meta-Key-7>]
emDefineKey <Meta-Key-8>    {emAdjustCount 10 8 0}
emDefineKey <Control-Key-8> [emDefineKey <Meta-Key-8>]
emDefineKey <Escape><Key-8> [emDefineKey <Meta-Key-8>]
emDefineKey <Meta-Key-9>    {emAdjustCount 10 9 0}
emDefineKey <Control-Key-9> [emDefineKey <Meta-Key-9>]
emDefineKey <Escape><Key-9> [emDefineKey <Meta-Key-9>]
emDefineKey <Control-u> {emAdjustCount 4 0 1}

# numeric keys normally just insert but will increment counter
# if already in progress
emDefineKey <Key-0>    {emSelfInsertNumber {} 0}
emDefineKey <Key-1>    {emSelfInsertNumber {} 1}
emDefineKey <Key-2>    {emSelfInsertNumber {} 2}
emDefineKey <Key-3>    {emSelfInsertNumber {} 3}
emDefineKey <Key-4>    {emSelfInsertNumber {} 4}
emDefineKey <Key-5>    {emSelfInsertNumber {} 5}
emDefineKey <Key-6>    {emSelfInsertNumber {} 6}
emDefineKey <Key-7>    {emSelfInsertNumber {} 7}
emDefineKey <Key-8>    {emSelfInsertNumber {} 8}
emDefineKey <Key-9>    {emSelfInsertNumber {} 9}

# Kludge list keys have to be turned off to allow chorded keys
set emSetSyntaxKludgeList {Control_L Control_R Meta_L Meta_R Shift_L Shift_R}
foreach key $emSetSyntaxKludgeList { 
  bind Text  <$key> {;} 
  bind Entry <$key> {;} 
}
proc emSetSyntaxTable {{rootkey {}}} {
  # puts stderr "emSetSyntaxTable $rootkey"
  global emSetSyntaxKludgeList
  set syntaxTable "em${rootkey}SyntaxTable"
  global $syntaxTable
  set wclass [winfo class [focus]]
  foreach key [bind $wclass] { bind $wclass $key {}}
  case $syntaxTable {
    emSyntaxTable { 
      set prefix ""
      bind $wclass <Any-Key> {emSelfInsertCommand {} %A}
      [focus] configure -insertbackground Black
      foreach key $emSetSyntaxKludgeList { bind $wclass <$key> {;} }
      grab release [focus]
    }
    * { 
      set prefix "emSetSyntaxTable;"
      if {[catch {set ${syntaxTable}(<Any-Key>)}]} { set ${syntaxTable}(<Any-Key>) bell }
      [focus] configure -insertbackground Red
      foreach key $emSetSyntaxKludgeList { bind $wclass <$key> {;} }
      grab [focus]
    }
  }
  set id [array startsearch $syntaxTable]
  while {[array anymore $syntaxTable $id]} {
    set e [array nextelement $syntaxTable $id]
    bind $wclass $e "${prefix}[set ${syntaxTable}($e)]"
  }
  array donesearch $syntaxTable $id
  return $syntaxTable
}

set emMacro {}

proc emExecuteMacro {} { emGetCount; bell }

proc emCallInteractively {cmd} {
  global emMacro
  set omacro $emMacro
  set result [eval $cmd]
  if {"$omacro"!={}} { set emMacro [format "%s\n  %s" $emMacro $cmd] }
  return $result
}

proc emBeginMacro {} {
  global emMacro
  set emMacro [format "proc emExecuteMacro {} \{"]
}
proc emEndMacro {} {
  global  emMacro
  if {"$emMacro"=={}} { bell; return {} }
  set emMacro [format "%s\n\}" $emMacro]
  eval $emMacro
  set emMacro {}
}

if {[lsearch [info commands bell] bell]<0} {
  # define visible bell if no bell exists
  set emBellP 0
  global emBellP
  proc bell {args} {
    global emBellP emMacro emMacroP
    set emMacroP 0; set emMacro {}
    if {$emBellP} return
    set emBellP 1
    set window [focus]
    catch {
      set fg [lindex [$window configure -fg] 4]
      set bg [lindex [$window configure -bg] 4]
      $window configure -bg $fg -fg $bg
      catch {update}
      after 100 "$window configure -bg $bg -fg $fg"
    }
    set emBellP 0
  }
}

set emCount {}
set emSign  {}
proc emAdjustCount {{times 0} {plus 0} {default 0}} {
  global emCount emSign
  if {$times<0} { 
    set times [expr -$times]
    if {"$emSign"=={}} { 
      set emSign -
    } else { 
      set emSign {}
    }
  }
  if {"$emCount"=={}} { set emCount $default }
  set emCount [expr ($emCount*$times)+${plus}]
  [focus] configure -insertbackground Green
  return $emSign$emCount
}
proc emGetCount {{count {}} {default 1}} {
  global emCount emSign
  if {"$count"=={}} {
    case $emCount {
      {} { set count $default }
      *  { set count $emSign$emCount }
    }
  }
  if {"$emSign"!={} || "$emCount"!={}} {
    [focus] configure -insertbackground Black
  }
  set emCount {}
  set emSign  {}
  return $count
}
proc emSelfInsertNumber {{window {}} {number 0}} {
  global emCount emSign	
  if {$emSign=={} && $emCount=={}} {
    return [emSelfInsertCommand $window $number]
  } else {
    return [emAdjustCount 10 $number]
  }
}

set emLastCommand ""
set emGoalColumn ""

proc emLast  {l {n 1}} { return [lindex $l [expr [llength $l]-$n]] }
proc emConfiguration {window option} {
  if {"$window"=={}} { set window [focus] }
  return [emLast [$window configure $option]]
}

proc emLine {loc} {
  if {[regexp {([0-9]+)[.]} $loc ignore line]} { return $line }
  return 1
}
proc emCol {loc} {
  if {[regexp {[0-9]+[.]([0-9]+)} $loc ignore col]} { return $col }
  return $loc
}
proc emMark {{window {}}} {
  if {"$window"=={}} { set window [focus] }
  global $window.Anchor
  case [winfo class $window] {
    Text  { if {[catch {$window index anchor}]} { return {} } else { return [$window index anchor] } }
    Entry { if {![info exists $window.Anchor]}   { return {} } else { return [set $window.Anchor] } }
    *     { error "Unknown winfo class for $window, [winfo class $window]" }
  }
}
proc emSetMark {{window {}} {mark insert}} {
  if {"$window"=={}} { set window [focus] }
  emGetCount
  global $window.Anchor
  if {"$mark"!={}} { set mark [emPoint $window $mark] }
  case [winfo class $window] {
    Text  { if {"$mark"=={}} {catch {$window mark unset anchor}} else {$window mark set anchor $mark}}
    Entry {
      if {"$mark"=={}} {
	if {[info exists $window.Anchor]} {unset $window.Anchor}
      } else {
	set $window.Anchor $mark
      }
    }
  }
  return [emMark $window]
}
proc emOrderPoints {{window {}} {start anchor} {end insert}} {
  if {"$window"=={}} { set window [focus] }
  set start [emPoint $window $start]
  set end   [emPoint $window $end]
  case [winfo class $window] {
    Text {
      if {[$window compare $end < $start]} {
	set tmp $start; set start $end; set end $tmp;
      }
    }
    Entry {
      if {$end < $start} {
	set tmp $start; set start $end; set end $tmp;
      }
    }
  }
  return "[emPoint $window $start] [emPoint $window $end]"
}
proc emPoint {{window {}} {point insert}} {
  if {"$window"=={}} { set window [focus] }
  case [winfo class $window] {
    Text  { 
      set currpoint [$window index insert]
      case $point {
	{+* -* *.+* *.-*} {
	  ## Allow relative offsets
	  set col  [emCol $currpoint]
	  set line [emLine $currpoint] 
	  set lineOffset {}
	  set colOffset {+0}
	  if {![regexp {^([+-][0-9]+)[.]} $point ignore lineOffset]} {
	    regexp {^([^.]+)[.]} $point ignore line
	  }
	  if {![regexp {^[+-][0-9]+$} $point colOffset] && ![regexp {^[^.]*[.]([+-][0-9]+)$} $point ignore colOffset]} {
	    if {![regexp {^[^.]+$} $point col]} {regexp {^[^.]*[.]([^.]+)$} $point ignore col}
	  }
	  set currpoint [$window index [expr $line$lineOffset].$col${colOffset}chars]
	}
	* { 
	  if {[regexp {^[0-9]+$} $point]} {
	    set currpoint [$window index [emLine $currpoint].$point]
	  } else { return [$window index $point] }
	}
      }
      if {[$window compare $currpoint < 1.0]} { set currpoint 1.0 }
      if {[$window compare $currpoint > end]} { set currpoint [$window index end] }
      return $currpoint
    }
    Entry {
      case $point {
	anchor  { return [emMark $window] }
	insert  { return [$window index insert] }
	@-*     { return [emPointMin $window] }
	* {
	  if {[regexp {^[+-]} $point]} {
	    set currpoint [expr [emPoint $window]$point]
	  } else { 
	    regexp {(@[0-9]+),} $point ignore point
	    if {"$point"=={}} { set point 0 }
	    set currpoint [$window index [emCol $point]]
	    if {$currpoint<0} { set currpoint 0 }
	    if {$currpoint>[$window index end]} { set currrpoint [$window index end] }
	  }
	  return $currpoint
	}
      }
    }
  }
}
proc emSetPoint {{window {}} {point insert}} {
  if {"$window"=={}} { set window [focus] }
  emGetCount
  set point [emPoint $window $point]
  case [winfo class $window] {
    Text {
      $window mark set insert $point
      $window yview -pickplace insert
    }
    Entry {
      $window icursor $point
      set width [lindex [$window configure -width] 4]
      set start [expr ([emCol $point]/$width)*$width-($width/2)]
      if {$start<0} { set start 0 }
      $window view $start
    }
  }
  emMarkRegion $window
  set newPoint [emPoint $window]
  if {$newPoint!=$point} { set undo($window:NoJoin) TRUE }
  return $newPoint
}
proc emPointMin {{window {}}} {
  case [winfo class $window] {
    Text { return 1.0 }
    Entry { return 0 }
  }
}
proc emPointMax {{window {}}} {
  return [emPoint $window end]
}
proc emPointChar {{window {}}} {
  if {"$window"=={}} { set window [focus] }
  case [winfo class $window] {
    Text  { return [$window get [emPoint $window]] }
    Entry { 
      set text [$window get]
      return [string index $text [emPoint $window]]
    }
  }
}
proc emBOL {{window {}} {point insert}} {
  if {"$window"=={}} { set window [focus] }
  case [winfo class $window] {
    Text  { return [emLine [emPoint $window $point]].0 }
    Entry { return [emPointMin $window] }
  }
}
proc emEOL {{window {}} {point insert}} {
  if {"$window"=={}} { set window [focus] }
  case [winfo class $window] {
    Text  { return [emPoint $window [emLine [emPoint $window $point]].0lineend] }
    Entry { return [emPointMax $window] }
  }
}
proc emAbort {{window {}}} {
  global emLastCommand
  emGetCount
  if {"$window"=={}} { set window [focus] }
  emSetSyntaxTable
  case [winfo class $window] {
    Text  { catch { $window tag remove sel [emPointMin $window] [emPointMax $window] } }
    Entry { $window select clear }
  }
  emSetMark $window {}
  bell
  set emLastCommand emAbort
  return [emPoint $window]
}
proc emMarkRegion {{window {}} {point insert} {mark anchor}} {
  catch {
    case [winfo class $window] {
      Text  { 
	if {"$window"=={}} { set window [focus] }
	set points [emOrderPoints $window $point $mark]
	set start [lindex $points 0]
	set end   [lindex $points 1]
	$window tag remove sel [emPointMin $window] $start
	$window tag remove sel $end [emPointMax $window]
	$window tag add sel $start $end
      }
      Entry { 
	set mark [emPoint $window $mark]
	set point [emPoint $window $point]
	## This was a bug in the release! add following line to fix
	if {"$mark"=={}} { set mark $point }
	set points [emOrderPoints $window $point $mark]
	set start [lindex $points 0]
	if {$start<0} { set start [emPointMin $window] }
	set end   [lindex $points 1]
	if {$end<0} { set end [emPointMin $window] }
	if {$start==$end} {
	  $window select clear
	} else {
	  $window select from $start
	  $window select to   [expr $end-1]
	}
      }
    }
    return [emPoint $window]
  }
  return {}
}
proc emBufferSubstring {{window {}} {start insert} {end anchor}} {
  if {"$window"=={}} { set window [focus] }
  set result {}
  catch {
    set points [emOrderPoints $window $start $end]
    set start [lindex $points 0]
    set end   [lindex $points 1]
    case [winfo class $window] {
      Text  { set result [$window get $start $end] }
      Entry { set text [$window get]; set result [string range $text $start [expr $end-1]] }
    }
  }
  return $result
}
proc emLookingAt {{window {}} {pattern {}} {count 256} {submatchp 0} {endp 0}} {
  if {"$window"=={}} { set window [focus] }
  if {$count<0 || "$count"=={}} { set count 256 }
  set result -1
  if {[regexp -indices $pattern [emBufferSubstring $window [emPoint $window] [emPoint $window +$count]] match submatch]} {
    if {$submatchp} { set match $submatch }
    set result [lindex $match $endp]
    if {"$result"=={}} { set result -1 } else { if {$endp} { incr result } }
  }
  return $result
}

catch { unset emUndo }
proc emFlushUndo {{window {}}} {
  global emUndo
  if {"$window"=={}} { set window [focus] }
  if {[info exists emUndo($window:Max)]} {
    for {set n $emUndo($window:Max)} {[info exists emUndo($window:Action:$n)]} {incr n -1} {
      unset emUndo($window:Action:$n)
    }
    unset emUndo($window:Max)
    if {[info exists emUndo($window:Ptr)]} { unset emUndo($window:Ptr) }
  }
}
proc emUndo {{window {}} {count {}}} {
  ## With an COUNT of 0, this will delete all the
  ## existing undo history.
  global emLastCommand emUndo
  if {"$window"=={}} { set window [focus] }
  set count [abs [emGetCount $count]]
  if {!$count} { return [emFlushUndo $window] }
  for {set n 0} {$n<$count} {incr n} {
    if {[info exists emUndo($window:Ptr)]} { set ptr $emUndo($window:Ptr) } { set ptr {} }
    if {[info exists emUndo($window:Max)]} { set max $emUndo($window:Max) } { set max {} }
    if {$emLastCommand=={emUndo}} { set undo($window:NoJoin) TRUE }
    if {$emLastCommand!={emUndo} || "$ptr"=={} } { set ptr $max }
    if {"$ptr"=={}} { bell; break }
    set action $emUndo($window:Action:$ptr)
    if {[info exists emUndo($window:Action:[expr $ptr-1])]} {
      set emUndo($window:Ptr) [incr ptr -1]
    } else {
      if {[info exists emUndo($window:Ptr)]} { unset emUndo($window:Ptr) }
    }
    if {[regexp {^\+ ([^ ]+) (.*)$} $action ignore point text]} {
      emInsertAtPoint $window $point $text
    } else {
      if {[regexp {^\- ([^ ]+) ([^ ]+)$} $action ignore point mark]} {
	emSetPoint $window $point
	emDeleteRegion $window $point $mark;
      } else bell
    }
  }
  set emLastCommand emUndo
}
proc emPushUndoInsert {window mark point} {
  #puts stderr "emPushUndoInsert $window $mark $point"
  global emUndo
  set max -1; set prev {} 
  if {[info exists emUndo($window:Max)]} {
    set max $emUndo($window:Max)
    set prev $emUndo($window:Action:$max)
  }
  set omark [lindex $prev 1]
  if {[lindex $prev 0]=={-} && $mark==[lindex $prev 2] && \
      ![info exists undo($window:NoJoin)] && \
      [emLine $omark]==[emLine $point] && [expr [emCol $point]-[emCol $omark]]<16} {
    set cmd "- $omark $point"
  } else {
    set emUndo($window:Max) [incr max]
    set cmd "- $mark $point"
  }
  #puts stderr "set emUndo($window:Action:$max) $cmd"
  if {[info exists undo($window:NoJoin)]} { unset undo($window:NoJoin) }
  set emUndo($window:Action:$max) $cmd
}
proc emPushUndoDelete {window point mark} {
  #puts stderr "emPushUndoDelete $window $point $mark"
  global emUndo
  set max -1; set prev {} 
  if {[info exists emUndo($window:Max)]} {
    set max $emUndo($window:Max)
    set prev $emUndo($window:Action:$max)
  }
  set text [emBufferSubstring $window $point $mark]
  if {"$text"=={}} return;
  if {[regexp {^\+ ([^ ]+) (.*)$} $prev ignore opoint prev] && \
	![info exists undo($window:NoJoin)] && \
	$mark==$opoint && \
	[expr [string length $text]+[string length $prev]]<16\
	} {
    set cmd "+ $point ${text}$prev"
  } else {
    set emUndo($window:Max) [incr max]
    set cmd "+ $point $text"
  }
  #puts stderr "set emUndo($window:Action:$max) $cmd"
  if {[info exists undo($window:NoJoin)]} { unset undo($window:NoJoin) }
  set emUndo($window:Action:$max) $cmd
}
catch { unset emKill }

# The kill ring may be any size, I arbitrarily chose 24
set emKill(Size) 24
set emKill(Length) 0
set emKill(Next) 0
set emKill(YankPtr) 0
proc emPushKill {kill} {
  global emKill
  set size $emKill(Size)
  set next $emKill(Next)
  set length $emKill(Length)
  if ($next>=$size) { set next 0 }
  set emKill($next) $kill
  incr next
  if {$length<$size} { incr emKill(Length) }
  set emKill(YankPtr) $next
  set emKill(Next) $next
  return $kill
}
proc emPopKill {{count {}}} {
  global emKill
  set emKill(YankPtr) $emKill(Next)
  return [emNextKill $count]
}
proc emNextKill {{count {}}} {
  global emKill
  if {![set count [emGetCount $count]]} { bell; return }
  set ptr $emKill(YankPtr)
  set size $emKill(Length)
  set inc [expr ($count<0)?1:-1]
  set count [abs $count]
  for {set n 0} {$n<$count} {incr n} {
    incr ptr $inc
    if {$ptr<0} { set ptr [expr $size-1] }
    if {$ptr>=$size} { set ptr 0 }
  }
  if {![info exists emKill($ptr)]} { bell; return }
  set emKill(YankPtr) $ptr
  return $emKill($ptr)
}

set emErrorP 0
## Insert hooks are provided here so that you can have windows 
## that only accept upcase, numeric or some other special format
## text.  These functions accept the text to be inserted and may
## filter or modify it prior to insertion.
## Some insert hook functions that can be used hacking input
## These are called by emInsert if an entry emInsertHook($window) exists.
## They normally return the filtered text for insertion.
## If the input is no good, they set emErrorP to 1 and return {}
proc emUpperCaseHook {window point text} { return [string toupper $text ] }
proc emLowerCaseHook {window point text} { return [string tolower $text ] }
proc emNumericHook   {window point text} { 
  if {[regexp {^[0-9]+$} $text]} { return $text }
  global emInsertP
  set emErrorP 1
  return {}
}
proc emAlphaHook   {window point text} { 
  if {[regexp {^[A-Za-z]+$} $text]} { return $text }
  global emErrorP
  set emErrorP 1
  return {}
}
proc emAlphaNumericHook   {window point text} { 
  if {[regexp {^[A-Za-z0-9]+$} $text]} { return $text }
  global emErrorP
  set emErrorP 1
  return {}
}
proc emPrintableHook   {window text} { 
  if {[regexp {^[	 -~]+$} $text]} { return $text }
  global emErrorP
  set emErrorP 1
  return {}
}

proc emInsert {{window {}} text {count {}}} {
  ## emInsert is used to add text to a buffer  It has the capability of
  ## preprocessing typed input such as capitalization or
  ## ignoring non-printing characters.  
  ## The array: emInsertHook($window) where window
  ## is the name of the window being filtered.  The hook
  ## function takes the window, point and text as args.  
  ## Some example insert hook functions are listed above.
  global emLastCommand
  if {"$window"=={}} { set window [focus] }
  set count [emGetCount $count]
  set mark [emPoint $window]
  global emInsertHook emErrorP
  set hookp [info exists emInsertHook($window)]
  for {set n 0} {$n<$count} {incr n} {
    set emErrorP 0
    if {$hookp} { set text [$emInsertHook($window) $window [emPoint $window] $text] }
    if {$emErrorP} bell
    if {$text!={}} { 
      set point [emPoint $window]
      $window insert $point $text 
    }
  }
  set point [emSetPoint $window]
  emPushUndoInsert $window $mark $point
  set emLastCommand emInsert
  return $point
}
proc emDeleteRegion {{window {}} {point insert} {mark anchor}} {
  global emLastCommand
  emGetCount
  if {"$window"=={}} { set window [focus] }
  set points [emOrderPoints $window $point $mark]
  set start [lindex $points 0]
  set end   [lindex $points 1]

  global emDeleteHook emErrorP
  set hookp [info exists emDeleteHook($window)]
  set emErrorP 0
  if {$hookp} { $emDeleteHook($window) $window [emPoint $window] $mark }
  if {$emErrorP} bell
  if {$start==$end} { return [emPoint $window] }
  emPushUndoDelete $window $start $end
  catch {
    case [winfo class $window] {
      Text  { $window delete $start $end }
      Entry { $window delete $start [expr $end-1] }
    }
  }
  set emLastCommand emDeleteRegion
  return [emPoint $window]
}

proc emSelfInsertCommand {{window {}} {key {}} {count {}}} {
  ## emSelfInsertCommand is the basic way keys are inserted
  ## by typing into a window.
  global emLastCommand
  set point [emInsert $window $key $count]
  set emLastCommand emSelfInsertCommand
  return $point
}
proc emNewline {{window {}} {count {}}} {
  global emLastCommand
  if {"$window"=={}} { set window [focus] }
  case [winfo class $window] {
    Entry { bell; return }
  }
  set point [emSelfInsertCommand $window "\n" $count]
  set emLastCommand emNewline
  return $point
}
set emExcursion {}
proc emPushExcursion {} {
  global emExcursion
  return [lindex [set emExcursion "{{[focus]} {[emMark]} {[emPoint]}} $emExcursion"] 0]
}
proc emPopExcursion {} {
  global emExcursion
  set excursion [lindex $emExcursion 0]
  set emExcursion [lrange $emExcursion 1 end]
  set window [lindex $excursion 0]
  set mark [lindex $excursion 1]
  set point [lindex $excursion 2]
  focus $window
  emSetMark $window $mark
  emSetPoint $window $point
  return $window
}

proc emChildren {frame} {
  case [winfo class $frame] {
    Canvas { return [winfo children $frame] }
    * {
      if {[catch {set l [pack info $frame]}]} { return {} }
      set children ""
      set ll [llength $l]
      for {set n 0} {$n<$ll} {incr n 2} { lappend children [lindex $l $n] }
      return $children
    }
  }
}

proc emPreviousEntry {{window {}} {count {}}} {
  global emLastCommand
  set count [expr 0-[emGetCount $count]]
  set result [emNextEntry $window $count]
  set emLastCommand emPreviousEntry
  return $result
}

proc emNextEntry {{window {}} {count {}}} {
  global emLastCommand
  set emLastCommand emNextEntry
  if {"$window"=={}} { set window [focus] }
  set count [emGetCount $count]
  set reversep [expr $count<0]
  set count [abs $count]
  set foundp 0
  set currlist {}
  for {set x 0} {$x<1000} {incr x} {
    if {$currlist=={}} { 
      set currlist [emChildren [winfo toplevel $window]] 
      if {$reversep} { set currlist [lreverse $currlist] }
    }
    set curr [lindex $currlist 0]
    set currlist [lrange $currlist 1 end]
    case [winfo class $curr] {
      {Entry Text}  { 
	if {$foundp} {
	  incr count -1
	  if {$count<=0} { return $curr }
	}
	if {$curr==$window} { set foundp 1 }
      }
      {Canvas Frame}  { 
	set children [emChildren $curr]
	if {$reversep} { set children [lreverse $children] }
	set currlist [concat $children $currlist]
      }
    }
  }
  ## Should never get here
  return $window
}
proc emDrag {window x {startp 0}} {
  if {$startp} {
    $window scan mark $x
  } else {
    $window scan dragto $x
  }
}

proc emInsertSelection {{window {}} {point insert} {count {}}} {
  if {"$window"=={}} { set window [focus] }
  if {"[lindex [$window configure -state] 4]"=="normal"} {
    if {![catch { set sel [selection get]}]} {
      emPushExcursion
      catch {
	emSetPoint $window $point
	emInsert $window $sel $count
      }
      emPopExcursion
    }
  }
}
proc emInsertAtPoint {{window {}} {point insert} {text {}}} {
  global emLastCommand
  if {"$window"=={}} { set window [focus] }
  emSetPoint $window $point
  emInsert $window $text 1
  return [emPoint $window]
}
proc emDeleteChar {{window {}} {count {}}} {
  global emLastCommand
  if {"$window"=={}} { set window [focus] }
  set count [emGetCount $count]
  set emLastCommand emDeleteChar
  emDeleteRegion $window [emPoint $window] [emForwardChar $window $count]
  set emLastCommand emDeleteChar
  return [emPoint $window]
}
proc emBeginningOfLine {{window {}} {point insert}} {
  global emLastCommand
  emGetCount
  if {"$window"=={}} { set window [focus] }
  set emLastCommand emBeginningOfLine
  return [emSetPoint $window [emBOL $window $point]]
}
proc emEndOfLine {{window {}} {point insert}} {
  global emLastCommand
  emGetCount
  if {"$window"=={}} { set window [focus] }
  set emLastCommand emEndOfLine
  return [emSetPoint $window [emEOL $window $point]]
}
proc emForwardChar {{window {}} {count {}}} {
  global emLastCommand
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  if {[regexp {^[0-9]} $count]} { set count +$count }
  catch { emSetPoint $window $count }
  set emLastCommand emForwardChar
  return [emPoint $window]
}
proc emBackwardChar {{window {}} {count {}}} {
  set count [emGetCount $count]
  return [emForwardChar $window [expr -$count]]
}
proc emDeleteBackwardChar {{window {}} {count {}}} {
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  global emLastCommand
  emDeleteRegion $window [emPoint $window -$count] [emPoint $window]
  set emLastCommand emDeleteBackwardChar
  return [emPoint $window]
}
proc emKillLine {{window {}} {count {}}} {
  global emLastCommand
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  set kill {}
  for {set n 0} {$n<$count} {incr n} {
    set kill ${kill}[emBufferSubstring $window [emPoint $window] [emEOL $window]]
    set point [emPoint $window]
    set mark  [emEOL $window]
    if {$point==$mark && $point!="[emPointMax $window]"} {
      set mark [emPoint $window +1]
    }
    emDeleteRegion $window $point $mark
  }
  emPushKill $kill
  set emLastCommand emKillLine
  return [emPoint $window]
}
proc emRecenter {{window {}} {count {}}} {
  global emLastCommand
  set count [emGetCount $count center]
  if {"$window"=={}} { set window [focus] }
  case [winfo class $window] {
    Text  { 
      set numlines [emConfiguration $window -height]
      if {"$count"=="center"} { set count [expr $numlines/2] }
      if {$count<0} { set count [expr $numlines+$count] }
      if {$count<0} { set count 0 }
      if {$count>=$numlines} { set count [expr $numlines-1] }
      $window yview [emPoint $window -$count.0] 
    }
    Entry { $window view  [emCol [emPoint $window]] }
  }
  set emLastCommand emRecenter
  return [emPoint $window]
}
proc emOpenLine {{window {}} {count {}}} {
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  case [winfo class $window] { Entry { bell; return {} }}
  global emLastCommand
  set point [emPoint $window]
  emInsert $window "\n" $count
  set emLastCommand emOpenLine
  return [emSetPoint $window $point]
}
proc emNextLine {{window {}} {count {}}} {
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  if {[regexp {^[0-9]} $count]} { set count +$count }
  case [winfo class $window] { Entry { bell; return {} }}
  global emLastCommand emGoalColumn
  set opoint [emPoint $window]
  if {"$emLastCommand"!="emNextLine"} { set emGoalColumn [emCol $opoint] }
  set npoint [lindex [emOrderPoints $window [emPoint $window $count.$emGoalColumn] [emEOL $window $count.0]] 0]
  if {[emLine $opoint]!=[emLine $npoint]} { emSetPoint $window $npoint }
  set emLastCommand emNextLine
  return [emPoint $window]
}
proc emPreviousLine {{window {}} {count {}}} {
  set count [emGetCount $count]
  return [emNextLine $window [expr -$count]]
}
proc emTransposeChars {{window {}} {count {}}} {
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  if {[emPoint $window]==[emPointMin $window]} { bell; return {} }
  global emLastCommand
  for {set n 0} {$n<$count} {incr n} {
    emBackwardChar $window
    set char [emPointChar $window]
    emDeleteChar $window
    emForwardChar $window
    emInsert $window $char
  }
  set emLastCommand emTransposeChars
  return [emSetPoint $window]
}
proc emScrollDown {{window {}} {count {}}} {
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  global emLastCommand
  case [winfo class $window] { Entry { bell; return {} }}
  if {$count<0} { return [emScrollUp $window [expr -$count]] }
  if {"$window"=={}} { set window [focus] }
  set numlines [expr $count*[emConfiguration $window -height]]
  if {$numlines} { emPreviousLine $window [expr $numlines-1] }
  emRecenter $window
  set emLastCommand emScrollDown
  return [emPoint $window]
}
proc emScrollUp {{window {}} {count {}}} {
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  global emLastCommand
  case [winfo class $window] { Entry { bell; return {} }}
  if {$count<0} { return [emScrollDown $window [expr -$count]] }
  if {"$window"=={}} { set window [focus] }
  set numlines [expr $count*[emConfiguration $window -height]]
  if {$numlines} { emNextLine $window [expr $numlines-1] }
  emRecenter $window
  set emLastCommand emScrollUp
  return [emPoint $window]
}
proc emExchangePointAndMark {{window {}}} {
  global emLastCommand
  emGetCount
  if {"$window"=={}} { set window [focus] }
  set mark [emMark $window]
  if {"$mark"=={}} { bell; return {} }
  set point [emPoint $window]
  emSetMark  $window $point
  emSetPoint $window $mark
  set emLastCommand emExchangePointAndMark
  return $mark
}
proc emKillWord {{window {}} {count {}}} {
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  global emLastCommand
  emSetMark $window
  emForwardWord $window $count
  emKillRegion $window [emMark $window] [emPoint $window]
  set emLastCommand emKillWord
  return [emPoint $window]
}
proc emKillRegion {{window {}} {start insert} {end anchor}} {
  if {"$window"=={}} { set window [focus] }
  emGetCount
  global emLastCommand
  emPushKill [emBufferSubstring $window $start $end]
  emDeleteRegion $window $start $end
  set emLastCommand emKillRegion
  return [emSetPoint $window]
}
proc emYank {{window {}} {count {}}} {
  if {"$window"=={}} { set window [focus] }
  global emLastCommand
  set kill [emPopKill $count]
  set opos [emPoint $window]
  emInsert $window $kill
  emSetMark $window $opos
  set emLastCommand emYank
  return [emSetPoint $window]
}
proc emYankPop {{window {}} {count {}}} {
  if {"$window"=={}} { set window [focus] }
  global emLastCommand
  if {"$emLastCommand"!="emYank"} { return [emYank $window $count] }
  emDeleteRegion $window
  set kill [emNextKill $count]
  set opos [emPoint $window]
  emInsert $window $kill
  emSetMark $window $opos
  set emLastCommand emYank
  return [emSetPoint $window]
}
proc emForwardWord {{window {}} {count {}}} {
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  global emLastCommand
  if {$count<0} { return [emBackwardWord $window [expr -$count]] }
  set end [emPointMax $window]
  for {set n 0} {$n<$count} { incr n } {
    emForwardChar $window [emLookingAt $window {[^a-zA-Z0-9?]*([a-zA-Z0-9?]+)} {} 1 1]
  }
  set emLastCommand emForwardWord
  return [emPoint $window]
}
proc emBackwardWord {{window {}} {count {}}} {
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  global emLastCommand
  if {$count<0} { return [emForwardWord $window [expr -$count]] }
  set point [emPoint $window]
  set text  [emBufferSubstring $window [emBOL $window] $point]
  set end   [emCol $point]
  while {$count>0} {
    if {[regexp -indices {^(|.*[^a-zA-Z0-9?])[a-zA-Z0-9?]+[^a-zA-Z0-9?]*$} $text ignore whitespace]} {
      set end [expr [lindex $whitespace 1]+1]
      set text [string range $text 0 [expr $end-1]]
      incr count -1
    } else { if {[emLine $point]>[emLine [emPointMin $window]]} {
      emPreviousLine $window
      emEndOfLine $window
      set point [emPoint $window]
      set text  [emBufferSubstring $window [emBOL $window] $point]
      set end   [emCol $point]
    } else { bell; break } }
  }
  set emLastCommand emBackwardWord
  return [emSetPoint $window [emLine [emPoint $window]].$end]
}
proc emCapitalizeWord {{window {}} {count {}}} {
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  global emLastCommand
  set mark [emPoint $window]
  set point [emForwardWord $window $count]
  set otext [emBufferSubstring $window $mark [emPoint $window]]
  set ntext {}
  while {[regexp -indices {[^a-zA-Z0-9?]*([a-zA-Z0-9?]+)} $otext ignore word]} {
    set wstart [lindex $word 0]
    set wend   [lindex $word 1]
    set whitespace [string range $otext 0 [expr $wstart-1]]
    set cap        [string toupper [string range $otext $wstart $wstart]]
    set rest       [string tolower [string range $otext [expr $wstart+1] $wend]]
    set ntext "${ntext}${whitespace}${cap}${rest}"
    set otext [string range $otext [expr $wend+1] end]
  }
  set ntext "${ntext}${otext}"
  emDeleteRegion $window $mark $point
  emInsert $window $ntext
  set emLastCommand emCapitalizeWord
  return [emSetPoint $window]
}
proc emTransposeWords {{window {}} {count {}}} {
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  global emLastCommand
  set point [emPoint $window]
  set p0 [emBackwardWord $window 1]
  set e0 [emForwardWord $window 1]
  set word0 [emBufferSubstring $window $p0 $e0]
  set e1 [emForwardWord $window $count]
  set p1 [emBackwardWord $window 1]
  set word1 [emBufferSubstring $window $p1 $e1]
  if {![regexp {^[a-zA-Z0-9]+} $word0] || ![regexp {^[a-zA-Z0-9]+} $word0]} {bell; return [emSetPoint $window $point] }
  emDeleteRegion $window $p1 $e1
  emSetPoint $window $p1
  emInsert $window $word0
  emSetPoint $window $p0
  emDeleteRegion $window $p0 $e0
  emInsert $window $word1
  set emLastCommand emTransposeWords
  return [emSetPoint $window $e1]
}
proc emDowncaseWord {{window {}} {count {}}} {
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  global emLastCommand
  set mark [emPoint $window]
  set point [emForwardWord $window $count]
  set text [string tolower [emBufferSubstring $window $mark $point]]
  emDeleteRegion $window $mark $point
  emInsert $window $text
  set emLastCommand emDowncaseWord
  return [emSetPoint $window]
}
proc emUpcaseWord {{window {}} {count {}}} {
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  global emLastCommand
  set mark [emPoint $window]
  set point [emForwardWord $window $count]
  set text [string toupper [emBufferSubstring $window $mark $point]]
  emDeleteRegion $window $mark $point
  emInsert $window $text
  set emLastCommand emUpcaseWord
  return [emSetPoint $window]
}
proc emBackToIndentation {{window {}} {count {}}} {
  set count [emGetCount $count]
  if {"$window"=={}} { set window [focus] }
  global emLastCommand
  emBeginningOfLine $window
  while {[emPoint $window]!=[emEOL $window] && [emLookingAt $window {[ 	]} 1]>=0} {
    emForwardChar $window
  }
  set emLastCommmand emBackToIndentation
  return [emSetPoint $window]
}
proc emForwardSentence {{window {}} {count {}}} {
  # Just a stub to be defined
  if {"$window"=={}} { set window [focus] }
  set count [emGetCount $count]
  bell
  return [emPoint $window]
}
proc emBackwardSentence {{window {}} {count {}}} {
  # Just a stub to be defined
  if {"$window"=={}} { set window [focus] }
  set count [emGetCount $count]
  bell
  return [emPoint $window]
}
