/*-
 * Copyright (c) 1993, 1994 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
extern "C" {
#include <tcl.h>
}
#include <stdlib.h>
#include <iostream.h>
#include <string.h>
#include <ctype.h>
#include "tclmidi.h"
#include "TclmIntp.h"
#include "Song.h"
#ifdef USE_MPU401
#include "MPU401.h"
#endif
#ifdef USE_MPU401THREAD
#include "MPU401Thread.h"
#endif

#ifdef _WINDOWS
extern "C" Tcl_CmdProc __loadds Tclm_MidiPlay;
extern "C" Tcl_CmdProc __loadds Tclm_MidiRecord;
extern "C" Tcl_CmdProc __loadds Tclm_MidiStop;
extern "C" Tcl_CmdProc __loadds Tclm_MidiWait;
extern "C" Tcl_CmdProc __loadds Tclm_MidiDevice;
extern "C" Tcl_CmdProc __loadds Tclm_MidiTime;
#else
static int Tclm_MidiPlay(ClientData client_data, Tcl_Interp *interp, int argc,
    char **argv);
static int Tclm_MidiRecord(ClientData client_data, Tcl_Interp *interp, int argc,
    char **argv);
static int Tclm_MidiStop(ClientData client_data, Tcl_Interp *interp, int argc,
    char **argv);
static int Tclm_MidiWait(ClientData client_data, Tcl_Interp *interp, int argc,
    char **argv);
static int Tclm_MidiDevice(ClientData client_data, Tcl_Interp *interp, int argc,
    char **argv);
static int Tclm_MidiTime(ClientData client_data, Tcl_Interp *interp, int argc,
    char **argv);
#endif
static void GetTime(ostrstream &buf, MidiDevice *dev, TimeType type);

int
Tclm_PlayInit(Tcl_Interp *interp, TclmInterp *tclm_interp)
{

#if defined(USE_MPU401)
	tclm_interp->SetMidiDevice(new MPU401("/dev/midi0"));
#elif defined(USE_MPU401THREAD)
	tclm_interp->SetMidiDevice(new MPU401Thread("/dev/midi0"));
#else
	tclm_interp->SetMidiDevice(0);
#endif

	Tcl_CreateCommand(interp, "midiplay", Tclm_MidiPlay, tclm_interp, 0);
	Tcl_CreateCommand(interp, "midirecord", Tclm_MidiRecord,
	    tclm_interp, 0);
	Tcl_CreateCommand(interp, "midistop", Tclm_MidiStop, tclm_interp, 0);
	Tcl_CreateCommand(interp, "midiwait", Tclm_MidiWait, tclm_interp, 0);
	Tcl_CreateCommand(interp, "mididevice", Tclm_MidiDevice,
	    tclm_interp, 0);
	Tcl_CreateCommand(interp, "miditime", Tclm_MidiTime, tclm_interp, 0);
	return (TCL_OK);
}

int
#ifdef _WINDOWS
FAR PASCAL __loadds
#endif
Tclm_MidiPlay(ClientData client_data, Tcl_Interp *interp, int argc,
    char **argv)
{
	TclmInterp *tclm_interp;
	Song *song;
	MidiDevice *dev;
	int repeat;

	if (argc != 2 && argc != 3) {
		Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " MidiID ?repeat?\"", 0);
		return (TCL_ERROR);
	}
	tclm_interp = (TclmInterp *)client_data;

	dev = tclm_interp->GetMidiDevice();
	if (dev == 0) {
		Tcl_SetResult(interp, "0", TCL_STATIC);
		return (TCL_OK);
	}

	if ((song = tclm_interp->GetSong(argv[1])) == 0) {
		Tcl_AppendResult(interp, "bad key ", argv[1], 0);
		return (TCL_ERROR);
	}

	repeat = 0;
	if (argc == 3 && strlen(argv[2]) != 0) {
		if (strcmp(argv[2], "repeat") == 0)
			repeat = 1;
		else {
			Tcl_AppendResult(interp, "bad repeat option: should "
			    "be \"", argv[0], " MidiID ?repeat?\"", 0);
			return (TCL_ERROR);
		}
	}

	if (!dev->Play(song, repeat)) {
		Tcl_AppendResult(interp, "couldn't play song \n",
		    dev->GetError(), 0);
		return (TCL_ERROR);
	}
	Tcl_SetResult(interp, "1", TCL_STATIC);
	return (TCL_OK);
}

int
#ifdef _WINDOWS
FAR PASCAL __loadds
#endif
Tclm_MidiRecord(ClientData client_data, Tcl_Interp *interp, int argc,
    char **argv)
{
	TclmInterp *tclm_interp;
	Song *rsong, *psong;
	MidiDevice *dev;
	int repeat;

	if (argc < 2 || argc > 4) {
		Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " RecMidiID ?PlayMidiID ?repeat??\"", 0);
		return (TCL_ERROR);
	}
	tclm_interp = (TclmInterp *)client_data;

	dev = tclm_interp->GetMidiDevice();
	if (dev == 0) {
		Tcl_SetResult(interp, "0", TCL_STATIC);
		return (TCL_OK);
	}

	if ((rsong = tclm_interp->GetSong(argv[1])) == 0) {
		Tcl_AppendResult(interp, "bad key ", argv[1], 0);
		return (TCL_ERROR);
	}

	psong = 0;
	repeat = 0;
	if (argc > 2) {
		if ((psong = tclm_interp->GetSong(argv[2])) == 0) {
			Tcl_AppendResult(interp, "bad key ", argv[2], 0);
			return (TCL_ERROR);
		}

		if (argc == 4 && strlen(argv[3]) != 0) {
			if (strcmp(argv[3], "repeat") == 0)
				repeat = 1;
			else {
				Tcl_AppendResult(interp, "bad repeat flag: ",
				    argv[3], 0);
				return (TCL_ERROR);
			}
		}
	}

	if (!dev->Record(rsong, psong, repeat)) {
		Tcl_AppendResult(interp, "Couldn't record song\n",
		    dev->GetError(), 0);
		return (TCL_ERROR);
	}
	Tcl_SetResult(interp, "1", TCL_STATIC);
	return (TCL_OK);
}

int
#ifdef _WINDOWS
FAR PASCAL __loadds
#endif
Tclm_MidiStop(ClientData client_data, Tcl_Interp *interp, int argc,
    char **argv)
{
	TclmInterp *tclm_interp;
	MidiDevice *dev;

	if (argc != 1) {
		Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], "\"", 0);
		return (TCL_ERROR);
	}
	tclm_interp = (TclmInterp *)client_data;

	dev = tclm_interp->GetMidiDevice();
	if (dev == 0) {
		Tcl_SetResult(interp, "0", TCL_STATIC);
		return (TCL_OK);
	}

	if (!dev->Stop()) {
		Tcl_AppendResult(interp, "Couldn't stop playing/recording\n",
		    dev->GetError(), 0);
		return (TCL_ERROR);
	}
	Tcl_SetResult(interp, "1", TCL_STATIC);
	return (TCL_OK);
}

int
#ifdef _WINDOWS
FAR PASCAL __loadds
#endif
Tclm_MidiWait(ClientData client_data, Tcl_Interp *interp, int argc,
    char **argv)
{
	TclmInterp *tclm_interp;
	MidiDevice *dev;

	if (argc != 1) {
		Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], 0);
		return (TCL_ERROR);
	}
	tclm_interp = (TclmInterp *)client_data;
	dev = tclm_interp->GetMidiDevice();
	if (dev == 0) {
		Tcl_SetResult(interp, "0", TCL_STATIC);
		return (TCL_OK);
	}
	if (!dev->Wait()) {
		Tcl_AppendResult(interp, "Couldn't wait for playing/recording "
		    "to stop\n", dev->GetError(), 0);
		return (TCL_ERROR);
	}
	Tcl_SetResult(interp, "1", TCL_STATIC);
	return (TCL_OK);
}

int
#ifdef _WINDOWS
FAR PASCAL __loadds
#endif
Tclm_MidiDevice(ClientData client_data, Tcl_Interp *interp, int argc,
    char **argv)
{
	TclmInterp *tclm_interp;
	MidiDevice *dev;
	ostrstream *buf;
	char *str, **sub_argv;
	int i, sub_argc, value;

	tclm_interp = (TclmInterp *)client_data;
	dev = tclm_interp->GetMidiDevice();
	if (dev == 0) {
		Tcl_SetResult(interp, "0", TCL_STATIC);
		return (TCL_OK);
	}

	/*
	 * mididevice with no args means just check for device
	 * support.
	 */
	if (argc == 1) {
		Tcl_SetResult(interp, "1", TCL_STATIC);
		return (TCL_OK);
	}

	for (i = 1; i < argc; i++) {
		// loop through each arg and either set or return values
		if (Tcl_SplitList(interp, argv[i], &sub_argc, &sub_argv)
		    != TCL_OK)
			return (TCL_ERROR);
		if (strcmp(sub_argv[0], "name") == 0) {
			switch (sub_argc) {
			case 1:
				/* return the value */
				buf = new ostrstream;
				*buf << "name \"" << dev->GetName() << "\""
				    << ends;
				str = buf->str();
				Tcl_AppendElement(interp, str);
				delete str;
				delete buf;
				break;
			case 2:
				/* set the value */
				dev->SetName(sub_argv[1]);
				break;
			default:
				/* wrong args */
				Tcl_AppendResult(interp, "Wrong number of "
				    "args: should be \"", argv[0], " name "
				    "?DevName?\"", 0);
				return (0);
			}
		} else if (strcmp(sub_argv[0], "midithru") == 0) {
			switch (sub_argc) {
			case 1:
				/* return the value */
				buf = new ostrstream;
				*buf << "midithru " <<
				    (dev->GetMidiThru() ? "on" : "off")
				    << ends;
				str = buf->str();
				Tcl_AppendElement(interp, str);
				delete str;
				delete buf;
				break;
			case 2:
				/* set the value */
				if (Tcl_GetBoolean(interp, sub_argv[1], &value)
				    != TCL_OK)
					return (TCL_ERROR);
				dev->SetMidiThru(value);
				break;
			default:
				/* wrong args */
				Tcl_AppendResult(interp, "Wrong number of "
				    "args: should be \"", argv[0], " midithru "
				    "?on|off?\"", 0);
				return (0);
			}
		} else {
			Tcl_AppendResult(interp, "Bad parameter \"",
			    sub_argv[0], "\"", 0);
			return (TCL_ERROR);
		}
		free(sub_argv);
	}
	return (TCL_OK);
}

int
#ifdef _WINDOWS
FAR PASCAL __loadds
#endif
Tclm_MidiTime(ClientData client_data, Tcl_Interp *interp, int argc,
    char **argv)
{
	ostrstream tbuf;
	TimeType tt;
	TclmInterp *tclm_interp;
	MidiDevice *dev;
	char *str;

	if (argc < 2 || argc > 3) {
		Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " {smf|smpte} ?value?\"", 0);
		return (TCL_ERROR);
	}
	tclm_interp = (TclmInterp *)client_data;
	dev = tclm_interp->GetMidiDevice();
	if (dev == 0) {
		Tcl_SetResult(interp, "0", TCL_STATIC);
		return (TCL_OK);
	}
	if (strcmp(argv[1], "smf") == 0)
		tt = SMFTIME;
	else if (strcmp(argv[1], "smpte") == 0)
		tt = SMPTETIME;
	else {
		Tcl_AppendResult(interp, "bad time type \"", argv[1],
		    "\" must be smf or smpte", 0);
		return (TCL_ERROR);
	}

	if (argc == 3) {
		Tcl_SetResult(interp, "Setting time not currently supported",
		    0);
		return (TCL_ERROR);
	} else {
		GetTime(tbuf, dev, tt);
		str = tbuf.str();
		Tcl_SetResult(interp, str, TCL_VOLATILE);
		delete str;
	}
	return (TCL_OK);
}

void
GetTime(ostrstream &buf, MidiDevice *dev, TimeType type)
{
	SMPTETime smpte_t;
	unsigned long smf_t;

	switch (type) {
	case SMFTIME:
		if (!dev->GetTime(SMFTIME, &smf_t))
			buf << "ERR";
		else
			buf << smf_t;
		buf << ends;
		break;
	case SMPTETIME:
		if (!dev->GetTime(SMPTETIME, &smpte_t))
			buf << "ERR";
		else {
			if (!(smpte_t.GetFlags() & SMPTE_INSYNC))
				buf << "NOSYNC";
			else
				buf << smpte_t;
			buf << ends;
		}
		break;
	default:
		buf << "Bad TimeType" << ends;
		break;
	}
}
