/*-
 * Copyright (c) 1993, 1994 Michael B. Durian.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Michael B. Durian.
 * 4. The name of the the Author may be used to endorse or promote 
 *    products derived from this software without specific prior written 
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
extern "C" {
#include <tcl.h>
}
#include <strstream.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#include "tclmidi.h"
#include "tclmEvnt.h"

void
Tclm_PrintEvent(ostream &buf, Event *e)
{
	char *str;

	switch (e->GetType()) {
	case NOTEOFF:
		if (((NoteEvent *)e)->GetNotePair() != 0) {
			buf << ends;
			return;
		}
		str = Tclm_PrintNoteOff((NoteOffEvent *)e);
		break;
	case NOTEON:
		if (((NoteEvent *)e)->GetNotePair() == 0)
			str = Tclm_PrintNoteOn((NoteOnEvent *)e);
		else {
			if (((NoteEvent *)e)->GetVelocity() == 0) {
				buf << ends;
				return;
			}
			str = Tclm_PrintNote((NoteOnEvent *)e);
		}
		break;
	case KEYPRESSURE:
		str = Tclm_PrintKeyPressure((KeyPressureEvent *)e);
		break;
	case PARAMETER:
		str = Tclm_PrintParameter((ParameterEvent *)e);
		break;
	case PROGRAM:
		str = Tclm_PrintProgram((ProgramEvent *)e);
		break;
	case CHANNELPRESSURE:
		str = Tclm_PrintChannelPressure((ChannelPressureEvent *)e);
		break;
	case PITCHWHEEL:
		str = Tclm_PrintPitchWheel((PitchWheelEvent *)e);
		break;
	case SYSTEMEXCLUSIVE:
		str = Tclm_PrintSystemExclusive((SystemExclusiveEvent *)e);
		break;
	case METASEQUENCENUMBER:
		str = Tclm_PrintMetaSequenceNumber(
		    (MetaSequenceNumberEvent *)e);
		break;
	case METATEXT:
		str = Tclm_PrintMetaText((MetaTextEvent *)e);
		break;
	case METACOPYRIGHT:
		str = Tclm_PrintMetaCopyright((MetaCopyrightEvent *)e);
		break;
	case METASEQUENCENAME:
		str = Tclm_PrintMetaSequenceName((MetaSequenceNameEvent *)e);
		break;
	case METAINSTRUMENTNAME:
		str = Tclm_PrintMetaInstrumentName(
		    (MetaInstrumentNameEvent *)e);
		break;
	case METALYRIC:
		str = Tclm_PrintMetaLyric((MetaLyricEvent *)e);
		break;
	case METAMARKER:
		str = Tclm_PrintMetaMarker((MetaMarkerEvent *)e);
		break;
	case METACUE:
		str = Tclm_PrintMetaCue((MetaCueEvent *)e);
		break;
	case METACHANNELPREFIX:
		str = Tclm_PrintMetaChannelPrefix((MetaChannelPrefixEvent *)e);
		break;
	case METAPORTNUMBER:
		str = Tclm_PrintMetaPortNumber((MetaPortNumberEvent *)e);
		break;
	case METAENDOFTRACK:
		str = Tclm_PrintMetaEndOfTrack((MetaEndOfTrackEvent *)e);
		break;
	case METATEMPO:
		str = Tclm_PrintMetaTempo((MetaTempoEvent *)e);
		break;
	case METASMPTE:
		str = Tclm_PrintMetaSMPTE((MetaSMPTEEvent *)e);
		break;
	case METATIME:
		str = Tclm_PrintMetaTime((MetaTimeEvent *)e);
		break;
	case METAKEY:
		str = Tclm_PrintMetaKey((MetaKeyEvent *)e);
		break;
	case METASEQUENCERSPECIFIC:
		str = Tclm_PrintMetaSequencerSpecific(
		    (MetaSequencerSpecificEvent *)e);
		break;
	case METAUNKNOWN:
		str = Tclm_PrintMetaUnknown((MetaUnknownEvent *)e);
		break;
	default:
		str = 0;
		break;
	}
	buf << e->GetTime() << " " << str << ends;
	delete str;
}

char *
Tclm_PrintNoteOff(NoteOffEvent *e)
{
	ostrstream buf;

	buf << "NoteOff " << (int)e->GetChannel() << " " << (int)e->GetPitch()
	    << " " << (int)e->GetVelocity() << ends;
	return (buf.str());
}

char *
Tclm_PrintNoteOn(NoteOnEvent *e)
{
	ostrstream buf;

	buf << "NoteOn " << (int)e->GetChannel() << " " << (int)e->GetPitch()
	    << " " << (int)e->GetVelocity() << ends;
	return (buf.str());
}

char *
Tclm_PrintNote(NoteOnEvent *e)
{
	ostrstream buf;

	buf << "Note " << (int)e->GetChannel() << " " << (int)e->GetPitch()
	    << " " << (int)e->GetVelocity() << " " <<
	    (e->GetNotePair()->GetTime() - e->GetTime()) << ends;
	return (buf.str());
}

char *
Tclm_PrintKeyPressure(KeyPressureEvent *e)
{
	ostrstream buf;

	buf << "KeyPressure " << (int)e->GetChannel() << " "
	    << (int)e->GetPitch() << " " << (int)e->GetPressure() << ends;
	return (buf.str());
}

char *
Tclm_PrintParameter(ParameterEvent *e)
{
	ostrstream buf;

	buf << "Parameter " << (int)e->GetChannel() << " "
	    << (int)e->GetParameter() << " " << (int)e->GetValue() << ends;
	return (buf.str());
}

char *
Tclm_PrintProgram(ProgramEvent *e)
{
	ostrstream buf;

	buf << "Program " << (int)e->GetChannel() << " "
	    << (int)e->GetValue() << ends;
	return (buf.str());
}

char *
Tclm_PrintChannelPressure(ChannelPressureEvent *e)
{
	ostrstream buf;

	buf << "ChannelPressure " << (int)e->GetChannel() << " "
	    << (int)e->GetPressure() << ends;
	return (buf.str());
}

char *
Tclm_PrintPitchWheel(PitchWheelEvent *e)
{
	ostrstream buf;

	buf << "PitchWheel " << (int)e->GetChannel() << " " <<
	    e->GetValue() << ends;
	return (buf.str());
}

char *
Tclm_PrintSystemExclusive(SystemExclusiveEvent *e)
{
	ostrstream buf;

	buf << "SystemExclusive ";
	if (e->GetContinued() == 1)
		buf << "continued ";
	buf << "{";
	Tclm_PrintData(buf, e->GetData(), e->GetLength());
	buf << "}" << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaSequenceNumber(MetaSequenceNumberEvent *e)
{
	ostrstream buf;

	buf << "MetaSequenceNumber " << e->GetNumber() << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaText(MetaTextEvent *e)
{
	ostrstream buf;

	buf << "MetaText \"" << e->GetString() << "\"" << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaCopyright(MetaCopyrightEvent *e)
{
	ostrstream buf;

	buf << "MetaCopyright \"" << e->GetString() << "\"" << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaSequenceName(MetaSequenceNameEvent *e)
{
	ostrstream buf;

	buf << "MetaSequenceName \"" << e->GetString() << "\"" << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaInstrumentName(MetaInstrumentNameEvent *e)
{
	ostrstream buf;

	buf << "MetaInstrumentName \"" << e->GetString() << "\"" << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaLyric(MetaLyricEvent *e)
{
	ostrstream buf;

	buf << "MetaLyric \"" << e->GetString() << "\"" << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaMarker(MetaMarkerEvent *e)
{
	ostrstream buf;

	buf << "MetaMarker \"" << e->GetString() << "\"" << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaCue(MetaCueEvent *e)
{
	ostrstream buf;

	buf << "MetaCue \"" << e->GetString() << "\"" << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaChannelPrefix(MetaChannelPrefixEvent *e)
{
	ostrstream buf;

	buf << "MetaChannelPrefix {";
	Tclm_PrintData(buf, e->GetData(), e->GetLength());
	buf << "}" << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaPortNumber(MetaPortNumberEvent *e)
{
	ostrstream buf;

	buf << "MetaPortNumber " << (int)e->GetPort() << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaEndOfTrack(MetaEndOfTrackEvent *e)
{
	ostrstream buf;

	buf << "MetaEndOfTrack" << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaTempo(MetaTempoEvent *e)
{
	ostrstream buf;

	buf << "MetaTempo " << e->GetTempo() << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaSMPTE(MetaSMPTEEvent *e)
{
	ostrstream buf;

	buf << "MetaSMPTE " << (int)e->GetHour() << " " << (int)e->GetMinute()
	    << " " << (int)e->GetSecond() << " " << (int)e->GetFrame()
	    << " " << (int)e->GetFractionalFrame() << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaTime(MetaTimeEvent *e)
{
	ostrstream buf;

	buf << "MetaTime " << (int)e->GetNumerator()
	    << " " << (int)e->GetDenominator()
	    << " " << (int)e->GetClocksPerBeat()
	    << " " << (int)e->Get32ndNotesPerQuarterNote() << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaKey(MetaKeyEvent *e)
{
	ostrstream buf;

	buf << "MetaKey \"" << e->GetKeyStr() << "\" " << e->GetModeStr()
	    << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaSequencerSpecific(MetaSequencerSpecificEvent *e)
{
	ostrstream buf;

	buf << "MetaSequencerSpecific {";
	Tclm_PrintData(buf, e->GetData(), e->GetLength());
	buf << "}" << ends;
	return (buf.str());
}

char *
Tclm_PrintMetaUnknown(MetaUnknownEvent *e)
{
	ostrstream buf;

	buf << "MetaUnknown " << (int)e->GetMetaType() << " {";
	Tclm_PrintData(buf, e->GetData(), e->GetLength());
	buf << "}" << ends;
	return (buf.str());
}

Event *
Tclm_ParseEvent(Tcl_Interp *interp, char *str)
{
	Event *event;
	Event *(*pfunc)(Tcl_Interp *, long, int, char **);
	char **argv, **aptr;;
	char *name;
	long time;
	int argc, i, length;

	if (Tcl_SplitList(interp, str, &argc, &argv) != TCL_OK)
		return (0);
	aptr = argv;

	if (Tcl_GetLong(interp, argv[0], &time) != TCL_OK)
		return (0);

	length = strlen(argv[1]);
	name = new char[length + 1];
	for (i = 0; i < length; i++)
		name[i] = tolower(argv[1][i]);
	name[i] = '\0';

	argv++;
	argc--;
	
	pfunc = 0;
	switch (name[0]) {
	case 'c':
		if (strncmp(name, "channelpressure", length) == 0)
			pfunc = Tclm_ParseChannelPressure;
		break;
	case 'k':
		if (strncmp(name, "keypressure", length) == 0)
			pfunc = Tclm_ParseKeyPressure;
		break;
	case 'm':
		// meta events
		switch (name[4]) {
		case 'c':
			if (strncmp(name, "metachannelprefix", length) == 0)
				pfunc = Tclm_ParseMetaChannelPrefix;
			else if (strncmp(name, "metacopyright", length) == 0)
				pfunc = Tclm_ParseMetaCopyright;
			else if (strncmp(name, "metacue", length) == 0)
				pfunc = Tclm_ParseMetaCue;
			break;
		case 'e':
			if (strncmp(name, "metaendoftrack", length) == 0)
				pfunc = Tclm_ParseMetaEndOfTrack;
			break;
		case 'i':
			if (strncmp(name, "metainstrumentname", length) == 0)
				pfunc = Tclm_ParseMetaInstrumentName;
			break;
		case 'k':
			if (strncmp(name, "metakey", length) == 0)
				pfunc = Tclm_ParseMetaKey;
			break;
		case 'l':
			if (strncmp(name, "metalyric", length) == 0)
				pfunc = Tclm_ParseMetaLyric;
			break;
		case 'm':
			if (strncmp(name, "metamarker", length) == 0)
				pfunc = Tclm_ParseMetaMarker;
			break;
		case 'p':
			if (strncmp(name, "metaportnumber", length) == 0)
				pfunc = Tclm_ParseMetaPortNumber;
			break;
		case 's':
			if (strncmp(name, "metasequencename", length) == 0)
				pfunc = Tclm_ParseMetaSequenceName;
			else if (strncmp(name, "metasequencenumber", length)
			    == 0)
				pfunc = Tclm_ParseMetaSequenceNumber;
			else if (strncmp(name, "metasequencerspecific", length)
			    == 0)
				pfunc = Tclm_ParseMetaSequencerSpecific;
			else if (strncmp(name, "metasmpte", length) == 0)
				pfunc = Tclm_ParseMetaSMPTE;
			break;
		case 't':
			if (strncmp(name, "metatempo", length) == 0)
				pfunc = Tclm_ParseMetaTempo;
			else if (strncmp(name, "metatext", length) == 0)
				pfunc = Tclm_ParseMetaText;
			else if (strncmp(name, "metatime", length) == 0)
				pfunc = Tclm_ParseMetaTime;
			break;
		case 'u':
			if (strncmp(name, "metaunknown", length) == 0)
				pfunc = Tclm_ParseMetaUnknown;
			break;
		}
		break;
	case 'n':
		if (strncmp(name, "note", length) == 0)
			pfunc = Tclm_ParseNote;
		else if (strncmp(name, "noteoff", length) == 0)
			pfunc = Tclm_ParseNoteOff;
		else if (strncmp(name, "noteon", length) == 0)
			pfunc = Tclm_ParseNoteOn;
		break;
	case 'p':
		if (strncmp(name, "parameter", length) == 0)
			pfunc = Tclm_ParseParameter;
		else if (strncmp(name, "pitchwheel", length) == 0)
			pfunc = Tclm_ParsePitchWheel;
		else if (strncmp(name, "program", length) == 0)
			pfunc = Tclm_ParseProgram;
		break;
	case 's':
		if (strncmp(name, "systemexclusive", length) == 0)
			pfunc = Tclm_ParseSystemExclusive;
		break;
	}

	if (pfunc == 0) {
		Tcl_AppendResult(interp, "bad event type ", argv[0], 0);
		free(aptr);
		delete name;
		return (0);
	}
	event = pfunc(interp, time, argc, argv);
	free(aptr);
	delete name;
	return (event);
}

Event *
Tclm_ParseNoteOff(Tcl_Interp *interp, long time, int argc, char **argv)
{
	unsigned char channel, pitch, velocity;

	if (argc != 3 && argc != 4) {
		Tcl_SetResult(interp, "bad event: should be \"time NoteOff "
		    "channel pitch ?velocity?\"", TCL_STATIC);
		return (0);
	}

	if (!Tclm_ParseDataByte(interp, argv[1], &channel))
		return (0);
	if (!Tclm_ParseDataByte(interp, argv[2], &pitch))
		return (0);
	if (argc == 3)
		velocity = 0;
	else if (!Tclm_ParseDataByte(interp, argv[3], &velocity))
		return (0);

	return (new NoteOffEvent(time, channel, pitch, velocity));
}

Event *
Tclm_ParseNoteOn(Tcl_Interp *interp, long time, int argc, char **argv)
{
	unsigned char channel, pitch, velocity;

	if (argc != 4) {
		Tcl_SetResult(interp, "bad event: should be \"time NoteOn "
		    "channel pitch velocity\"", TCL_STATIC);
		return (0);
	}

	if (!Tclm_ParseDataByte(interp, argv[1], &channel))
		return (0);
	if (!Tclm_ParseDataByte(interp, argv[2], &pitch))
		return (0);
	if (!Tclm_ParseDataByte(interp, argv[3], &velocity))
		return (0);

	return (new NoteOnEvent(time, channel, pitch, velocity));
}

Event *
Tclm_ParseNote(Tcl_Interp *interp, long time, int argc, char **argv)
{
	NoteOnEvent *event;
	NoteOffEvent *off;
	long duration;
	unsigned char channel, pitch, velocity;

	if (argc != 5) {
		Tcl_SetResult(interp, "bad event: should be \"time Note "
		    "channel pitch velocity duration\"", TCL_STATIC);
		return (0);
	}

	if (!Tclm_ParseDataByte(interp, argv[1], &channel))
		return (0);
	if (!Tclm_ParseDataByte(interp, argv[2], &pitch))
		return (0);
	if (!Tclm_ParseDataByte(interp, argv[3], &velocity))
		return (0);
	if (Tcl_GetLong(interp, argv[4], &duration) != TCL_OK)
		return (0);

	event = new NoteOnEvent();
	event->SetTime(time);
	event->SetChannel(channel);
	event->SetPitch(pitch);
	event->SetVelocity(velocity);

	off = new NoteOffEvent();
	off->SetTime(time + duration);
	off->SetChannel(channel);
	off->SetPitch(pitch);
	event->SetNotePair(off);
	off->SetNotePair(event);

	return (event);
}

Event *
Tclm_ParseKeyPressure(Tcl_Interp *interp, long time, int argc, char **argv)
{
	unsigned char channel, pitch, pressure;

	if (argc != 4) {
		Tcl_SetResult(interp, "bad event: should be "
		    "\"time KeyPressure channel pitch pressure\"", TCL_STATIC);
		return (0);
	}

	if (!Tclm_ParseDataByte(interp, argv[1], &channel))
		return (0);
	if (!Tclm_ParseDataByte(interp, argv[2], &pitch))
		return (0);
	if (!Tclm_ParseDataByte(interp, argv[3], &pressure))
		return (0);

	return (new KeyPressureEvent(time, channel, pitch, pressure));
}

Event *
Tclm_ParseParameter(Tcl_Interp *interp, long time, int argc, char **argv)
{
	unsigned char channel, parameter, value;

	if (argc != 4) {
		Tcl_SetResult(interp, "bad event: should be \"time Parameter "
		    "channel parameter value\"", TCL_STATIC);
		return (0);
	}

	if (!Tclm_ParseDataByte(interp, argv[1], &channel))
		return (0);
	if (!Tclm_ParseDataByte(interp, argv[2], &parameter))
		return (0);
	if (!Tclm_ParseDataByte(interp, argv[3], &value))
		return (0);

	return (new ParameterEvent(time, channel, parameter, value));
}

Event *
Tclm_ParseProgram(Tcl_Interp *interp, long time, int argc, char **argv)
{
	unsigned char channel, value;

	if (argc != 3) {
		Tcl_SetResult(interp, "bad event: should be \"time Program "
		    "channel value\"", TCL_STATIC);
		return (0);
	}

	if (!Tclm_ParseDataByte(interp, argv[1], &channel))
		return (0);
	if (!Tclm_ParseDataByte(interp, argv[2], &value))
		return (0);

	return (new ProgramEvent(time, channel, value));
}

Event *
Tclm_ParseChannelPressure(Tcl_Interp *interp, long time, int argc, char **argv)
{
	unsigned char channel, pressure;

	if (argc != 3) {
		Tcl_SetResult(interp, "bad event: should be "
		    "\"time ChannelPressure channel pressure\"", TCL_STATIC);
		return (0);
	}

	if (!Tclm_ParseDataByte(interp, argv[1], &channel))
		return (0);
	if (!Tclm_ParseDataByte(interp, argv[2], &pressure))
		return (0);

	return (new ChannelPressureEvent(time, channel, pressure));
}

Event *
Tclm_ParsePitchWheel(Tcl_Interp *interp, long time, int argc, char **argv)
{
	unsigned char channel;
	int value;

	if (argc != 3) {
		Tcl_SetResult(interp, "bad event: should be \"time PitchWheel "
		    "channel value\"", TCL_STATIC);
		return (0);
	}

	if (!Tclm_ParseDataByte(interp, argv[1], &channel))
		return (0);
	if (Tcl_GetInt(interp, argv[2], &value) != TCL_OK)
		return (0);

	return (new PitchWheelEvent(time, channel, value));
}

Event *
Tclm_ParseSystemExclusive(Tcl_Interp *interp, long time, int argc, char **argv)
{
	char **str;
	SystemExclusiveEvent *event;
	unsigned char *data;
	int i, len, val;

	if ((argc != 2 && argc != 3) || (argc == 3 && strncmp(argv[1], "cont",
	    4) != 0)) {
		Tcl_SetResult(interp, "bad event: should be "
		    "\"time SystemExclusive ?continued? {data ?data ...?}\"",
		    TCL_STATIC);
		return (0);
	}

	if (argc == 2) {
		if (Tcl_SplitList(interp, argv[1], &len, &str) != TCL_OK)
			return (0);
	} else {
		if (Tcl_SplitList(interp, argv[2], &len, &str) != TCL_OK)
			return (0);
	}

	data = new unsigned char[len];
	if (data == 0)
		return (0);

	for (i = 0; i < len; i++) {
		if (Tcl_GetInt(interp, str[i], &val) != TCL_OK)
			return (0);
		data[i] = val;
	}

	free(str);
	event = new SystemExclusiveEvent(time, data, len);
	if (argc == 3)
		event->SetContinued(1);
	delete data;

	return (event);
}

Event *
Tclm_ParseMetaSequenceNumber(Tcl_Interp *interp, long time, int argc,
    char **argv)
{
	int num;

	if (argc != 2) {
		Tcl_SetResult(interp, "bad event: should be "
		    "\"time MetaSequenceNumber number\"", TCL_STATIC);
		return (0);
	}

	if (Tcl_GetInt(interp, argv[1], &num) != TCL_OK)
		return (0);

	return (new MetaSequenceNumberEvent(time, num));
}

Event *
Tclm_ParseMetaText(Tcl_Interp *interp, long time, int argc, char **argv)
{

	if (argc != 2) {
		Tcl_SetResult(interp, "bad event: should be \"time MetaText "
		    "string\"", TCL_STATIC);
		return (0);
	}

	return (new MetaTextEvent(time, argv[1]));
}

Event *
Tclm_ParseMetaCopyright(Tcl_Interp *interp, long time, int argc, char **argv)
{

	if (argc != 2) {
		Tcl_SetResult(interp, "bad event: should be "
		    "\"time MetaCopyright string\"", TCL_STATIC);
		return (0);
	}

	return (new MetaCopyrightEvent(time, argv[1]));
}

Event *
Tclm_ParseMetaSequenceName(Tcl_Interp *interp, long time, int argc,
    char **argv)
{

	if (argc != 2) {
		Tcl_SetResult(interp, "bad event: should be "
		    "\"time MetaSequenceName string\"", TCL_STATIC);
		return (0);
	}

	return (new MetaSequenceNameEvent(time, argv[1]));
}

Event *
Tclm_ParseMetaInstrumentName(Tcl_Interp *interp, long time, int argc,
    char **argv)
{

	if (argc != 2) {
		Tcl_SetResult(interp, "bad event: should be "
		    "\"time MetaInstrumentName string\"", TCL_STATIC);
		return (0);
	}

	return (new MetaInstrumentNameEvent(time, argv[1]));
}

Event *
Tclm_ParseMetaLyric(Tcl_Interp *interp, long time, int argc, char **argv)
{

	if (argc != 2) {
		Tcl_SetResult(interp, "bad event: should be \"time MetaLyric "
		    "string\"", TCL_STATIC);
		return (0);
	}

	return (new MetaLyricEvent(time, argv[1]));
}

Event *
Tclm_ParseMetaMarker(Tcl_Interp *interp, long time, int argc, char **argv)
{

	if (argc != 2) {
		Tcl_SetResult(interp, "bad event: should be \"time MetaMarker "
		    "string\"", TCL_STATIC);
		return (0);
	}

	return (new MetaMarkerEvent(time, argv[1]));
}

Event *
Tclm_ParseMetaCue(Tcl_Interp *interp, long time, int argc, char **argv)
{

	if (argc != 2) {
		Tcl_SetResult(interp, "bad event: should be \"time MetaCue "
		    "string\"", TCL_STATIC);
		return (0);
	}

	return (new MetaCueEvent(time, argv[1]));
}

Event *
Tclm_ParseMetaChannelPrefix(Tcl_Interp *interp, long time, int argc,
    char **argv)
{
	char **str;
	MetaChannelPrefixEvent *event;
	unsigned char *data;
	int i, len, val;

	if (argc != 2) {
		Tcl_SetResult(interp, "bad event: should be "
		    "\"time MetaChannelPrefix {data ?data ...?}\"", TCL_STATIC);
		return (0);
	}

	if (Tcl_SplitList(interp, argv[1], &len, &str) != TCL_OK)
		return (0);

	data = new unsigned char[len];
	if (data == 0)
		return (0);

	for (i = 0; i < len; i++) {
		if (Tcl_GetInt(interp, str[i], &val) != TCL_OK)
			return (0);
		data[i] = val;
	}

	free(str);
	event = new MetaChannelPrefixEvent(time, data, len);
	delete data;

	return (event);
}

Event *
Tclm_ParseMetaPortNumber(Tcl_Interp *interp, long time, int argc, char **argv)
{
	int port;

	if (argc != 2) {
		Tcl_SetResult(interp, "bad event: should be "
		    "\"time MetaPortNumber port\"", TCL_STATIC);
		return (0);
	}

	if (Tcl_GetInt(interp, argv[1], &port) != TCL_OK)
		return (0);

	return (new MetaPortNumberEvent(time, port));
}

Event *
Tclm_ParseMetaEndOfTrack(Tcl_Interp *interp, long time, int argc, char **argv)
{

	if (argc != 1) {
		Tcl_SetResult(interp, "bad event: should be "
		    "\"time MetaEndOfTrack\"", TCL_STATIC);
		return (0);
	}

	return (new MetaEndOfTrackEvent(time));
}

Event *
Tclm_ParseMetaTempo(Tcl_Interp *interp, long time, int argc, char **argv)
{
	int tempo;

	if (argc != 2) {
		Tcl_SetResult(interp, "bad event: should be \"time MetaTempo "
		    "tempo\"", TCL_STATIC);
		return (0);
	}

	if (Tcl_GetInt(interp, argv[1], &tempo) != TCL_OK)
		return (0);

	return (new MetaTempoEvent(time, tempo));
}

Event *
Tclm_ParseMetaSMPTE(Tcl_Interp *interp, long time, int argc, char **argv)
{
	int hour, minute, second, frame, fractional_frame;

	if (argc != 6) {
		Tcl_SetResult(interp, "bad event: should be \"time MetaSMPTE "
		    "hour minute second frame fractional_frame\"", TCL_STATIC);
		return (0);
	}

	if (Tcl_GetInt(interp, argv[1], &hour) != TCL_OK)
		return (0);
	if (Tcl_GetInt(interp, argv[2], &minute) != TCL_OK)
		return (0);
	if (Tcl_GetInt(interp, argv[3], &second) != TCL_OK)
		return (0);
	if (Tcl_GetInt(interp, argv[4], &frame) != TCL_OK)
		return (0);
	if (Tcl_GetInt(interp, argv[5], &fractional_frame) != TCL_OK)
		return (0);

	return (new MetaSMPTEEvent(time, hour, minute, second, frame,
	    fractional_frame));
}

Event *
Tclm_ParseMetaTime(Tcl_Interp *interp, long time, int argc, char **argv)
{
	int numerator, denominator, clocks, thirty_seconds;

	if (argc != 5) {
		Tcl_SetResult(interp, "bad event: should be \"time MetaTime "
		    "numerator denominator clocks/beat 32nds/quarter\"",
		    TCL_STATIC);
		return (0);
	}

	if (Tcl_GetInt(interp, argv[1], &numerator) != TCL_OK)
		return (0);
	if (Tcl_GetInt(interp, argv[2], &denominator) != TCL_OK)
		return (0);
	if (Tcl_GetInt(interp, argv[3], &clocks) != TCL_OK)
		return (0);
	if (Tcl_GetInt(interp, argv[4], &thirty_seconds) != TCL_OK)
		return (0);

	return (new MetaTimeEvent(time, numerator, denominator, clocks,
	    thirty_seconds));
}

Event *
Tclm_ParseMetaKey(Tcl_Interp *interp, long time, int argc, char **argv)
{
	Key key;
	Mode mode;
	int match;

	if (argc != 3) {
		Tcl_SetResult(interp, "bad event: should be \"time MetaKey "
		    "key mode\"", TCL_STATIC);
		return (0);
	}

	key = StrToKey(argv[1], &match);
	if (!match) {
		Tcl_AppendResult(interp, "bad key: ", argv[1], 0);
		return (0);
	}
	mode = StrToMode(argv[2], &match);
	if (!match) {
		Tcl_AppendResult(interp, "bad mode: ", argv[2], 0);
		return (0);
	}

	return (new MetaKeyEvent(time, key, mode));
}

Event *
Tclm_ParseMetaSequencerSpecific(Tcl_Interp *interp, long time, int argc,
    char **argv)
{
	char **str;
	MetaSequencerSpecificEvent *event;
	unsigned char *data;
	int i, len, val;

	if (argc != 2) {
		Tcl_SetResult(interp, "bad event: should be "
		    "\"time MetaSequencerSpecific {data ?data ...?}\"",
		    TCL_STATIC);
		return (0);
	}

	if (Tcl_SplitList(interp, argv[1], &len, &str) != TCL_OK)
		return (0);

	data = new unsigned char[len];
	if (data == 0)
		return (0);

	for (i = 0; i < len; i++) {
		if (Tcl_GetInt(interp, str[i], &val) != TCL_OK)
			return (0);
		data[i] = val;
	}

	free(str);
	event = new MetaSequencerSpecificEvent(time, data, len);
	delete data;

	return (event);
}

Event *
Tclm_ParseMetaUnknown(Tcl_Interp *interp, long time, int argc, char **argv)
{
	char **str;
	MetaUnknownEvent *event;
	unsigned char *data;
	int i, len, type, val;

	if (argc != 3) {
		Tcl_SetResult(interp, "bad event: should be \"time MetaUnknown "
		    "type {data ?data ...?}\"", TCL_STATIC);
		return (0);
	}

	if (Tcl_GetInt(interp, argv[1], &type) != TCL_OK)
		return (0);

	if (Tcl_SplitList(interp, argv[2], &len, &str) != TCL_OK)
		return (0);

	data = new unsigned char[len];
	if (data == 0)
		return (0);

	for (i = 0; i < len; i++) {
		if (Tcl_GetInt(interp, str[i], &val) != TCL_OK)
			return (0);
		data[i] = val;
	}

	free(str);
	event = new MetaUnknownEvent(time, data, len, type);
	delete data;

	return (event);
}
