/************************************************************************
 ** Copyright (c) 1994, Aaron Jackson
 ** All rights reserved
 **
 ** Permission to use, copy, modify, and distribute this software and
 ** its documentation for any purpose and without fee is hereby granted,
 ** provided that the above copyright notice appear in all copies and
 ** that both that copyright notice and this permission notice appear in
 ** supporting documentation.
 **
 ** In no event shall the author(s) be liable to any party for direct,
 ** indirect, special, incidental, or consequential damages arising out
 ** of the use of this software and its documentation, even if the
 ** authors(s) have been advised of the possibility of such damage.
 **
 ** The authors(s) specifically disclaim any warranties, including but
 ** not limited to, the implied warranties of merchantability and
 ** fitness for a particular purpose.  The software provided hereunder
 ** is on an "as is" basis, and the author(s) have no obligation to
 ** provide maintenance, support, updates, enhancements, or modifications
 ************************************************************************/

static char rcsid[] = "$Header: /home/edsdev/cvstree/tcldev/vwtable/vwTableSet.c,v 1.2 1994/11/11 21:37:10 aajackso Exp $" ;

/**
 ** $Log: vwTableSet.c,v $
 * Revision 1.2  1994/11/11  21:37:10  aajackso
 * *** empty log message ***
 *
 * Revision 1.1.1.1  1994/10/21  13:35:28  aajackso
 * Table Widget
 *
 * Revision 1.1  1994/08/24  00:55:59  aaron
 * *** empty log message ***
 *
 * Revision 1.0.0.1  1994/08/18  00:05:48  aaron
 * *** empty log message ***
 *
 * Revision 1.0  1994/08/17  23:54:06  aaron
 * Initial revision
 *
 **/

#include <stdlib.h>
#include <memory.h>
#include <string.h>
#include <malloc.h>

#include <tk.h>
#include <tcl.h>

#include <vwTable.h>
#include <vwTableCell.h>

#define Allocate_Cell(xx) \
   if ((xx)[0] == NULL) (xx)[0] = (VWTableCell *) calloc(1, sizeof(VWTableCell))

typedef struct StandardHandler StandardHandler ;
struct StandardHandler
{
  char *arg ;
  int (*proc) _ANSI_ARGS_((VWTable*, VWTableCell**, Tcl_Interp*, char*, Tk_Anchor)) ;
};

/************************************************************************
 ** void
 ** VWT_FreeData( VWTable     *vwptr,
 **               VWTableCell *vwcell )
 **
 ** Frees data associated with a cell.
 ************************************************************************/

#ifdef __STDC__
void
VWT_FreeData( VWTable*        vwptr,
              VWTableCell*    vwcell )
#else /** __STDC__ **/
void
VWT_FreeData( vwptr, vwcell )
    VWTable*        vwptr ;
    VWTableCell*    vwcell ;
#endif /** __STDC__ **/
{
  if ( vwcell == NULL )
    return ;

  switch( vwcell->dataType )
    {
    case VWTableTextType:
      if ( vwcell->data.text.data != NULL )
	{
	  vwcell->changed = TRUE ;
	  free(vwcell->data.text.data) ;
	}

      break ;

    case VWTableBitmapType:
      if ( vwcell->data.bitmap.data != VWT_NullPixmap )
	{
	  vwcell->changed = TRUE ;
	  VWT_FreeBitmap( vwptr->display, vwcell->data.bitmap.data ) ;
	}

      break ;
    }

  memset((char *) &vwcell->data, 0x0, sizeof(vwcell->data)) ;
}

/************************************************************************
 ** int
 ** VWT_SetBitmap( VWTable       *vwptr,
 **                VWTableCell  **vwcell,
 **                Tcl_Interp    *interp,
 **                char          *arg,
 **                Tk_Anchor      anchor )
 **
 ** Assigns a bitmap to a cell
 ************************************************************************/

#ifdef __STDC__
int
VWT_SetBitmap( VWTable*                  vwptr,
               VWTableCell**             vwcell,
               Tcl_Interp*               interp,
               char*                     arg,
               Tk_Anchor                 anchor )
#else /** __STDC__ **/
int
VWT_SetBitmap( vwptr, vwcell, interp, arg, anchor )
    VWTable*                  vwptr ;
    VWTableCell**             vwcell ;
    Tcl_Interp*               interp ;
    char*                     arg ;
    Tk_Anchor                 anchor ;
#endif /** __STDC__ **/
{
  VWAddr*         ref ;
  Pixmap          bitmap ;

  unsigned int    w ;
  unsigned int    h ;  

  bitmap = VWT_GetBitmap(interp, vwptr->tkWin, arg, &ref, &w, &h) ;
  if (bitmap == VWT_BadPixmap)
    return TCL_ERROR ;

  if (bitmap == VWT_NullPixmap)
    return TCL_OK ;

  ref->refs++ ;

  Allocate_Cell(vwcell) ;

  vwcell[0]->changed = TRUE ;
  vwcell[0]->dataType = VWTableBitmapType ;
  vwcell[0]->data.bitmap.data = bitmap ;
  vwcell[0]->data.bitmap.width = w ;
  vwcell[0]->data.bitmap.height = h ;
  vwcell[0]->data.bitmap.name = (char *) Tk_GetUid(arg) ;
  vwcell[0]->data.bitmap.namelen = strlen(arg) ;
  vwcell[0]->data.bitmap.anchor = anchor ;

  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_SetPixmap( VWTable       *vwptr,
 **                VWTableCell  **vwcell,
 **                Tcl_Interp    *interp,
 **                char          *arg,
 **                Tk_Anchor      anchor )
 **
 ** Assigns a bitmap to a cell
 ************************************************************************/

#ifdef __STDC__
int
VWT_SetPixmap( VWTable*                  vwptr,
               VWTableCell**             vwcell,
               Tcl_Interp*               interp,
               char*                     arg,
               Tk_Anchor                 anchor )
#else /** __STDC__ **/
int
VWT_SetPixmap( vwptr, vwcell, interp, arg, anchor )
    VWTable*                  vwptr ;
    VWTableCell**             vwcell ;
    Tcl_Interp*               interp ;
    char*                     arg ;
    Tk_Anchor                 anchor ;
#endif /** __STDC__ **/
{
  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_SetText( VWTable*        vwptr,
 **              VWTableCell**   vwcell,
 **              Tcl_Interp*     interp,
 **              char*           arg,
 **              Tk_Anchor       anchor )
 **
 ** Assigns a string to a cell.  Or possibly a number of strings to
 ** a range of cells
 ************************************************************************/

#ifdef __STDC__
int
VWT_SetText( VWTable*                  vwptr,
             VWTableCell**             vwcell,
             Tcl_Interp*               interp,
             char*                     arg,
             Tk_Anchor                 anchor )
#else /** __STDC__ **/
int
VWT_SetText( vwptr, vwcell, interp, arg, anchor )
    VWTable*                  vwptr ;
    VWTableCell**             vwcell ;
    Tcl_Interp*               interp ;
    char*                     arg ;
    Tk_Anchor                 anchor ;
#endif /** __STDC__ **/
{
  Allocate_Cell(vwcell) ;

  vwcell[0]->changed = TRUE ;
  vwcell[0]->dataType = VWTableTextType ;
  vwcell[0]->data.text.anchor = anchor ;
  vwcell[0]->data.text.data = (char *) strdup(arg) ;
  vwcell[0]->data.text.datalen = strlen(arg) ;

  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_SetElement( VWTable       *vwptr,
 **                 VWTableCell  **vwcell,
 **                 Tcl_Interp    *interp,
 **                 char          *element )
 **
 ** Processes a single cell given element.  This function is responsible
 ** for parsing and processing information in that element and then
 ** creating a useable type of cell
 ************************************************************************/

#ifdef __STDC__
int
VWT_SetElement( VWTable*               vwptr,
                VWTableCell**          vwcell,
                Tcl_Interp*            interp,
                char*                  element )
#else /** __STDC__ **/
int
VWT_SetElement( vwptr, vwcell, interp, element )
    VWTable*               vwptr ;
    VWTableCell**          vwcell ;
    Tcl_Interp*            interp ;
    char*                  element ;
#endif /** __STDC__ **/
{
  char *argv[3] ;
  int   argLen[2] ;
  int   error ;

  register int i ;
  static StandardHandler Handler[] =
    {
      { "BITMAP",   VWT_SetBitmap },
#ifdef HAVE_XPM
      { "PIXMAP",   VWT_SetPixmap },
#endif /** HAVE_XPM **/
      { "TEXT",     VWT_SetText },
    };

  static int HandlerSize = sizeof(Handler) / sizeof(Handler[0]) ;

  /** Do any legwork up front **/


  /**
   ** Free old data residing in the cell.  The type of data is unknown,
   ** please use VWT_FreeData()
   **/

  VWT_FreeData(vwptr, *vwcell) ;

  /**
   ** If there is no string, simply return.  The cell is presumed
   ** empty upon return from VWT_FreeData()
   **/

  if (! strncmp("", element, strlen(element)))
    return TCL_OK ;

  error = TclFindElement(interp, element, &argv[0], &argv[1], &argLen[0], NULL) ;
  if (error != TCL_OK)
    return error ;

  /** Add smart hashing here **/

  if (! argLen[0]) return TCL_OK ;

  /**
   ** Determine the information for the handler
   **/

  for ( i = 0 ; i < HandlerSize ; i++ )
    if (! strncmp(argv[0], Handler[i].arg, argLen[0]))
      {
	Tk_Anchor anchor ;
	
	error = TclFindElement(interp, argv[1], &argv[1], &argv[2], &argLen[1], NULL) ;
	if (error != TCL_OK)
	  return error ;
	
	argv[2][-1] = 0x0 ;
	
	error = Tk_GetAnchor(interp, argv[1], &anchor) ;
	if (error != TCL_OK)
	  return error ;

	return Handler[i].proc(vwptr, vwcell, interp, argv[2], anchor) ;
      }

  return TCL_ERROR ;
}

/************************************************************************
 ** static int
 ** Handle_SetSingle( VWTable       *vwptr,
 **                   Tcl_Interp    *interp,
 **                   int            argc,
 **                   char          *argv[] )
 **
 ** Sets a single cell.  One might wish to think of it as a subset of
 ** a range, but the arguments are different and this is not the
 ** function which deals with dismantling and such.
 ************************************************************************/

#ifdef __STDC__
static int
Handle_SetSingle( VWTable*             vwptr,
                  Tcl_Interp*          interp,
                  int                  argc,
                  char*                argv[] )
#else /** __STDC__ **/
static int
Handle_SetSingle( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  int             x ;
  int             y ;
  int             result ;
  VWTableCell**   vwcell ;
  char            indx[21] ;  /** 10x10 **/

  if ((VWT_GetBoundingPoint(interp, argv[0], &x, vwptr->columns, TRUE) != TCL_OK) ||
      (VWT_GetBoundingPoint(interp, argv[1], &y, vwptr->rows, TRUE) != TCL_OK))
    return TCL_ERROR ;

  VWT_ResetBinding(vwptr, interp, FALSE) ;

  vwcell = VWMatrixIndex(vwptr->matrix, VWTableCell*, x, y, 0) ;
  result = VWT_SetElement(vwptr, vwcell, interp, argv[2]) ;
  if (( vwptr->varName ) && ( result == TCL_OK ))
    {
      char*         data = NULL ;
      unsigned long alloc = 0 ;

      sprintf(indx, "%d,%d", x, y),
      VWT_GetData(vwptr, *vwcell, &data, &alloc) ;
      Tcl_SetVar2(interp, vwptr->varName, indx, data, TCL_GLOBAL_ONLY) ;
      free(data) ;
    }

  VWT_ResetBinding(vwptr, interp, TRUE) ;
  if (( x == vwptr->focus.col ) &&
      ( y == vwptr->focus.row ))
    {
      VWT_InstantiateFocus(vwptr, x, y) ;
    }

  return result ;
}

/************************************************************************
 ** static int
 ** Handle_SetRange( VWTable       *vwptr,
 **                  Tcl_Interp    *interp,
 **                  int            argc,
 **                  char          *argv[] )
 **
 ** Handles the setting of ranges.  The process is more complicated when
 ** dealing with a range than when dealing with a single cell
 ************************************************************************/

#ifdef __STDC__
static int
Handle_SetRange( VWTable*             vwptr,
                 Tcl_Interp*          interp,
                 int                  argc,
                 char*                argv[] )
#else /** __STDC__ **/
static int
Handle_SetRange( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  int             x, x1, x2 ;
  int             y, y1, y2 ;
  
  int             listArgc ;
  char**          listArgv ;

  int             result ;

  VWTableCell**   vwcell ;

  unsigned long   alloc = 0 ;
  char*           data = NULL ;
  char            indx[21] ;  /** 10x10 **/

  /**
   ** Get the bounding region (x1, y1, x2, y2) to
   ** apply the highlight to
   **/

  if (VWT_GetBoundingBox( interp,
			  vwptr,
			  &x1, &y1,
			  &x2, &y2,
			  &argv[0],
			  TRUE ) != TCL_OK )
    {
      return TCL_ERROR ;
    }

  if (Tcl_SplitList(interp, argv[4], &listArgc, &listArgv) == TCL_ERROR)
    return TCL_ERROR ;

  result = TCL_OK ;

  VWT_ResetBinding(vwptr, interp, FALSE) ;

  for ( y = y1 ; y <= y2 ; y++ )
    {
      int    lineArgc = 0 ;
      char **lineArgv = NULL ;
      char  *listLine = (( y - y1 ) >= listArgc) ? NULL : listArgv[ y - y1 ] ;

      if (listLine)
	{
	  if (Tcl_SplitList(interp, listLine, &lineArgc, &lineArgv) == TCL_ERROR)
	    {
	      result = TCL_ERROR ;
	      break ;
	    }
	}

      for ( x = x1 ; x <= x2 ; x++ )
	{
	  vwcell = VWMatrixIndex(vwptr->matrix, VWTableCell*, x, y, 0) ;

	  if (( x - x1 ) >= lineArgc)
	    continue ;

	  result = VWT_SetElement(vwptr, vwcell, interp, lineArgv[x - x1]) ;
	  if (( vwptr->varName ) && ( result == TCL_OK ))
	    {
	      sprintf(indx, "%d,%d", x, y),
	      VWT_GetData(vwptr, *vwcell, &data, &alloc) ;
	      Tcl_SetVar2(interp, vwptr->varName, indx, data, TCL_GLOBAL_ONLY) ;
	    }

	  if (( x == vwptr->focus.col ) &&
	      ( y == vwptr->focus.row ))
	    {
	      VWT_InstantiateFocus(vwptr, x, y) ;
	    }
	}

      free((char *) lineArgv) ;
    }

  free((char *) listArgv) ;

  if (data)
    free(data) ;

  VWT_ResetBinding(vwptr, interp, TRUE) ;
  return result ;
}

/************************************************************************
 ** int
 ** VWT_HandleSet( VWTable     *vwptr,
 **                    Tcl_Interp  *interp,
 **                    int          argc,
 **                    char        *argv[] )
 **
 ** Attempts to set members of the table
 ************************************************************************/

#ifdef __STDC__
int
VWT_HandleSet( VWTable*             vwptr,
               Tcl_Interp*          interp,
               int                  argc,
               char*                argv[] )
#else /** __STDC__ **/
int
VWT_HandleSet( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  if ((argc < 5) || (argc == 6))
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " ", argv[1],
		       " x1 y1 ? x2 y2 ? elements\"",
		       NULL);
      return TCL_ERROR ;
    }

  return (( argc != 5 ) ?
	  ( Handle_SetRange ( vwptr, interp, argc - 2, argv + 2 )) :
	  ( Handle_SetSingle( vwptr, interp, argc - 2, argv + 2 ))) ;
}
