/************************************************************************
 ** Copyright (c) 1994, Aaron Jackson
 ** All rights reserved
 **
 ** Permission to use, copy, modify, and distribute this software and
 ** its documentation for any purpose and without fee is hereby granted,
 ** provided that the above copyright notice appear in all copies and
 ** that both that copyright notice and this permission notice appear in
 ** supporting documentation.
 **
 ** In no event shall the author(s) be liable to any party for direct,
 ** indirect, special, incidental, or consequential damages arising out
 ** of the use of this software and its documentation, even if the
 ** authors(s) have been advised of the possibility of such damage.
 **
 ** The authors(s) specifically disclaim any warranties, including but
 ** not limited to, the implied warranties of merchantability and
 ** fitness for a particular purpose.  The software provided hereunder
 ** is on an "as is" basis, and the author(s) have no obligation to
 ** provide maintenance, support, updates, enhancements, or modifications
 ************************************************************************/

static char rcsid[] = "$Header: /home/edsdev/cvstree/tcldev/vwtable/vwTableFocus.c,v 1.1.1.1 1994/10/21 13:35:25 aajackso Exp $" ;

/**
 ** $Log: vwTableFocus.c,v $
 * Revision 1.1.1.1  1994/10/21  13:35:25  aajackso
 * Table Widget
 *
 **/

#include <tcl.h>
#include <tk.h>

#include <malloc.h>
#include <vwTable.h>

#ifdef Min
#undef Min
#endif /** Min **/

#define Min(xx,yy) (((xx) > (yy)) ? (yy) : (xx))

#ifdef Max
#undef Max
#endif /** Max **/

#define Max(xx,yy) (((xx) < (yy)) ? (yy) : (xx))

/************************************************************************
 ** static void
 ** Handle_Blink(ClientData clientData)
 **
 ** This procedures is called as a timer handler to blink the cursor
 ** on and off.  Most of ideas behind this come from tkEntry.c
 ************************************************************************/

#ifdef __STDC__
static void
Handle_Blink( ClientData    clientData )
#else /** __STDC__ **/
static void
Handle_Blink( clientData )
    ClientData    clientData ;
#endif /** __STDC__ **/
{
  VWTable* vwptr = (VWTable *) clientData ;

  if (vwptr->focus.offTime == 0)
    return ;

  if (vwptr->flags & VWT_CursorOn)
    {
#ifdef STRICT_DEBUG
      fprintf(stderr, "Handle_Blink.. Blink handler recreated (0)\n") ;
#endif /** STRICT_DEBUG **/

      vwptr->flags &= ~VWT_CursorOn ;
      vwptr->focus.blinkHandler = 
	Tk_CreateTimerHandler(vwptr->focus.offTime,
			      Handle_Blink,
			      (ClientData) vwptr) ;
    }
  else
    {
#ifdef STRICT_DEBUG
      fprintf(stderr, "Handle_Blink.. Blink handler recreated (1)\n") ;
#endif /** STRICT_DEBUG **/

      vwptr->flags |= VWT_CursorOn ;
      vwptr->focus.blinkHandler = 
	Tk_CreateTimerHandler(vwptr->focus.onTime,
			      Handle_Blink,
			      (ClientData) vwptr) ;
    }

  Tk_DoWhenIdle(VWT_Display, clientData) ;
}

/************************************************************************
 ** static int
 ** GetIndex( VWTable     *vwptr,
 **           Tcl_Interp  *interp,
 **           char        *position,
 **           int         *rpos )
 **
 ** Finds the index given <position> and returns it in <rpos>.
 ** This ensures that consistent methods are used to get an
 ** index.
 **
 ** Values supported for <position>
 **      <integer>    -- Absolute position
 **      @<integer>   -- Absolute pixel position
 **      @@<integer>  -- Relative pixel position
 **      end          -- Last position
 **      insert       -- Insertion position
 ************************************************************************/

#ifdef __STDC__
static int
GetIndex( VWTable*             vwptr,
          Tcl_Interp*          interp,
          char*                position,
          int*                 rpos )
#else /** __STDC__ **/
static int
GetIndex( vwptr, interp, position, rpos )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    char*                position ;
    int*                 rpos ;
#endif /** __STDC__ **/
{
  int           err ;
  int           max ;
  VWTableCell** vwcell ;

  char*         ldata ;
  int           ldatalen ;

  /**
   ** To ensure that the user is not requesting an invalid point we
   ** need to determine the length of the visible string and then
   ** compare the values.
   **/

  vwcell = VWMatrixIndex(vwptr->matrix, VWTableCell*, vwptr->focus.col, vwptr->focus.row, 0) ;

  ldata = vwptr->focus.data ;
  ldatalen = vwptr->focus.len ;
  max = vwptr->focus.len ;

  /**
   ** Array index 0 may not be the leftmost character.  Check what
   ** the focus information says
   **/

  if (ldata)
    {
      ldata += vwptr->focus.vpos ;
      ldatalen -= vwptr->focus.vpos ;
    }
  
  /**
   ** Determine the point they have requested and then change the
   ** focus icursor position (vwptr->focus.curp)
   **/

  switch(position[0])
    {
    case '@':
	{
	  int pixel ;
	  int width ;
	  int relative ;
	  VWTableImageElement* elem ;

	  relative = (position[1] == '@') ;

	  /**
	   ** Extract the pixel position from the <index> value
	   **/

	  if ((err = Tcl_GetInt(interp, position + 1 + relative, &pixel)) != TCL_OK)
	    return err ;
      
	  if (! relative)
	    {
	      /**
	       ** Determine where the correct index given <pixel>
	       **/

	      elem = VWT_FindElementAtPixel(vwptr->image.elements[VWT_Horizontal], pixel) ;
	      if (elem == NULL)
		{
		  *rpos = 0 ;
		  return TCL_OK ;
		}

	      width = Min(pixel - elem->pixel, elem->size) ;
	    } else {

	      /**
	       ** Determine where the correct index given <pixel>
	       **/

	      elem = VWT_FindElementByPosition(vwptr->image.elements[VWT_Horizontal], vwptr->focus.col) ;
	      width = ((elem == NULL) ? (vwptr->columnWidth[vwptr->focus.col]) : (elem->size)) ;
	    }

	  *rpos = VWT_GetDisplayIndex(vwptr, vwcell[0], width, ldata, ldatalen) + vwptr->focus.vpos ;
	  return TCL_OK ;
	}

    case 'i':
      if (! strcmp(position, "insert"))
	{
	  *rpos = vwptr->focus.curp ;
	  return TCL_OK ;
	}
      break ;

    case 's':
      if (! strcmp(position, "sel.first"))
	{
	  *rpos = Min(vwptr->focus.selFrom, vwptr->focus.selTo) ;
	  return TCL_OK ;
	}
      else if (! strcmp(position, "sel.last"))
	{
	  *rpos = Max(vwptr->focus.selFrom, vwptr->focus.selTo) ;
	  return TCL_OK ;
	}
      break ;
    }

  if ((err = VWT_GetBoundingPoint(interp, position, rpos, max + 1, TRUE)) != TCL_OK)
    return err ;

  return TCL_OK ;
}

/************************************************************************
 ** static void
 ** ChangeFocusCache( VWTableFocus  *focus,
 **                   unsigned int   bytes )
 **
 ** This function is here to ensure that the cache can hold at least
 ** "bytes".
 **
 ** If more memory is required the cache will try to allocate at least
 ** 1 block above bytes.  At this time we consider one block to be 512
 ** bytes.
 **
 ** If less memory is required the cache will look at the current
 ** allocation compared to the amount of overhead and adjust the
 ** cache downwards as less blocks are used
 ************************************************************************/

static CONST int CacheBlock = 512 ;

#ifdef __STDC__
static void
ChangeFocusCache( VWTableFocus*       focus,
                  unsigned int        bytes )
#else /** __STDC__ **/
static void
ChangeFocusCache( focus, bytes )
    VWTableFocus*       focus ;
    unsigned int        bytes ;
#endif /** __STDC__ **/
{
  int blocks = (bytes / CacheBlock) + ((bytes % CacheBlock) != 0) + 1 ;
  
  if (focus->alloc < blocks)
    {
      if (focus->data)
	focus->data = realloc(focus->data, blocks * CacheBlock) ;
      else
	focus->data = calloc(blocks, CacheBlock) ;
      
      focus->alloc = blocks ;
      return ;
    }

  /**
   ** Memory requirements are decreasing.  Ensure that were not
   ** being an overhog
   **/

  if ((focus->alloc - 2) > blocks)
    {
      focus->data = realloc(focus->data, blocks * CacheBlock) ;
      focus->alloc -= 2 ;
    }
}

/************************************************************************
 ** void
 ** VWT_ResetFocus( VWTable       *vwptr )
 **
 ** Resets the table focus policy
 ************************************************************************/

#ifdef __STDC__
void
VWT_ResetFocus( VWTable*    vwptr )
#else /** __STDC__ **/
void
VWT_ResetFocus( vwptr )
    VWTable*    vwptr ;
#endif /** __STDC__ **/
{
  /**
   ** Set focus markers
   **/

  vwptr->focus.focusing  = 1 ;
  vwptr->focus.vpos      = 0 ;
  vwptr->focus.curp      = 0 ;
  vwptr->focus.selTo     = -1 ;
  vwptr->focus.selFrom   = -1 ;
  vwptr->focus.scanMark  = 0 ;
  vwptr->focus.scanIndex = 0 ;
}

/************************************************************************
 ** void
 ** VWT_InstantiateFocus( VWTable       *vwptr,
 **                       int            insx,
 **                       int            insy )
 **
 ** Performs the initial memory allocation.  Also copies data over from
 ** the base object.
 ************************************************************************/

#ifdef __STDC__
void
VWT_InstantiateFocus( VWTable*          vwptr,
                      int               insx,
                      int               insy )
#else /** __STDC__ **/
void
VWT_InstantiateFocus( vwptr, insx, insy )
    VWTable*          vwptr ;
    int               insx ;
    int               insy ;
#endif /** __STDC__ **/
{
  VWTableCell   **vwcell ;
  char           *source = NULL ;
  int             bytes = 0 ;

  vwptr->focus.len = 0 ;    /** This is regardless **/

  vwcell = VWMatrixIndex(vwptr->matrix, VWTableCell*, insx, insy, 0) ;
  if (vwcell[0] == NULL)
    return ;

  switch(vwcell[0]->dataType)
    {
    case VWTableTextType:
      source = vwcell[0]->data.text.data ;
      bytes = vwcell[0]->data.text.datalen ;
      break ;

    case VWTableBitmapType:
      source = vwcell[0]->data.bitmap.name ;
      bytes = vwcell[0]->data.bitmap.namelen ;
      break ;
    }

  ChangeFocusCache(&vwptr->focus, bytes + 1) ;
  memcpy(vwptr->focus.data, source, bytes) ;
  vwptr->focus.len = bytes ;
}

/************************************************************************
 ** static void
 ** RelinquishFocus( VWTable       *vwptr,
 **                  Tcl_Interp    *interp,
 **                  int            relx,
 **                  int            rely )
 **
 ** Relinquishes focus data to relx, rely
 ************************************************************************/

#ifdef __STDC__
static void
RelinquishFocus( VWTable*             vwptr,
                 Tcl_Interp*          interp,
                 int                  relx,
                 int                  rely )
#else /** __STDC__ **/
static void
RelinquishFocus( vwptr, interp, relx, rely )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  relx ;
    int                  rely ;
#endif /** __STDC__ **/
{
  VWTableCell   **vwcell ;
  int             anchor ;
  int             dataType ;

  if (( relx < 0 ) || ( rely < 0 ))
    return ;

  vwcell = VWMatrixIndex(vwptr->matrix, VWTableCell*, relx, rely, 0) ;

  /**
   ** Hmm, if there is no data then AND the cell is originally empty
   ** then the cell must be cleared to ensure that the blink cursor
   ** is not still active
   **/

  if ((! vwptr->focus.len) && (vwcell[0] == NULL))
    {
      VWT_ClearCell(vwptr, relx, rely) ;
      return ;
    }

  /**
   ** Interesting problem has been posed.  The user does not have
   ** anything declared for the cell.  We could assume a default
   ** and go from there but that makes application assumptions.
   **/

  dataType = (((vwcell[0] == NULL) || (vwcell[0]->dataType == 0)) ? 
	      (VWTableTextType) :
	      (vwcell[0]->dataType)) ;

  switch(dataType)
    {
    case VWTableTextType:
      anchor = ((vwcell[0] != NULL) ?
		(vwcell[0]->data.text.anchor) :
		(TK_ANCHOR_W)) ;

      VWT_FreeData(vwptr, vwcell[0]) ;
      if (vwptr->focus.data && vwptr->focus.len)
	{
	  vwptr->focus.data[vwptr->focus.len] = 0 ;
	  VWT_SetText(vwptr, vwcell, interp, vwptr->focus.data, anchor) ;
	}

      break ;

    case VWTableBitmapType:
      anchor = ((vwcell[0] != NULL) ?
		(vwcell[0]->data.bitmap.anchor) :
		(TK_ANCHOR_W)) ;

      VWT_FreeData(vwptr, vwcell[0]) ;
      if (vwptr->focus.data && vwptr->focus.len)
	{
	  vwptr->focus.data[vwptr->focus.len] = 0 ;
	  VWT_SetBitmap(vwptr, vwcell, interp, vwptr->focus.data, anchor) ;
	}

      break ;
    }
}

/************************************************************************
 ** int
 ** VWT_SetFocus( VWTable       *vwptr,
 **               Tcl_Interp    *interp )
 **
 ** Turns focus flags on & starts handlers
 ************************************************************************/

#ifdef __STDC__
int
VWT_SetFocus( VWTable*       vwptr,
              Tcl_Interp*    interp )
#else /** __STDC__ **/
int
VWT_SetFocus( vwptr, interp )
    VWTable*       vwptr ;
    Tcl_Interp*    interp ;
#endif /** __STDC__ **/
{
  /**
   ** Delete old blink handler
   **/

  if (vwptr->focus.blinkHandler)
    {
#ifdef STRICT_DEBUG
      fprintf(stderr, "SetFocus.. Blink handler destroyed\n") ;
#endif /** STRICT_DEBUG **/
      Tk_DeleteTimerHandler(vwptr->focus.blinkHandler) ;
    }

  /**
   ** Mark the flags so we don't get lost
   **/

  vwptr->flags |= VWT_CursorOn ;

  /**
   ** Create the new blink handler
   **/

  if (vwptr->focus.offTime != 0)
    {
#ifdef STRICT_DEBUG
      fprintf(stderr, "SetFocus... BlinkHandler installed\n") ;
#endif /** STRICT_DEBUG **/
      vwptr->focus.blinkHandler = 
	Tk_CreateTimerHandler(vwptr->focus.onTime,
			      Handle_Blink,
			      (ClientData) vwptr) ;
    }

  /**
   ** Have Tk move the focus to us
   **/

  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_UnsetFocus( VWTable     *vwptr,
 **                 Tcl_Interp  *interp )
 **
 ** Removes focus from the widget
 ************************************************************************/

#ifdef __STDC__
int
VWT_UnsetFocus( VWTable*       vwptr,
                Tcl_Interp*    interp )
#else /** __STDC__ **/
int
VWT_UnsetFocus( vwptr, interp )
    VWTable*       vwptr ;
    Tcl_Interp*    interp ;
#endif /** __STDC__ **/
{
  /**
   ** Delete old blink handler
   **/

  if (vwptr->focus.blinkHandler)
    {
#ifdef STRICT_DEBUG
      fprintf(stderr, "UnsetFocus.. BlinkHandler destroyed\n");
#endif /** STRICT_DEBUG **/
      Tk_DeleteTimerHandler(vwptr->focus.blinkHandler) ;
    }

  /**
   ** Mark the flags so we don't get lost
   **/

  vwptr->flags &= ~VWT_CursorOn ;

  Tk_DoWhenIdle(VWT_Display, (ClientData) vwptr) ;
  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_HandleFocusSelClear( VWTable     *vwptr,
 **                          Tcl_Interp  *interp,
 **                          int          argc,
 **                          char        *argv[] )
 **
 ** Usage:
 **    <object> focus select clear
 ************************************************************************/

#ifdef __STDC__
int
VWT_HandleFocusSelClear( VWTable*             vwptr,
                         Tcl_Interp*          interp,
                         int                  argc,
                         char*                argv[] )
#else /** __STDC__ **/
int
VWT_HandleFocusSelClear( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  int err ;
  int indx ;

  /**
   ** Your standard usage error on incomplete information
   **/

  if (argc < 4)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " ", argv[1], " ", argv[2], " ", argv[3],
		       NULL) ;

      return TCL_ERROR ;
    }

  /**
   ** For non-focusing entries just return TCL_OK since there
   ** really isn't an error.
   **/

  if (! vwptr->focus.focusing)
    return TCL_OK ;

  vwptr->focus.selTo = -1 ;
  vwptr->focus.selFrom = -1 ;
  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_HandleFocusSelTo( VWTable     *vwptr,
 **                       Tcl_Interp  *interp,
 **                       int          argc,
 **                       char        *argv[] )
 **
 ** Usage:
 **    <object> focus select to index
 ************************************************************************/

#ifdef __STDC__
int
VWT_HandleFocusSelTo( VWTable*             vwptr,
                      Tcl_Interp*          interp,
                      int                  argc,
                      char*                argv[] )
#else /** __STDC__ **/
int
VWT_HandleFocusSelTo( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  int err ;
  int indx ;

  /**
   ** Your standard usage error on incomplete information
   **/

  if (argc < 5)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " ", argv[1], " ", argv[2], " ", argv[3],
		       " index\"",
		       NULL) ;

      return TCL_ERROR ;
    }

  /**
   ** For non-focusing entries just return TCL_OK since there
   ** really isn't an error.
   **/

  if (! vwptr->focus.focusing)
    return TCL_OK ;

  /**
   ** Find the <index> requested
   **/

  if ((err = GetIndex(vwptr, interp, argv[4], &indx)) != TCL_OK)
    return err ;

  vwptr->focus.selTo = indx ;
  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_HandleFocusSelFrom( VWTable     *vwptr,
 **                         Tcl_Interp  *interp,
 **                         int          argc,
 **                         char        *argv[] )
 **
 ** Usage:
 **    <object> focus select from index
 ************************************************************************/

#ifdef __STDC__
int
VWT_HandleFocusSelFrom( VWTable*             vwptr,
                        Tcl_Interp*          interp,
                        int                  argc,
                        char*                argv[] )
#else /** __STDC__ **/
int
VWT_HandleFocusSelFrom( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  int err ;
  int indx ;

  /**
   ** Your standard usage error on incomplete information
   **/

  if (argc < 5)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " ", argv[1], " ", argv[2], " ", argv[3],
		       " index\"",
		       NULL) ;

      return TCL_ERROR ;
    }

  /**
   ** For non-focusing entries just return TCL_OK since there
   ** really isn't an error.
   **/

  if (! vwptr->focus.focusing)
    return TCL_OK ;

  /**
   ** Find the <index> requested
   **/

  if ((err = GetIndex(vwptr, interp, argv[4], &indx)) != TCL_OK)
    return err ;

  vwptr->focus.selFrom = indx ;
  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_HandleFocusScanMark( VWTable     *vwptr,
 **                          Tcl_Interp  *interp,
 **                          int          argc,
 **                          char        *argv[] )
 **
 ** Usage:
 **    <object> focus scan mark x
 ************************************************************************/

#ifdef __STDC__
int
VWT_HandleFocusScanMark( VWTable*             vwptr,
                         Tcl_Interp*          interp,
                         int                  argc,
                         char*                argv[] )
#else /** __STDC__ **/
int
VWT_HandleFocusScanMark( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  int  err ;
  int  indx ;
  int  pixel ;

  VWTableImageElement* elem ;

  /**
   ** Your standard usage error on incomplete information
   **/

  if (argc < 5)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " ", argv[1], " ", argv[2], " ", argv[3],
		       " x\"",
		       NULL) ;

      return TCL_ERROR ;
    }

  /**
   ** Extract the pixel position from argv[4]
   **/

  if ((err = Tcl_GetInt(interp, argv[4], &pixel)) != TCL_OK)
    return err ;

  /**
   ** For non-focusing entries just return TCL_OK since there
   ** really isn't an error.
   **/

  if (! vwptr->focus.focusing)
    return TCL_OK ;

  /**
   ** Determine where the correct index given <pixel>
   **/

  elem = VWT_FindElementAtPixel(vwptr->image.elements[VWT_Horizontal], pixel) ;
  if (elem == NULL)
    return TCL_OK ;

  vwptr->focus.scanMark = pixel - elem->pixel ;
  vwptr->focus.scanIndex = vwptr->focus.vpos ;
  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_HandleFocusScanDrag( VWTable     *vwptr,
 **                          Tcl_Interp  *interp,
 **                          int          argc,
 **                          char        *argv[] )
 **
 ** Usage:
 **    <object> focus scan dragto x
 ************************************************************************/

#ifdef __STDC__
int
VWT_HandleFocusScanDrag( VWTable*             vwptr,
                         Tcl_Interp*          interp,
                         int                  argc,
                         char*                argv[] )
#else /** __STDC__ **/
int
VWT_HandleFocusScanDrag( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  int  err ;
  int  indx ;
  int  pixel ;
  int  newLeftIndex ;

  VWTableImageElement* elem ;

  /**
   ** Your standard usage error on incomplete information
   **/

  if (argc < 5)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " ", argv[1], " ", argv[2], " ", argv[3],
		       " x\"",
		       NULL) ;

      return TCL_ERROR ;
    }

  /**
   ** Extract the pixel position from argv[4]
   **/

  if ((err = Tcl_GetInt(interp, argv[4], &pixel)) != TCL_OK)
    return err ;

  /**
   ** For non-focusing entries just return TCL_OK since there
   ** really isn't an error.
   **/

  if (! vwptr->focus.focusing)
    return TCL_OK ;

  /**
   ** Determine where the correct index given <pixel>
   **/

  elem = VWT_FindElementByPosition(vwptr->image.elements[VWT_Horizontal], vwptr->focus.col) ;
  if (elem == NULL)
    return TCL_OK ;

  /**
   ** Compute new leftIndex for entry by amplifying the difference
   ** between the current position and the place where the scan
   ** started (the "mark" position).  If we run off the left or right
   ** side of the entry, then reset the mark point so that the current
   ** position continues to correspond to the edge of the window.
   ** This means that the picture will start dragging as soon as the
   ** mouse reverses direction (without this reset, might have to slide
   ** mouse a long ways back before the picture starts moving again).
   **/

  pixel -= elem->pixel ;
  newLeftIndex = vwptr->focus.scanIndex - (10 * (pixel - vwptr->focus.scanMark)) / vwptr->focus.avgWidth ;
  if (newLeftIndex >= vwptr->focus.len)
    newLeftIndex = vwptr->focus.scanIndex = vwptr->focus.len - 1 ;
  else if (newLeftIndex < 0)
    newLeftIndex = vwptr->focus.scanIndex = 0;

  if (newLeftIndex != vwptr->focus.vpos)
    vwptr->focus.vpos = newLeftIndex ;

  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_HandleFocusIndex( VWTable     *vwptr,
 **                       Tcl_Interp  *interp,
 **                       int          argc,
 **                       char        *argv[] )
 **
 ** Usage:
 **    <object> focus index <index>
 ************************************************************************/

#ifdef __STDC__
int
VWT_HandleFocusIndex( VWTable*             vwptr,
                      Tcl_Interp*          interp,
                      int                  argc,
                      char*                argv[] )
#else /** __STDC__ **/
int
VWT_HandleFocusIndex( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  int  err ;
  int  indx ;
  char str[40] ;

  /**
   ** Your standard usage error on incomplete information
   **/

  if (argc < 4)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " ", argv[1], " ", argv[2], 
		       " index\"",
		       NULL) ;

      return TCL_ERROR ;
    }

  /**
   ** For non-focusing entries just return TCL_OK since there
   ** really isn't an error.
   **/

  if (! vwptr->focus.focusing)
    {
      Tcl_AppendResult(interp, "0", NULL) ;
      return TCL_OK ;
    }

  /**
   ** Find the <index> requested
   **/

  if ((err = GetIndex(vwptr, interp, argv[3], &indx)) != TCL_OK)
    return err ;

  Tcl_AppendResult(interp, sprintf(str, "%d", indx), NULL) ;
  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_HandleFocusDelete( VWTable     *vwptr,
 **                        Tcl_Interp  *interp,
 **                        int          argc,
 **                        char        *argv[] )
 **
 ** Usage:
 **    <object> focus delete <first> ? <last> ?
 ************************************************************************/

#ifdef __STDC__
int
VWT_HandleFocusDelete( VWTable*             vwptr,
                       Tcl_Interp*          interp,
                       int                  argc,
                       char*                argv[] )
#else /** __STDC__ **/
int
VWT_HandleFocusDelete( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  int err ;
  int last ;
  int first ;
  int bytes ;

  register char *p0 ;
  register char *p1 ;
  register char *p2 ;

  /**
   ** Your standard usage error on incomplete information
   **/

  if (argc < 4)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " ", argv[1], " ", argv[2], 
		       " first ? last ?\"",
		       NULL) ;

      return TCL_ERROR ;
    }

  /**
   ** For non-focusing entries just return TCL_OK since there
   ** really isn't an error.
   **/

  if (! vwptr->focus.focusing)
    return TCL_OK ;

  /**
   ** Find the indices requested
   **/

  if ((err = GetIndex(vwptr, interp, argv[3], &first)) != TCL_OK)
    return err ;

  if (argc > 4)
    err = GetIndex(vwptr, interp, argv[4], &last) ;
  else
    last = first ;

  if (last < first)
    {
      int swap = last ;
      last = first ;
      first = swap ;
    }

  bytes = last - first ;

  /**
   ** Becareful, bcopy() will work fine here but memcpy()
   ** will not always behave properly.  Also since bcopy()
   ** is not available on all systems take appropriate
   ** action
   **/

  p0 = vwptr->focus.data + first ;
  p1 = vwptr->focus.data + last ;
  p2 = vwptr->focus.data + vwptr->focus.len ;

  while (p1 <= p2)
    {
      p0[0] = p1[0] ;
      p0++ ;
      p1++ ;
    }

  ChangeFocusCache(&vwptr->focus, vwptr->focus.len - bytes + 1) ;

  vwptr->focus.len -= bytes ;
  if (vwptr->focus.len < 0)
    vwptr->focus.len = 0 ;

  /**
   ** Many of the indices will be incorrect
   **/

#define DecrementVar(var, barrier, decr) \
  if (var > (barrier)) \
    { \
      var -= decr ; \
      if (var < 0) \
	var = 0 ; \
    }

  DecrementVar(vwptr->focus.vpos, first, bytes) ;
  DecrementVar(vwptr->focus.curp, first, bytes) ;
  DecrementVar(vwptr->focus.selTo, first, bytes) ;
  DecrementVar(vwptr->focus.selFrom, first, bytes) ;

#undef DecrementVar

  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_HandleFocusInsert( VWTable     *vwptr,
 **                        Tcl_Interp  *interp,
 **                        int          argc,
 **                        char        *argv[] )
 **
 ** Usage:
 **    <object> focus insert <index> <string>
 ************************************************************************/

#ifdef __STDC__
int
VWT_HandleFocusInsert( VWTable*             vwptr,
                       Tcl_Interp*          interp,
                       int                  argc,
                       char*                argv[] )
#else /** __STDC__ **/
int
VWT_HandleFocusInsert( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  int err ;
  int indx ;
  int bytes ;

  register char *p0 ;
  register char *p1 ;
  register char *p2 ;

  /**
   ** Your standard usage error on incomplete information
   **/

  if (argc < 5)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " ", argv[1], " ", argv[2], 
		       " index string\"",
		       NULL) ;

      return TCL_ERROR ;
    }

  /**
   ** For non-focusing entries just return TCL_OK since there
   ** really isn't an error.
   **/

  if (! vwptr->focus.focusing)
    return TCL_OK ;

  /**
   ** Find the <index> requested
   **/

  if ((err = GetIndex(vwptr, interp, argv[3], &indx)) != TCL_OK)
    return err ;

  bytes = strlen(argv[4]) ;
  ChangeFocusCache(&vwptr->focus, vwptr->focus.len + bytes + 1) ;

  /**
   ** Becareful, bcopy() will work fine here but memcpy()
   ** will not always behave properly.  Also since bcopy()
   ** is not available on all systems take appropriate
   ** action
   **/

  p0 = vwptr->focus.data + vwptr->focus.len + bytes ;
  p1 = vwptr->focus.data + vwptr->focus.len ;
  p2 = vwptr->focus.data + indx ;

  while (p1 >= p2)
    {
      p0[0] = p1[0] ;
      p0-- ;
      p1-- ;
    }

  memcpy(p2, argv[4], bytes) ;
  vwptr->focus.len += bytes ;

  /**
   ** Many of the indices will be incorrect
   **/

#define IncrementVar(var, barrier, incr) \
  if (var >= (barrier)) \
      var += incr

  IncrementVar(vwptr->focus.curp, indx, bytes) ;
  IncrementVar(vwptr->focus.selTo, indx, bytes) ;
  IncrementVar(vwptr->focus.selFrom, indx, bytes) ;

#undef IncrementVar

  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_HandleFocusView( VWTable     *vwptr,
 **                      Tcl_Interp  *interp,
 **                      int          argc,
 **                      char        *argv[] )
 **
 ** Usage:
 **    <object> focus view <index>
 ************************************************************************/

#ifdef __STDC__
int
VWT_HandleFocusView( VWTable*             vwptr,
                     Tcl_Interp*          interp,
                     int                  argc,
                     char*                argv[] )
#else /** __STDC__ **/
int
VWT_HandleFocusView( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  int  err ;
  int  indx ;

  /**
   ** Your standard usage error on incomplete information
   **/

  if (argc < 4)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " ", argv[1], " ", argv[2], 
		       " index\"",
		       NULL) ;

      return TCL_ERROR ;
    }

  /**
   ** For non-focusing entries just return TCL_OK since there
   ** really isn't an error.
   **/

  if (! vwptr->focus.focusing)
    return TCL_OK ;

  /**
   ** Find the <index> requested
   **/

  if ((err = GetIndex(vwptr, interp, argv[3], &indx)) != TCL_OK)
    return err ;

  vwptr->focus.vpos = indx ;

  Tk_DoWhenIdle(VWT_Display, (ClientData) vwptr) ;
  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_HandleFocusIcursor( VWTable     *vwptr,
 **                         Tcl_Interp  *interp,
 **                         int          argc,
 **                         char        *argv[] )
 **
 ** Usage:
 **    <object> focus icursor <index>
 ************************************************************************/

#ifdef __STDC__
int
VWT_HandleFocusIcursor( VWTable*             vwptr,
                        Tcl_Interp*          interp,
                        int                  argc,
                        char*                argv[] )
#else /** __STDC__ **/
int
VWT_HandleFocusIcursor( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  int  err ;
  int  icursor ;

  /**
   ** Your standard usage error on incomplete information
   **/

  if (argc < 4)
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " ", argv[1], " ", argv[2], 
		       " index\"",
		       NULL) ;

      return TCL_ERROR ;
    }

  /**
   ** For non-focusing entries just return TCL_OK since there
   ** really isn't an error.
   **/

  if (! vwptr->focus.focusing)
    return TCL_OK ;

  /**
   ** Find the <index> requested
   **/

  if ((err = GetIndex(vwptr, interp, argv[3], &icursor)) != TCL_OK)
    return err ;

  vwptr->focus.curp = icursor ;

  Tk_DoWhenIdle(VWT_Display, (ClientData) vwptr) ;
  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_HandleFocusWhere( VWTable     *vwptr,
 **                       Tcl_Interp  *interp,
 **                       int          argc,
 **                       char        *argv[] )
 **
 ** Gives you information about the cell you are focused on.
 **
 ** Usage:
 **    <object> focus where
 **
 ** Returns:
 **    <column> <row> <column pixel> <row pixel> <column width> <row height>
 ************************************************************************/

#ifdef __STDC__
int
VWT_HandleFocusWhere( VWTable*             vwptr,
                      Tcl_Interp*          interp,
                      int                  argc,
                      char*                argv[] )
#else /** __STDC__ **/
int
VWT_HandleFocusWhere( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  int                  pixel_x ;
  int                  pixel_y ;
  int                  dimen_x ;
  int                  dimen_y ;
  char                 str[128] ;
  VWTableImageElement* object_x ;
  VWTableImageElement* object_y ;

  /** There is no standard error handle, because there are no errors **/

  if (( vwptr->focus.col == -1 ) || ( vwptr->focus.row == -1 ))
    {
      Tcl_SetResult(interp, "-1 -1 -1 -1 -1 -1") ;
      return TCL_OK ;
    }

  object_x = VWT_FindElementByPosition(vwptr->image.elements[VWT_Horizontal], vwptr->focus.col) ;
  object_y = VWT_FindElementByPosition(vwptr->image.elements[VWT_Vertical], vwptr->focus.row) ;

  dimen_x = ((object_x != NULL) ? (object_x->size) : (vwptr->columnWidth[vwptr->focus.col])) ;
  dimen_y = ((object_y != NULL) ? (object_y->size) : (vwptr->rowHeight[vwptr->focus.row])) ;

  pixel_x = ((object_x != NULL) ? (object_x->pixel) : (-1)) ;
  pixel_y = ((object_y != NULL) ? (object_y->pixel) : (-1)) ;

  sprintf(str, "%d %d %d %d %d %d",
	  vwptr->focus.col,
	  vwptr->focus.row,
	  pixel_x,
	  pixel_y,
	  dimen_x,
	  dimen_y) ;

  Tcl_AppendResult(interp, str, NULL) ;
  return TCL_OK ;
}

/************************************************************************
 ** int
 ** VWT_HandleFocusSet( VWTable     *vwptr,
 **                     Tcl_Interp  *interp,
 **                     int          argc,
 **                     char        *argv[] )
 **
 ** Causes the matrix widget to give the appearance of focus on a cell.
 ** The result is that the user can edit the cell.
 **
 ** Usage:
 **    <object> focus set ? x y ?
 ************************************************************************/

#ifdef __STDC__
int
VWT_HandleFocusSet( VWTable*             vwptr,
                    Tcl_Interp*          interp,
                    int                  argc,
                    char*                argv[] )
#else /** __STDC__ **/
int
VWT_HandleFocusSet( vwptr, interp, argc, argv )
    VWTable*             vwptr ;
    Tcl_Interp*          interp ;
    int                  argc ;
    char*                argv[] ;
#endif /** __STDC__ **/
{
  int           x ;
  int           y ;
  int          ox ;
  int          oy ;

  if ((argc < 3) || (argc == 4))
    {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " ", argv[1], " ", argv[2], 
		       " ? x y pixel ?\"",
		       NULL) ;

      return TCL_ERROR ;
    }

  /**
   ** If "<object> focus set" then turn focus policy off
   **/

  if (argc == 3)
    {
      vwptr->focus.focusing = 0 ;

      /**
       ** Since focus is now being relinquished then set the underlying element
       ** and leave
       **/

      RelinquishFocus(vwptr, interp, vwptr->focus.col, vwptr->focus.row) ;

      vwptr->focus.col = -1 ;
      vwptr->focus.row = -1 ;
      return VWT_UnsetFocus(vwptr, interp) ;
    }

  /**
   ** Form is "<object> focus set x y", so turn on focus
   ** policy if everything is correct
   **/

  ox = vwptr->focus.col ;
  oy = vwptr->focus.row ;

  if ((VWT_GetBoundingPoint(interp, argv[3], &x, vwptr->columns, TRUE) != TCL_OK) ||
      (VWT_GetBoundingPoint(interp, argv[4], &y, vwptr->rows, TRUE) != TCL_OK))
    return TCL_ERROR ;

  if (( ox == x ) && ( oy == y ))
    return TCL_OK ;

  /**
   ** Since focus is now being relinquished then set the underlying element
   ** and leave
   **/

  RelinquishFocus(vwptr, interp, ox, oy) ;
  VWT_InstantiateFocus(vwptr, x, y) ;
  VWT_ResetFocus(vwptr) ;

  vwptr->focus.col = x ;
  vwptr->focus.row = y ;

  return VWT_SetFocus(vwptr, interp) ;
}
