static char copyright []
= "$Id: xessRange.c,v 1.1 1994/08/26 14:12:58 johnsonm Exp $\n\
   Copyright (c) 1992 General Electric.  All rights reserved.";

/*
 *   Permission to use, copy, modify, and distribute this
 *   software and its documentation for any purpose and without
 *   fee is hereby granted, provided that the above copyright
 *   notice appear in all copies and that both that copyright
 *   notice and this permission notice appear in supporting
 *   documentation, and that the name of General Electric not be used in
 *   advertising or publicity pertaining to distribution of the
 *   software without specific, written prior permission.
 *   General Electric makes no representations about the suitability of
 *   this software for any purpose.  It is provided "as is"
 *   without express or implied warranty.
 *
 *   This work was supported by the DARPA Initiative in Concurrent
 *   Engineering (DICE) through DARPA Contract MDA972-88-C-0047.
 *
 * $Log: xessRange.c,v $
 * Revision 1.1  1994/08/26  14:12:58  johnsonm
 * Initial revision
 *
 *
 * Old log:
 * Revision 1.3  1994/03/02  22:58:55  kennykb
 * Changed some `ints' to `unsigneds' in order to get around Xess
 * library inconsistencies.
 *
 * Revision 1.2  1994/02/28  20:59:25  kennykb
 * Removed #includes that are now obtained from tclXessInt.h
 * Changed Tk_ParseArgv calls to obtain application main window in the
 * standard way.
 *
 * Revision 1.1  1992/10/09  19:09:07  kennykb
 * Initial revision
 *
 *
 * xessRange.c --
 *
 *	This file contains the C code for the functions that implement
 *	the Xess connection object commands that deal with managing ranges
 *	on a spreadsheet
 */

#include "tclXessInt.h"

/*
 * XessConnObj_get_range_cmd
 *
 *	Block while the user enters a range on the spreadsheet, then return
 *	it.
 *
 * Syntax:
 *	connection get_range
 *
 * Where:
 *	connection is an Xess connection object
 *
 * Result:
 *	Returns a standard Tcl result, which normally contains the
 *	selected range (e.g., B2..D7)
 *
 * Notes:
 *	This procedure blocks for user interaction.  Its use is *not*
 *	recommended -- use `get_selected' in connection with user prompts
 *	instead.
 */

int
XessConnObj_get_range_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  int r0, c0, r1, c1;
  char * p;

  /* Check syntax */

  if (argc != 1) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], "\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Get the selection */

  status = xess_get_range (conn -> port, &r0, &c0, &r1, &c1);
  if (status == 0) {
    Tcl_SetResult (interp, "no range selected", TCL_STATIC);
    return TCL_ERROR;
  }

  /* Translate the range */

  status = xessStoreCell (interp, r0, c0, 0, interp -> result, &p);
  if (status != TCL_OK)
    return status;
  *p++ = '.';
  *p++ = '.';
  status = xessStoreCell (interp, r1, c1, 0, p, (char * *) NULL);
  return status;
}

/*
 * XessConnObj_get_selected_cmd
 *
 *	Determine the currently selected range on the spreadsheet.
 *
 * Syntax:
 *	connection get_selected
 *
 * Where:
 *	connection is an Xess connection object
 *
 * Result:
 *	Returns a standard Tcl result, which normally contains the
 *	selected range (e.g., B2..D7)
 */

int
XessConnObj_get_selected_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  int r0, c0, r1, c1;
  char * p;

  /* Check syntax */

  if (argc != 1) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], "\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Get the selection */

  status = xess_get_selected (conn -> port, &r0, &c0, &r1, &c1);
  if (status == 0) {
    Tcl_SetResult (interp, "no range selected", TCL_STATIC);
    return TCL_ERROR;
  }

  /* Translate the range */

  status = xessStoreCell (interp, r0, c0, 0, interp -> result, &p);
  if (status != TCL_OK)
    return status;
  *p++ = '.';
  *p++ = '.';
  status = xessStoreCell (interp, r1, c1, 0, p, (char * *) NULL);
  return status;
}

/*
 * XessConnObj_get_extent_cmd
 *
 *	Determine the range that defines the bounding box of the
 *	non-empty cells in a spreadsheet.
 *
 * Syntax:
 *	connection get_extent
 *
 * Where:
 *	connection is an Xess connection object
 *
 * Result:
 *	Returns a standard Tcl result, which normally contains the
 *	selected range (e.g., B2..D7)
 */

int
XessConnObj_get_extent_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  int r0, c0, r1, c1;
  char * p;

  /* Check syntax */

  if (argc != 1) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], "\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Get the selection */

  status = xess_get_extent (conn -> port, &r0, &c0, &r1, &c1);
  if (status == 0) {
    Tcl_SetResult (interp, "empty sheet", TCL_STATIC);
    return TCL_ERROR;
  }

  /* Translate the range */

  status = xessStoreCell (interp, r0, c0, 0, interp -> result, &p);
  if (status != TCL_OK)
    return status;
  *p++ = '.';
  *p++ = '.';
  status = xessStoreCell (interp, r1, c1, 0, p, (char * *) NULL);
  return status;
}

/*
 * XessConnObj_scroll_range_cmd --
 *
 *	Tcl command on a Xess connection object to cause it to scroll a
 *	range
 *
 * Syntax:
 *	connection scroll_range range direction
 * -or- connection scroll_range direction range
 *
 * Where:
 *	connection is the name of an Xess connection object
 *	range is a range on the spreadsheet
 *	direction is one of the following:
 *		-up -down -left -right
 *		and specifies the direction of scrolling.
 *
 * Bugs:
 *	-left and -right are reversed in the connection library.
 */

static int scrollDirection;

/* Option table for the scroll command */

static Tk_ArgvInfo scrollRangeOpts [] = {
  {"-up", TK_ARGV_CONSTANT, (char *) XESS_SCROLL_UP,
     (char *) & scrollDirection, "Scroll the specified range upward"},
  {"-down", TK_ARGV_CONSTANT, (char *) XESS_SCROLL_DOWN,
     (char *) & scrollDirection, "Scroll the specified range downward"},
  {"-left", TK_ARGV_CONSTANT, (char *) XESS_SCROLL_LEFT,
     (char *) & scrollDirection, "Scroll the specified range leftward"},
  {"-right", TK_ARGV_CONSTANT, (char *) XESS_SCROLL_RIGHT,
     (char *) & scrollDirection, "Scroll the specified range rightward"},
  {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL}
};

int
XessConnObj_scroll_range_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  Range range;

  /* First off, do the direction flag. */

  
  scrollDirection = -1;
  status = Tk_ParseArgv (interp, Tk_MainWindow (interp), &argc, argv,
			 scrollRangeOpts, 0);
  if (status != TCL_OK)
    return status;
  if (scrollDirection == -1) {
    Tcl_SetResult (interp, "scroll direction not specified", TCL_STATIC);
    return TCL_ERROR;
  }

  /* Now do the range */

  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " range -direction\"", (char *) NULL);
    return TCL_ERROR;
  }
  status = xessGetRange (interp, argv [1], &range);
  if (status != TCL_OK)
    return status;

  /* Make the Xess call */

  xess_scroll_range (conn -> port, range.r0, range.c0, range.r1, range.c1,
		     scrollDirection);
  return TCL_OK;
}

/*
 * XessConnObj_copy_range_cmd --
 *
 *	This function implements the `copy_range' command on an Xess
 *	connection object.
 *
 * Syntax:
 *	connection copy_range -from range -to range ?-values | -formulas?
 *
 * Where:
 *	connection is the name of an Xess connection object
 *
 * Options:
 *	-from takes a range specification describing the range to copy from
 *	-to takes a range specification describing the range to copy to
 *	-values -- Copies the values in the cells between the two ranges.
 *	-formulas -- [Default] Copies the formulas between the two ranges.
 *
 * Result:
 *	Returns a standard Tcl result, normally empty.
 */

/* Option table for the command */

static int copyRangeFlag;
static Range copyFromRange;
static Range copyToRange;
static Tk_ArgvInfo copyRangeOpts [] = {
  {"-from", TK_ARGV_GENFUNC, (char *) xessGetRangeArgv,
     (char *) &copyFromRange,
     "Range to copy from"},
  {"-to", TK_ARGV_GENFUNC, (char *) xessGetRangeArgv, (char *) &copyToRange,
     "Range to copy to"},
  {"-values", TK_ARGV_CONSTANT, (char *) 1, (char *) &copyRangeFlag,
     "Copy the values in the cells."},
  {"-formulas", TK_ARGV_CONSTANT, (char *) 0, (char *) &copyRangeFlag,
     "Copy the formulae in the cells (default)."},
  {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL}
};

int
XessConnObj_copy_range_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;

  int status;

  /* Parse command arguments */

  copyRangeFlag = 0;
  copyFromRange.r0 = 0;
  copyToRange.r0 = 0;
  status = Tk_ParseArgv (interp, Tk_MainWindow (interp),
			 &argc, argv, copyRangeOpts,
			 TK_ARGV_NO_LEFTOVERS);
  if (status != TCL_OK)
    return status;
  if (copyFromRange.r0 == 0) {
    Tcl_SetResult (interp, "-from argument must be supplied", TCL_STATIC);
    return TCL_ERROR;
  }
  if (copyToRange.r0 == 0) {
    Tcl_SetResult (interp, "-to argument must be supplied", TCL_STATIC);
    return TCL_ERROR;
  }

  /* Copy the range */

  status = xess_copy_range (conn -> port, copyRangeFlag,
			    copyFromRange, copyToRange);
  switch (status)
    {
    case 0:
      Tcl_SetResult (interp, "couldn't copy range", TCL_STATIC);
      return TCL_ERROR;
    case 1:
      return TCL_OK;
    default:
      sprintf (interp -> result,
	       "xess_copy_range returned unexpected status %d",
	       status);
      return TCL_ERROR;
    }
}

/*
 * XessConnObj_erase_range_cmd --
 *
 *	This function implements a Tcl command on an Xess connection object
 *	to erase a specified range on a spreadsheet.
 *
 * Syntax:
 *	connection erase_range range
 *
 * Where:
 *	connection is an Xess connection object
 *	range is the range to be erased.
 *
 * Result:
 *	Returns a standard Tcl result, normally empty.
 */

int
XessConnObj_erase_range_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;

  int status;
  Range range;

  /* Check syntax */

  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " range\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Decode the range */

  status = xessGetRange (interp, argv [1], &range);
  if (status != TCL_OK)
    return status;

  /* Erase the range */

  status = xess_erase_range (conn -> port, range);
  switch (status)
    {
    case 0:
      Tcl_SetResult (interp, "range not erased", TCL_STATIC);
      return TCL_ERROR;
    case 1:
      return TCL_OK;
    default:
      sprintf (interp -> result,
	       "xess_erase_range returned unexpected status %d",
	       status);
      return TCL_ERROR;
    }
}

/*
 * xessConnObj_move_range_cmd
 *
 *	This function implements the `move_range' Tcl command on an
 *	Xess connection object.
 *
 * Syntax:
 *	connection move_range -from RANGE -to CELL
 *
 * Where:
 *	connection is the name of an Xess connection object
 *	RANGE is the range to be moved
 *	CELL is the cell co-ordinates of the upper left corner of
 *		the place to move it to.
 *
 * Result:
 *	Returns a standard Tcl result, normally empty.
 */

/* Option table for the command */

static Range moveFromRange;
static XessCell moveToCell;
static Tk_ArgvInfo moveRangeOpts [] = {
  {"-from", TK_ARGV_GENFUNC, (char *) xessGetRangeArgv,
     (char *) &moveFromRange,
     "Range to move from"},
  {"-to", TK_ARGV_GENFUNC, (char *) xessGetCellArgv,
     (char *) &moveToCell,
     "Cell number of upper left corner of range to move to."},
  {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL}
};

int
XessConnObj_move_range_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;

  int status;

  /* Parse the argument vector */

  moveFromRange.r0 = 0;
  moveToCell.row = 0;
  status = Tk_ParseArgv (interp, Tk_MainWindow (interp), &argc, argv,
			 moveRangeOpts, TK_ARGV_NO_LEFTOVERS);
  if (status != TCL_OK)
    return status;
  if (moveFromRange.r0 == 0) {
    Tcl_SetResult (interp, "-from option must be supplied", TCL_STATIC);
    return TCL_ERROR;
  }
  if (moveToCell.row == 0) {
    Tcl_SetResult (interp, "-to option must be supplied", TCL_STATIC);
    return TCL_ERROR;
  }

  /* Do the move */

  status = xess_move_range (conn -> port, moveFromRange,
			    moveToCell.row, moveToCell.col);
  switch (status)
    {
    case 0:
      Tcl_SetResult (interp, "range not moved", TCL_STATIC);
      return TCL_ERROR;
    case 1:
      return TCL_OK;
    default:
      sprintf (interp -> result,
	       "xess_move_range returned unexpected status %d",
	       status);
      return TCL_ERROR;
    }
}

/*
 * XessConnObj_insert_rows_cmd --
 *
 *	Tcl command on an Xess connection object to insert rows into the
 *	spreadsheet
 *
 * Syntax:
 *	connection insert_rows -count N -at N
 *
 * Where:
 *	connection is the name of an Xess connection object
 *
 * Options:
 *	-count N
 *		Specifies the number of rows to insert.  Default is 1.
 *	-at N
 *		Specifies the row numbner in the sheet where the rows are
 *		to be inserted.  This option must be supplied.
 *
 * Results:
 *	Returns a standard Tcl result, normally empty.
 */

/* Options for insert_rows */
static XessCell insertAtRow;
static int insertNumRows;
static Tk_ArgvInfo insertRowsOpts [] = {
  {"-count", TK_ARGV_INT, (char *) NULL, (char *) &insertNumRows,
     "Number of rows to insert"},
  {"-at", TK_ARGV_GENFUNC, (char *) xessGetRowArgv, (char *) &insertAtRow,
     "Row number at which to insert"},
  {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL}
};

int
XessConnObj_insert_rows_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;

  int status;

  /* Parse the argument vector */

  insertAtRow.row = 0;
  insertNumRows = 1;
  status = Tk_ParseArgv (interp, Tk_MainWindow (interp), &argc, argv,
			 insertRowsOpts, TK_ARGV_NO_LEFTOVERS);
  if (status != TCL_OK)
    return status;
  if (insertAtRow.row == 0) {
    Tcl_SetResult (interp, "-at option must be supplied", TCL_STATIC);
    return TCL_ERROR;
  }

  /* Do the insert */

  status = xess_insert_rows (conn -> port, insertAtRow.row,
			     insertNumRows);
    switch (status)
    {
    case 0:
      Tcl_SetResult (interp, "row(s) not inserted", TCL_STATIC);
      return TCL_ERROR;
    case 1:
      return TCL_OK;
    default:
      sprintf (interp -> result,
	       "xess_insert_rows returned unexpected status %d",
	       status);
      return TCL_ERROR;
    }
}

/*
 * XessConnObj_insert_columns_cmd --
 *
 *	Tcl command on an Xess connection object to insert columns into the
 *	spreadsheet
 *
 * Syntax:
 *	connection insert_columns -count N -at N
 *
 * Where:
 *	connection is the name of an Xess connection object
 *
 * Options:
 *	-count N
 *		Specifies the number of columns to insert.  Default is 1.
 *	-at N
 *		Specifies the column numbner in the sheet where the columns are
 *		to be inserted.  This option must be supplied.
 *
 * Results:
 *	Returns a standard Tcl result, normally empty.
 */

/* Options for insert_columns */
static XessCell insertAtColumn;
static int insertNumColumns;
static Tk_ArgvInfo insertColumnsOpts [] = {
  {"-count", TK_ARGV_INT, (char *) NULL, (char *) &insertNumColumns,
     "Number of columns to insert"},
  {"-at", TK_ARGV_GENFUNC, (char *) xessGetColumnArgv,
     (char *) &insertAtColumn,
     "Column number at which to insert"},
  {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL}
};

int
XessConnObj_insert_columns_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;

  int status;

  /* Parse the argument vector */

  insertAtColumn.col = -1;
  insertNumColumns = 1;
  status = Tk_ParseArgv (interp, Tk_MainWindow (interp), &argc, argv,
			 insertColumnsOpts, TK_ARGV_NO_LEFTOVERS);
  if (status != TCL_OK)
    return status;
  if (insertAtColumn.col == -1) {
    Tcl_SetResult (interp, "-at option must be supplied", TCL_STATIC);
    return TCL_ERROR;
  }

  /* Do the insert */

  status = xess_insert_cols (conn -> port, insertAtColumn.col,
			     insertNumColumns);
  switch (status)
    {
    case 0:
      Tcl_SetResult (interp, "column(s) not inserted", TCL_STATIC);
      return TCL_ERROR;
    case 1:
      return TCL_OK;
    default:
      sprintf (interp -> result,
	       "xess_insert_cols returned unexpected status %d",
	       status);
      return TCL_ERROR;
    }
}

/*
 * XessConnObj_delete_rows_cmd --
 *
 *	Tcl connad on an Xess connection object to delete rows from a
 *	spreadsheet
 *
 * Syntax:
 *	connection delete_rows from to
 *
 * Where:
 *	connection is the name of an Xess connection object.
 *	from is the first row to delete
 *	to is the last row to delete
 *
 * Result:
 *	Returns a standard Tcl result, normally empty.
 */

int
XessConnObj_delete_rows_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  unsigned from;
  unsigned to;
  int status;

  /* Check syntax */

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " from to\"", (char *) NULL);
    return TCL_ERROR;
  }
  status = xessGetRow (interp, argv [1], 0, &from,
		       (int *) NULL, (char * *) NULL);
  if (status != TCL_OK)
    return status;
  status = xessGetRow (interp, argv [2], 0, &to,
		       (int *) NULL, (char * *) NULL);
  if (status != TCL_OK)
    return status;
  if (from > to) {
    Tcl_SetResult (interp, "rows out of sequence", TCL_STATIC);
    return TCL_ERROR;
  }

  /* Delete the rows */

  status = xess_delete_rows (conn -> port, (int) from, (int) to);
  switch (status)
    {
    case 0:
      Tcl_SetResult (interp, "row(s) not deleted", TCL_STATIC);
      return TCL_ERROR;
    case 1:
      return TCL_OK;
    default:
      sprintf (interp -> result,
	       "xess_delete_rows returned unexpected status %d",
	       status);
      return TCL_ERROR;
    }
}

/*
 * XessConnObj_delete_columns_cmd --
 *
 *	Tcl connad on an Xess connection object to delete columns from a
 *	spreadsheet
 *
 * Syntax:
 *	connection delete_columns from to
 *
 * Where:
 *	connection is the name of an Xess connection object.
 *	from is the first column to delete
 *	to is the last column to delete
 *
 * Result:
 *	Returns a standard Tcl result, normally empty.
 */

int
XessConnObj_delete_columns_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  unsigned from;
  unsigned to;
  int status;

  /* Check syntax */

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " from to\"", (char *) NULL);
    return TCL_ERROR;
  }
  status = xessGetColumn (interp, argv [1], 0, &from,
			  (int *) NULL, (char * *) NULL);
  if (status != TCL_OK)
    return status;
  status = xessGetColumn (interp, argv [2], 0, &to,
			  (int *) NULL, (char * *) NULL);
  if (status != TCL_OK)
    return status;
  if (from > to) {
    Tcl_SetResult (interp, "columns out of sequence", TCL_STATIC);
    return TCL_ERROR;
  }

  /* Delete the columns */

  status = xess_delete_cols (conn -> port, (int) from, (int) to);
  switch (status)
    {
    case 0:
      Tcl_SetResult (interp, "column(s) not deleted", TCL_STATIC);
      return TCL_ERROR;
    case 1:
      return TCL_OK;
    default:
      sprintf (interp -> result,
	       "xess_delete_cols returned unexpected status %d",
	       status);
      return TCL_ERROR;
    }
}

/*
 * XessConnObj_clear_sheet_cmd --
 *
 *	Tcl command on an Xess connection object to clear the spreadsheet.
 *
 * Syntax:
 *	connection clear_sheet
 *
 * Where:
 *	connection is the name of an Xess connection object.
 *
 * Results:
 *	Returns a standard Tcl result, normally empty.
 */

int
XessConnObj_clear_sheet_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;

  /* Check syntax */

  if (argc != 1) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1],
		      " ", argv [0], "\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Clear the sheet */

  status = xess_clear_sheet (conn -> port);
  switch (status)
    {
    case 0:
      Tcl_SetResult (interp, "sheet not cleared", TCL_STATIC);
      return TCL_ERROR;
    case 1:
      return TCL_OK;
    default:
      sprintf (interp -> result,
	       "xess_clear_sheet returned unexpected status %d",
	       status);
      return TCL_ERROR;
    }

}
