static char copyright []
= "$Id: xessCell.c,v 1.1 1994/08/26 13:35:48 johnsonm Exp $\n\
   Copyright (c) 1992 General Electric.  All rights reserved.";

/*
 *   Permission to use, copy, modify, and distribute this
 *   software and its documentation for any purpose and without
 *   fee is hereby granted, provided that the above copyright
 *   notice appear in all copies and that both that copyright
 *   notice and this permission notice appear in supporting
 *   documentation, and that the name of General Electric not be used in
 *   advertising or publicity pertaining to distribution of the
 *   software without specific, written prior permission.
 *   General Electric makes no representations about the suitability of
 *   this software for any purpose.  It is provided "as is"
 *   without express or implied warranty.
 *
 *   This work was supported by the DARPA Initiative in Concurrent
 *   Engineering (DICE) through DARPA Contract MDA972-88-C-0047.
 *
 * $Log: xessCell.c,v $
 * Revision 1.1  1994/08/26  13:35:48  johnsonm
 * Initial revision
 *
 *
 * Old log:
 * Revision 1.3  1994/03/02  22:52:53  kennykb
 * Changed xessGetRow and xessGetColumn to used unsigned row and column
 * numbers.  The Xess connection library is inconsistent.
 *
 * Revision 1.2  1994/02/28  20:44:50  kennykb
 * Removed #includes that are now obtained from tclXessInt.h
 *
 * Revision 1.1  1992/10/09  19:02:49  kennykb
 * Initial revision
 *
 *
 * xessCell.c --
 *
 *	This file contains functions that parse and synthesize Xess row,
 * column, cell and range addresses.
 */

#include <ctype.h>
#include "tclXessInt.h"

/* Cell parser */

static int xessParseCell _ANSI_ARGS_((Tcl_Interp * interp, char * cellname,
				      int flags,
				      unsigned * row, unsigned * col,
				      int * retflags, char ** next));

/* Flags for the above */

#define XESS_CELL_PARTIAL (1<<0)
				/* Cell name is only part of the string */

/*
 * xessGetRangeArgv --
 *
 *	Given an argument "key", and a pointer to a argument vector "argv"
 *	with "argc" elements, this function processes a single range from the
 *	head of argv, compacting it and returning a count of remaining
 *	arguments.  If it encounters an error it places it in the Tcl result
 *	of the interpreter `interp' and returns -1.
 *
 *	The `dest' pointer designates a Range structure that receives
 *	the address of the range.
 */

int
xessGetRangeArgv (dest, interp, key, argc, argv)
     char * dest;
     Tcl_Interp * interp;
     char * key;
     int argc;
     char * * argv;
{
  int status;
  int i;

  /* Make sure that the range is supplied */

  if (argc == 0) {
    sprintf (interp -> result, "%s option requires a range specifier", key);
    return -1;
  }

  /* Parse the range */

  status = xessGetRange (interp, argv [0], (Range *) dest);
  if (status != TCL_OK) {
    return -1;
  }

  /* Compact the argument vector */

  --argc;
  for (i = 0; i < argc; ++i) {
    argv [i] = argv [i+1];
  }
  return argc;
}

/*
 * xessGetRange --
 *
 *	Given a string `string', that is expected to contain a range
 *	(e.g., `A10..B16') within a spreadsheet, this procedure parses it
 *	and places the result in the structure `range'.  A status return
 *	reflecting the success or failure of the parse is returned as
 *	a normal Tcl result.
 */

int
xessGetRange (interp, string, range)
     Tcl_Interp * interp;
     char * string;
     Range *range;
{
  unsigned row, col;
  int flags;
  char * fstring;
  int status;

  range -> flags = 0;

  /* Parse the first cell */

  status = xessParseCell (interp, string, XESS_CELL_PARTIAL,
			&row, &col, &flags, &fstring);
  if (status != TCL_OK)
    return status;

  range -> r0 = row;
  range -> c0 = col;
  if (flags & XESS_CELL_ABSROW)
    range -> flags |= ABS_R0;
  if (flags & XESS_CELL_ABSCOL)
    range -> flags |= ABS_C0;
  
  /* Get the `..' delimiter, if there is one */

  switch (*fstring++)
    {
    case '\0':
      range -> r1 = range -> r0;
      range -> c1 = range -> c0;
      if (range -> flags & ABS_R0)
	range -> flags |= ABS_R1;
      if (range -> flags & ABS_C0)
	range -> flags |= ABS_C1;
      range -> flags |= CELL_ONLY;
      return TCL_OK;

    case '.':
      if (*fstring++ != '.') {
	Tcl_AppendResult (interp, "syntax error in cell range \"", string,
			  "\"", (char *) NULL);
	return TCL_ERROR;
      }

      /* Parse the ending cell address */

      status = xessParseCell (interp, fstring, 0,
			    &row, &col, &flags, (char * *) NULL);
      if (status != TCL_OK)
	return status;

      if (row < range -> r0) {
	range -> r1 = range -> r0;
	range -> r0 = row;
	if (range -> flags & ABS_R0) {
	  range -> flags |= ABS_R1;
	  range -> flags &= ~ ABS_R0;
	}
	if (flags & XESS_CELL_ABSROW)
	  range -> flags |= ABS_R0;
      } else {
	range -> r1 = row;
	if (flags & XESS_CELL_ABSROW)
	  range -> flags |= ABS_R1;
      }

      if (col < range -> c0) {
	range -> c1 = range -> c0;
	range -> c0 = col;
	if (range -> flags & ABS_C0) {
	  range -> flags |= ABS_C1;
	  range -> flags &= ~ ABS_C0;
	}
	if (flags & XESS_CELL_ABSCOL)
	  range -> flags |= ABS_C0;
      } else {
	range -> c1 = col;
	if (flags & XESS_CELL_ABSCOL)
	  range -> flags |= ABS_C1;
      }

      return TCL_OK;

    default:
      Tcl_AppendResult (interp, "syntax error in cell range \"", string,
			  "\"", (char *) NULL);
      return TCL_ERROR;
    }
}

/*
 * xessGetCellArgv --
 *
 *	Given an argument "key", and a pointer to a argument vector "argv"
 *	with "argc" elements, this function processes a single cell from the
 *	head of argv, compacting it and returning a count of remaining
 *	arguments.  If it encounters an error it places it in the Tcl result
 *	of the interpreter `interp' and returns -1.
 *
 *	The `dest' pointer designates the XessCell data structure that is
 *	filled in with the cell's address.
 */

int
xessGetCellArgv (dest, interp, key, argc, argv)
     char * dest;
     Tcl_Interp * interp;
     char * key;
     int argc;
     char * * argv;
{
  int status;
  int i;
  XessCell * results = (XessCell *) dest;

  /* Make sure that the cell is supplied */

  if (argc == 0) {
    sprintf (interp -> result, "%s option requires a cell specifier", key);
    return -1;
  }

  /* Parse the cell */

  status = xessGetCell (interp, argv [0], results);
  if (status != TCL_OK) {
    return -1;
  }

  /* Compact the argument vector */

  --argc;
  for (i = 0; i < argc; ++i) {
    argv [i] = argv [i+1];
  }
  return argc;
}

/*
 * xessGetCell --
 *
 *	Given a string, `string', containing a cell address in a spreadsheet,
 *	parse the cell address into the structure, `cell', and return a
 *	standard Tcl result using `interp'.
 */

int
xessGetCell (interp, string, cell)
     Tcl_Interp * interp;
     char * string;
     XessCell * cell;
{
  unsigned row, col;
  int flags;
  int status;

  flags = 0;
  status = xessParseCell (interp, string, 0,
			  &row, &col, &flags, (char * *) NULL);
  if (status == TCL_OK) {
    cell -> row = row;
    cell -> col = col;
    cell -> flags = flags;
  }

  return status;
}

/* xessParseCell --
 *
 *	Given a string `string', that is expected to contain a cell address
 *	(e.g., `AC37') in an Xess spreadsheet, this procedure extracts the
 *	row and column number into the variables `row' and `column'.
 *	The `flags' argument is special.  If it has the XESS_CELL_PARTIAL
 *	bit set, the cell name need not be null-terminated.  Instead, any
 *	prefix that is a cell name is accepted.  In any case, if the `next'
 *	argument is non-NULL, a pointer to one beyond the last acceptable
 * 	character of the string will be stored in it.  `next' will point
 *	to the terminating '\0' if the string consisted entirely of a cell
 *	name, to the first unacceptable character if the string began with
 *	a cell name and contained other data, or to the beginning of the
 *	string if the string could not be parsed.  `Retflags' contains some
 *	combination of the bits XESS_CELL_ABSROW and XESS_CELL_ABSCOL to
 *	indicate absolute addressing.
 */

static int
xessParseCell (interp, string, flags, row, col, retflags, next)
     Tcl_Interp * interp;
     char * string;
     int flags;
     unsigned * row;
     unsigned * col;
     int * retflags;
     char * * next;
{
  int status;
  int rflags;
  char * rstring;

  /* Set up the continuation pointer in case of parse failure */

  if (next != NULL) {
    *next = string;
  }
  if (retflags != NULL) {
    *retflags = 0;
  }

  /* Parse the column number */

  status = xessGetColumn (interp, string, XESS_CELL_PARTIAL,
			  col, &rflags, &rstring);
  if (status != TCL_OK) {
    *row = 0;
    return status;
  }
  if (retflags != NULL) {
    *retflags |= rflags;
  }
  string = rstring;

  /* Parse the row number */

  status = xessGetRow (interp, string, (flags & XESS_CELL_PARTIAL),
		       row, &rflags, &rstring);
  if (status != TCL_OK) {
    return status;
  }
  if (retflags != NULL) {
    *retflags |= rflags;
  }
  string = rstring;

  /* Parse OK */

  if (next != NULL) {
    *next = string;
  }
  return TCL_OK;
}

/*
 * xessGetColumnArgv --
 *
 *	Given an argument "key", and a pointer to a argument vector "argv"
 *	with "argc" elements, this function processes a single column number
 * 	from the head of argv, compacting it and returning a count of remaining
 *	arguments.  If it encounters an error it places it in the Tcl result
 *	of the interpreter `interp' and returns -1.
 *
 *	The `dest' pointer designates an XessCell structure that is
 *	filled in with the column number and ABSCOL flag; the row number
 *	in the structure is left untouched.
 */

int
xessGetColumnArgv (dest, interp, key, argc, argv)
     char * dest;
     Tcl_Interp * interp;
     char * key;
     int argc;
     char * * argv;
{
  int status;
  int i;
  unsigned col;
  XessCell * results = (XessCell *) dest;
  results -> flags = 0;

  /* Make sure that the column is supplied */

  if (argc == 0) {
    sprintf (interp -> result, "%s option requires a column specifier", key);
    return -1;
  }

  /* Parse the column */

  status = xessGetColumn (interp, argv [0], 0, &col,
		       &(results -> flags), (char * *) NULL);
  if (status != TCL_OK) {
    return -1;
  }
  results -> col = col;

  /* Compact the argument vector */

  --argc;
  for (i = 0; i < argc; ++i) {
    argv [i] = argv [i+1];
  }
  return argc;
}

/*
 * xessGetColumn --
 *
 *	Given a string `string', this function extracts and returns an
 * Xess column number in `col'.  If the column number is absolute, the
 * XESS_CELL_ABSCOL bit is set in `retflags.'  If the XESS_CELL_PARTIAL bit
 * is set in `flags,' then the column number need not be null-terminated.
 * Instead, any prefix that is a valid column number is accepted.  In any case
 * if the `next' argument is non-NULL, a pointer to one location beyind the
 * last acceptable character of the string will be stored therein.  `next'
 * will point to the terminating '\0' if the string consisted entirely of
 * a column number, to the first unacceptable character if the string begins
 * with a column number and is followed by other data, or to the beginning of
 * the string if no column number is found.
 */

int
xessGetColumn (interp, string, flags, col, retflags, next)
     Tcl_Interp * interp;
     char * string;
     int flags;
     unsigned * col;
     int * retflags;
     char * * next;
{
  char c;
  char * ostring = string;

  /* Set up the continuation pointer in case of parse failure */

  if (next != NULL) {
    *next = string;
  }
  if (retflags != NULL) {
    *retflags = 0;
  }

  /* Get the first character */

  c = *string++;

  /* Leading $ marks an absolute column address */

  if (c == '$') {
    c = *string++;
    if (retflags != NULL) {
      *retflags |= XESS_CELL_ABSCOL;
    }
  }
 
  /* Parse the column */

  if (isalpha (c)) {
    if (islower (c)) c = toupper (c);
    *col = c - 'A';
    c = *string++;
    if (isalpha (c)) {
      if (islower (c)) c = toupper (c);
      *col = (*col + 1) * 26 + (c - 'A');
      c = *string++;
    }

    /* Parse succeeded */

    if ((c == '\0') || (flags & XESS_CELL_PARTIAL)) {
      if (next != NULL) {
	*next = string - 1;
      }
      return TCL_OK;
    }
  }

  /* Parse error */

  *col = -1;
  Tcl_AppendResult (interp, "invalid column number in \"", ostring, "\"",
		    (char *) NULL);
  return TCL_ERROR;
}

/*
 * xessGetRowArgv --
 *
 *	Given an argument "key", and a pointer to a argument vector "argv"
 *	with "argc" elements, this function processes a single row number
 * 	from the head of argv, compacting it and returning a count of remaining
 *	arguments.  If it encounters an error it places it in the Tcl result
 *	of the interpreter `interp' and returns -1.
 *
 *	The `dest' pointer designates an XessCell structure that is
 *	filled in with the row number and ABSROW flag; the column number in the
 *	structure is left untouched.
 */

int
xessGetRowArgv (dest, interp, key, argc, argv)
     char * dest;
     Tcl_Interp * interp;
     char * key;
     int argc;
     char * * argv;
{
  int status;
  int i;
  unsigned row;
  XessCell * results = (XessCell *) dest;
  results -> flags = 0;

  /* Make sure that the row is supplied */

  if (argc <= 0) {
    sprintf (interp -> result, "%s option requires a row specifier", key);
    return -1;
  }

  /* Parse the row */

  status = xessGetRow (interp, argv [0], 0, &row,
		       &(results -> flags), (char * *) NULL);
  if (status != TCL_OK) {
    return -1;
  }
  results -> row = row;

  /* Compact the argument vector */

  --argc;
  for (i = 0; i < argc; ++i) {
    argv [i] = argv [i+1];
  }
  return argc;
}

/*
 * xessGetRow --
 *
 *	Given a string `string', this function extracts and returns an
 * Xess row number in `row'.  If the row number is absolute, the
 * XESS_CELL_ABSROW bit is set in `retflags.'  If the XESS_CELL_PARTIAL bit
 * is set in `flags,' then the row number need not be null-terminated.
 * Instead, any prefix that is a valid row number is accepted.  In any case
 * if the `next' argument is non-NULL, a pointer to one location beyind the
 * last acceptable character of the string will be stored therein.  `next'
 * will point to the terminating '\0' if the string consisted entirely of
 * a row number, to the first unacceptable character if the string begins
 * with a row number and is followed by other data, or to the beginning of
 * the string if no row number is found.
 */

int
xessGetRow (interp, string, flags, row, retflags, next)
     Tcl_Interp * interp;
     char * string;
     int flags;
     unsigned * row;
     int * retflags;
     char * * next;
{
  char c;
  char * ostring = string;

  /* Set up the continuation pointer in case of parse failure */

  if (next != NULL) {
    *next = string;
  }

  /* Get the first character */

  c = *string++;

  /* Leading $ marks an absolute row address */

  if (c == '$') {
    c = *string++;
    if (retflags != NULL) {
      *retflags |= XESS_CELL_ABSROW;
    }
  }
 
  /* Parse the row */

  *row = 0;
  if (isdigit (c)) {
    do {
      *row = 10 * *row + (c - '0');
      c = *string++;
    } while (isdigit(c) && *row <= 999);

    /* Parsed ok -- return TCL_OK if everything was accepted, or if the
       user wants to continue. */

    if ((flags & XESS_CELL_PARTIAL) || (c == '\0')) {
      if (next != NULL) {
	*next = string - 1;
      }
      return TCL_OK;
    }
  }

  /* Parse error */

  *row = 0;
  Tcl_AppendResult (interp, "invalid row number in \"", ostring, "\"",
		    (char *) NULL);
  return TCL_ERROR;
}

/* xessStoreRange --
 *
 *	Given a range of cells in the spreadsheet specified by `range', this
 * function translates their addresses to alphanumeric and stores the result
 * in `string.'  If the `next' argument is non-NULL, a pointer to the 
 * terminating `\0' in `string' is stored in `next'.
 */

int
xessStoreRange (interp, range, string, next)
     Tcl_Interp * interp;
     Range * range;
     char * string;
     char * * next;
{
  int flags;
  int status;
  char * p;

  if (next != (char * *) NULL)
    *next = string;

  /* Store the starting cell */

  flags = 0;
  if (range -> flags & ABS_R0)
    flags |= XESS_CELL_ABSROW;
  if (range -> flags & ABS_C0)
    flags |= XESS_CELL_ABSCOL;
  status = xessStoreCell (interp, range -> r0, range -> c0, flags,
			  string, &p);
  if (status != TCL_OK)
    return status;

  /* Store the `..' and the ending cell */

  *p++ = '.';
  *p++ = '.';
  flags = 0;
  if (range -> flags & ABS_R1)
    flags |= XESS_CELL_ABSROW;
  if (range -> flags & ABS_C1)
    flags |= XESS_CELL_ABSCOL;
  status = xessStoreCell (interp, range -> r1, range -> c1, flags,
			  p, next);

  return status;
}

/*
 * xessStoreCell --
 *
 *	Given a cell address specified by `row' and `col', this function
 * translates the address to alphanumeric and stores it in `string.'  If the
 * `next' argument is non-NULL, a pointer to the terminating '\0' in `string'
 * is stored in `next'.  The `XESS_CELL_ABSROW' and `XESS_CELL_ABSCOL' bits
 * in `flags' control whether '$' characters are stored to mark absolute
 * row and column addresses.
 */

int
xessStoreCell (interp, row, col, flags, string, next)
     Tcl_Interp * interp;
     int row;
     int col;
     int flags;
     char * string;
     char * * next;
{
  int status;

  status = xessStoreColumn (interp, col, flags, string, &string);
  if (status != 0)
    return status;

  status = xessStoreRow (interp, row, flags, string, next);
  return status;
}

/*
 * xessStoreColumn --
 *
 * 	Given a column number `col', this function convers it to alphabetics
 * and stores it in `string'.  If the `next' argument is non-NULL, a pointer
 * to the terminating '\0' in `string' is stored in `next'.  The 
 * `XESS_CELL_ABSCOL' bit in `flags' controls whether a `$' character is
 * required at the start of the string.
 */

int
xessStoreColumn (interp, col, flags, string, next)
     Tcl_Interp * interp;
     int col;
     int flags;
     char * string;
     char * * next;
{
  if (col < 0 || col > 701) {
    sprintf (interp -> result, "column number %d out of range", (int) col);
    return TCL_ERROR;
  }

  if (flags & XESS_CELL_ABSCOL)
    *string++ = '$';
  if (col >= 26) {
    *string++ = col / 26 - 1 + 'A';
    *string++ = col % 26 + 'A';
  }
  else {
    *string++ = col + 'A';
  }
  *string = '\0';

  if (next != NULL) {
    *next = string;
  }

  return TCL_OK;
}

/*
 * xessStoreRow --
 *
 * 	Given a row number `row', this function convers it to alphabetics
 * and stores it in `string'.  If the `next' argument is non-NULL, a pointer
 * to the terminating '\0' in `string' is stored in `next'.  The 
 * `XESS_CELL_ABSROW' bit in `flags' controls whether a `$' character is
 * required at the start of the string.
 */

int
xessStoreRow (interp, row, flags, string, next)
     Tcl_Interp * interp;
     int row;
     int flags;
     char * string;
     char * * next;
{
  if (row < 1 || row > 9999) {
    sprintf (interp -> result, "row number %d out of range", (int) row);
    return TCL_ERROR;
  }

  if (flags & XESS_CELL_ABSROW)
    *string++ = '$';
  sprintf (string, "%d", (int) row);
  string += strlen (string);

  if (next != NULL) {
    *next = string;
  }

  return TCL_OK;
}
