static char copyright []
= "$Id: xessCOStore.c,v 1.1 1994/08/26 13:29:53 johnsonm Exp $\n\
   Copyright (c) 1992 General Electric.  All rights reserved.";

/*
 *   Permission to use, copy, modify, and distribute this
 *   software and its documentation for any purpose and without
 *   fee is hereby granted, provided that the above copyright
 *   notice appear in all copies and that both that copyright
 *   notice and this permission notice appear in supporting
 *   documentation, and that the name of General Electric not be used in
 *   advertising or publicity pertaining to distribution of the
 *   software without specific, written prior permission.
 *   General Electric makes no representations about the suitability of
 *   this software for any purpose.  It is provided "as is"
 *   without express or implied warranty.
 *
 *   This work was supported by the DARPA Initiative in Concurrent
 *   Engineering (DICE) through DARPA Contract MDA972-88-C-0047.
 *
 * $Log: xessCOStore.c,v $
 * Revision 1.1  1994/08/26  13:29:53  johnsonm
 * Initial revision
 *
 *
 * Old log:
 * Revision 1.2  1994/02/28  20:43:29  kennykb
 * Changed call to Tk_ParseArgv to use the application main window when
 * parsing arguments to `$connection store_formatted_number', rather than
 * requiring the XessConnection object to carry a window.
 *
 * Revision 1.1  1992/10/09  19:01:54  kennykb
 * Initial revision
 *
 *
 * xessCOStore.c --
 *
 *	This file contains the code that processes the `store' family of
 * commands on Xess connection objects.
 */

#include <tcl.h>
#include <tk.h>
#include "tclXessInt.h"

/*
 * XessConnObj_store_cmd --
 *
 *	This function processes the `store' command on an Xess connection
 *	object.
 *
 * Syntax:
 *	connName store cellAddress value
 *
 *	where cellAddress is the address of a cell in the spreadsheet.
 *
 *	      value is the value to store there.
 *
 *	If value is numeric, it is stored as a number.
 *
 *	Otherwise, if the first character of value is =, it is stored as
 *	a formula.
 *
 *	Otherwise, it is stored as a label.
 */

int
XessConnObj_store_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  XessCell cell;
  double value;

  /* Check argument count */

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " cellAddress value\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Get cell address */

  status = xessGetCell (interp, argv[1], &cell);
  if (status != TCL_OK)
    return status;

  /* Try to parse a number */

  status = Tcl_GetDouble (interp, argv [2], &value);
  if (status == TCL_OK) {
    xess_store_number (conn -> port, cell.row, cell.col, value);
    return TCL_OK;
  }

  /* No number -- formula or string */

  Tcl_ResetResult (interp);
  if (argv [2] [0] == '=') {
    xess_store_formula (conn -> port, cell.row, cell.col, argv [2]);
  } else {
    xess_store_label (conn -> port, cell.row, cell.col, argv [2]);
  }

  return TCL_OK;
}

/*
 * XessConnObj_store_formatted_number_cmd --
 *
 *	This function processes the `store_formatted_number' command
 * on an Xess connection object.
 *
 * Syntax:
 *	connName store_formatted_number cellAddress value ?-option value...?
 *
 *	where cellAddress is the address of a cell in the spreadsheet.
 *
 *	      value is the value to store there.
 *
 * Options:
 *
 *	-places <n> - Number of decimal places.  Default is 8.
 *	-font NAME - Name for the font style.  Default is `default', which
 *		uses the Xess default font.  Other possibilities are `normal'
 *		`bold', `italic', and `bolditalic'.
 *	-format CHAR - Character describing the format to use.  Possibilities
 *		include:
 *			'*'	[Default]	Xess default format
 *			'e'			Scientific notation
 *			'f'			Fixed point
 *			'g'			General
 *			'$'			Dollar amount
 *			','			Comma-delimited
 *			'h'			Hex
 *			'l'			Logical
 *			'a'			dd-mmm-yy
 *			'b'			dd-mmm
 *			'c'			mmm-yy
 *			'd'			mm/dd/yy
 *			'i'			Hidden
 *			't'			hh:mm:ss
 *			'%'			Percentage
 *			'x'			Text
 */

/* Option table for `store formatted number' */

static int places;
static int fontid;
static int fmtid;
static Tk_ArgvInfo argTable [] = {
  {"-places", TK_ARGV_INT, (char *) NULL, (char *) &places,
     "Number of decimal places to display"},
  {"-font", TK_ARGV_GENFUNC, (char *) xessGetFontArgv, (char *) &fontid,
     "Font to use"},
  {"-format", TK_ARGV_GENFUNC, (char *) xessGetFormatArgv, (char *) &fmtid,
     "Format to use"},
  {(char *) NULL, TK_ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL}};

int
XessConnObj_store_formatted_number_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  XessCell cell;
  double value;

  /* Check argument count */

  if (argc < 3) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " cellAddress value\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Get cell address */

  status = xessGetCell (interp, argv[1], &cell);
  if (status != TCL_OK)
    return status;

  /* Try to parse a number */

  status = Tcl_GetDouble (interp, argv [2], &value);
  if (status != TCL_OK) {
    return status;
  }

  /* Parse off the remaining arguments */

  places = 0;
  fmtid = FMT_DEF;
  fontid = FNT_DEF;

  argc -= 3;
  argv += 3;
  status = Tk_ParseArgv (interp, Tk_MainWindow (interp), &argc, argv, argTable,
			 TK_ARGV_DONT_SKIP_FIRST_ARG | TK_ARGV_NO_LEFTOVERS);

  if (status != TCL_OK)
    return status;

  xess_store_formatted_number (conn -> port, cell.row, cell.col, value,
			       fontid, fmtid, places);

  return TCL_OK;
}

/*
 * XessConnObj_store_formula_cmd --
 *
 *	This function processes the `store_formula' command on an Xess
 *	connection object.
 *
 * Syntax:
 *	connName store_formula cellAddress value
 *
 *	where cellAddress is the address of a cell in the spreadsheet.
 *
 *	      value is the value to store there.
 *
 */

int
XessConnObj_store_formula_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  XessCell cell;

  /* Check argument count */

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " cellAddress value\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Get cell address */

  status = xessGetCell (interp, argv[1], &cell);
  if (status != TCL_OK)
    return status;

  /* Store the formula */

  xess_store_formula (conn -> port, cell.row, cell.col, argv [2]);

  return TCL_OK;
}

/*
 * XessConnObj_store_label_cmd --
 *
 *	This function processes the `store_label' command on an Xess
 *	connection object.
 *
 * Syntax:
 *	connName store_label cellAddress value
 *
 *	where cellAddress is the address of a cell in the spreadsheet.
 *
 *	      value is the value to store there.
 *
 */

int
XessConnObj_store_label_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  XessCell cell;

  /* Check argument count */

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " cellAddress value\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Get cell address */

  status = xessGetCell (interp, argv[1], &cell);
  if (status != TCL_OK)
    return status;

  /* Store the label */

  xess_store_label (conn -> port, cell.row, cell.col, argv [2]);

  return TCL_OK;
}

/*
 * XessConnObj_store_number_cmd --
 *
 *	This function processes the `store_number' command on an Xess
 *	connection object.
 *
 * Syntax:
 *	connName store_number cellAddress value
 *
 *	where cellAddress is the address of a cell in the spreadsheet.
 *
 *	      value is the value to store there.
 *
 */

int
XessConnObj_store_number_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  XessCell cell;
  double value;

  /* Check argument count */

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " cellAddress value\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Get cell address */

  status = xessGetCell (interp, argv[1], &cell);
  if (status != TCL_OK)
    return status;

  /* Get the value */

  status = Tcl_GetDouble (interp, argv [2], &value);
  if (status != TCL_OK)
    return status;

  /* Store the number */

  xess_store_number (conn -> port, cell.row, cell.col, value);

  return TCL_OK;
}

/*
 * XessConnObj_store_tagged_number_cmd --
 *
 *	This function processes the `store_tagged_number' command on an Xess
 *	connection object.
 *
 * Syntax:
 *	connName store_tagged_number cellAddress value
 *
 *	where cellAddress is the address of a cell in the spreadsheet.
 *
 *	      value is the value to store there.
 *
 */

int
XessConnObj_store_tagged_number_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  XessCell cell;
  double value;

  /* Check argument count */

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1], " ",
		      argv [0], " cellAddress value\"", (char *) NULL);
    return TCL_ERROR;
  }

  /* Get cell address */

  status = xessGetCell (interp, argv[1], &cell);
  if (status != TCL_OK)
    return status;

  /* Get the value */

  status = Tcl_GetDouble (interp, argv [2], &value);
  if (status != TCL_OK)
    return status;

  /* Store the number */

  xess_store_tagged_number (conn -> port, cell.row, cell.col, value);

  return TCL_OK;
}
