static char copyright []
= "$Id: xessCOGet.c,v 1.1 1994/08/26 13:26:06 johnsonm Exp $\n\
   Copyright (c) 1992 General Electric.  All rights reserved.";

/*
 *   Permission to use, copy, modify, and distribute this
 *   software and its documentation for any purpose and without
 *   fee is hereby granted, provided that the above copyright
 *   notice appear in all copies and that both that copyright
 *   notice and this permission notice appear in supporting
 *   documentation, and that the name of General Electric not be used in
 *   advertising or publicity pertaining to distribution of the
 *   software without specific, written prior permission.
 *   General Electric makes no representations about the suitability of
 *   this software for any purpose.  It is provided "as is"
 *   without express or implied warranty.
 *
 *   This work was supported by the DARPA Initiative in Concurrent
 *   Engineering (DICE) through DARPA Contract MDA972-88-C-0047.
 *
 * $Log: xessCOGet.c,v $
 * Revision 1.1  1994/08/26  13:26:06  johnsonm
 * Initial revision
 *
 *
 * Old Log:
 * Revision 1.2  1994/02/28  20:41:58  kennykb
 * Removed #includes that are now obtained from tclXessInt.h
 *
 * Revision 1.1  1992/10/09  19:01:19  kennykb
 * Initial revision
 *
 *
 * xessCOGet.c --
 *
 *	This file contains the command processing routines for the
 * `get value', `get string', `get number', and `get formula' commands
 * on an Xess connection object.  All of them accept a cell address and
 * return either an appropriate value or an error.
 */

#include "tclXessInt.h"

/*
 * XessConnObj_get_value_cmd --
 *
 *      Return an arbitrary value from a spreadsheet cell
 *
 * Syntax:
 *
 *	connectionName get_value cellAddress
 *
 * Result:
 *
 *	Returns a standard Tcl result.  Return string is either the
 * string value of the cell or the numeric value of the cell's content.
 */

int
XessConnObj_get_value_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  XessCell cell;
  int xessStatus;
  double value;
  char * string;

  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1],
		      " ", argv [0], " cellAddress\"", (char *) NULL);
    return TCL_ERROR;
  }

  status = xessGetCell (interp, argv [1], &cell);

  if (status != TCL_OK)
    return status;

  xessStatus = xess_get_value (conn -> port, cell.row, cell.col,
			       &value, &string);

  switch (xessStatus)
    {
    case XESS_EMPTY:
      Tcl_AppendResult (interp, "empty cell ", argv [1], (char *) NULL);
      return TCL_ERROR;
    case XESS_NUMBER:
      sprintf (interp -> result, "%#g", value);
      return TCL_OK;
    case XESS_STRING:
      Tcl_SetResult (interp, string, TCL_DYNAMIC);
      return TCL_OK;
    case XESS_ERROR:
      Tcl_SetResult (interp, "xess connection library returned XESS_ERROR",
		     TCL_STATIC);
      return TCL_ERROR;
    default:
      sprintf (interp -> result, "unexpected return %d from Xess library",
	       xessStatus);
      return TCL_ERROR;
    }
}

/*
 * XessConnObj_get_string_cmd --
 *
 *      Return a string value from a spreadsheet cell
 *
 * Syntax:
 *
 *	connectionName get_string cellAddress
 *
 * Result:
 *
 *	Returns a standard Tcl result.  Return string is the
 * string value of the cell, even if it contains a number.
 *
 * Bugs:
 *	Returns rubbish if invoked on an empty cell or a cell with a `hidden'
 *	format.
 */

int
XessConnObj_get_string_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  XessCell cell;
  int xessStatus;
  char * string;

  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1],
		      " ", argv [0], " cellAddress\"", (char *) NULL);
    return TCL_ERROR;
  }

  status = xessGetCell (interp, argv [1], &cell);

  if (status != TCL_OK)
    return status;

  xessStatus = xess_get_string (conn -> port, cell.row, cell.col, &string);

  switch (xessStatus)
    {
    case XESS_EMPTY:
      Tcl_AppendResult (interp, "empty cell ", argv [1], (char *) NULL);
      return TCL_ERROR;
    case XESS_STRING:
      Tcl_SetResult (interp, string, TCL_DYNAMIC);
      return TCL_OK;
    case XESS_ERROR:
      Tcl_SetResult (interp, "xess connection library returned XESS_ERROR",
		     TCL_STATIC);
      return TCL_ERROR;
    default:
      break;
    }

  sprintf (interp -> result, "unexpected return %d from Xess library",
	   xessStatus);
  return TCL_ERROR;
}

/*
 * XessConnObj_get_number_cmd --
 *
 *      Return a numeric value from a spreadsheet cell
 *
 * Syntax:
 *
 *	connectionName get_number cellAddress
 *
 * Result:
 *
 *	Returns a standard Tcl result.  Return string is the
 * numeric value of the cell.  Use on a cell without a numeric value is
 * an error.
 */

int
XessConnObj_get_number_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  XessCell cell;
  int xessStatus;
  double value;

  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1],
		      " ", argv [0], " cellAddress\"", (char *) NULL);
    return TCL_ERROR;
  }

  status = xessGetCell (interp, argv [1], &cell);
  if (status != TCL_OK)
    return status;

  xessStatus = xess_get_number (conn -> port, cell.row, cell.col, &value);

  switch (xessStatus)
    {
    case XESS_EMPTY:
      Tcl_AppendResult (interp, "empty or non-numeric cell ",
			argv [1], (char *) NULL);
      return TCL_ERROR;
    case XESS_NUMBER:
      sprintf (interp -> result, "%#g", value);
      return TCL_OK;
    case XESS_ERROR:
      Tcl_SetResult (interp, "xess connection library returned XESS_ERROR",
		     TCL_STATIC);
      return TCL_ERROR;
    default:
      sprintf (interp -> result, "unexpected return %d from Xess library",
	       xessStatus);
      return TCL_ERROR;
    }
}

/*
 * XessConnObj_get_formula_cmd --
 *
 *      Return a formula value from a spreadsheet cell
 *
 * Syntax:
 *
 *	connectionName get_formula cellAddress
 *
 * Result:
 *
 *	Returns a standard Tcl result.  Return string is the
 * formula value of the cell.  The result includes the equal sign,
 * double quote, single quote, or caret that flags the type of
 * formula.
 */

int
XessConnObj_get_formula_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;
  int status;
  XessCell cell;
  int xessStatus;
  char * formula;

  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1],
		      " ", argv [0], " cellAddress\"", (char *) NULL);
    return TCL_ERROR;
  }

  status = xessGetCell (interp, argv [1], &cell);
  if (status != TCL_OK)
    return status;

  xessStatus = xess_get_formula (conn -> port, cell.row, cell.col, &formula);

  switch (xessStatus)
    {
    case XESS_EMPTY:
      Tcl_AppendResult (interp, "empty cell ", argv [1], (char *) NULL);
      return TCL_ERROR;
    case XESS_LABEL:
    case XESS_FORMULA:
      Tcl_SetResult (interp, formula, TCL_DYNAMIC);
      return TCL_OK;
    case XESS_ERROR:
      Tcl_SetResult (interp, "xess connection library returned XESS_ERROR",
		     TCL_STATIC);
      return TCL_ERROR;
    default:
      sprintf (interp -> result, "unexpected return %d from Xess library",
	       xessStatus);
      return TCL_ERROR;
    }
}
