/*
 *   Permission to use, copy, modify, and distribute this
 *   software and its documentation for any purpose and without
 *   fee is hereby granted, provided that the above copyright
 *   notice appear in all copies and that both that copyright
 *   notice and this permission notice appear in supporting
 *   documentation, and that the name of General Electric not be used in
 *   advertising or publicity pertaining to distribution of the
 *   software without specific, written prior permission.
 *   General Electric makes no representations about the suitability of
 *   this software for any purpose.  It is provided "as is"
 *   without express or implied warranty.
 *
 *   This work was supported by the DARPA Initiative in Concurrent
 *   Engineering (DICE) through DARPA Contract MDA972-88-C-0047.
 *
 * xessCursor.c--
 *
 *	This file contains the functions that provide the Tcl bindings for
 * the Xess functions that access the spreadsheet cursor.
 *
 */

static char copyright []
= "$Id: xessCOCursor.c,v 1.1 1994/08/25 21:05:44 johnsonm Exp $\
   Copyright (c) 1992 General Electric.  All rights reserved.";
char RCSid_xessCursor_c []
= "$Id: xessCOCursor.c,v 1.1 1994/08/25 21:05:44 johnsonm Exp $";
/* $Log: xessCOCursor.c,v $
 * Revision 1.1  1994/08/25  21:05:44  johnsonm
 * Initial revision
 *
 *
 */

#include <tcl.h>
#include <tk.h>
#include "tclXessInt.h"

/*
 * XessConnObj_get_location_cmd --
 *
 *	Command on an Xess connection object to get the current cursor
 *	position.
 *
 * Syntax:
 *
 *	$connection get_location
 *
 * Result:
 *
 *	A standard Tcl result, whose string value is the current location
 * of the cursor.
 */

int
XessConnObj_get_location_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;

  int row;
  int col;
  int status;

  if (argc != 1) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1],
		      " ", argv [0], "\"", (char *) NULL);
    return TCL_ERROR;
  }

  status = xess_get_location (conn -> port, &row, &col);

  if (status == 0) {
    Tcl_SetResult (interp, "connection closed unexpectedly", TCL_STATIC);
    XessCloseCallback (conn -> port);
    return TCL_ERROR;
  }

  return xessStoreCell (interp, row, col, 0, interp -> result,
			(char * *) NULL);
}

/*
 * XessConnObj_goto_cmd --
 *
 *	Command on an Xess connection object to go to a specified cursor
 *	position, scrolling it to upper left.
 *
 * Syntax:
 *
 *	$connection goto position
 *
 * Result:
 *	A standard Tcl result
 *
 */

int
XessConnObj_goto_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;

  XessCell cell;
  int status;

  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1],
		      " ", argv [0], " cellname\"", (char *) NULL);
    return TCL_ERROR;
  }

  status = xessGetCell (interp, argv[1], &cell);

  if (status != TCL_OK)
    return TCL_ERROR;

  xess_goto (conn -> port, cell.row, cell.col);

  return TCL_OK;
}

/*
 * XessConnObj_moveto_cmd --
 *
 *	Command on an Xess connection object to move to a specified cursor
 *	position, without scrolling the sheet.
 *
 * Syntax:
 *
 *	$connection moveto position
 *
 * Result:
 *	A standard Tcl result
 *
 */

int
XessConnObj_moveto_cmd (clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp * interp;
     int argc;
     char * * argv;
{
  XessConnection * conn = (XessConnection *) clientData;

  XessCell cell;
  int status;

  if (argc != 2) {
    Tcl_AppendResult (interp, "wrong # args, should be \"", argv [-1],
		      " ", argv [0], " cellname\"", (char *) NULL);
    return TCL_ERROR;
  }

  status = xessGetCell (interp, argv [1], &cell);
  if (status != TCL_OK)
    return status;

  xess_moveto (conn -> port, cell.row, cell.col);

  return TCL_OK;
}
