#     $Id: cdisp.tcl,v 1.2 1994/11/14 17:05:36 johnsonm Exp $

#     Copyright (c) 1992 General Electric.  All rights reserved.

#     Permission to use, copy, modify, and distribute this software and
#     its documentation for any purpose and without fee is hereby
#     granted, provided that the above copyright notice appear in all
#     copies and that both that copyright notice and this permission
#     notice appear in supporting documentation, and that the name of
#     General Electric not be used in advertising or publicity
#     pertaining to distribution of the software without specific,
#     written prior permission.  General Electric makes no
#     representations about the suitability of this software for any
#     purpose.  It is provided "as is" without express or implied
#     warranty.

#     This work was supported by the DARPA Initiative in Concurrent
#     Engineering (DICE) through DARPA Contract MDA972-88-C-0047.

#     $Log: cdisp.tcl,v $
# Revision 1.2  1994/11/14  17:05:36  johnsonm
# Added space to "wrong # args" error message where one was needed.
#
# Revision 1.1  1994/08/26  16:25:09  johnsonm
# Initial revision
#
#

# cdisp.tcl

# This program pre-processes a C source file, handling several common
# Tcl conventions.

# %kwd name ?static?
# Produces a ?static? function called `name_cmd', that dispatches on a list of
# possible keys in argv[1].  The list of keys follows the %kwd
# directive, in the form:
#	key			- Call the function named `name_key_cmd'
#	key fname		- Call the function named `fname'
#	key %{
#		C code
#	%}			- Execut the specified C code
# The list of keys is terminated with %end.

# Things that give syntax problems

set open "%{"
set close "%}"
set eos "\\0"

# Error handling

set errors 0

# Make the default action for a command dispatch

proc make_action {func} {
	return "\treturn ${func}( clientData, interp, argc-1, argv+1 );\n"
}

# Get a user-specified action for a command
# (Just the lines from infile until a %} is read)

proc get_action {} {
	global infile
	global close
	global errors
	set action ""
	for {} {1} {} {
		set linelen [gets $infile line]
		if [eof $infile] {
			puts stderr "missing $close"
			incr errors
			break;
		}
		if {$line == $close} {
			break;
		}
		append action $line \n
	}
	return $action
}

# Add a command to the trie under construction

# trie(x) is the entry for state x.  It is a three-element list:
#	character to compare
#	trie entry number to go to on match after shifting input, if
#		character is non-null, or key number if character is null
#	trie entry number to go to on mismatch, or -1 if a mismatch is
#		an error.
# yesbranch(x) is 1 if trie entry x was reached by a true branch.  Used for
#	unique-prefix processing.

proc add_kwd_cmd {cmdName actionNum} {
	global trie
	global triesize
	global yesbranch
	global fname
	global errors

	if {$triesize == 0} {

		# First command in the trie

		set yesbranch(0) 1
		for {set i 0} {$i < [string length $cmdName]} {incr i} {
			set c [string index $cmdName $i]
			set trie($i) [list $c [expr $i+1] -1]
			set yesbranch([expr $i+1]) 1
		}
		set trie($i) [list "" $actionNum -1]
		set triesize [expr $i+1]
	} else {

		# Not the first command.  Walk through the trie until
		# a mismatch is found.

		set t 0
		for {set i 0} {$t != -1} {} {
			set c [string index $cmdName $i]
			set x [lindex $trie($t) 0]
			set oldt $t
			if {$x == $c} {
				if {$c == ""} {
					# No mismatch - this is a problem
					puts stderr "duplicate entry for \"$cmdName\" in \"$fname\""
					incr errors
					return
				}
				set t [lindex $trie($t) 1]
				incr i
			} else {
				set t [lindex $trie($t) 2]
			}
		}

		# Found a mismatch.  The trie entry at oldt has to be fixed
		# up so its `no' branch designates the entry at t.

		set t $triesize
		set trie($oldt) [lrange $trie($oldt) 0 1]
		lappend trie($oldt) $t
		set yesbranch($t) 0

		# Fill in the rest of the string

		for {} {$i < [string length $cmdName]} {incr i} {
			set c [string index $cmdName $i]
			set trie($t) [list $c [expr $t+1] -1]
			incr t
			set yesbranch($t) 1
		}
		set trie($t) [list "" $actionNum -1]
		set triesize [expr $t+1]
	}
}

# Process a command name within a %kwd directive

proc process_kwd_cmd {line} {
	global actions
	global actionsize
	global open
	global fname

	# Parse the line, and find the command action.

	set cmdName [lindex $line 0]
	if {[llength $line] > 1} {
		if {[lindex $line 1] == $open} {
			set action [get_action]
		} else {
			set action [make_action [lrange $line 1 end]]
		}
	} else {
		set action [make_action ${fname}_${cmdName}_cmd]
	}
	set actions($actionsize) [list $cmdName $action]

	# Add the command to the trie

	add_kwd_cmd $cmdName $actionsize

	# Bump the action count

	incr actionsize
}

# Do the unique prefix handling of a trie

proc unique_prefixes {} {
	global trie
	global triesize
	global yesbranch

	# What this does is to redirect `error' branches to `end of string'
	# comparisons if there is only one suffix that matches the command.
	# It goes through the table in reverse order.

	for {set i [expr $triesize-1]} {$i >= 0} {incr i -1} {
		set entry $trie($i)
		set c [lindex $entry 0]
		set yes [lindex $entry 1]
		set no [lindex $entry 2]

		if {$no == -1} {
			# If a table entry has an error on a mismatched 
			# character, then it's a candidate for having its
			# error branch redirected.

			if {$c == ""} {
				# Comparing against end of string.  Can't
				# fix this entry.  This entry should be the
				# error branch target, though, for any
				# state that has this end-of-string as its
				# unique destination.

				set uniq($i) $i
			} else {
				# Not comparing against end of string.  The
				# `yes' branch of this state is its unique
				# suffix if we got here from a `yes' branch.
				# Otherwise, this state has no unique suffix.
				# We don't care about the unique suffices of
				# states reached on `no' branches.

				if {$yesbranch($i)} {
					set uniq($i) $uniq($yes)
					set no $uniq($yes)
				} 
			}
		} else {
			# A table entry that doesn't get an error on a
			# mismatched character has no unique suffix.

			set uniq($i) -1
		}

		# Redirect the trie entry if necessary

		set trie($i) [list $c $yes $no]
	}
}

# Process the end of a list of comamnds to dispatch

proc process_kwd_end {} {
	global outfile
	global trie
	global triesize
	global actions
	global actionsize
	global eos

	# Do unique prefix handling

	unique_prefixes

	# Put out the trie

	puts $outfile "  static struct {"
	puts $outfile "    char c;"
	puts $outfile "    short yes;"
	puts $outfile "    short no;"
	puts $outfile "  } trie \[\] = {"

	for {set i 0} {$i < $triesize} {incr i} {
		set entry $trie($i)
		set c [lindex $entry 0]
		set yes [lindex $entry 1]
		set no [lindex $entry 2]
		if {$c == ""} {
			set c $eos
		}
		puts $outfile "    { '$c', $yes, $no }" nonewline
		if {$i < $triesize-1} {
			puts $outfile "," nonewline
		}
		puts $outfile "\t/* $i */"
	}
	puts $outfile "  };"

	# Put out the trie search

	puts $outfile "  if (argc < 2) {"
	puts $outfile "    Tcl_AppendResult (interp, \"wrong # args, should be \\\"\","
	puts $outfile "                      argv \[0\], \" command ?args?\\\"\","
	puts $outfile "                      (char *) NULL);"
	puts $outfile "    return TCL_ERROR;"
	puts $outfile "  }"

	puts $outfile "  arg = argv \[1\];"
	puts $outfile "  i = 0;"
	puts $outfile "  while (i >= 0) {"
	puts $outfile "    if (*arg == trie \[i\] . c) {"
	puts $outfile "      i = trie \[i\] . yes;"
	puts $outfile "      if (*arg == 0) break;"
	puts $outfile "      ++arg;"
	puts $outfile "    } else {"
	puts $outfile "      i = trie \[i\] . no;"
	puts $outfile "    }"
	puts $outfile "  }"

	# Put out the dispatching switch

	puts $outfile "  switch (i) {"
	for {set i 0} {$i < $actionsize} {incr i} {
		set entry $actions($i)
		set cname [lindex $entry 0]
		set act [lindex $entry 1]
		puts $outfile "    case $i:  /* $cname */"
		puts $outfile $act nonewline
	}
	puts $outfile "    default:"
	puts $outfile "\tTcl_AppendResult (interp, \"unknown command \\\"\","
	puts $outfile "\t                  argv \[0\], \" \", argv \[1\], \"\\\"\","
	puts $outfile "\t                  (char *) NULL);"
	puts $outfile "\treturn TCL_ERROR;"
	puts $outfile "  }"
}

# Process a %kwd directive

proc process_kwd {directive} {
	global trie
	global yesbranch
	global triesize
	global commands
	global actions
	global actionsize
	global infile
	global outfile
	global fname

	# Set the function name, and put out the function header

	set fname [lindex $directive 1]
	set line1 int
	if {[llength $directive] > 2 && [lindex $directive 2] == "static"} {
		set line1 "static int"
	}
	puts $outfile $line1
	puts $outfile "${fname}_cmd( clientData, interp, argc, argv )"
	puts $outfile "    ClientData clientData;"
	puts $outfile "    Tcl_Interp * interp;"
	puts $outfile "    int argc;"
	puts $outfile "    char * * argv;"
	puts $outfile "{"
	puts $outfile "  char *arg;"
	puts $outfile "  int i;"

	# Clear the trie and the keyword action tables

	catch {unset trie}
	catch {unset yesbranch}
	set triesize 0
	catch {unset commands}
	catch {unset actions}
	set actionsize 0

	# Process the keyword lines

	for {} {1} {} {
		set linelen [gets $infile line]
		if [eof $infile] {
			global errors
			puts stderr "missing %end in $fname"
			incr errors
			break;
		}
		if {$line == "%end"} {
			break;
		}
		process_kwd_cmd $line
	}

	# Put out the function body when %end is reached.

	process_kwd_end

	# Close the function definition.

	puts $outfile "}"
}

# Main program

# Open input and output files

if {[llength $argv] == 0} {
	set infile stdin
	set outfile stdout
} else {
	set infile [open [lindex $argv 0] r]
	if {[llength $argv] == 1} {
		set outfilename [file rootname [lindex $argv 0]].c
	} else {
		set outfilename [lindex $argv 1]
	}
	set outfile [open $outfilename w]
}

# Process the file

for {} {1} {} {
	set linelen [gets $infile line]
	if [eof $infile] {
		break
	}
	if {[string range $line 0 3] == "%kwd"} {
		process_kwd $line
	} else {
		puts $outfile $line
	}
}

exit $errors
