/*
 * TkTreeNode.cc - member routines for class TkTreeNode
 * 
 * Class TkTreeNode redefines virtual functions from its base class,
 * TreeNode, to allow drawing nodes on a tk canvas.
 * 
 * -----------------------------------------------------------------------------
 * Copyright 1994 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  
 * Allan Brighton make no representations about the suitability of this 
 * software for any purpose. It is provided "as is" without express or 
 * implied warranty.
 * -----------------------------------------------------------------------------
 */

#include <stdlib.h>
#include <string.h>
#include <strstream.h>
#include "TkTreeNode.h"
#include "TkTree.h"
#include "Point.h"


/*
 * construct a node with given tag, pos and size
 */
TkTreeNode::TkTreeNode(TkTree* tree, const char* tag, const char* line, 
    int x, int y, int w, int h, int border)
: TreeNode(tag, x, y, w, h, border), 
  tree_(tree), 
  line_(strdup(line)),
  removeCmd_(NULL)
{
}

/* 
 * destructor
 */
TkTreeNode::~TkTreeNode() {
    if (removeCmd_ != NULL)
	free(removeCmd_);
    free(line_);
}


/*
 * return distance from parent node
 */
int TkTreeNode::parentDistance() const
{
    return tree_->parentDistance();
}
    

/*
 * Return true if the tree's orientation is horizontal
 */
int TkTreeNode::horizontal() const 
{
    return tree_->horizontal();
}

    
/*
 * Draw the tree node using Tk 
 */
void TkTreeNode::drawNode() const
{
    if (tag() != NULL) 
        tree_->tcl() << tree_->canvas() << " move " 
	    << "{" << tag() << "}" << " " << (pos() - prev_pos()) << eval;
}


/*
 * Remove a tree node using Tk 
 */
void TkTreeNode::removeNode() const
{
    if (tag() != NULL)
        tree_->tcl() << tree_->canvas() << " delete {" << tag() << "}" << eval;
    
    if (removeCmd_ != NULL)
        tree_->tcl() << removeCmd_ << eval;
}


/*
 * Draw the line between this node and its parent 
 */
void TkTreeNode::drawLine(const TreePoint& p1, const TreePoint& p2) const
{
    // make sure the lines are close enough to the nodes in tk 
    // (I'm not sure why this is needed...)
    if (horizontal()) {
        tree_->tcl() << tree_->canvas() << " coords " << line_ << " " 
           << (p1.x-3) << " " << (p1.y) << " " << (p2.x-1) << " " << p2.y << eval;
    }
    else {
	tree_->tcl() << tree_->canvas() << " coords " << line_ << " " 
           << p1.x << " " << (p1.y-3) << " " << p2.x << " " << (p2.y-1) << eval;
    }
}


/*
 * Remove the line from this nodes parent  
 */
void TkTreeNode::removeLine() const
{
    tree_->tcl() << tree_->canvas() << " delete " << line_ << eval;
}


/*
 * This routine is called with the bounding box of the tree before drawing
 * We use this information to change the scrollregion of the tree's canvas
 */
void TkTreeNode::setDrawArea(int x1, int y1, int x2, int y2) const
{
    int b = tree_->borderWidth();	// border width around tree
    int p = tree_->parentDistance();	// distance to parent nodes

    if (tree_->horizontal())  
	x1 += p;
    else
	y1 += p;
    tree_->tcl() << tree_->canvas() << " config -scrollregion {" 
        << (x1-b) << " " << (y1-b) << " " << (x2+b) << " " << (y2+b) << "}" << eval;
}

void TkTreeNode::removeCmd(const char* s) {
    if (removeCmd_ != NULL)
	free(removeCmd_);
    removeCmd_ = strdup(s);
}
