/*
 * TclCommand.C - base class definitions for tcl command classes
 *
 * -----------------------------------------------------------------------------
 * Copyright 1994 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  
 * Allan Brighton make no representations about the suitability of this 
 * software for any purpose. It is provided "as is" without express or 
 * implied warranty.
 * -----------------------------------------------------------------------------
 *
 */


#include <stdlib.h>
#include <string.h>
#include <iostream.h>
#include "TclCommand.h"


// static member var
int TclCommand::seq_ = 0;


/*
 * Constructor - install the named tcl command in the interpreter
 * and initialize a hash table of tcl subcommands for this command.
 */
TclCommand::TclCommand(Tcl_Interp* interp, const char* cmdname, const char* instname)
: interp_(interp),
  tcl_(interp),
  status_(TCL_OK)
{
    // save the command name
    cmdname_ = strdup(cmdname);

    // generate a name if necessary
    if (strcmp(instname, "#auto") == 0) {
	instname_ = new char[strlen(cmdname_)+16];
	sprintf(instname_, "%s%d", cmdname_, seq_++);
    } else {
	instname_ = new char[strlen(instname)+1];
	strcpy(instname_, instname);
    }

    // create the basic tcl command and return its name
    Tcl_CreateCommand(interp, instname_, TclCommand::tclCmdProc, (ClientData)this, 
		      TclCommand::tclDeleteProc);

    // The result of the comamnd is its name
    // Note: if you access the tcl interpreter in a subclass constructor,
    // you will have to reset the result again before returning.
    Tcl_SetResult(interp, instname_, TCL_STATIC);
}


/* 
 * Dxestructor - called when tcl command is deleted
 */
TclCommand::~TclCommand() 
{
    free((char*)cmdname_);
    delete instname_; 
    instname_ = NULL;
}


/*
 * check the arg count for a subcommand and return an error in
 * Tcl if it is out of range
 */
int TclCommand::check_args(const char* name, int argc, int min_args, int max_args)
{
    if (argc >= min_args && argc <= max_args) 
	return TCL_OK;
    Tcl_AppendResult(interp_, "wrong number of args, should be \"",
		     instname_, " ", name, " ?args?\"", NULL);
    return TCL_ERROR;
}


/*
 * Call the given method in this class with the given arguments
 * (in this case there is only one method defined: "delete"
 */
int TclCommand::call(const char* name, int argc, char* argv[])
{
    if (strcmp(name, "delete") == 0) {
	return delete_(argc, argv);
    }
    Tcl_AppendResult(interp_, "unknown ", cmdname_, 
		     " subcommand: \"", name, "\"", NULL);
    return TCL_ERROR;
}


/*
 * This method is called for operations on the tcl objects created above
 * - just call a member function to be defined in the subclass
 */
int TclCommand::tclCmdProc(ClientData thisPtr, Tcl_Interp* interp, int argc, char* argv[])
{
    // saved this ptr for command
    TclCommand* tclcmd = (TclCommand*)thisPtr;

    // must be at least "cmd subcmd"
    if (argc >= 2) {
	tclcmd->tcl_.reset();
	return tclcmd->call(argv[1], argc-2, argv+2);
    }

    // error: arg count wrong
    Tcl_AppendResult(interp, "wrong number of args, should be \"",
		     argv[0], " ", argv[1], " ?args?\"", NULL);
    return TCL_ERROR;
}


/*
 * This method is called when a tcl object is deleted.
 * Delete the C++ object passed as client data.
 */
void TclCommand::tclDeleteProc(ClientData thisPtr)
{
    // pointer to current object
    TclCommand* tclcmd = (TclCommand*)thisPtr;
    delete tclcmd;
}


/*
 * delete subcommand - Remove the Tcl command from the interpreter
 * (the C++ object is deleted when the tclDeleteProc is called)
 */
int TclCommand::delete_(int, char**) {
    return Tcl_DeleteCommand(interp_, instname_);
}


/* 
 * append an integer result in tcl
 */
int TclCommand::append_result(int i) 
{
    char buf[32];
    sprintf(buf, "%d", i);
    Tcl_AppendResult(interp_, buf, NULL);
    return TCL_OK;
}


/* 
 * append a string result in tcl
 */
int TclCommand::append_result(char* s) 
{
    Tcl_AppendResult(interp_, s, NULL);
    return TCL_OK;
}



/* 
 * return an integer value in tcl
 */
int TclCommand::set_result(int i) 
{
    char buf[32];
    sprintf(buf, "%d", i);
    Tcl_SetResult(interp_, buf, TCL_VOLATILE);
    return TCL_OK;
}


/* 
 * return a string value in tcl
 */
int TclCommand::set_result(char* s) 
{
    Tcl_SetResult(interp_, s, TCL_VOLATILE);
    return TCL_OK;
}


/* 
 * report an error message in tcl
 */
int TclCommand::error(char* s) 
{
    Tcl_SetResult(interp_, s, TCL_VOLATILE);
    return TCL_ERROR;
}
