
# atclose -- command to install a Tcl callback to be invoked when
#	  -- the close command is evalutated.
#
#   close -- command to close process, after all callbacks installed by
#	  -- the atclose command have been invoked.
#

#######################################################################
#
# atclose -- manages atclose callbacks.
#

proc atclose {fileId {option list} args} {

  # The option may be append, prepend, insert, delete, clear, set, or list.
  # The args depends on the option specified.
  #

  # The atclose_callbacks array holds the installed atclose callbacks,
  # indexed by fileId.
  #
  global atclose_callbacks;
  if {[catch {set atclose_callbacks($fileId)}]} {
    set atclose_callbacks($fileId) {};
  }

  case $option in {
    set {
      #
      # set callbacks list.
      #
      set atclose_callbacks($fileId) $args;
    }
    append {
      #
      # append callback to end of the callbacks list.
      #
      if {[llength $args] != 1} {
	error {wrong # args: try "atclose fileId append callback"};
      }
      set callback [lindex $args 0];
      lappend atclose_callbacks($fileId) $callback;
    }
    prepend {
      #
      # prepend callback to front of the callbacks list.
      #
      if {[llength $args] != 1} {
	error {wrong # args: try "atclose fileId prepend callback"};
      }
      set callback [lindex $args 0];
      set atclose_callbacks($fileId) \
	"\{$callback\} $atclose_callbacks($fileId)";
    }
    insert {
      #
      # insert callback before the "before" callback in the callbacks list.
      #
      if {[llength $args] != 2} {
	error {wrong # args: try "atclose fileId insert before callback"};
      }
      set before   [lindex $args 0];
      set callback [lindex $args 1];
      set l {};
      foreach c $atclose_callbacks($fileId) {
	if {[string compare $before $c] == 0} {
	  lappend l $callback;
	}
	lappend l $c;
      }
      set atclose_callbacks($fileId) $l;
    }
    delete {
      #
      # delete callback from the callbacks list.
      #
      if {[llength $args] != 1} {
	error {wrong # args : should be "atclose fileId delete callback"};
      }
      set callback [lindex $args 0];
      set l {};
      foreach c $atclose_callbacks($fileId) {
	if {[string compare $callback $c] != 0} {
	  lappend l $c;
	}
      }
      set atclose_callbacks($fileId) $l;
    }
    clear {
      #
      # clear callbacks list.
      #
      if {[llength $args] != 0} {
	error {wrong # args : should be "atclose fileId clear"};
      }
      set atclose_callbacks($fileId) {};
    }
    list {
      #
      # list currently installed callbacks.
      #
    }
    default {
      error {options: append, prepend, insert, delete, clear, set, or list};
    }
  }
  return $atclose_callbacks($fileId);
}

#######################################################################
#
# Hide real close command.
#

rename close atclose_close;

#######################################################################
#
# close -- Wrapper close command that first invokes all callbacks installed
#       -- by the atclose command before doing real close.
#

proc close {fileId} {
  global atclose_callbacks;

  while {1} {

    # Every iteration, we rescan atclose_callbacks, in case
    # some callback modifies it.
    #
    if {[catch {set atclose_callbacks($fileId)} callbacks]} {
      break;
    }
    if {[llength $callbacks] <= 0} {
      break;
    }
    set         callback           [lindex $callbacks 0];
    set atclose_callbacks($fileId) [lrange $callbacks 1 end];

    catch {uplevel #0 $callback};
  }

  catch {unset atclose_callbacks($fileId)};
  catch {atclose_close $fileId};
}

