/* $Id: interp.c,v 1.4 1993/06/08 06:11:03 david Exp $
 *
 * interp.c -- TCL Commands to create/delete/manipulate interpretors.
 *
 * AUTHOR:	David Herron <david@davids.mmdf.com> (home)
 *
 *
 * Copyright 1993 David Herron.
 * Permission to use, copy, modify and distribute this software
 * and its documentation for any purpose and without fee is
 * hereby granted, provided that the above copyright notice
 * appear in all copies.  David Herron makes no representations
 * about the suitability of this software for any purpose save
 * printing it out and using the paper as bird cage lining.
 *
 * INTRO:
 *
 * In one of the Usenix papers delivered by Dr. Ousterhout is a
 * statement to the effect that "Interpretors are light weight
 * things and programmers should feel free to make as many
 * as s/he likes".  Unfortunately there was no facilities in the
 * TCL language for manipulating interpretors.  Instead the attributes
 * of interpretors are not described very well, and the only
 * tools to manipulate them are C functions in the library.
 *
 * This module is a first attempt at putting an interpretor facility
 * into TCL.  It keeps a hash table of of {interp,name,destroyHook}
 * tuples and provides a few TCL commands for manipulating them.
 *
 * Since each interpretor has its own namespace this is useful for
 *
 *	Creating `modules' with an pseudo-exported interface.
 *
 *	Separate namespaces avoiding problems with name collisions.
 *
 *	Provides an easy unit of modularity.
 *
 * The initial inspiration for creating this module was the
 * multiple Tk applications in one core image.  It was thought
 * up during a brainstorming session on a potential port of
 * TCL/Tk to MS-DOS using X/DOS.  X/DOS creates a DOS program
 * which makes the VGA screen look like an X session, but the
 * X session only lasts for the existance of that one program.
 *
 * So the thought was to have a Tk script be a sort of application
 * launcher.  It would need to be able to create new interpretors
 * so that each launched application did not interfere with others
 * and had `.' as its main window.  etc.  None of this has been
 * done since other facets of the project were so interesting.
 *
 * Facilities are also present for doing something like object
 * oriented programming.  Above we have discussed abstraction,
 * encapsulation and modularity.  Another facet of OOP we can
 * do is hierarchy creation.
 *
 * $Log: interp.c,v $
 * Revision 1.4  1993/06/08  06:11:03  david
 * Remove `minterp_library' and `thisInterpretor' and add [interp library]
 * and [interp this].
 *
 * Revision 1.3  1993/06/06  06:31:54  david
 * Add minterp_library variable.
 *
 * Revision 1.2  1993/06/04  06:12:09  david
 * Mark everything to be at release 0.8.
 *
 * Revision 1.1.1.1  1993/06/03  06:23:46  david
 * Initial revision of `minterp' as its own package.
 *
 * Revision 1.6  1993/06/01  05:09:15  david
 * Massive cleanup and rearrangement.  Removed the policy decisions which had
 * been made in C code before, and moved them to interpBaseC.tcl.  Left behind
 * the necessary underlying functionality.  Created ability to prevent -exec
 * from outside the interpretor.  Etc.
 *
 * Revision 1.5  1993/04/14  07:17:44  david
 * Massive changes in latest round of changing.  Am moving many
 * things into TCL code to allow greatest flexibility, while leaving
 * behind lower level operations as C functions.  The changes are not
 * finished nor finalized, but they are in a Working State.
 *...
 * Revision 1.2  1993/02/02  04:28:04  david
 * Verified the needed functionality was there.  Removed old code
 * and a general cleanup.  Changed `-chainCommand' to just `-chain'.
 * Likewise `-setParent' became `-parent'.
 *
 * Revision 1.1  1992/10/24  20:11:43  david
 * Initial revision.
 *
 *
 */

#include <stdlib.h>
#include <memory.h>	/* Pick up memset() and friends */
#include <tclInt.h>
/*
#include <tclHash.h>
*/
#include "interp.h"

static int cmdInterp	     _ANSI_ARGS_((ClientData,Tcl_Interp*,int,char **));
static int cmdExitInterpCMD  _ANSI_ARGS_((ClientData,Tcl_Interp*,int,char **));
static int cmdDoExecHere     _ANSI_ARGS_((ClientData,Tcl_Interp*,int,char **));
static int cmdDoMethod	     _ANSI_ARGS_((ClientData,Tcl_Interp*,int,char **));
static int cmdMakeMethod     _ANSI_ARGS_((ClientData,Tcl_Interp*,int,char **));
static int cmdDestroyHookCMD _ANSI_ARGS_((ClientData,Tcl_Interp*,int,char **));
static int cmdAllowExec      _ANSI_ARGS_((ClientData,Tcl_Interp*,int,char **));
static int cmdInterp         _ANSI_ARGS_((ClientData,Tcl_Interp*,int,char **));

static Tcl_HashTable iList;
static short iListInited = (1==0);

static char *createHookText = (char *)NULL;

static struct interpInfo main_interp_info;

static char *nm_MainInterp = "MainInterp";
extern char *strdup();

void init_interp(interp)
Tcl_Interp *interp;
{
	Tcl_CreateCommand(interp, "interp", cmdInterp, (ClientData) NULL,
            (Tcl_CmdDeleteProc *) NULL);
	if (!iListInited) {
		memset(&main_interp_info, 0, sizeof(main_interp_info));
		main_interp_info.name = strdup(nm_MainInterp);
		Tcl_InitHashTable(&iList, TCL_STRING_KEYS);
		Tcl_InitHashTable(&(main_interp_info.methods),TCL_STRING_KEYS);
		iListInited = (1==1);
	}
}

/*
 * Give an interpretor a complete set of the commands necessary
 * to run this module.
 */
static void add_our_commands(info)
struct interpInfo *info;
{
	Tcl_HashEntry *hp;
	Tcl_HashSearch hs;
	struct interpInfo *iinfo;

/*printf("add_our_commands: '%s'\n", info->name);*/

	/*
	 * Add the name of this new interp to every other interp's commands.
	 * Also add the name of those interp's to this interp's commands.
	 */
	for (hp = Tcl_FirstHashEntry(&iList, &hs);
	     hp;
	     hp = Tcl_NextHashEntry(&hs)) {
		iinfo = (struct interpInfo *)Tcl_GetHashValue(hp);
		if (!iinfo)
			continue;
/*
 * printf("Adding '%s' to '%s'\n", info->name, iinfo->name);
 * fflush(stdout);
 */
		Tcl_CreateCommand(iinfo->interp, info->name, cmdDoMethod,
				 (ClientData) info,
			         (Tcl_CmdDeleteProc *) NULL);
/*
 * printf("Adding '%s' to '%s'\n", iinfo->name, info->name);
 * fflush(stdout);
 */
		Tcl_CreateCommand(info->interp, iinfo->name, cmdDoMethod,
				 (ClientData) iinfo,
			         (Tcl_CmdDeleteProc *) NULL);
	}

	Tcl_CreateCommand(info->interp, "interp", cmdInterp,
			 (ClientData) info,
		         (Tcl_CmdDeleteProc *) NULL);
	Tcl_CreateCommand(info->interp, nm_MainInterp, cmdDoMethod,
			 (ClientData) &main_interp_info,
		         (Tcl_CmdDeleteProc *) NULL);

	if (main_interp_info.interp) {
		Tcl_CreateCommand(main_interp_info.interp, info->name, cmdDoMethod,
				 (ClientData) info,
			         (Tcl_CmdDeleteProc *) NULL);
	}

	Tcl_CreateCommand(info->interp, "-AddMethod", cmdMakeMethod,
			 (ClientData) info,
		         (Tcl_CmdDeleteProc *) NULL);

	Tcl_CreateCommand(info->interp, "-DelMethod", cmdMakeMethod,
			 (ClientData) info,
		         (Tcl_CmdDeleteProc *) NULL);

	Tcl_CreateCommand(info->interp, "-IsMethod", cmdMakeMethod,
			 (ClientData) info,
		         (Tcl_CmdDeleteProc *) NULL);

	Tcl_CreateCommand(info->interp, "-Methods", cmdMakeMethod,
			 (ClientData) info,
		         (Tcl_CmdDeleteProc *) NULL);

	Int_AddMethod(info, "-Methods");
	Int_AddMethod(info, "-IsMethod");
	Int_AddMethod(info, "-exec");

	Tcl_CreateCommand(info->interp, "-execHere", cmdDoExecHere,
			 (ClientData) info,
		         (Tcl_CmdDeleteProc *) NULL);

	Tcl_CreateCommand(info->interp, "-AllowExec", cmdAllowExec,
			 (ClientData) info,
		         (Tcl_CmdDeleteProc *) NULL);

	/*
	 * These are not put into MainInterp.
	 */
	if (info != &main_interp_info && info->interp != main_interp_info.interp) {
		Tcl_CreateCommand(info->interp, "-exit", cmdExitInterpCMD,
			 (ClientData) info,
		         (Tcl_CmdDeleteProc *) NULL);
		Tcl_CreateCommand(info->interp, "-destroyHook", cmdDestroyHookCMD,
			 (ClientData) info,
		         (Tcl_CmdDeleteProc *) NULL);
	}
}

/* 
**			C Programmers Interface (API)
**
**/



int Int_MainInterp(interp)
Tcl_Interp *interp;
{
	main_interp_info.interp = interp;
	main_interp_info.allowExec = 1;
	add_our_commands(&main_interp_info);
	return TCL_OK;
}

/*
 * Int_CreateInterp() -- Create one of our special interpreters.
 */
struct interpInfo *Int_CreateInterp(interp, new_interp, name)
Tcl_Interp *interp, *new_interp;
char *name;
{
	int new;
	Tcl_HashEntry *hPtr;
	struct interpInfo *info;
	char *c;

	hPtr = Tcl_CreateHashEntry(&iList, name, &new);
	if (!new) {
		Tcl_AppendResult(interp, "An interpretor named ",
					  name, " already exists.", NULL);
		return (struct interpInfo *)NULL;
	}

	if (!main_interp_info.interp) {
		Tcl_AppendResult(interp, "No MainInterp.", NULL);
		Tcl_DeleteHashEntry(hPtr);
		return (struct interpInfo *)NULL;
	}

	if (!name || !name[0]) {
		Tcl_AppendResult(interp, "No interpretor name given!", NULL);
		Tcl_DeleteHashEntry(hPtr);
		return (struct interpInfo *)NULL;
	}

	for (c = name; c && c[0] && isascii(c[0]); c++)
		;
	if (c && c[0] && !isalnum(c[0])) {
		Tcl_AppendResult(interp, "Bad interpretor name: ", name, NULL);
		Tcl_DeleteHashEntry(hPtr);
		return (struct interpInfo *)NULL;
	}

	if (!new_interp)
		new_interp = Tcl_CreateInterp();
	if (!new_interp) {
		Tcl_AppendResult(interp, "Could not create interpretor ", name, NULL);
		Tcl_DeleteHashEntry(hPtr);
		return (struct interpInfo *)NULL;
	}

	info = (struct interpInfo *)malloc(sizeof(*info));
	memset(info, 0, sizeof(*info));
	Tcl_InitHashTable(&(info->methods), TCL_STRING_KEYS);
	Tcl_SetHashValue(hPtr, info);
	info->interp = new_interp;
	info->name = strdup(name);
	info->allowExec = 1;

	add_our_commands(info);

	/*
	 * The intention of the createHook is so the
	 * new interpreters can be customized.
	 */
	if (isstr(createHookText)) {
		Tcl_GlobalEval(info->interp, createHookText);
	}

	return info;
}

/*
 * Int_findInterp() -- Find the named interpretor info structure if
 *	it exists.
 *
 * HACK: We recognize `MainInterp' as a special name and fake up
 *	 an info structure for it.
 */
struct interpInfo *Int_findInterp(name)
char *name;
{
	Tcl_HashEntry *hPtr;
	struct interpInfo *info;

	if (!isstr(name))
		return (struct interpInfo *)NULL;

	if (strcmp(name, nm_MainInterp) == 0 && main_interp_info.interp) {
		return &main_interp_info;
	}

	hPtr = Tcl_FindHashEntry(&iList, name);
	if (!hPtr)
		return (struct interpInfo *)NULL;
	info = (struct interpInfo *)Tcl_GetHashValue(hPtr);
	return info;
}

/*
 * Int_whoIs() -- Find the name for a particular interpretor.
 */
struct interpInfo *Int_whoIs(interp)
Tcl_Interp *interp;
{
	Tcl_HashEntry *hPtr;
	Tcl_HashSearch hs;
	struct interpInfo *info;

	if (interp == main_interp_info.interp)
		return &main_interp_info;
	for (hPtr = Tcl_FirstHashEntry(&iList, &hs);
	     hPtr;
	     hPtr = Tcl_NextHashEntry(&hs)) {
		info = (struct interpInfo *)Tcl_GetHashValue(hPtr);
		if (info->interp == interp)
			return info;
	}
	return (struct interpInfo *)NULL;
}

/*
 * Int_delInterp() -- Clean up after an interpretor.
 */
int Int_delInterp(interp, info)
Tcl_Interp *interp;
struct interpInfo *info;
{
	Tcl_HashEntry *hPtr;
	Tcl_HashEntry *hp;
	Tcl_HashSearch hs;
	struct interpInfo *iinfo;


	if (isstr(info->destroyHook)) {
		int res;
		res = Tcl_GlobalEval(info->interp,info->destroyHook);
		if (res	!= TCL_OK) {
			Int_copyErrorInfo(info, interp);
			Tcl_AppendResult(interp,
				"\nERROR: Not allowed to delete interpretor ",
				info->name,
				NULL);
			return res;
		}
		free(info->destroyHook);
	}

	hPtr = Tcl_FindHashEntry(&iList, info->name);
	for (hp = Tcl_FirstHashEntry(&iList, &hs);
	     hp;
	     hp = Tcl_NextHashEntry(&hs)) {
		iinfo = (struct interpInfo *)Tcl_GetHashValue(hp);
		Tcl_DeleteCommand(iinfo->interp, info->name);
	}
	if (main_interp_info.interp)
		Tcl_DeleteCommand(main_interp_info.interp, info->name);

	Tcl_DeleteHashTable(&(info->methods));
	Tcl_DeleteInterp(info->interp);
	memset(info, 0, sizeof(*info));	/* Mark it so people can tell it's gone */
	free(info);
	Tcl_DeleteHashEntry(hPtr);

	return TCL_OK;
}
/*
 *----------------------------------------------------------------------
 *
 * TclFindCmd --
 *
 *      Given the name of a command, return a pointer to the
 *      structure describing the command.
 *
 * Results:
 *      NULL is returned if the name doesn't correspond to any
 *      command.  Otherwise the return value is a pointer to
 *      the command's structure.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
Command *
TclFindCmd(interp, cmdName)
    Tcl_Interp *interp;         /* Interpreter in which to look. */
    char *cmdName;              /* Name of desired command. */
{
    Interp *iPtr = (Interp*)interp;
    Tcl_HashEntry *entryPtr;
    Command *cmdPtr;

    entryPtr = Tcl_FindHashEntry(&iPtr->commandTable, cmdName);
    if (entryPtr == NULL) {
        return (NULL);
    }
    cmdPtr = (Command *) Tcl_GetHashValue(entryPtr);
    return (cmdPtr);
}

/*
 * Int_ExecCommand() -- Execute a command (already parsed into argc/argv)
 *	in another interpretor.  If we cannot find the command over there
 *	attempt to execute `unknown' over there.
 *
 * HEAVILY derived from Tcl_Eval(), but we cannot call trace procedures.
 */
int Int_ExecCommand(this, other, argc, argv)
Tcl_Interp *this, *other;
int   argc;
char *argv[];
{
	int result = TCL_OK;                         /* Return value. */
	register Interp *iPtr = (Interp *) this;
	Command *cmdPtr;
	char **new_argv = (char **)NULL;
	int    new_argc = -1;

	if (!other) {
		Tcl_AppendResult(this, "ERROR: Int_ExecCommand() called with NULL `other' interpretor.", NULL);
		return TCL_ERROR;
	}
	/*
	 * This is taken from tclBasic.c:Tcl_Eval().
	 *//*
         * Find the procedure to execute this command.  If there isn't
         * one, then see if there is a command "unknown".  If so,
         * invoke it instead, passing it the words of the original
         * command as arguments.
         */
	cmdPtr = TclFindCmd(other, argv[0]);
	if (cmdPtr == (Command *)NULL) {
		int i;
		cmdPtr = TclFindCmd(this, "unknown");
		if (cmdPtr == (Command *)NULL) {
			Tcl_ResetResult(this);
			Tcl_AppendResult(this, "ERROR: unknown command name: ", argv[0], (char *) NULL);
			result = TCL_ERROR;
			goto done;
		}
		new_argc = argc + 1;
		new_argv = (char **)malloc((new_argc+1) * sizeof(char *));
		new_argv[0] = "unknown";
		for (i = 0; i < new_argc; i++) {
			new_argv[i+1] = argv[i];
		}
        }
	/*
	 * Call trace procedures, if any.
	 *//*
	 * ... We can't do trace procedures since the original
	 * ... command line isn't available.
	 */
 
        /*
         * At long last, invoke the command procedure.  Reset the
         * result to its default empty value first (it could have
         * gotten changed by earlier commands in the same command
         * string).
         */
        iPtr->cmdCount++;
        Tcl_FreeResult(iPtr);
        iPtr->result = iPtr->resultSpace;
        iPtr->resultSpace[0] = 0;
        result = (*(cmdPtr->proc)) ((ClientData)(cmdPtr->clientData), this,
					new_argv ? new_argc : argc,
					new_argv ? new_argv : argv);

done:
	if (new_argv) free(new_argv);
	return result;
}

void Int_copyErrorInfo(src, dest)
struct interpInfo *src;
Tcl_Interp *dest;
{
	char *errorInfo;
	char *errorCode;

	if (src->interp == dest)
		return;

	Tcl_SetResult(dest, src->interp->result, TCL_STATIC);
	errorCode = Tcl_GetVar(src->interp, "errorCode", TCL_GLOBAL_ONLY);
	errorInfo = Tcl_GetVar(src->interp, "errorInfo", TCL_GLOBAL_ONLY);

	if (errorInfo) {
		Tcl_AddErrorInfo(dest, "\n    ***** Preceeding is copied from interpretor ");
		Tcl_AddErrorInfo(dest, src->name);
		Tcl_AddErrorInfo(dest, " *****");
	}
	if (errorCode)
		Tcl_SetVar(dest, "errorCode", errorCode, TCL_GLOBAL_ONLY);
}

int Int_AddMethod(info, name)
struct interpInfo *info;
char *name;
{
	int new;
	Tcl_HashEntry *hPtr;

	hPtr = Tcl_CreateHashEntry(&(info->methods), name, &new);
/*
 * printf("AddMethod: %s, name: %s, new: %d, is: %d, hp: 0x%x, tbl: 0x%x\n",
 *  info->name, name, new, Int_IsMethod(info, name), hPtr, &(info->methods));
 */
	if (!new) {
		Tcl_AppendResult(info->interp, "Interpretor ", info->name,
			" already has a method named ", name, NULL);
		return TCL_ERROR;
	}
	Tcl_SetHashValue(hPtr, (ClientData)NULL);

	return TCL_OK;
}

int Int_DelMethod(info, name)
struct interpInfo *info;
char *name;
{
	Tcl_HashEntry *hPtr;

	hPtr = Tcl_FindHashEntry(&(info->methods), info->name);
	Tcl_DeleteHashEntry(hPtr);
	return TCL_OK;
}

int Int_IsMethod(info, name)
struct interpInfo *info;
char *name;
{
	Tcl_HashEntry *hPtr;

	hPtr = Tcl_FindHashEntry(&(info->methods), name);
	return hPtr != (Tcl_HashEntry *)NULL;
}


/* 
 * 			Command handling procedures.
 */


static int cmdExitInterpCMD(clientData, interp, argc, argv)
ClientData clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
	Tcl_DeleteCommand(interp, ((struct interpInfo *)clientData)->name);
	return Int_delInterp(interp, (struct interpInfo *)clientData);
}


static int cmdMakeMethod(clientData, interp, argc, argv)
ClientData clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
	struct interpInfo *info = (struct interpInfo *)clientData;


	if (!isstr(argv[0]))
		goto usage;
	if (argv[0][1] != 'M' && !isstr(argv[1]))
		goto usage;

	/*
	 * -AddMethod name
	 */
	if (argv[0][1] == 'A' && strcmp(argv[0], "-AddMethod") == 0) {
		return Int_AddMethod(info, argv[1]);
	}
	/*
	 * -DelMethod name
	 */
	else if (argv[0][1] == 'D' && strcmp(argv[0], "-DelMethod") == 0) {
		return Int_DelMethod(info, argv[1]);
	}
	/*
	 * -IsMethod name
	 */
	else if (argv[0][1] == 'I' && strcmp(argv[0], "-IsMethod") == 0) {
		sprintf(interp->result, "%d",
			Int_IsMethod(info, argv[1]) ? 1 : 0);
		return TCL_OK;
	}
	/*
	 * -Methods
	 */
	else if (argv[0][1] == 'M' && strcmp(argv[0], "-Methods") == 0) {
		Tcl_HashEntry *hp;
		Tcl_HashSearch hs;
	
		for (hp = Tcl_FirstHashEntry(&(info->methods), &hs);
		     hp;
		     hp = Tcl_NextHashEntry(&hs)) {
			char *key = Tcl_GetHashKey(&(info->methods), hp);
			Tcl_AppendElement(interp, key, 0);
		}
		return TCL_OK;
	}

usage:
	Tcl_AppendResult(interp, "USAGE: -AddMethod name\n", 
				 "USAGE: -DelMethod name\n",
				 "USAGE: -IsMethod name\n",
				 "USAGE: -Methods\n", NULL);
	return TCL_ERROR;
}

/*
 * -execHere destInterp method [ arg ... ]
 */
static int cmdDoExecHere(clientData, interp, argc, argv)
ClientData clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
	struct interpInfo *info = (struct interpInfo *)clientData, *dest;

	if (argc < 3) {
		Tcl_AppendResult(interp,
			"USAGE: ", argv[0], " destination_interp method_name [ arg ... ]",
			NULL);
		return TCL_ERROR;
	}

	if (info != Int_whoIs(interp)) {
		Tcl_AppendResult(interp,
			"Passed interp pointer does not make sense.", NULL);
		return TCL_ERROR;
	}

	if (strcmp(argv[0], "-execHere") == 0) {
		dest = Int_findInterp(argv[1]);
		if (!dest) {
			Tcl_AppendResult(interp,
				"-execHere: Cannot find interp '", argv[1],
				"' from interp '", info->name, "'.", NULL);
			return TCL_ERROR;
		}
		return Int_ExecCommand(info->interp, dest->interp, argc-2,&(argv[2]));
	}
	else {
		Tcl_AppendResult(interp, "Unknown command ", argv[0], NULL);
		return TCL_ERROR;
	}

	/*NOTREACHED*/
}

/*
 * -AllowExec true|false|0|1
 */
static int cmdAllowExec(clientData, interp, argc, argv)
ClientData clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
	struct interpInfo *info = (struct interpInfo *)clientData;
	int r, res;

	if (argc < 1 || argc > 2) {
		Tcl_AppendResult(interp, "USAGE: -AllowExec ?true|false|0|1?",
					 NULL);
		return TCL_ERROR;
	}

	if (argc == 2) {
		res = Tcl_GetBoolean(interp, argv[1], &r);
		if (res == TCL_OK) {
			info->allowExec = r;
			goto set_result;
		}
	}
	else {
	set_result:
		sprintf(interp->result, "%d", info->allowExec);
		res = TCL_OK;
	}

	return res;
}


/*
 *----------------------------------------------------------------------
 *
 * cmdDoMethod --
 *
 *	Send commands to another interp.  There are two forms to this
 *	command:
 *
 * interp-name method-name arg ...
 *
 *	Execute a command whose name is registered in the `method list'.  The
 *	methods are commands executable from outside of the interp.  See -AddMethod
 *	and such for manipulating the method list.
 *
 * interp-name -exec command-string
 *
 *	Execute an arbitrary command line.  This is only allowed if interp->allowExec
 *	is set to true (manipulated with -AllowExec command).
 *
 * NOTE:
 *	It would be nice if TCL internals offered a Tcl_xxxEval()
 *	which took an argv/argc.  That way we could avoid some
 *	difficulties in Int_ExecCommand().
 *
 * Results:
 *      A standard Tcl return result.  An error message or other
 *      result may be left in interp->result.
 *
 * Side effects:
 *      Depends on what was done by the command.
 *
 * The command line looks like:
 *
 *	interp-name method-name arg ...
 * or	interp-name -exec command-string
 *
 */
static int cmdDoMethod(clientData, interp, argc, argv)
ClientData clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
	struct interpInfo *info = (struct interpInfo *)clientData;
	/* char *cmd; */
	int result;

/*printf("cmdDoMethod: interp: '%s', arg0: '%s'\n", info->name, argv[0]);*/

	if (!argv[1] || !argv[1][0]) {
usage:
		Tcl_AppendResult(interp, "USAGE: interp-name method-name args", NULL);
		if (info->allowExec)
			Tcl_AppendResult(interp, "USAGE: interp-name -exec commnd-line", NULL);
		return TCL_ERROR;
	}

	if (strcmp(argv[1], "-exec") == 0) {
		if (argc != 3)
			goto usage;
/*
 * printf("do -exec: interp: %s, cmd: %s\n", info->name, argv[2]);
 * fflush(stdout);
 */
		if (info->allowExec) {
			result = Tcl_Eval(info->interp, argv[2],
						0, (char **)NULL);
			goto copy_result;
		}
		else {
			Tcl_AppendResult(interp,
	"ERROR: Not currently allowed to execute commands (with -exec) in interp ",
				info->name, NULL);
			return TCL_ERROR;
		}
	}
	else {
		if (!Int_IsMethod(info, argv[1])) {
			Tcl_AppendResult(interp,
				"ERROR: ", argv[1],
				" is not a method of interpretor ",
				info->name, NULL);
			return TCL_ERROR;
		}

		result = Int_ExecCommand(info->interp, info->interp, argc-1, argv+1);

		/*
		 * If the the interpretor was deleted during that command, then
		 * info->interp will be NULL.
		 */
	copy_result:
		if (info->interp) {
			if (result != TCL_OK) {
				Int_copyErrorInfo(info, interp);
			}
			else {
				Tcl_SetResult(interp,info->interp->result,TCL_STATIC);
			}
		}

		return result;
	}
}

/*
 * USAGE:	-destroyHook ?command?
 *
 * Manipulates the command string executed just before
 * destruction of the interpreter.
 */
static int cmdDestroyHookCMD(clientData, interp, argc, argv)
ClientData clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
	struct interpInfo *info = (struct interpInfo *)clientData;

	if (isstr(info->destroyHook)) free(info->destroyHook);
	info->destroyHook = (char *)NULL;
	if (isstr(argv[1])) info->destroyHook = strdup(argv[1]);

	return TCL_OK;
}

static int cmdInterp(clientData, interp, argc, argv)
ClientData clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
	Tcl_HashEntry *hPtr;
	struct interpInfo *info;



	Tcl_ResetResult(interp);

	if (argc < 2) {
		Tcl_AppendResult(interp, "USAGE: interp new|MainInterp|exists|list|createHook",
			NULL);
		return TCL_ERROR;
	}

	/*
	 * interp new name
	 */
	if (argv[1][0] == 'n' && strcmp(argv[1], "new") == 0) {
		info = Int_CreateInterp(interp, (Tcl_Interp *)NULL, argv[2]);
		if (!info) {
			Tcl_AddErrorInfo(interp, "   While executing 'interp new'");
			return TCL_ERROR;
		}
		else return TCL_OK;
	}

	/*
	 * interp MainInterp
	 */
	if (argv[1][0] == nm_MainInterp[0] && strcmp(argv[1], nm_MainInterp) == 0) {
		return Int_MainInterp(interp);
	}

	/*
	 * interp exists name
	 */
	else if (argv[1][0] == 'e' && strcmp(argv[1], "exists") == 0) {
		if (!isstr(argv[2])) goto no_interp_name_given;
		sprintf(interp->result, "%d",
			Int_findInterp(argv[2]) ? 1 : 0);
		return TCL_OK;
	}

	/*
	 * interp list
	 */
	else if (argv[1][0] == 'l' && strcmp(argv[1], "list") == 0) {
		Tcl_HashSearch hs;
		for (hPtr = Tcl_FirstHashEntry(&iList, &hs);
		     hPtr;
		     hPtr = Tcl_NextHashEntry(&hs)) {
			Tcl_AppendElement(interp, Tcl_GetHashKey(&iList, hPtr), 0);
		}
		if (main_interp_info.interp)
			Tcl_AppendElement(interp, "MainInterp", 0);
		return TCL_OK;
	}

	/*
	 * interp createHook ?command?
	 */
	else if (argv[1][0] == 'c' && strcmp(argv[1], "createHook") == 0) {
		if (createHookText) free(createHookText);
		createHookText = isstr(argv[2]) ? strdup(argv[2]) : NULL;
		return TCL_OK;
	}

	/*
	 * interp result name
	 */
	else if (argv[1][0] == 'r' && strcmp(argv[1], "result") == 0) {
		if (!isstr(argv[2])) goto no_interp_name_given;
		info = Int_findInterp(argv[2]);
		if (!info) {
			Tcl_AppendResult(interp, "ERROR: Could not find interpreter ",
				argv[2], NULL);
			return TCL_ERROR;
		}

		Tcl_SetResult(interp, info->interp->result, TCL_STATIC);
		return TCL_OK;
	}

	/*
	 * interp this
	 */
	else if (argv[1][0] == 't' && strcmp(argv[1], "this") == 0) {
		Tcl_SetResult(interp, ((struct interpInfo *)clientData)->name,
					TCL_STATIC);
		return TCL_OK;
	}

	/*
	 * interp library
	 */
	else if (argv[1][0] == 'l' && strcmp(argv[1], "library") == 0) {
		char buf[BUFSIZ];
		char* p = Tcl_GetVar(interp,"minterp_library",TCL_GLOBAL_ONLY);
		strcpy(buf,p);
		Tcl_SetResult(interp, buf, TCL_STATIC);
		/*
		Tcl_SetResult(interp, MINTERP_LIBRARY, TCL_STATIC);
		*/
		return TCL_OK;
	}

	Tcl_AppendResult(interp, "ERROR: Unknown interp subcommand: ",
				argv[1], NULL);
	return TCL_ERROR;

no_interp_name_given:
	Tcl_AppendResult(interp, "USAGE: ", argv[0], " ", argv[1], " interp-name",
					NULL);
	return TCL_ERROR;
}

