/* Bos_Storage.c - Object storage/retrieval
 *
 * Copyright (C) 1992,1993 Engineering Design Research Center
 *
 * Authors: Suresh L. Konda (slk@sei.cmu.edu)
 *          Sean Levy (snl+@cmu.edu)
 *          n-dim Group
 *          Engineering Design Research Center
 *          Carnegie Mellon University
 *          5000 Forbes Ave / PGH, PA / 51221
 *
 *          Fax: (412) 268-5229
 *          Voice: (412) 268-5226
 */

/*
 * Bos_Storage.c,v 1.17 1992/07/30 17:53:42 snl Exp
 *
 * Bos_Storage.c,v
 * Revision 1.17  1992/07/30  17:53:42  snl
 * Re-implemented GDBM-based storage scheme
 *
 * Revision 1.16  1992/07/29  22:50:48  snl
 * Fixed nasty bug with evanescence and load
 *
 * Revision 1.15  1992/07/29  18:27:59  snl
 * added lsobj utility plus cosmetic changes
 *
 * Revision 1.14  1992/07/22  17:15:08  snl
 * More tweaks for storage and evanescence
 *
 * Revision 1.13  1992/07/21  15:56:40  snl
 * Made reporting of GDBM errors better
 *
 * Revision 1.12  1992/07/14  03:01:50  snl
 * Added evanescence everywhere, plus small changes to Storage
 *
 * Revision 1.11  1992/07/10  19:31:26  snl
 * Fixed major bug in load, added ifdefed DEBUGGING code
 *
 * Revision 1.10  1992/07/10  18:37:52  snl
 * Added Bos_InStoreCmd and related Tcl prim
 *
 * Revision 1.9  1992/07/08  16:39:53  slk
 * fixed bug in handling METHOD slots in bosLoad()
 *
 * Revision 1.8  1992/07/07  12:55:27  slk
 * Made Bos_CloseGdbm impervious to multiple calls
 *
 * Revision 1.7  1992/07/06  19:54:10  slk
 * Added registry of bos_CloseGdbm with atexit() to ensure proper closing of all gdbm files
 *
 * Revision 1.6  1992/06/29  02:22:22  snl
 * fixed frame hackery and caught bad call to free()
 *
 * Revision 1.5  1992/06/28  15:51:00  slk
 * Propogated snl's changes in _bos_loadCmd() to _bossaveCmd() to avoid same problems
 *
 * Revision 1.4  1992/06/28  06:25:34  snl
 * Fixed bug in _bos_LoadCmd() that was probably due to cut/paste
 * hacking... arg[CV]Ptr were local (auto) vars when they should've
 * been static, since they were being passed between recursive invocations.
 * Renamed them recurse_nSlots, recurse_slotNames, resp.
 *
 * Revision 1.3  1992/06/28  05:30:23  slk
 * bug fixes to storage stuff
 *
 * Revision 1.2  1992/06/23  18:08:31  slk
 * bug fixes, patches, checked in by snl masquerading as slk
 *
 * Revision 1.1  1992/06/19  06:49:52  slk
 * Added Version 0 of storage stuff
 *
 */

static char rcsID[] = "Bos_Storage.c,v 1.17 1992/07/30 17:53:42 snl Exp";

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>
#include "bosInt.h"
#include <gdbm.h>

#define	STORAGE_VERSION	1
#ifndef STORAGE_FILE_NAME
#define STORAGE_FILE_NAME "bosobjects.gdbm"
#endif /* STORAGE_FILE_NAME */

typedef unsigned char byte;
typedef struct bos_gdbm_data {
  GDBM_FILE *gdbmf;
  char *directory;
  byte mode;
  byte lock_file;
  struct dbgm_data *next;
}Gdbm_Data;

#define TCL_ERR_RETURN(s, a) {sprintf(tcl->interp, s, a);return(TCL_ERROR);}

/* required externals */
extern gdbm_error gdbm_errno;  

/* local functions */
static GDBM_FILE *bos_OpenGdbm _ARGS_((char *directory, char *lock_file,
				       mode));
static get_buffer_space();
static increase_buffer_space();
static char *_gdbmErrMsg();
     
/* local variables */
#if	EVEN_BOUNDARY
static int ibuf[100];			/* testing even boundary hypothesis */
#endif	
static byte *buffer = (byte *)0;
static int bindex;
static int blen = 2048;
static int saw_a_frameSlot = FALSE;
static int recursing = FALSE;
static char frameSlot_object[124];
static Gdbm_Data *last_used = (Gdbm_Data *)0;
static Gdbm_Data *head = (Gdbm_Data *)0;

/* for ndim frame fundas */

static int recurse_nSlots = 0;
static char **recurse_slotNames = (char **)0;
     
int Bos_SaveCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  if (argc < 4)
    {
      sprintf(interp->result,
	      "wrong # args: should be \"%.50s object directory do_lock\"",
	      argv[0]);
      return TCL_ERROR;
    }
  
  return _bos_saveCmd(clientData, interp, argv[0], argv[1],
		      argv[2], argv[3]);
}

int _bos_saveCmd(clientData, interp, caller, object, directory, do_lock)
     ClientData clientData;
     Tcl_Interp *interp;
     char *caller;
     char *directory;
     char *object;
     char *do_lock;
{
  
  Bos_World *world = (Bos_World *)clientData;
  GDBM_FILE *gdbmf;
  datum gdbm_key;
  datum gdbm_content;
  Tcl_HashSearch search;
  Tcl_HashEntry *entry;
  Bos_Object *obj;
  u_long slot_count;
  u_short temp;
  byte *vp;
  Bos_Slot *slot;
  int name_size, required;
  byte do_free;

  if(!buffer)
    get_buffer_space();
  
  obj = Bos_Find(world, object);
  if(obj == (Bos_Object*)0)
    {
      sprintf(interp->result,
	      "cannot find object: \"%.50s\"", object);
      return TCL_ERROR;
    }
  
  /* open the gdbm file */
  if(( gdbmf = bos_OpenGdbm(directory, do_lock, GDBM_WRCREAT))
     == (GDBM_FILE  *)0)
    {
      sprintf(interp->result,
	      "cannot open gdbm file in:\"%.50s\" %.50s", caller, directory);
      return TCL_ERROR;
    }
  
  
  /* since the object name is the key, we dont have to store it again */
  /*
    set the version number and move the pointer to account for the
    version number and the number of slots
    */     
  buffer[0] = STORAGE_VERSION;
  bindex = 5;
  slot_count = 0;
  saw_a_frameSlot = FALSE;
  
  for (entry = Tcl_FirstHashEntry(obj->slots, &search);
       entry != (Tcl_HashEntry *)0;
       entry = Tcl_NextHashEntry(&search))
    {

      register Bos_Slot_Type plain_type;

      slot = (Bos_Slot *)Tcl_GetHashValue(entry);
      
      /* XXX
       *
       * New deal with evanescent slots:
       *
       * We DO store them, but we store an zero-length value string.
       * When we load them, we restore an evanescent slot iff it doesn't
       * already exist in the object.
       *
       *        -- snl
       */

      plain_type = Bos_PlainSlotType(slot->type);
      
      slot_count++;
      
      name_size = strlen(slot->name) + 1;
      required = name_size + 2 /* type */ + 2 /* pri */;
      if(bindex + required > blen)
	if(!increase_buffer_space(required))
	  {
	    sprintf(interp->result, "out of space in bos_saveCmd: \"%.50s\"",
		    caller);
	    return TCL_ERROR;
	  }
      
      /* this will store it as a c string */
      memcpy(buffer + bindex, slot->name, name_size);
      bindex += name_size;
      
      temp = htons(slot->type);
      memcpy(buffer+bindex, &temp, 2);
      bindex += 2;
      
      temp = htons(slot->pri);
      memcpy(buffer+bindex, &temp, 2);
      bindex += 2;

      /* XXX
       *
       * Rest of evanescent fundas for save.
       */

      if(slot->value == (_VoidPtr)0 || IS_EVANESCENT(slot))
	{
	  /* set to zero since the string will be null */

	  /* XXX
	   *
	   * Since BOS doesn't distinguish between int and string
	   * slots, we need to here; if an evanescent slot's value
	   * is all digits, then store the value as the string "0";
	   * otherwise, store an empty string ("").
	   */

	  if (IS_INTEGER(slot))
	    buffer[bindex++] = '0';
	  buffer[bindex++] = '\0';
	  continue;	/* process the next slot */
	}
      
      do_free = FALSE;
      switch(plain_type)
	{
	case Bos_SLOT_CMETHOD:
	  vp = Bos_GetCMethodName(slot->value);
	  break;
	  
	case Bos_SLOT_METHOD:
	  vp = ((Bos_Method *)(slot->value))->body;
	  break;
	  
	case Bos_SLOT_OBJECT:
	case Bos_SLOT_REFERENCE:
	case Bos_SLOT_NORMAL:
	  vp = slot->value;
	  break;
	  
	case Bos_SLOT_FOREIGN:
	  vp = Bos_GetCSlotString(slot->value, slot->pri);
	  do_free = TRUE;
	  break;
	  
	default:
	  sprintf(interp->result,
		  "bad slot type: \"%.50s\" %.50s %.50s, type %u (%u)",
		  caller, object, slot->name, plain_type,
		  slot->type);
	  return TCL_ERROR;
	}
      
      required = strlen(vp) + 1;         /* include the null */
      
      if(bindex + required > blen)
	if(!increase_buffer_space(required))
	  {
	    sprintf(interp->result, "out of space in bos_savecmd: \"%.50s\"",
		    caller);
	    return TCL_ERROR;
	  }
      
      memcpy(buffer + bindex, vp, required);
      bindex += required;
      
      /* free the string if obtained form a foreign slot */
      if(do_free)
	ckfree(vp);
      
      /* now check for the special case of a frame slot */
      if(!strcmp(slot->name, "frameSlots"))
	{
	  saw_a_frameSlot = TRUE;
	  strcpy(frameSlot_object, slot->value);
	}
      
      if(recursing && !strcmp(slot->name, "contents") &&
	 plain_type == Bos_SLOT_FOREIGN)
	{
	  /* now remember the object names to save them recursively */
	  
	  if(Tcl_SplitList(interp, vp, &recurse_nSlots, &recurse_slotNames) != TCL_OK)
	    {
	      sprintf(interp->result, "error in split_list: \"%.50s\"",
		      caller);
	      recurse_nSlots = 0;
	      recurse_slotNames = (char **)0;
	      return TCL_ERROR;
	    }
	}
      
    } /* for loop */
  
  /* add in the slot count after the version number */
  slot_count = htonl(slot_count);
  memcpy(buffer + 1, &slot_count, 4);
  
  gdbm_key.dptr = object;
  gdbm_key.dsize = strlen(object);
  
  gdbm_content.dptr = (char *)buffer;
  gdbm_content.dsize = bindex;
  
  if(gdbm_store(gdbmf, gdbm_key, gdbm_content, GDBM_REPLACE) < 0)
    {
      sprintf(interp->result, "gdbm save error: \"%.50s\" (%d: %s)", caller,
	      gdbm_errno, _gdbmErrMsg(gdbm_errno));
      return TCL_ERROR;
    }
  
  if(saw_a_frameSlot)
    {
      int i;
      saw_a_frameSlot = FALSE;
      recursing = TRUE;
      if(_bos_saveCmd(clientData, interp, caller,
		      frameSlot_object, directory, do_lock) == TCL_ERROR)
	return TCL_ERROR;
      recursing = FALSE;
      for(i = 0; i < recurse_nSlots; i++)
	if(_bos_saveCmd(clientData, interp, caller,
			recurse_slotNames[i], directory, do_lock) == TCL_ERROR)
	  {
	    ckfree((char *) recurse_slotNames);
	    recurse_nSlots = 0;
	    recurse_slotNames = (char **)0;
	    return TCL_ERROR;
	  }
      ckfree((char *) recurse_slotNames);
      recurse_nSlots = 0;
      recurse_slotNames = (char **)0;
    }
  
  
  return TCL_OK;
}

int Bos_LoadCmd(clientData, interp, argc, argv)
     ClientData clientData;
     Tcl_Interp *interp;
     int argc;
     char **argv;
{
  if (argc < 4)
    {
      sprintf(interp->result,
	      "wrong # args: should be \"%.50s object directory do_lock\"",
	      argv[0]);
      return TCL_ERROR;
    }
  return(_bos_loadCmd(clientData, interp, argv[0], argv[1],argv[2], argv[3]));
}

int _bos_loadCmd(clientData, interp, caller, object, directory, do_lock)
v     ClientData clientData;
     Tcl_Interp *interp;
     char *caller;
     char *directory;
     char *object;
     char *do_lock;
     
{
  
  Bos_World *world = (Bos_World *)clientData;
  GDBM_FILE *gdbmf;
  datum gdbm_key;
  datum gdbm_content;
  Tcl_HashSearch search;
  Tcl_HashEntry *entry;
  Bos_Object *obj;
  u_long slot_count;
  u_short temp;
  byte *bp;
  Bos_Slot slot;
  
  if(!buffer)
    get_buffer_space();
  
  /* get the gdbm file */
  gdbmf = bos_OpenGdbm(directory, do_lock, GDBM_READER);
  if( gdbmf == (GDBM_FILE  *)0 )
    {
      sprintf(interp->result,
	      "cannot open gdbm file in:\"%.50s\" %.50s", caller, directory);
      return TCL_ERROR;
    }
  
  
  gdbm_key.dptr = object;
  gdbm_key.dsize = strlen(object);
  
  gdbm_content = gdbm_fetch(gdbmf, gdbm_key);
  
  if(gdbm_content.dptr == NULL)
    {
      sprintf(interp->result, "object not in gdbm: \"%.50s\" %.50s (%d: %s)",
	      caller, object, gdbm_errno, _gdbmErrMsg(gdbm_errno));
      return TCL_ERROR;
    }
  
  bp = (byte *)gdbm_content.dptr;
  
  /* check  version number */
  if(*bp != STORAGE_VERSION)
    {
      sprintf(interp->result, "object wrong version: \"%.50s\" %.50s",
	      caller, object);
      free(gdbm_content.dptr);
      return TCL_ERROR;
    }
  
  bindex = 1;			/* skip past the version byte */
  
  obj = Bos_CreateNewObject(world, object);
  if(obj == (Bos_Object *)0)
    {
      sprintf(interp->result, "could not create: \"%.50s\" %.50s",
	      caller, object);
      free(gdbm_content.dptr);
      return TCL_ERROR;
    }
  
  
  /* get number of slots */
  memcpy(&slot_count, bp + bindex, 4);
  slot_count = ntohl(slot_count);
  bindex += 4;
  saw_a_frameSlot = FALSE;
  
  while(slot_count--)
    {    
      int skip_this_slot;
      Bos_Slot_Type plain_type;
      int add_stat, is_evanescent;

      skip_this_slot = 0;

      slot.name = (char *)bp + bindex;
      bindex += 1 + strlen(slot.name);           /* past the name */
      
      memcpy(&slot.type, bp + bindex, 2);
      slot.type = ntohs(slot.type);
      bindex += 2;
      plain_type = Bos_PlainSlotType(slot.type);
      is_evanescent = (slot.type & Bos_SLOT_EVANESCENT_MASK);

      memcpy(&slot.pri, bp + bindex, 2);
      slot.pri = ntohs(slot.pri);
      bindex += 2;

      /* XXX
       *
       * Here is the rest of the evanescent funda. --snl
       */
      if (is_evanescent && (_BosFindSlot(obj, slot.name) != (Bos_Slot *)0)) {
	bindex += strlen(bp + bindex) + 1;
	continue;
      }
      /* else, we'll restore it with the null value that the saveCmd
       * code, above, saved it with.... fall through. Note that the
       * null value is either "0" for integer slots or "" for everything
       * else. Yuck.
       */

#ifdef DEBUGGING
      printf("_bos_loadCmd() object %s slot %s type %u(%u) pri %u\n",
	     object, slot.name, slot.type, plain_type, slot.pri);
#endif

      switch(plain_type)
	{
	case Bos_SLOT_CMETHOD:
	  if (*(bp + bindex) == 0) {
	    fprintf(stderr,
		    "WARNING: Bos_Load(%s,%s,%s) CMETHOD %s is NULL - skipping...\n",
		    object, directory, do_lock, slot.name);
	    skip_this_slot = 1;
	  } else
	    slot.value = Bos_GetCMethodPointer(bp + bindex);
	  break;
	  
	case Bos_SLOT_METHOD:	/* treat like other -- per Sean */
	case Bos_SLOT_OBJECT:
	case Bos_SLOT_REFERENCE:
	case Bos_SLOT_NORMAL:
	  slot.value = bp + bindex;
	  break;
	  
	case Bos_SLOT_FOREIGN:
	  slot.value = Bos_ParseCSlotString(bp + bindex, slot.pri);
#ifdef DEBUGGING
	  printf("foreign slot %s (%s) => 0x%x\n", slot.name,
		 bp + bindex, slot.value);
	  fflush(stdout);
#endif
	  break;

        default:
#ifdef SKIP_BAD_SLOTS
	  printf("WARNING: bad slot type %u(%u) object %s slot %s\n",
		 plain_type, slot.type, object, slot.name);
	  skip_this_slot = 1;
	  break;
#else
	  sprintf(interp->result,
		  "bad slot type: \"%.50s\" %.50s %.50s %u (%u)",
		  caller, object, slot.name, plain_type,
		  slot.type);
	  free(gdbm_content.dptr);
	  return TCL_ERROR;
#endif
	  break;
	}

#ifdef notdef      
      /* XXX This is fucked for binary data (FOREIGN and CMETHOD).
	 */
#if	EVEN_BOUNDARY
      /* testing even boundary */
      strcpy((char *)ibuf, slot.value);
      slot.value = (_VoidPtr)ibuf;
#endif
#endif

      if (skip_this_slot)
	goto adv_bindex;

#ifdef DEBUGGING
      printf("adding %sslot %s to %s: type=%d pri=%d val=0x%x\n",
	     is_evanescent? "evanescent ": "", slot.name,
	     object, slot.type, slot.pri, slot.value);
      fflush(stdout);
#endif
      /* add the slot */

      add_stat = Bos_AddSlot(obj, slot.name, slot.type, slot.pri, slot.value);
      if(add_stat == BOS_ALREADY) {
	int op_mask;

	op_mask = Bos_SET_TYPE|Bos_SET_PRI|Bos_SET_VALUE;

	/* XXX
	 *
	 * Should we set E OFF if it isn't ON on disk because the
	 * user has asked us to (re)load the object from disk?
	 * Maybe evanescence should be a "sticky" attribute of slots?
	 *
	 *      --snl
	 */

	if (is_evanescent)
	  op_mask |= Bos_SET_EVANESCENT_ON;
#ifdef notdef /* XXX */
	else
	  op_mask |= Bos_SET_EVANESCENT_OFF;
#endif
	add_stat = Bos_SetSlot(obj, op_mask,
			       slot.name, slot.type, slot.pri, slot.value);
      }
      if(add_stat != BOS_OK)
	{
	  sprintf(interp->result,
		  "cant add slot: \"%.50s\" object %s slot %s type %d pri %d stat %d",
		  caller, object, slot.name, slot.type, slot.pri, add_stat);
	  free(gdbm_content.dptr);
	  return TCL_ERROR;
	}
      
      /* now check for the special case of a frame slot */
      if(!strcmp(slot.name, "frameSlots"))
	{
	  saw_a_frameSlot = TRUE;
	  strcpy(frameSlot_object, slot.value);
	}
      
      if(recursing && !strcmp(slot.name, "contents") &&
	 (plain_type == Bos_SLOT_FOREIGN))
	{
	  char *str_val = bp + bindex;
	  
	  /* now remember the object names to load them recursively */
	  
	  if(Tcl_SplitList(interp, str_val, &recurse_nSlots,
			   &recurse_slotNames) != TCL_OK)
	    {
	      sprintf(interp->result, "error in split_list: \"%.50s\"",
		      caller);
	      free(gdbm_content.dptr);
	      recurse_nSlots = 0;
	      recurse_slotNames = (char **)0;
	      return TCL_ERROR;
	    }
#ifdef DEBUGGING
	  printf("_bos_LoadCmd: SplitList(%s) = %d elts @%x\n", str_val,
		 recurse_nSlots, recurse_slotNames);
	  fflush(stdout);
#endif
	}

    adv_bindex:

      /* move the index */
      bindex += strlen(bp + bindex) + 1;

    }/* for loop */
  
  if(saw_a_frameSlot)
    {
      int i;
      
      saw_a_frameSlot = FALSE;
      recursing = TRUE;
      if(_bos_loadCmd(clientData, interp, caller,
		      frameSlot_object, directory, do_lock) == TCL_ERROR)
	{
	  free(gdbm_content.dptr);
	  return TCL_ERROR;
	}
      recursing = FALSE;
#ifdef DEBUGGING
      printf("_bos_LoadCmd recursing: nSlots=%d slotNames@%x\n",
	     recurse_nSlots, recurse_slotNames);
      fflush(stdout);
#endif
      for(i = 0; i < recurse_nSlots; i++) {
#ifdef DEBUGGING
	printf(".. recursing on slot %d: \"%s\"\n", i, recurse_slotNames[i]);
	fflush(stdout);
#endif
	if(_bos_loadCmd(clientData, interp, caller,
			recurse_slotNames[i], directory, do_lock) == TCL_ERROR)
	  {
	    ckfree((char *) recurse_slotNames);	    
	    recurse_slotNames = (char **)0;
	    recurse_nSlots = 0;
	    free(gdbm_content.dptr);
	    return TCL_ERROR;
	  }
      }
      ckfree((char *) recurse_slotNames);
      recurse_slotNames = (char **)0;
      recurse_nSlots = 0;
    }
  Tcl_CreateCommand(interp, object, Bos_ObjectCmd,
		    clientData, (void (*)())NULL);
  free(gdbm_content.dptr);
  return TCL_OK;
}

int Bos_InStoreCmd(clientData, interp, argc, argv)
  ClientData clientData;
  Tcl_Interp *interp;
  int argc;
  char **argv;
{
  Bos_World *world = (Bos_World *)clientData;
  GDBM_FILE *gdbmf;
  datum gdbm_key;
  datum gdbm_content;
  char *caller, *object, *directory, *do_lock;

  if (argc < 4) {
    sprintf(interp->result,
	    "wrong # args: should be \"%.50s object directory do_lock\"",
	    argv[0]);
    return TCL_ERROR;
  }

  caller = argv[0];
  object = argv[1];
  directory = argv[2];
  do_lock = argv[3];

  if(!buffer)
    get_buffer_space();
  
  /* get the gdbm file */
  gdbmf = bos_OpenGdbm(directory, do_lock, GDBM_READER);
  if( gdbmf == (GDBM_FILE  *)0 ) {
    sprintf(interp->result,
	    "cannot open gdbm file in:\"%.50s\" %.50s", caller, directory);
    return TCL_ERROR;
  }
  
  
  gdbm_key.dptr = object;
  gdbm_key.dsize = strlen(object);
  
  gdbm_content = gdbm_fetch(gdbmf, gdbm_key);
  
  if(gdbm_content.dptr == NULL)
    strcpy(interp->result, "0");
  else
    strcpy(interp->result, "1");

  return TCL_OK;
}

static GDBM_FILE *bos_OpenGdbm(directory, lock_file, mode)
     char *directory;
     char *lock_file;
     int mode;
{
  void fatal_func();
  char fname[1024];
  
  if(last_used == (Gdbm_Data *)0)
    {
      /* not opened this directory previously.  Open now */
      last_used = (Gdbm_Data *)ckalloc(sizeof(Gdbm_Data));
      last_used->directory = (char *)ckalloc(strlen(directory)+1);
      strcpy(last_used->directory, directory);
      last_used->mode = mode;
      last_used->lock_file = (lock_file[0] != '0');
      sprintf(fname, "%s/%s", directory, STORAGE_FILE_NAME);
      last_used->gdbmf = gdbm_open(fname, 512, mode, 0x1A4, fatal_func);
      if(last_used->gdbmf == (GDBM_FILE *)0)
 	{
 	  fprintf(stderr, "major fucking problem in GDBM (%d: %s)\n",
 		  gdbm_errno, _gdbmErrMsg(gdbm_errno));
 	  return((GDBM_FILE *)0);
 	}
      if(head)
	last_used->next = head;		/* make this the head */
      else
	{
	  /* make this the head */
	  last_used->next = (Gdbm_Data *)0;
	  head = last_used;
	}
      return(last_used->gdbmf);
    }

  /*
    check if the last_used was a locked file opened in write mode
    and is now being used in read mode.  If so, reopen in read mode to let
    others get a chance to write to it
  */
  if(last_used->lock_file && last_used->mode == GDBM_WRCREAT &&
     mode != GDBM_WRCREAT)
    {
      
      /* close and reopen in read mode */
      gdbm_close(last_used->gdbmf);
      last_used->mode = GDBM_READER;
      sprintf(fname, "%s/%s", last_used->directory, STORAGE_FILE_NAME);
      last_used->gdbmf = gdbm_open(fname, 512, mode, 0x1A4, fatal_func);
      if(last_used->gdbmf == (GDBM_FILE *)0)
	{
	  fprintf(stderr, "major fucking problem in GDBM (%d: %s)\n",
		  gdbm_errno, _gdbmErrMsg(gdbm_errno));
	  return((GDBM_FILE *)0);
	}
    }
  
  /* see if we have already opened the desired file */
  if(strcmp(last_used->directory, directory))
    {
      /* Not the last one used. Find one in the chain if it exists */
      for(last_used = head;
	  last_used != (Gdbm_Data *)0;
	  last_used = last_used->next)
	if(!strcmp(last_used->directory, directory))
	  break;
      if(!last_used)
	/* does not exist in the chain.  recurse */
	return(bos_OpenGdbm(directory, lock_file, mode)	);
    }
  /* use the one in the last_used */
  if(last_used->mode == mode)
    return(last_used->gdbmf);
  
  /* modes differ.  close and reopen only if mode is to write */
  if(mode == GDBM_WRCREAT)
    {
      gdbm_close(last_used->gdbmf);
      last_used->mode = mode;
      sprintf(fname, "%s/%s", last_used->directory, STORAGE_FILE_NAME);
      last_used->gdbmf = gdbm_open(fname, 512, mode, 0x1A4, fatal_func);
      if(last_used->gdbmf == (GDBM_FILE *)0)
	{
	  fprintf(stderr, "major problem in GDBM (%d: %s)\n",
		  gdbm_errno, _gdbmErrMsg(gdbm_errno));
	  return((GDBM_FILE *)0);
	}
    }
  return(last_used->gdbmf);
}

void bos_CloseGdbm()
{
  Gdbm_Data *f;
  
  for(f = head; f != (Gdbm_Data *)0; f = f->next)
    {
      gdbm_close(f->gdbmf);
      ckfree(f->directory);
      ckfree(f);
    }
  head = (Gdbm_Data *)0;
  return;
}

/* return true or false */
static int get_buffer_space()
{
  void bos_CloseGdbm();
  
  buffer = (byte *)ckalloc(blen);
  if(buffer == (byte*)0)
    return(FALSE);

#ifdef sunos41
  on_exit(bos_CloseGdbm, 0);
#else
  atexit(bos_CloseGdbm);	/* register to close all files properly */
#endif
  return(TRUE);
  
}


static int increase_buffer_space(min_additional)
     int min_additional;
     
{
  byte *temp;
  int new_len;
  
  new_len = 1024 * (1 +  (blen + min_additional)/1024);
  temp = (byte *)ckalloc(new_len);
  if(temp == (byte *)0 )
    return(FALSE);
  
  /* got it. reset, copy, free and return */
  memcpy(temp, buffer, blen);
  ckfree(buffer);
  blen = new_len;
  buffer = temp;
  return(TRUE);
}

static void fatal_func()
{
  extern gdbm_error gdbm_errno;
  
  fprintf(stderr, "major problem in GDBM (%d: %s)\n",
	  gdbm_errno, _gdbmErrMsg(gdbm_errno));
  abort();
}

static int IS_EVANESCENT(slot)
  Bos_Slot *slot;
{
  return (slot->type & Bos_SLOT_EVANESCENT_MASK);
}

static int IS_INTEGER(slot)
     Bos_Slot *slot;
{
  Bos_Slot_Type plain_type;
  register char *vs, c;
  
  plain_type = Bos_PlainSlotType(slot->type);
  if (plain_type != Bos_SLOT_NORMAL)
    return 0;
  vs = (char *)slot->value;
  if (vs == (char *)0)
    return 0;
  while ((c = *vs++) != '\0')
    if (c < '0' || c > '9')
      return 0;
  return 1;
}

static char *_gdbmErrMsg(code)
     gdbm_error code;
{
  static struct {gdbm_error code; char *text;} _msgs[] = {
    { GDBM_NO_ERROR, "No error" },
    { GDBM_MALLOC_ERROR, "Malloc error" },
    { GDBM_BLOCK_SIZE_ERROR, "Block size error" },
    { GDBM_FILE_OPEN_ERROR, "File open error" },
    { GDBM_FILE_WRITE_ERROR, "File write error" },
    { GDBM_FILE_SEEK_ERROR, "File seek error" },
    { GDBM_FILE_READ_ERROR, "File read error" },
    { GDBM_BAD_MAGIC_NUMBER, "Bad magic number" },
    { GDBM_EMPTY_DATABASE, "Empty database" },
    { GDBM_CANT_BE_READER, "Can't be reader" },
    { GDBM_CANT_BE_WRITER, "Can't be writer" },
    { GDBM_READER_CANT_DELETE, "Reader can't delete" },
    { GDBM_READER_CANT_STORE, "Reader can't store" },
    { GDBM_READER_CANT_REORGANIZE, "Reader can't reorganize" },
    { GDBM_UNKNOWN_UPDATE, "Unknown update" },
    { GDBM_ITEM_NOT_FOUND, "Item not found" },
    { GDBM_REORGANIZE_FAILED, "Reorganize failed" },
    { GDBM_CANNOT_REPLACE, "Cannot replace" },
    { GDBM_NO_ERROR, (char *)0 }
  };
  static char _unknMsg[200];
  int i;

  for (i = 0; _msgs[i].text != (char *)0; i++)
    if (_msgs[i].code == code)
      return _msgs[i].text;
  sprintf(_unknMsg, "Unknown GDBM error code: %d", code);
  return _unknMsg;
}
