/* Bos_Send.c - code dealing with message sending */

/* 
 *
 * /afs/cs/project/edrc/ndim/source/bos/libbos/Bos_Send.c,v 1.2 1992/07/14 03:01:38 snl Exp
 *
 * HISTORY
 *
 * Bos_Send.c,v
 * Revision 1.2  1992/07/14  03:01:38  snl
 * Added evanescence everywhere, plus small changes to Storage
 *
 * Revision 1.1.1.1  1992/05/08  19:45:37  snl
 * bos 1.2
 *
 * Revision 1.2  92/03/07  00:47:47  snl
 * rel 1.2
 * 
 * Revision 1.1  92/03/06  22:02:58  snl
 * Initial revision
 * 
 * Revision 1.2  92/01/27  16:19:29  snl
 * Port to new TCL
 * 
 * Revision 1.1  91/12/16  20:14:45  snl
 * Initial revision
 * 
 */

#ifndef lint
static char *_RCSId =
 "/afs/cs/project/edrc/ndim/source/bos/libbos/Bos_Send.c,v 1.2 1992/07/14 03:01:38 snl Exp";
#endif /* lint */

#include "bosInt.h"

#ifdef SEND_DEBUG
#include <stdio.h>
int _Bos_Send_Debug = 0;
#endif

typedef struct _Bos_Slot_List {
  List_Links links;
  Bos_Slot *s;
} Bos_Slot_List;

int Bos_Send(world, interp, obj_name, message)
     Bos_World *world;
     Tcl_Interp *interp;
     char *obj_name;
     char *message;
{
  int nfields = 0, s;
  char **fields = (char **)0;

  s = Tcl_SplitList(interp, message, &nfields, &fields);
  if (s != TCL_OK) {
    strcpy(interp->result, "SplitList failed");
    s = BOS_TCL_ERROR;
  } else {
    int i;

    s = Bos_Sendv(world, interp, obj_name, nfields, fields);
    for (i = 0; i < nfields; i++)
      ckfree(fields[i]);
    ckfree(fields);
  }
  return s;
}

int Bos_Sendv(world, interp, obj_name, argc, argv)
     Bos_World *world;
     Tcl_Interp *interp;
     char *obj_name;
     int argc;
     char **argv;
{
  int sendv();

  return sendv(world, interp, obj_name, obj_name, argc, argv);
}

static int sendv(world, interp, self, obj_name, argc, argv)
     Bos_World *world;
     Tcl_Interp *interp;
     char *self;
     char *obj_name;
     int argc;
     char **argv;
{
  Bos_Object *obj;
  int s;

  obj = Bos_Find(world, obj_name);
  if (obj == (Bos_Object *)0) {
#ifdef DISTRIBUTED_BOS
    int _Bos_SendOut();

    s = _Bos_SendOut(interp, obj_name, argc, argv);
#else
    s = BOS_NOT_FOUND;
#endif /* DISTRIBUTED_BOS */
  } else if (argc < 1) {
    strcpy(interp->result, "Bos_Send needs at least one arg");
    s = BOS_TCL_ERROR;
  } else {
    char *key = argv[0], k0, *path_sep, *index();
    Tcl_HashEntry *slot_entry;

    /*
     * First, try to find the slot in the object itself.
     */
    path_sep = index(key, ':');
    if (path_sep != (char *)0) {
      Bos_Object *path_obj;

      /*
       * The syntax <path>:<method> means that we want to chase the
       * pointer named <path> to look for <method>, explicitly.
       */
      *path_sep = '\0';
      path_obj = Bos_Find(world, key);
      *path_sep++ = ':';
      if (path_obj != (Bos_Object *)0) {
        strcpy(argv[0], path_sep);
	s = sendv(world, interp, self, path_obj->name, argc, argv);
      } else {
        sprintf(interp->result,
                "method key \"%.50s\" points at non-existent object", key);
	s = BOS_NOT_FOUND;
      }
    } else {
      k0 = key[0];
      slot_entry = Tcl_FindHashEntry(obj->slots, key);
      if (slot_entry == (Tcl_HashEntry *)0 && (k0 == '&' || k0 == '@'))
	slot_entry = Tcl_FindHashEntry(obj->slots, &key[1]);
      if (slot_entry != (Tcl_HashEntry *)0) {
	Bos_Slot *slot;

        slot = (Bos_Slot *)Tcl_GetHashValue(slot_entry);
	s = handleMessage(world, interp, self, obj,
			  slot, key, argc-1,&argv[1]);
      } else {
	Bos_Slot_List *pointers, *findObjectSlots();

	/* No luck; look for pointers to other objects. findObjectSlots()
	 * sorts the pointers in order of priority for us.
	 */
	pointers = findObjectSlots(obj);
	if (pointers == (Bos_Slot_List *)0)
	  s = BOS_NOT_FOUND;
	else {
	  List_Links *the_slots, *a_slot;

	  the_slots = (List_Links *)pointers;
	  LIST_FORALL(the_slots, a_slot) {
	    Bos_Slot_List *a_ptr = (Bos_Slot_List *)a_slot;

	    s = sendv(world, interp, self, (char *)a_ptr->s->value, argc, argv);
	    if (s != BOS_NOT_FOUND)
	      break;
	  }
	  LIST_FORALL(the_slots, a_slot) {
	    List_Remove(a_slot);
	    ckfree(a_slot);
	  }
	  ckfree(the_slots);
	}
      }
    }
  }
  return s;
}

static int handleMessage(world, interp, self, obj, slot, key, argc, argv)
     Bos_World *world;
     Tcl_Interp *interp;
     char *self;
     Bos_Object *obj;
     Bos_Slot *slot;
     char *key;
     int argc;
     char **argv;
{
  int s;
  char k0 = key[0];
  Bos_Slot_Type plain_type;

  plain_type = Bos_PlainSlotType(slot->type);
  s = BOS_OK;
  if (plain_type == Bos_SLOT_FOREIGN) {
    char *val;

    val = Bos_GetCSlotString(slot->value, slot->pri);
    if (val == (char *)0) {
      s = BOS_ERROR;
      Tcl_AddErrorInfo(interp, "error getting C slot string");
    } else {
      interp->result = val;
      interp->freeProc = TCL_DYNAMIC;
    }
  } else if ((plain_type != Bos_SLOT_METHOD && plain_type != Bos_SLOT_CMETHOD)
          || ((plain_type == Bos_SLOT_METHOD || plain_type == Bos_SLOT_CMETHOD)
          &&  (k0 == '&' || k0 == '@'))) {
    if (slot->value == (_VoidPtr)0)
      *interp->result = '\0';
    else {
      char *val = (char *)slot->value;
      int len;

      /*
       * If this is a method slot, then we must've gotten here because
       * the caller asked for the method itself via the & syntax.
       */
      if (plain_type == Bos_SLOT_METHOD) {
	Bos_Method *method = (Bos_Method *)slot->value;

        /*
	 * This syntax is peculiar to method-typed slots. If you send the
	 * message '&slot', you get the actual text of the method code;
	 * if you send '@slot' you get the interned name of the method,
	 * e.g. the TclProc bound to the method code.
	 */
        switch (k0) {
	case '&':
          val = method->body? method->body: "";
	  break;
	case '@':
	  if (!(method->flags & Bos_METHOD_INTERNED)) {
	    s = internMethod(world, interp, obj, &key[1], method);
	    if (s != BOS_OK) {
	      Tcl_AddErrorInfo(interp, "handleMessage()");
	      return s;
	    }
	  }
	  val = method->proc_name;
	  break;
	default:
	  Bos_FatalError("handleMessage: impossible # 1");
	  break;
	}
      } else if (plain_type == Bos_SLOT_CMETHOD) {
        val = Bos_GetCMethodName(slot->value);
	if (val == (char *)0)
	  val = "unregistered C method hook";
      }
      len = strlen(val);
      if (len < Bos_USE_DYNAMIC_RETURN) {
        strcpy(interp->result, val);
        interp->freeProc = 0;
      } else {
        interp->result = (char *)ckalloc(len + 1);
        strcpy(interp->result, val);
        interp->freeProc = TCL_DYNAMIC;
      }
    }
    s = BOS_OK;
  } else if (plain_type == Bos_SLOT_CMETHOD) {
    if (slot->value == (_VoidPtr)0) {
      *interp->result = '\0';
      s = BOS_OK;
    } else {
      int (*c_function)() = (int (*)())slot->value;
      Bos_CMethod_ClientData cd;

      cd.world = world;
      cd.object = obj;
      cd.self = Bos_Find(world, self);
      if (cd.self == (Bos_Object *)0) {
        sprintf(interp->result, "could not locate self (%.50s)", self);
        s = BOS_ERROR;
      } else
        s = (*c_function)((ClientData)&cd, interp, argc, argv);
    }
  } else {
    if (slot->value == (_VoidPtr)0) {
      *interp->result = '\0';
      s = BOS_OK;
    } else {
      Bos_Method *method = (Bos_Method *)slot->value;

      s = invokeMethod(world, interp, self, obj, method, key, argc, argv);
    }
  }
  return s;
}

static Bos_Slot_List *findObjectSlots(obj)
     Bos_Object *obj;
{
  Tcl_HashEntry *slot_entry;
  Tcl_HashSearch search;
  Bos_Slot_List *slot_list;

  slot_list = (Bos_Slot_List *)0;
  for (slot_entry = Tcl_FirstHashEntry(obj->slots, &search);
       slot_entry != (Tcl_HashEntry *)0;
       slot_entry = Tcl_NextHashEntry(&search)) {
    Bos_Slot *slot;
    Bos_Slot_Type plain_type;

    slot = (Bos_Slot *)Tcl_GetHashValue(slot_entry);
    plain_type = Bos_PlainSlotType(slot->type);
    if (plain_type == Bos_SLOT_OBJECT) {
      Bos_Slot_List *item;

      item = (Bos_Slot_List *)ckalloc(sizeof(Bos_Slot_List));
      List_InitElement((List_Links *)item);
      item->s = slot;
      if (slot_list == (Bos_Slot_List *)0) {
        slot_list = (Bos_Slot_List *)ckalloc(sizeof(Bos_Slot_List));
        List_Init((List_Links *)slot_list);
      }
      if (slot->pri == Bos_PRI_HIGHEST)
        List_Insert(item, LIST_ATFRONT(slot_list));
      else if (slot->pri == Bos_PRI_LOWEST)
        List_Insert(item, LIST_ATREAR(slot_list));
      else {
        List_Links *the_list = (List_Links *)slot_list, *an_item;
        int inserted = 0;

	LIST_FORALL(the_list, an_item) {
          Bos_Slot_List *i = (Bos_Slot_List *)an_item;

	  if (i->s->pri < slot->pri) {
	    List_Insert(item, LIST_BEFORE(an_item));
	    inserted = 1;
	    break;
	  }
	}
	if (!inserted)
	  List_Insert(item, LIST_ATREAR(slot_list));
      }
    }
  }
  return slot_list;
}

static int invokeMethod(world, interp, self, obj, method, key, argc, argv)
     Bos_World *world;
     Tcl_Interp *interp;
     char *self;
     Bos_Object *obj;
     Bos_Method *method;
     char *key;
     int argc;
     char **argv;
{
  int s;

  s = BOS_OK;
  if (!(method->flags & Bos_METHOD_INTERNED) ||
       (method->flags & Bos_METHOD_CHANGED))
    s = internMethod(world, interp, obj, key, method);
  if (s == BOS_OK) {
    int args_len;
    char *cmd, *args, *old_self;

    args = Tcl_Merge(argc, argv);
    args_len = strlen(args);
    cmd = (char *)ckalloc(args_len + strlen(method->proc_name) + 5);
    *cmd = '\0';
    sprintf(cmd, "%s %s", method->proc_name,  args);
#ifdef SEND_DEBUG
    if (_Bos_Send_Debug) {
      printf("invokeMethod(self=%s): %s\n", self, cmd);
      fflush(stdout);
    }
#endif
    old_self = Tcl_GetVar(Bos_Methods(world), "self", TCL_GLOBAL_ONLY);
    if (old_self != (char *)0) {
      char *save;

      save = (char *)ckalloc(strlen(old_self)+1);
      strcpy(save, old_self);
      old_self = save;
    }
#ifdef DEBUG_SELF
    printf(" *B4 invokeMethod(self=%s,obj=%s,key=%s) old_self=%s\n",
           self,obj->name,key,old_self?old_self:"<null>");
#endif /* DEBUG_SELF */
    Tcl_SetVar(Bos_Methods(world), "self", self, TCL_GLOBAL_ONLY);
    s = Tcl_Eval(Bos_Methods(world), cmd, 0, (char **)0);
    if (s != TCL_OK) {
      char errMsg[1000];

      sprintf(errMsg, "invokeMethod(%x,%x,%s,%s,%s,%s,{%s})", world,
              interp, self, obj->name, method->proc_name, key, args);
      Tcl_AddErrorInfo(interp, errMsg);
      s = BOS_TCL_ERROR;
    }
    Tcl_SetVar(Bos_Methods(world), "self", old_self? old_self: "",
	       TCL_GLOBAL_ONLY);
#ifdef DEBUG_SELF
    { char *__s;

      __s = Tcl_GetVar(Bos_Methods(world), "self", TCL_GLOBAL_ONLY);
      printf(" *AF invokeMethod(self=%s,old_self=%s) self=%s\n",
             self, old_self?old_self:"<null>", __s?__s:"<null>");
    }
#endif /* DEBUG_SELF */
    if (old_self != (char *)0)
      ckfree(old_self);
    ckfree(args);
    ckfree(cmd);
  }
  return s;
}

static int internMethod(world, interp, obj, key, method)
     Bos_World *world;
     Tcl_Interp *interp;
     Bos_Object *obj;
     char *key;
     Bos_Method *method;
{
  int s, nfields = 0;
  char **fields = (char **)0;

  if (!(method->flags & Bos_METHOD_INTERNED)) {
    char *methodProcName();

    method->proc_name = methodProcName(world, obj, key);
    method->flags |= Bos_METHOD_INTERNED;
  }
  s = Tcl_SplitList(interp, method->body, &nfields, &fields);
  if (s != TCL_OK || nfields != 2) {
    sprintf(interp->result, "badly formed method %.50s in %.50s (%.50s)",
            key, obj->name, method->body);
    s = BOS_TCL_ERROR;
  } else {
    int proc_cmd_len;
    char *proc_cmd;

    proc_cmd_len = strlen(method->proc_name) + strlen(fields[0]) +
                   strlen(fields[1]) + 100;
    proc_cmd = (char *)ckalloc(proc_cmd_len);
    sprintf(proc_cmd, "proc %s {%s} {global self;%s}",
            method->proc_name, fields[0], fields[1]);
    s = Tcl_Eval(interp, proc_cmd, 0, (char **)0);
    if (s != TCL_OK) {
      char errMsg[1000];

      sprintf(errMsg, "internMethod(%x,%x,%s,%s,%s;cmd=%s)",
              world, interp, obj->name, key, method->proc_name, proc_cmd);
      Tcl_AddErrorInfo(interp, errMsg);
      s = BOS_TCL_ERROR;
    } else
      method->flags &= ~Bos_METHOD_CHANGED;
    ckfree(proc_cmd);
  }
  ckfree((char *)fields);
  return s;
}

static char *methodProcName(world, obj, key)
     Bos_World *world;
     Bos_Object *obj;
     char *key;
{
  int len;
  char *name;

  len = strlen(obj->name) + strlen(key) + 20;
  name = (char *)ckalloc(len + 1);
  sprintf(name, "%s%s%x", obj->name, key, world);
  return name;
}
