/*
 *=============================================================================
 *                                  tSippObj.c
 *-----------------------------------------------------------------------------
 * Tcl commands to manage SIPP objects.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1993 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippObj.c,v 5.0 1994/09/05 01:23:18 markd Rel $
 *=============================================================================
 */

#include "tSippInt.h"

/*
 * Internal function prototypes.
 */
typedef void (*rotateFunc_pt) _ANSI_ARGS_((Object  *objectPtr,
                                           double   angle));

static void
ObjectHandleCleanup _ANSI_ARGS_((tSippGlob_pt   tSippGlobPtr));

static void
UnrefAllSubObjects _ANSI_ARGS_((Object *objectPtr));

static Object *
ObjectHandleCmdSetup _ANSI_ARGS_((tSippGlob_pt   tSippGlobPtr,
                                  int            argc,
                                  char         **argv));

static bool
CheckSubObjAdd _ANSI_ARGS_((Object  *objectPtr,
                            Object  *subObjectPtr));

static bool
ObjectAxisRotate _ANSI_ARGS_((tSippGlob_pt   tSippGlobPtr,
                              int            argc,
                              char         **argv,
                              rotateFunc_pt  rotateFunc));

/*=============================================================================
 * TSippBindObjectToHandle --
 *   Assigns a handle to the specified object.
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the Tcl SIPP globals. The handle is
 *     returned in interp->result.
 *   o objectPtr (I) - A pointer to the object.
 *-----------------------------------------------------------------------------
 */
void
TSippBindObjectToHandle (tSippGlobPtr, objectPtr)
    tSippGlob_pt    tSippGlobPtr;
    Object         *objectPtr;
{
    Object  **objectEntryPtr;

    objectEntryPtr = (Object **)
        Tcl_HandleAlloc (tSippGlobPtr->objectTblPtr, 
                         tSippGlobPtr->interp->result);
    *objectEntryPtr = objectPtr;

}

/*=============================================================================
 * TSippObjectHandleToPtr --
 *   Utility procedure to convert an object handle to an object pointer.
 * For use of by functions outside of this module. Checks for magic handle
 * "WORLD".
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A object handle.
 * Returns:
 *   A pointer to the object, or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
Object *
TSippObjectHandleToPtr (tSippGlobPtr, handle)
    tSippGlob_pt    tSippGlobPtr;
    char           *handle;
{
    Object **objectEntryPtr;

    if ((handle [0] == 'W') && (STREQU (handle, "WORLD")))
        return sipp_world;

    objectEntryPtr = (Object **)
        Tcl_HandleXlate (tSippGlobPtr->interp, 
                         tSippGlobPtr->objectTblPtr, handle);
    if (objectEntryPtr == NULL)
        return NULL;
    return *objectEntryPtr;

}

/*=============================================================================
 * ObjectHandleCleanup --
 *    Unreference all object handles that are defined.  This deletes all
 * objects except those that are sub-objects of the WORLD object.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static void
ObjectHandleCleanup (tSippGlobPtr)
    tSippGlob_pt   tSippGlobPtr;
{
    int      walkKey = -1;
    Object **objectEntryPtr;

    while (TRUE) {
        objectEntryPtr = Tcl_HandleWalk (tSippGlobPtr->objectTblPtr, &walkKey);
        if (objectEntryPtr == NULL)
            break;

        object_unref (*objectEntryPtr);
        Tcl_HandleFree (tSippGlobPtr->objectTblPtr, objectEntryPtr);
    }

}

/*=============================================================================
 * UnrefAllSubObjects --
 *    Recursively subtract and unreference all sub-objects of the specified
 * object.
 *
 * Parameters:
 *   o objectPtr (I) - A pointer to the object whose sub-objects are to be
 *     unreferenced.  This object is not unreferenced.
 *-----------------------------------------------------------------------------
 */
static void
UnrefAllSubObjects (objectPtr)
    Object *objectPtr;
{
    Object *subObjectPtr;

    while (objectPtr->num_sub_objs > 0) {
        subObjectPtr = objectPtr->sub_objs [0];
        object_sub_subobj (objectPtr, subObjectPtr);
    }

}

/*=============================================================================
 * ObjectHandleCmdSetup --
 *    Utility procedure for the commands that take a single argument of an
 * object handle.  Validates argv and retrieves the handle table entry.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o argc, argv (I) - Command argument vector.
 * Returns:
 *   A pointer to the object associated with the handle or NULL and an error
 *   in tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static Object *
ObjectHandleCmdSetup (tSippGlobPtr, argc, argv)
    tSippGlob_pt   tSippGlobPtr;
    int            argc;
    char         **argv;
{

    if (tSippGlobPtr->rendering) {
        TSippNotWhileRendering (tSippGlobPtr->interp);
        return NULL;
    }

    if (argc != 2) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " objecthandle", (char *) NULL);
        return NULL;
    }
    return TSippObjectHandleToPtr (tSippGlobPtr, argv [1]);

}

/*=============================================================================
 * SippObjectCreate --
 *   Implements the command:
 *     SippObjectCreate
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectCreate (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (((tSippGlob_pt) clientData)->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 1) {
        Tcl_AppendResult (interp, "wrong # args: ", argv[0], (char *) NULL);
        return TCL_ERROR;
    }                     

    TSippBindObjectToHandle ((tSippGlob_pt) clientData, object_create ());
    return TCL_OK;

}

/*=============================================================================
 * SippObjectUnref --
 *   Implements the command:
 *     SippObjectUnref objectlist|ALL
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectUnref (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    int             idx;
    handleList_t    objectList;
    handleList_t    objectEntryList;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " objectlist|ALL", (char *) NULL);
        return TCL_ERROR;
    }
    
    if (STREQU (argv [1], "ALL")) {
        ObjectHandleCleanup (tSippGlobPtr);
        return TCL_OK;
    }

    if (!TSippHandleListConvert (tSippGlobPtr, tSippGlobPtr->objectTblPtr,
                                 argv [1], &objectList, &objectEntryList,
                                 NULL))
        return TCL_ERROR;

    for (idx = 0; idx < objectList.len; idx++) {
        Object *objectPtr = (Object *) objectList.ptr [idx];

        object_unref (objectPtr);
        Tcl_HandleFree (tSippGlobPtr->objectTblPtr, objectEntryList.ptr [idx]);
    }

    TSippHandleListFree (&objectList);
    TSippHandleListFree (&objectEntryList);
    return TCL_OK;

}

/*=============================================================================
 * SippObjectInstance --
 *   Implements the command:
 *     SippObjectInstance objecthandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectInstance (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    Object  *objectPtr;

    objectPtr = ObjectHandleCmdSetup ((tSippGlob_pt) clientData, argc, argv);
    if (objectPtr == NULL)
        return TCL_ERROR;    

    TSippBindObjectToHandle ((tSippGlob_pt) clientData,
                             object_instance (objectPtr));
    return TCL_OK;

}

/*=============================================================================
 * SippObjectDup --
 *   Implements the command:
 *     SippObjectDup objecthandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectDup (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    Object  *objectPtr;

    objectPtr = ObjectHandleCmdSetup ((tSippGlob_pt) clientData, argc, argv);
    if (objectPtr == NULL)
        return TCL_ERROR;    

    TSippBindObjectToHandle ((tSippGlob_pt) clientData, 
                              object_dup (objectPtr));
    return TCL_OK;

}

/*=============================================================================
 * SippObjectDeepDup --
 *   Implements the command:
 *     SippObjectDeepDup objecthandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectDeepDup (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    Object  *objectPtr;

    objectPtr = ObjectHandleCmdSetup ((tSippGlob_pt) clientData, argc, argv);
    if (objectPtr == NULL)
        return TCL_ERROR;    

    TSippBindObjectToHandle ((tSippGlob_pt) clientData,
                              object_deep_dup (objectPtr));
    return TCL_OK;

}

/*=============================================================================
 * SippObjectGetTransf --
 *   Implements the command:
 *     SippObjectGetTransf objecthandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectGetTransf (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    Object         *objectPtr;
    Transf_mat      matrix;

    objectPtr = ObjectHandleCmdSetup ((tSippGlob_pt) clientData, argc, argv);
    if (objectPtr == NULL)
        return TCL_ERROR;    

    object_get_transf (objectPtr, &matrix);
    Tcl_SetResult (interp,
                   TSippFormatMatrix (tSippGlobPtr,
                                      &matrix),
                   TCL_DYNAMIC);

    return TCL_OK;

}

/*=============================================================================
 * SippObjectSetTransf --
 *   Implements the command:
 *     SippObjectSetTransf objectHandle matrix
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectSetTransf (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    Transf_mat      matrix;
    Object         *objectPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " objectHandle matrix", (char *) NULL);
        return TCL_ERROR;
    }
    objectPtr = TSippObjectHandleToPtr ((tSippGlob_pt) clientData, 
                                         argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    if (!TSippConvertMatrix (tSippGlobPtr, argv [2], &matrix))
        return TCL_ERROR;

    object_set_transf (objectPtr, &matrix);

    return TCL_OK;

}

/*=============================================================================
 * SippObjectClearTransf --
 *   Implements the command:
 *     SippObjectClearTransf objectHandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectClearTransf (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    Object        *objectPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " objectHandle", (char *) NULL);
        return TCL_ERROR;
    }                     
    objectPtr = TSippObjectHandleToPtr ((tSippGlob_pt) clientData, 
                                         argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    object_clear_transf (objectPtr);

    return TCL_OK;

}

/*=============================================================================
 * SippObjectTransform --
 *   Implements the command:
 *     SippObjectTransform objectHandle matrix
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectTransform (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    Transf_mat      matrix;
    Object        *objectPtr;
    
    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " objectHandle matrix", (char *) NULL);
        return TCL_ERROR;
    }
    objectPtr = TSippObjectHandleToPtr ((tSippGlob_pt) clientData, 
                                         argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    if (!TSippConvertMatrix (tSippGlobPtr, argv [2], &matrix))
        return TCL_ERROR;

    object_set_transf (objectPtr, &matrix);

    return TCL_OK;

}

/*=============================================================================
 * SippObjectAddSurface --
 *   Implements the command:
 *     SippObjectAddSurface objectHandle surfacelist
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectAddSurface (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt   tSippGlobPtr = (tSippGlob_pt) clientData;
    Object         *objectPtr;
    int             idx, surfIdx;
    handleList_t    surfaceList;
    char          **handleArgv;
    
    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " objectHandle surfacelist", (char *) NULL);
        return TCL_ERROR;
    }                     
    objectPtr = TSippObjectHandleToPtr ((tSippGlob_pt) clientData, 
                                         argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    if (!TSippHandleListConvert (tSippGlobPtr, tSippGlobPtr->surfaceTblPtr,
                                 argv [2], &surfaceList, NULL, &handleArgv))
        return TCL_ERROR;

    /*
     * Check to make sure they are not already surfaces of the object.
     */
    for (idx = 0; idx < surfaceList.len; idx++) {
        for (surfIdx = 0; surfIdx < objectPtr->num_surfaces; surfIdx++) {
            if (surfaceList.ptr [idx] == objectPtr->surfaces [surfIdx]) {
                Tcl_AppendResult (interp, handleArgv [idx],
                                  " is already a surface of ", argv [1],
                                  (char *) NULL);
                goto errorExit;
            }
        }
    }

    /*
     * Add the surfaces.
     */
    for (idx = 0; idx < surfaceList.len; idx++)
        object_add_surface (objectPtr, (Surface *) (surfaceList.ptr [idx]));

    TSippHandleListFree (&surfaceList);
    sfree (handleArgv);
    return TCL_OK;

  errorExit:
    TSippHandleListFree (&surfaceList);
    sfree (handleArgv);
    return TCL_ERROR;

}

/*=============================================================================
 * SippObjectSurfaceCreate --
 *   Implements the command:
 *     SippObjectSurfaceCreate objecthandle shaderhandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectSurfaceCreate (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    Object         *objectPtr;
    Surface        *surfacePtr;
    Shader         *shaderPtr;
    void           *surfDescPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " objecthandle shaderhandle", (char *) NULL);
        return TCL_ERROR;
    }                     

    objectPtr = TSippObjectHandleToPtr (tSippGlobPtr, argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;

    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [2],
                                        &surfDescPtr);
    if (shaderPtr == NULL)
        return TCL_ERROR;

    surfacePtr = surface_create (surfDescPtr, shaderPtr);
    if (surfacePtr == NULL) {
        Tcl_AppendResult (interp, "the polygon stack is empty",
                         (char *) NULL);
        return TCL_ERROR;
    }

    object_add_surface (objectPtr, surfacePtr);
    surface_unref (surfacePtr);

    return TCL_OK;

}

/*=============================================================================
 * SippObjectSubSurface --
 *   Implements the command:
 *     SippObjectSubSurface objecthandle surfacelist
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectSubSurface (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt   tSippGlobPtr = (tSippGlob_pt) clientData;
    Object        *objectPtr;
    int            idx, surfIdx;
    handleList_t   surfaceList;
    char          **handleArgv;
    
    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " objectHandle surfacelist", (char *) NULL);
        return TCL_ERROR;
    }                     
    objectPtr = TSippObjectHandleToPtr ((tSippGlob_pt) clientData, 
                                         argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    if (!TSippHandleListConvert (tSippGlobPtr, tSippGlobPtr->surfaceTblPtr,
                                 argv [2], &surfaceList, NULL, &handleArgv))
        return TCL_ERROR;

    /*
     * Check to make sure they are surfaces of the object.
     */
    for (idx = 0; idx < surfaceList.len; idx++) {
        for (surfIdx = 0; surfIdx < objectPtr->num_surfaces; surfIdx++) {
            if (surfaceList.ptr [idx] == objectPtr->surfaces [surfIdx])
                break;
        }
        if (surfIdx == objectPtr->num_surfaces) {
            Tcl_AppendResult (interp, handleArgv [idx],
                              " is not a surface of ", argv [1],
                              (char *) NULL);
            goto errorExit;
        }
    }

    /*
     * Subtract the surfaces.
     */
    for (idx = 0; idx < surfaceList.len; idx++)
       object_sub_surface (objectPtr, (Surface *) (surfaceList.ptr [idx]));

    TSippHandleListFree (&surfaceList);
    sfree (handleArgv);
    return TCL_OK;

  errorExit:
    TSippHandleListFree (&surfaceList);
    sfree (handleArgv);
    return TCL_ERROR;

}

/*=============================================================================
 * CheckSubObjAdd --
 *    Recursively that a object's new parent is not one of it's decendents.
 * This would lead to a circular object list.
 *
 * Parameters:
 *   o objectPtr (I) - Pointer to the object to add the object to.
 *   o subObjectPtr (I) - Pointer to the subobject.
 * Returns:
 *   TRUE if all is ok, FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
static bool
CheckSubObjAdd (objectPtr, subObjectPtr)
    Object  *objectPtr;
    Object  *subObjectPtr;
{
    register int     idx, numSubObjs = subObjectPtr->num_sub_objs; 
    register Object  **subObjTblPtr  = subObjectPtr->sub_objs;

    if (subObjectPtr == objectPtr)
        return FALSE;

    for (idx = 0; idx < numSubObjs; idx++) {
        if (!CheckSubObjAdd (objectPtr, subObjTblPtr [idx]))
            return FALSE;
    }
    return TRUE;

}

/*=============================================================================
 * SippObjectAddSubobj --
 *   Implements the command:
 *     SippObjectAddSubobj [-flag] objectHandle subobjlist
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectAddSubobj (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt   tSippGlobPtr = (tSippGlob_pt) clientData;
    Object        *objectPtr, *subObjectPtr;
    int            idx, objIdx;
    handleList_t   subObjList;
    char         **handleArgv;
    int            nextArg = 1;
    bool           check = TRUE;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc < 3 || argc > 4) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " [-flag] objectHandle subobjlist", (char *) NULL);
        return TCL_ERROR;
    }                     

    if (argc == 4) {
        if (STREQU (argv [1], "-check"))
            check = TRUE;
        else if (STREQU (argv [1], "-nocheck"))
            check = FALSE;
        else {
            Tcl_AppendResult (interp, "expected one of \"-check\" or ",
                              "\"-nocheck\", got \"", argv [1], "\"",
                              (char *) NULL);
            return TCL_ERROR;
        }
        nextArg = 2;
    }

    objectPtr = TSippObjectHandleToPtr ((tSippGlob_pt) clientData,
                                        argv [nextArg]);
    if (objectPtr == NULL)
        return TCL_ERROR;

    if (!TSippHandleListConvert (tSippGlobPtr, tSippGlobPtr->objectTblPtr,
                                 argv [nextArg + 1], &subObjList, NULL,
                                 &handleArgv))
        return TCL_ERROR;

    /*
     * Check to make sure none are already subobjects of the object.  If
     * checking is enabled, recursively check for a circular list (if we are
     * not adding to WORLD.
     */
    if (objectPtr == sipp_world)
        check = FALSE;

    for (idx = 0; idx < subObjList.len; idx++) {
        subObjectPtr = (Object *) subObjList.ptr [idx];

        for (objIdx = 0; objIdx < objectPtr->num_sub_objs; objIdx++) {
            if (subObjectPtr == objectPtr->sub_objs [objIdx]) {
                Tcl_AppendResult (interp, handleArgv [idx],
                                  " is already a subobject of ", argv [1],
                                  (char *) NULL);
                goto errorExit;
            }
        }
        if (check && !CheckSubObjAdd (objectPtr, subObjectPtr)) {
            Tcl_AppendResult (interp, argv [1], " is the same as or a ",
                              "decendent of ", handleArgv [idx],
                              " - would cause a circular object tree",
                              (char *) NULL);
            goto errorExit;
        }
    }

    /*
     * Add the subobjects.
     */
    for (idx = 0; idx < subObjList.len; idx++)
        object_add_subobj (objectPtr, (Object *) (subObjList.ptr [idx]));

    TSippHandleListFree (&subObjList);
    sfree (handleArgv);
    return TCL_OK;

  errorExit:
    TSippHandleListFree (&subObjList);
    sfree (handleArgv);
    return TCL_ERROR;

}

/*=============================================================================
 * SippObjectSubSubobj --
 *   Implements the command:
 *     SippObjectSubSubobj objectHandle subobjlist|ALL
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectSubSubobj (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    Object         *objectPtr;
    int             idx, objIdx;
    handleList_t    subObjList;
    char          **handleArgv;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " objectHandle subobjlist|ALL", (char *) NULL);
        return TCL_ERROR;
    }                     

    objectPtr = TSippObjectHandleToPtr ((tSippGlob_pt) clientData, argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;

    if (STREQU (argv [2], "ALL")) {
        UnrefAllSubObjects (objectPtr);
        return TCL_OK;
    }

    if (!TSippHandleListConvert (tSippGlobPtr, tSippGlobPtr->objectTblPtr,
                                 argv [2], &subObjList, NULL, &handleArgv))
        return TCL_ERROR;

    /*
     * Check to make sure they are subobjects of the object.
     */
    for (idx = 0; idx < subObjList.len; idx++) {
        for (objIdx = 0; objIdx < objectPtr->num_sub_objs; objIdx++) {
            if (subObjList.ptr [idx] == objectPtr->sub_objs [objIdx])
                break;
        }
        if (objIdx == objectPtr->num_sub_objs) {
            Tcl_AppendResult (interp, handleArgv [idx],
                              " is not a subobject of ", argv [1],
                              (char *) NULL);
            goto errorExit;
        }
    }

    /*
     * Subtract the subobjects.
     */
    for (idx = 0; idx < subObjList.len; idx++)
        object_sub_subobj (objectPtr, (Object *) (subObjList.ptr [idx]));

    TSippHandleListFree (&subObjList);
    sfree (handleArgv);
    return TCL_OK;

  errorExit:
    TSippHandleListFree (&subObjList);
    sfree (handleArgv);
    return TCL_ERROR;

}

/*=============================================================================
 * ObjectAxisRotate --
 *   Process parameters for the commands to rotate an object around an axis
 * and call the function to do the rotation. These commands have the
 * arguments: objecthandle angle
 *
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the Tcl SIPP globals.
 *   o argc, argv (I) - The arguments to the command.
 *   o rotateFunc (I) - The rotate function that is to be called.
 * Returns:
 *   TRUE if all is OK,  FALSE if there is an error.
 *-----------------------------------------------------------------------------
 */
static bool
ObjectAxisRotate (tSippGlobPtr, argc, argv, rotateFunc)
    tSippGlob_pt   tSippGlobPtr;
    int            argc;
    char         **argv;
    rotateFunc_pt  rotateFunc;
{
    Object  *objectPtr;
    double  angle;

    if (tSippGlobPtr->rendering) {
        TSippNotWhileRendering (tSippGlobPtr->interp);
        return FALSE;
    }

    if (argc != 3) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " objectHandle angle", (char *) NULL);
        return FALSE;
    }
    objectPtr = TSippObjectHandleToPtr (tSippGlobPtr, argv [1]);
    if (objectPtr == NULL)
        return FALSE;

    if (!TSippConvertAngleRad (tSippGlobPtr, argv [2], &angle))
        return FALSE;
    
    (*rotateFunc) (objectPtr, angle);
    return TRUE;

}

/*=============================================================================
 * SippObjectRotateX --
 *   Implements the command:
 *     SippObjectRotateX objectHandle angle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectRotateX (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (ObjectAxisRotate ((tSippGlob_pt) clientData, argc, argv, object_rot_x))
        return TCL_OK;
    else
        return TCL_ERROR;;

}

/*=============================================================================
 * SippObjectRotateY --
 *   Implements the command:
 *     SippObjectRotateY objectHandle angle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectRotateY (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (ObjectAxisRotate ((tSippGlob_pt) clientData, argc, argv, object_rot_y))
        return TCL_OK;
    else
        return TCL_ERROR;;

}

/*=============================================================================
 * SippObjectRotateZ --
 *   Implements the command:
 *     SippObjectRotateZ objectHandle angle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectRotateZ (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    if (ObjectAxisRotate ((tSippGlob_pt) clientData, argc, argv, object_rot_z))
        return TCL_OK;
    else
        return TCL_ERROR;;

}

/*=============================================================================
 * SippObjectRotate --
 *   Implements the command:
 *     SippObjectRotate objectHandle point vector angle
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectRotate (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt   tSippGlobPtr = (tSippGlob_pt) clientData;
    Object        *objectPtr;
    Vector         point, vector;
    double         angle;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 5) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " objectHandle point vector angle", (char *) NULL);
        return TCL_ERROR;
    }                     
    objectPtr =  TSippObjectHandleToPtr (tSippGlobPtr, argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [2], &point))
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [3], &vector))
        return TCL_ERROR;
    if (!TSippConvertAngleRad (tSippGlobPtr, argv [4], &angle))
        return TCL_ERROR;

    object_rot (objectPtr, &point, &vector, angle);

    return TCL_OK;

}

/*=============================================================================
 * SippObjectScale --
 *   Implements the command:
 *     SippObjectScale objectHandle factor|{xfactor yfactor zfactor}
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectScale (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt   tSippGlobPtr = (tSippGlob_pt) clientData;
    Object        *objectPtr;
    Vector         scale;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " objectHandle factor|{xfactor yfactor zfactor}",
                          (char *) NULL);
        return TCL_ERROR;
    }                     

    objectPtr =  TSippObjectHandleToPtr (tSippGlobPtr, argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    /*
     * Scale can be a list or a single factor.  If it contains any white space
     * assume its a list.
     */
    if (strpbrk (argv [2], " \f\t\n\r\v") == NULL) {
        if (Tcl_GetDouble (interp, argv [2], &scale.x) != TCL_OK)
            return TCL_ERROR;
        scale.y = scale.z = scale.x;
    } else {
        if (!TSippConvertVertex (tSippGlobPtr, argv [2], &scale))
            return TCL_ERROR;
    } 

    object_scale (objectPtr, scale.x, scale.y, scale.z);

    return TCL_OK;

}

/*=============================================================================
 * SippObjectMove --
 *   Implements the command:
 *     SippObjectMove objectHandle {xdist ydist zdist}
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippObjectMove (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt   tSippGlobPtr = (tSippGlob_pt) clientData;
    Object        *objectPtr;
    Vector         translation;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " objectHandle {xdist ydist zdist}",
                          (char *) NULL);
        return TCL_ERROR;
    }                     
    objectPtr =  TSippObjectHandleToPtr (tSippGlobPtr, argv [1]);
    if (objectPtr == NULL)
        return TCL_ERROR;
    if (!TSippConvertVertex (tSippGlobPtr, argv [2], &translation))
        return TCL_ERROR;

    object_move (objectPtr, translation.x, translation.y, translation.z);

    return TCL_OK;

}

/*=============================================================================
 * TSippObjectInit --
 *   Initialized the object commands, including creating the object table.
 *
 * Parameters:
 *   o tSippGlobP (I) - Pointer to the top level global data structure.
 *-----------------------------------------------------------------------------
 */
void
TSippObjectInit (tSippGlobPtr)
    tSippGlob_pt    tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippObjectCreate",         (Tcl_CmdProc *) SippObjectCreate},
        {"SippObjectUnref",          (Tcl_CmdProc *) SippObjectUnref},
        {"SippObjectInstance",       (Tcl_CmdProc *) SippObjectInstance},
        {"SippObjectDup",            (Tcl_CmdProc *) SippObjectDup},
        {"SippObjectDeepDup",        (Tcl_CmdProc *) SippObjectDeepDup},
        {"SippObjectGetTransf",      (Tcl_CmdProc *) SippObjectGetTransf},
        {"SippObjectSetTransf",      (Tcl_CmdProc *) SippObjectSetTransf},
        {"SippObjectClearTransf",    (Tcl_CmdProc *) SippObjectClearTransf},
        {"SippObjectTransform",      (Tcl_CmdProc *) SippObjectTransform},
        {"SippObjectUnref",          (Tcl_CmdProc *) SippObjectUnref},
        {"SippObjectAddSurface",     (Tcl_CmdProc *) SippObjectAddSurface},
        {"SippObjectSurfaceCreate",  (Tcl_CmdProc *) SippObjectSurfaceCreate},
        {"SippObjectSubSurface",     (Tcl_CmdProc *) SippObjectSubSurface},
        {"SippObjectAddSubobj",      (Tcl_CmdProc *) SippObjectAddSubobj},
        {"SippObjectSubSubobj",      (Tcl_CmdProc *) SippObjectSubSubobj},
        {"SippObjectRotateX",        (Tcl_CmdProc *) SippObjectRotateX},
        {"SippObjectRotateY",        (Tcl_CmdProc *) SippObjectRotateY},
        {"SippObjectRotateZ",        (Tcl_CmdProc *) SippObjectRotateZ},
        {"SippObjectRotate",         (Tcl_CmdProc *) SippObjectRotate},
        {"SippObjectScale",          (Tcl_CmdProc *) SippObjectScale},
        {"SippObjectMove",           (Tcl_CmdProc *) SippObjectMove},
        {NULL,                       NULL}
    };

    tSippGlobPtr->objectTblPtr = 
        Tcl_HandleTblInit ("object", sizeof (Object *), 24);

    TSippInitCmds (tSippGlobPtr, cmdTable);

}

/*=============================================================================
 * TSippObjectCleanUp --
 *   Cleanup the object table and release all associated resources.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippObjectCleanUp (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    UnrefAllSubObjects (sipp_world);

    ObjectHandleCleanup (tSippGlobPtr);

    Tcl_HandleTblRelease (tSippGlobPtr->objectTblPtr);
    tSippGlobPtr->objectTblPtr = NULL;

}


