/* stclsh.h
 * Fashioned by Ian Wilkinson.
 * Inception was Wed Jun 22 20:27:43 BST 1994.
 *
 * My History:
 *   Borrows structure from swish.
 *
 *   Tue Aug  9 19:21:35 BST 1994
 *   Changes made for Safe-Tcl 1.x (Nathaniel's recent
 *   changes).
 */

#ifndef NO_RLIMITS
#include <sys/time.h>
#include <sys/resource.h>
#endif
#include <tcl.h>

#ifdef	SYS5
#define	index	strchr
#define	rindex	strrchr
#endif

extern char *getenv(), *index(), *rindex ();
extern int  swish_ftruncate();

extern char *SafeTcl_message;
extern char *InitContents;

Tcl_Interp *unrestricted_interp;	/* Unrestricted interpreter */
Tcl_Interp *restricted_interp;		/* Restricted interpreter */

char *MsgBodyFile = NULL;
char *InitEval = NULL;
char *Mailbox = NULL;

static char *DefaultStclshAppName = "stclsh";
static Tcl_DString AppName;

static int nointerface = 0;

int runsafely = 0;
int messaging = 0;
int generic = 1; /* always in stclsh */

extern int main();
int *tclDummyMainPtr = (int *)main;

#define GET_OPTION(optVal) \
    do { \
	optVal = (char *)malloc(strlen(*++argv) + 1); \
	if (optVal == NULL) { \
	    fprintf(stderr, "%s: Problem with allocating memory.\n", Tcl_DStringValue(&AppName)); \
	    exit(1); \
	} \
	strcpy(optVal, *argv); \
	argc--; \
    } while (0)

InitTclLibs( interp )
    Tcl_Interp *interp;
{
    if (Tcl_VarEval(interp, InitContents, 0, (char **) NULL) != TCL_OK) {
	return(TCL_ERROR);
    }
    return(TCL_OK);
}

InitLimits( interp )
    Tcl_Interp *interp; /* Unrestricted interpreter */
{
#ifndef NO_RLIMITS
    struct rlimit rlp;
    char *s;

    getrlimit(RLIMIT_CPU, &rlp);
    rlp.rlim_cur = 0;
    s = Tcl_GetVar(interp, "swish_CPULimit", TCL_GLOBAL_ONLY);
    if (s) rlp.rlim_cur = atoi(s);
    if (rlp.rlim_cur <= 0) rlp.rlim_cur = 90;
    setrlimit(RLIMIT_CPU, &rlp);
#endif
}

int
Tcl_NoInterface()
{
    return nointerface;
}

int
Tcl_AppInit( interp )
    Tcl_Interp *interp;		/* Interpreter for application. */
{
    int argc, result;
    char *fileName = NULL, *name = NULL, buf[20], *args, **argv;
    char *tclVarVal;

    if (InitTclLibs(interp) == TCL_ERROR) {
	return TCL_ERROR;
    }
#ifdef USE_TCLX
    TclXCmd_Init (interp);
#endif
    Tcl_CreateCommand (interp, "ftruncate", swish_ftruncate,
                       (ClientData) 0, (void (*) ()) 0);
    Tcl_DStringInit(&AppName);
    if ((tclVarVal = Tcl_GetVar(interp, "argv0", TCL_GLOBAL_ONLY)) == NULL) {
	Tcl_DStringAppend(&AppName, DefaultStclshAppName, -1); 
    } else {
	char   *myname;

	if (!(myname = rindex (tclVarVal, '/')) || !*myname++)
	    myname = tclVarVal;
	Tcl_DStringAppend(&AppName, myname, -1); 
    }
    tclVarVal = Tcl_GetVar(interp, "argc", TCL_GLOBAL_ONLY);
    if (tclVarVal == NULL) {
	return TCL_ERROR;
    }
    if (Tcl_GetInt(interp, tclVarVal, &argc) != TCL_OK) {
	return TCL_ERROR;
    }
    Tcl_SplitList(interp, Tcl_GetVar(interp, "argv", TCL_GLOBAL_ONLY), &argc, &argv);
    for (; argc > 0; argc--, argv++) {
	if ((*argv)[0] == '-') {
	    if (strcmp(*argv, "-file") == 0) {
		if (argc > 0) {
		    GET_OPTION(fileName);
		} else {
		    fprintf(stderr,
			    "%s: Expecting a file from which to read commands.\n",
			    Tcl_DStringValue(&AppName));
		    exit(1);
		}
	    } else if (strcmp(*argv, "-name") == 0) {
		if (argc > 0) {
		    GET_OPTION(name);
		} else {
		    fprintf(stderr,
			    "%s: Expecting a name to call the application.\n",
			    Tcl_DStringValue(&AppName));
		    exit(1);
		}
	    } else if (strcmp(*argv, "-nointerface") == 0) {
		nointerface = 1;
	    } else if (strcmp(*argv, "-safe") == 0) {
		runsafely = 1;
	    } else if (strcmp(*argv, "-messaging") == 0) {
		messaging = 1;
	    } else if (strcmp(*argv, "-generic") == 0) {
		generic = 1;
	    } else if (strcmp(*argv, "-mailbox") == 0) {
		if (argc > 0) {
		    GET_OPTION(Mailbox);
		} else {
		    fprintf(stderr, "%s: Expecting a file indicating the location of the user's mailbox.\n", Tcl_DStringValue(&AppName));
		    exit(1);
		}
	    } else if (strcmp(*argv, "-messagebody") == 0
		           || strcmp(*argv, "-messagefile") == 0) {
		if (argc > 0) {
		    GET_OPTION(MsgBodyFile);
		} else {
		    fprintf(stderr, "%s: Expecting a file indicating the location of a mail message.\n", Tcl_DStringValue(&AppName));
		    exit(1);
		}
	    } else if (strcmp(*argv, "-initeval") == 0) {
		if (argc > 0) {
		    GET_OPTION(InitEval);
		} else {
		    fprintf(stderr, "%s: Expecting a Safe-Tcl fragment.\n", Tcl_DStringValue(&AppName));
		    exit(1);
		}
	    } else {
		fprintf (stderr, "%s: Unrecognized option '%s'.\n", *argv);
		exit (1);
	    }
	    continue;
	}
	break;
    }
    args = Tcl_Merge(argc, argv);
    Tcl_SetVar(interp, "argv", args, TCL_GLOBAL_ONLY);
    ckfree(args);
    sprintf(buf, "%d", argc);
    Tcl_SetVar(interp, "argc", buf, TCL_GLOBAL_ONLY);
    if (name == NULL && fileName != NULL) {
	if (!(name = rindex (fileName, '/')) || !*name++)
	    name = fileName;
	Tcl_DStringInit(&AppName);
	Tcl_DStringAppend (&AppName, name, -1);
    }
    
    if (Tcl_NoInterface()) {
	generic = 1;
    }
    
    if (runsafely) {
	restricted_interp = interp;
	unrestricted_interp = (Tcl_Interp *)Tcl_CreateInterp();
#ifdef TCL_MEM_DEBUG
        Tcl_InitMemory(unrestricted_interp);
#endif
        Tcl_SetVar(restricted_interp, "SafeTcl_evaluation_time",
	    Tcl_NoInterface () ? "delivery" : "activation",
	    TCL_GLOBAL_ONLY);
    } else {
	unrestricted_interp = interp; /* This variable should always be set */
	restricted_interp = NULL;
	if (Tcl_NoInterface() && messaging) {
	    InitReceipt(unrestricted_interp);
	    Tcl_SetVar(unrestricted_interp, "SafeTcl_evaluation_time", "receipt", TCL_GLOBAL_ONLY);
	}
    }
    
    if (InitEval && Tcl_GlobalEval(unrestricted_interp, InitEval) != TCL_OK) {
	return TCL_ERROR;
    }
    if (runsafely && Tcl_MakeInterpreterSafe(restricted_interp, unrestricted_interp, 0, generic) != TCL_OK) {
	return TCL_ERROR;
    }
    if (InitMessageBody(interp, Tcl_DStringValue(&AppName), MsgBodyFile)) {
	return TCL_ERROR;
    }
    if (messaging) {
	if (restricted_interp) {
	    InitMessaging(restricted_interp, Tcl_DStringValue(&AppName), Mailbox, 0);
        }
	InitMessaging(unrestricted_interp, Tcl_DStringValue(&AppName), Mailbox, 1);
    }
    if (Tcl_NoInterface()) {
        RemoveInterface(interp, interp);
    }
    if (runsafely && Tcl_Eval(unrestricted_interp, "init_safe_tcl") != TCL_OK) {
        char *m;
        /* Final initialization, once everything is defined */
	
        /* Propogate the error to the restricted interpreter for reporting */
        Tcl_SetResult(restricted_interp, unrestricted_interp->result, TCL_VOLATILE);
        m = Tcl_GetVar(unrestricted_interp, "errorInfo", TCL_GLOBAL_ONLY);
        Tcl_AddErrorInfo(restricted_interp, m ? m : "<no error info>");
        m = Tcl_GetVar(unrestricted_interp, "errorCode", TCL_GLOBAL_ONLY);
        if (m) Tcl_SetErrorCode(restricted_interp, m, 0);
        return TCL_ERROR;
    }
    InitLimits(unrestricted_interp);

    if (fileName != NULL) {
	int code = Tcl_EvalFile (interp, fileName);

	if (code != TCL_OK)
	    return code;

	if (Tcl_NoInterface()) {
	    Tcl_Eval (interp, "exit");
	    exit (1);
	}
    }

    return TCL_OK;
}

#ifdef	USE_TCLX
#include <errno.h>

/*
 * Declarations for various library procedures and variables (don't want
 * to include tclUnix.h here, because people might copy this file out of
 * the Tcl source directory to make their own modified versions).
 */

extern int		errno;
extern int		isatty _ANSI_ARGS_((int fd));

static Tcl_Interp *interp;	/* Interpreter for application. */
static Tcl_DString command;	/* Used to buffer incomplete commands being
				 * read from stdin. */
char *tcl_RcFileName = NULL;	/* Name of a user-specific startup script
				 * to source if the application is being run
				 * interactively (e.g. "~/.tclshrc").  Set
				 * by Tcl_AppInit.  NULL means don't source
				 * anything ever. */
#ifdef TCL_MEM_DEBUG
static char dumpFile[100];	/* Records where to dump memory allocation
				 * information. */
static int quitFlag = 0;	/* 1 means the "checkmem" command was
				 * invoked, so the application should quit
				 * and dump memory allocation information. */
#endif

/*
 * Forward references for procedures defined later in this file:
 */

static int		CheckmemCmd _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, int argc, char *argv[]));

/*
 *----------------------------------------------------------------------
 *
 * main --
 *
 *	This is the main program for a Tcl-based shell that reads
 *	Tcl commands from standard input.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Can be almost arbitrary, depending on what the Tcl commands do.
 *
 *----------------------------------------------------------------------
 */

int
main(argc, argv)
    int argc;				/* Number of arguments. */
    char **argv;			/* Array of argument strings. */
{
    char buffer[1000], *cmd, *args, *fileName;
    int code, gotPartial, tty;
    int exitCode = 0;

    interp = Tcl_CreateInterp();
#ifdef TCL_MEM_DEBUG
    Tcl_InitMemory(interp);
    Tcl_CreateCommand(interp, "checkmem", CheckmemCmd, (ClientData) 0,
	    (Tcl_CmdDeleteProc *) NULL);
#endif

    /*
     * Make command-line arguments available in the Tcl variables "argc"
     * and "argv".  If the first argument doesn't start with a "-" then
     * strip it off and use it as the name of a script file to process.
     */

    fileName = NULL;
    if ((argc > 1) && (argv[1][0] != '-')) {
	fileName = argv[1];
	argc--;
	argv++;
    }
    args = Tcl_Merge(argc-1, argv+1);
    Tcl_SetVar(interp, "argv", args, TCL_GLOBAL_ONLY);
    ckfree(args);
    sprintf(buffer, "%d", argc-1);
    Tcl_SetVar(interp, "argc", buffer, TCL_GLOBAL_ONLY);
    Tcl_SetVar(interp, "argv0", (fileName != NULL) ? fileName : argv[0],
	    TCL_GLOBAL_ONLY);

    /*
     * Set the "tcl_interactive" variable.
     */

    tty = isatty(0);
    Tcl_SetVar(interp, "tcl_interactive",
	    ((fileName == NULL) && tty) ? "1" : "0", TCL_GLOBAL_ONLY);

    /*
     * Invoke application-specific initialization.
     */

    if (Tcl_AppInit(interp) != TCL_OK) {
	fprintf(stderr, "Tcl_AppInit failed: %s\n", interp->result);
    }

    /*
     * If a script file was specified then just source that file
     * and quit.
     */

    if (fileName != NULL) {
	code = Tcl_EvalFile(interp, fileName);
	if (code != TCL_OK) {
	    fprintf(stderr, "%s\n", interp->result);
	    exitCode = 1;
	}
	goto done;
    }

    /*
     * We're running interactively.  Source a user-specific startup
     * file if Tcl_AppInit specified one and if the file exists.
     */

    if (tcl_RcFileName != NULL) {
	Tcl_DString buffer;
	char *fullName;
	FILE *f;

	fullName = Tcl_TildeSubst(interp, tcl_RcFileName, &buffer);
	if (fullName == NULL) {
	    fprintf(stderr, "%s\n", interp->result);
	} else {
	    f = fopen(fullName, "r");
	    if (f != NULL) {
		code = Tcl_EvalFile(interp, fullName);
		if (code != TCL_OK) {
		    fprintf(stderr, "%s\n", interp->result);
		}
		fclose(f);
	    }
	}
	Tcl_DStringFree(&buffer);
    }

    /*
     * Process commands from stdin until there's an end-of-file.
     */

    gotPartial = 0;
    Tcl_DStringInit(&command);
    while (1) {
	clearerr(stdin);
	if (tty) {
	    char *promptCmd;

	    promptCmd = Tcl_GetVar(interp,
		gotPartial ? "tcl_prompt2" : "tcl_prompt1", TCL_GLOBAL_ONLY);
	    if (promptCmd == NULL) {
		defaultPrompt:
		if (!gotPartial) {
		    fputs("% ", stdout);
		}
	    } else {
		code = Tcl_Eval(interp, promptCmd);
		if (code != TCL_OK) {
		    fprintf(stderr, "%s\n", interp->result);
		    Tcl_AddErrorInfo(interp,
			    "\n    (script that generates prompt)");
		    goto defaultPrompt;
		}
	    }
	    fflush(stdout);
	}
	if (fgets(buffer, 1000, stdin) == NULL) {
	    if (ferror(stdin)) {
		if (errno == EINTR) {
		    if (tcl_AsyncReady) {
			(void) Tcl_AsyncInvoke((Tcl_Interp *) NULL, 0);
		    }
		    clearerr(stdin);
		} else {
		    goto done;
		}
	    } else {
		if (!gotPartial) {
		    goto done;
		}
	    }
	    buffer[0] = 0;
	}
	cmd = Tcl_DStringAppend(&command, buffer, -1);
	if ((buffer[0] != 0) && !Tcl_CommandComplete(cmd)) {
	    gotPartial = 1;
	    continue;
	}

	gotPartial = 0;
	code = Tcl_RecordAndEval(interp, cmd, 0);
	Tcl_DStringFree(&command);
	if (code != TCL_OK) {
	    fprintf(stderr, "%s\n", interp->result);
	} else if (tty && (*interp->result != 0)) {
	    printf("%s\n", interp->result);
	}
#ifdef TCL_MEM_DEBUG
	if (quitFlag) {
	    Tcl_DeleteInterp(interp);
	    Tcl_DumpActiveMemory(dumpFile);
	    exit(0);
	}
#endif
    }

    /*
     * Rather than calling exit, invoke the "exit" command so that
     * users can replace "exit" with some other command to do additional
     * cleanup on exit.  The Tcl_Eval call should never return.
     */

    done:
    sprintf(buffer, "exit %d", exitCode);
    Tcl_Eval(interp, buffer);
    return 1;
}

/*
 *----------------------------------------------------------------------
 *
 * CheckmemCmd --
 *
 *	This is the command procedure for the "checkmem" command, which
 *	causes the application to exit after printing information about
 *	memory usage to the file passed to this command as its first
 *	argument.
 *
 * Results:
 *	Returns a standard Tcl completion code.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */
#ifdef TCL_MEM_DEBUG

	/* ARGSUSED */
static int
CheckmemCmd(clientData, interp, argc, argv)
    ClientData clientData;		/* Not used. */
    Tcl_Interp *interp;			/* Interpreter for evaluation. */
    int argc;				/* Number of arguments. */
    char *argv[];			/* String values of arguments. */
{
    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" fileName\"", (char *) NULL);
	return TCL_ERROR;
    }
    strcpy(dumpFile, argv[1]);
    quitFlag = 1;
    return TCL_OK;
}
#endif
#endif
