/*
 * Copyright 1994 The University of Newcastle upon Tyne
 *
 * Permission to use, copy, modify and distribute this software and its
 * documentation for any purpose other than its commercial exploitation
 * is hereby granted without fee, provided that the above copyright
 * notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that
 * the name of The University of Newcastle upon Tyne not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission. The University of
 * Newcastle upon Tyne makes no representations about the suitability of
 * this software for any purpose. It is provided "as is" without express
 * or implied warranty.
 *
 * THE UNIVERSITY OF NEWCASTLE UPON TYNE DISCLAIMS ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF
 * NEWCASTLE UPON TYNE BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * If you wish to use this software commercially please contact the author
 * for information as to how this can be arranged.
 *
 * Author:   Ian Campbell
 *          (Ian.Campbell@ncl.ac.uk)
 *
 *          Department of Computing Science
 *          University of Newcastle upon Tyne, UK
 */


#ifndef lint
static char rcsid[] = "$Header: /disk2/tclsrc/tk3.6/tkMovie-0.01A/movieSrc/RCS/tkMovie.c,v 1.4 1995/03/16 23:10:46 wilf Exp $";
#endif

#include "tkConfig.h"
#include "tk.h"
#include "mpeg.h"

/*
 * A data structure of the following type is kept for each movie widget
 * managed by this file:
 */
/*
 * This flag is set to 1 for debuging on
 */

#define MDEBUG 0


typedef struct
  {
    Tk_Window tkwin;		/* Window that embodies the movie.  NULL
				 * means window has been deleted but
				 * widget record hasn't been cleaned up
				 * yet. */
    Display *display;		/* X's token for the window's display. */
    Tcl_Interp *interp;		/* Interpreter associated with widget. */

    /*
     * Information about mpeg stream
     */
    FILE *mpeg;
    ImageDesc img;
    char *pixels;
    Boolean moreframes;
    char *newFile;
    char *dither;

    /*
     * Info needed for X dsiplay
     */

    XImage *ximage;
    int sample;
    int loopFlag;
    /*
     * Information used when displaying widget:
     */

    int borderWidth;		/* Width of 3-D border around whole
				 * widget. */
    Tk_3DBorder bgBorder;	/* Used for drawing background. */
    int relief;			/* Indicates whether window as a whole is
				 * raised, sunken, or flat. */
    GC gc;			/* Graphics context for copying from
				 * off-screen pixmap onto screen. */

    int updatePending;		/* Non-zero means a call to
					 * MovieDisplay has already been
					 * scheduled. */

  }

Movie;

/*
 * Information used for argv parsing.
 */

#define DEF_MOVIE_DITHER      "ordered2"


static Tk_ConfigSpec configSpecs[] =
{
  {TK_CONFIG_BORDER, "-background", "background", "Background",
   "#cdb79e", Tk_Offset (Movie, bgBorder), TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_BORDER, "-background", "background", "Background",
   "white", Tk_Offset (Movie, bgBorder), TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_SYNONYM, "-bd", "borderWidth", (char *) NULL,
   (char *) NULL, 0, 0},
  {TK_CONFIG_SYNONYM, "-bg", "background", (char *) NULL,
   (char *) NULL, 0, 0},
  {TK_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
   "2", Tk_Offset (Movie, borderWidth), 0},
  {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
   "raised", Tk_Offset (Movie, relief), 0},
  {TK_CONFIG_STRING, "-file", (char *) NULL, (char *) NULL,
   "", Tk_Offset (Movie, newFile), 0},
  {TK_CONFIG_STRING, "-dither", (char *) NULL, (char *) NULL,
   DEF_MOVIE_DITHER, Tk_Offset (Movie, dither), 0},
  {TK_CONFIG_BOOLEAN, "-loop", "loop", "Loop",
   "0", Tk_Offset (Movie, loopFlag), 0},
  {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
   (char *) NULL, 0, 0}
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static int MovieConfigure _ANSI_ARGS_ ((Tcl_Interp * interp,
				    Movie * moviePtr, int argc, char **argv,
					int flags));
static void MovieDestroy _ANSI_ARGS_ ((ClientData clientData));
static void MovieDisplay _ANSI_ARGS_ ((ClientData clientData));
static void MovieEventProc _ANSI_ARGS_ ((ClientData clientData,
					 XEvent * eventPtr));
static int MovieWidgetCmd _ANSI_ARGS_ ((ClientData clientData,
				      Tcl_Interp *, int argc, char **argv));
static int MovieCmd _ANSI_ARGS_ ((ClientData clientData,
				  Tcl_Interp *, int argc, char **argv));

/*
 * --------------------------------------------------------------
 *
 * MovieCmd --
 *
 * This procedure is invoked to process the "movie" Tcl command.  It creates
 * a new "movie" widget.
 *
 * Results: A standard Tcl result.
 *
 * Side effects: A new widget is created and configured.
 *
 * --------------------------------------------------------------
 */

static int
MovieCmd (clientData, interp, argc, argv)
     ClientData clientData;	/* Main window associated with
				 * interpreter. */
     Tcl_Interp *interp;	/* Current interpreter. */
     int argc;			/* Number of arguments. */
     char **argv;		/* Argument strings. */
{
  Tk_Window main = (Tk_Window) clientData;
  Movie *moviePtr;
  Tk_Window tkwin;

  if (argc < 2)
    {
      Tcl_AppendResult (interp, "wrong # args:  should be \"",
			argv[0], " pathName ?options?\"", (char *) NULL);
      return TCL_ERROR;
    }
  if (MDEBUG)
    printf ("MovieCmd Entered \n");
  tkwin = Tk_CreateWindowFromPath (interp, main, argv[1], (char *) NULL);
  if (tkwin == NULL)
    {
      return TCL_ERROR;
    }
  Tk_SetClass (tkwin, "Movie");

  /*
     * Allocate and initialize the widget record.
     */
  if (MDEBUG)
    printf ("Allocating tkRecord\n");

  moviePtr = (Movie *) ckalloc (sizeof (Movie));
  moviePtr->tkwin = tkwin;
  moviePtr->display = Tk_Display (tkwin);
  moviePtr->interp = interp;
  moviePtr->borderWidth = 0;
  moviePtr->bgBorder = NULL;
  moviePtr->relief = TK_RELIEF_FLAT;
  moviePtr->gc = None;
  moviePtr->newFile = NULL;
  moviePtr->updatePending = 0;
  moviePtr->sample = 1;
  moviePtr->moreframes = TRUE;
  moviePtr->ximage = NULL;
  moviePtr->mpeg = NULL;
  moviePtr->img.ditherType = ORDERED2_DITHER;
  moviePtr->dither = NULL;
  moviePtr->pixels = NULL;

  Tk_CreateEventHandler (moviePtr->tkwin, ExposureMask | StructureNotifyMask,
			 MovieEventProc, (ClientData) moviePtr);
  Tcl_CreateCommand (interp, Tk_PathName (moviePtr->tkwin), MovieWidgetCmd,
		     (ClientData) moviePtr, (void (*)()) NULL);
  if (MovieConfigure (interp, moviePtr, argc - 2, argv + 2, 0) != TCL_OK)
    {
      Tk_DestroyWindow (moviePtr->tkwin);
      return TCL_ERROR;
    }
  if (MDEBUG)
    printf ("Leaving MovieCmd...\n");
  interp->result = Tk_PathName (moviePtr->tkwin);
  return TCL_OK;
}

/*
 * --------------------------------------------------------------
 *
 * MovieWidgetCmd --
 *
 * This procedure is invoked to process the Tcl command that corresponds to a
 * widget managed by this module. See the user documentation for details
 * on what it does.
 *
 * Results: A standard Tcl result.
 *
 * Side effects: See the user documentation.
 *
 * --------------------------------------------------------------
 */

static int
MovieWidgetCmd (clientData, interp, argc, argv)
     ClientData clientData;	/* Information about movie widget. */
     Tcl_Interp *interp;	/* Current interpreter. */
     int argc;			/* Number of arguments. */
     char **argv;		/* Argument strings. */
{
  Movie *moviePtr = (Movie *) clientData;
  int result = TCL_OK;
  int length;
  char c;

  if (MDEBUG)
    printf ("Entering MovieWidgetCmd.....\n");

  if (argc < 2)
    {
      Tcl_AppendResult (interp, "wrong # args: should be \"",
			argv[0], " option ?arg arg ...?\"", (char *) NULL);
      return TCL_ERROR;
    }
  Tk_Preserve ((ClientData) moviePtr);
  c = argv[1][0];
  length = strlen (argv[1]);
  if ((c == 'c') && (strncmp (argv[1], "configure", length) == 0))
    {
      if (argc == 2)
	{
	  result = Tk_ConfigureInfo (interp, moviePtr->tkwin, configSpecs,
				     (char *) moviePtr, (char *) NULL, 0);
	}
      else if (argc == 3)
	{
	  result = Tk_ConfigureInfo (interp, moviePtr->tkwin, configSpecs,
				     (char *) moviePtr, argv[2], 0);
	}
      else
	{
	  result = MovieConfigure (interp, moviePtr, argc - 2, argv + 2,
				   TK_CONFIG_ARGV_ONLY);
	}
    }
  else if ((c == 'p') && (strncmp (argv[1], "play", length) == 0))
    {
      if ((argc != 2) && (argc != 3))
	{
	  Tcl_AppendResult (interp, "wrong # args: should be \"",
			    argv[0], " play ?amount?\"", (char *) NULL);
	  goto error;
	}
      if (argc == 3)
	{
	  moviePtr->sample = atoi (argv[2]);
	}
      if (argc == 2)
	{
	  moviePtr->sample = 1;	/* play one frame if play is on itself ??*/
	}
    }
  else if ((c == 'r') && (strncmp (argv[1], "rewind", length) == 0))
    {
      if (argc != 2)
	{
	  Tcl_AppendResult (interp, "wrong # args: should be \"",
			    argv[0], " rewind\"", (char *) NULL);
	  goto error;
	}
      if (argc == 2)
	{
	  RewindMPEG (&moviePtr->img);
	  moviePtr->moreframes = TRUE;
	  moviePtr->sample++;
	}
    }

  else if ((c == 'f') && (strncmp (argv[1], "framenumber", length) == 0))
    {
      if (argc != 2)
	{
	  Tcl_AppendResult (interp, "wrong # args: should be \"",
			    argv[0], " framenumber\"", (char *) NULL);
	  goto error;
	}
      if (argc == 2)
	{
	  sprintf(interp->result, "%d", moviePtr->img.totNumFrames); 
	}
    }
  else
    {
      Tcl_AppendResult (interp, "bad option \"", argv[1],
		   "\":  must be configure, play or rewind", (char *) NULL);
      goto error;
    }
  if (!moviePtr->updatePending)
    {
      Tk_DoWhenIdle (MovieDisplay, (ClientData) moviePtr);
      moviePtr->updatePending = 1;
    }
  Tk_Release ((ClientData) moviePtr);
  if (MDEBUG)
    printf ("Leaving MovieWidgetCmd...\n");

  return result;

error:
  Tk_Release ((ClientData) moviePtr);
  return TCL_ERROR;
}

/*
 * ----------------------------------------------------------------------
 *
 * MovieConfigure --
 *
 * This procedure is called to process an argv/argc list in conjunction with
 * the Tk option database to configure (or reconfigure) a movie widget.
 *
 * Results: The return value is a standard Tcl result.  If TCL_ERROR is
 * returned, then interp->result contains an error message.
 *
 * Side effects: Configuration information, such as colors, border width,
 * etc. get set for moviePtr;  old resources get freed, if there were any.
 *
 * ----------------------------------------------------------------------
 */

static int
MovieConfigure (interp, moviePtr, argc, argv, flags)
     Tcl_Interp *interp;	/* Used for error reporting. */
     Movie *moviePtr;		/* Information about widget. */
     int argc;			/* Number of valid entries in argv. */
     char **argv;		/* Arguments. */
     int flags;			/* Flags to pass to	 *
				 * Tk_ConfigureWidget. */
{
  /* Local variables */
  int width, height;
  char dummy;

  if (MDEBUG)
    printf ("Entering Movie Configure...\n");
  if (Tk_ConfigureWidget (interp, moviePtr->tkwin, configSpecs,
			  argc, argv, (char *) moviePtr, flags) != TCL_OK)
    {
      return TCL_ERROR;
    }
  /*
     * Set the background for the window and create a graphics context for
     * use during redisplay.
     */
  if (MDEBUG)
    printf ("Tk_Configure widgit function complete...\n");
  Tk_SetWindowBackground (moviePtr->tkwin,
			  Tk_3DBorderColor (moviePtr->bgBorder)->pixel);
  if ((moviePtr->gc == None))
    {
      XGCValues gcValues;
      gcValues.function = GXcopy;
      gcValues.graphics_exposures = False;
      moviePtr->gc = Tk_GetGC (moviePtr->tkwin,
			       GCFunction | GCGraphicsExposures, &gcValues);
    }

  /*
    * Process the dither type for the stream
    */

  if (moviePtr->dither != NULL)
    {
      if (MDEBUG)
	printf ("Processing dither type...\n");
      if (strcmp (moviePtr->dither, "hybrid") == 0)
	{

	  moviePtr->img.ditherType = HYBRID_DITHER;
	}
      else if (strcmp (moviePtr->dither, "hybrid2") == 0)
	{

	  moviePtr->img.ditherType = HYBRID2_DITHER;
	}
      else if (strcmp (moviePtr->dither, "fs4") == 0)
	{

	  moviePtr->img.ditherType = FS4_DITHER;
	}
      else if (strcmp (moviePtr->dither, "fs2") == 0)
	{

	  moviePtr->img.ditherType = FS2_DITHER;
	}
      else if (strcmp (moviePtr->dither, "fs2fast") == 0)
	{

	  moviePtr->img.ditherType = FS2FAST_DITHER;
	}
      else if (strcmp (moviePtr->dither, "hybrid2") == 0)
	{

	  moviePtr->img.ditherType = HYBRID2_DITHER;
	}
      else if (strcmp (moviePtr->dither, "2x2") == 0)
	{

	  moviePtr->img.ditherType = Twox2_DITHER;
	}
      else if (strcmp (moviePtr->dither, "gray") == 0)
	{

	  moviePtr->img.ditherType = GRAY_DITHER;
	}
      else if (strcmp (moviePtr->dither, "color") == 0)
	{

	  moviePtr->img.ditherType = FULL_COLOR_DITHER;
	}
      else if (strcmp (moviePtr->dither, "none") == 0)
	{

	  moviePtr->img.ditherType = NO_DITHER;
	}
      else if (strcmp (moviePtr->dither, "ordered") == 0)
	{

	  moviePtr->img.ditherType = ORDERED_DITHER;
	}
      else if (strcmp (moviePtr->dither, "ordered2") == 0)
	{

	  moviePtr->img.ditherType = ORDERED2_DITHER;
	}
      else if (strcmp (moviePtr->dither, "mbordered") == 0)
	{

	  moviePtr->img.ditherType = MBORDERED_DITHER;
	}
      else if (strcmp (moviePtr->dither, "mono") == 0)
	{

	  moviePtr->img.ditherType = MONO_DITHER;
	}
      else if (strcmp (moviePtr->dither, "threshold") == 0)
	{

	  moviePtr->img.ditherType = MONO_THRESHOLD;
	}
      else
	{
	   Tcl_AppendResult(interp, "bad dither value \"", moviePtr->dither,
                    "\":  see man page", (char *) NULL); 
	  return TCL_ERROR;
	}

      if (moviePtr->mpeg != NULL)
	{
	  if (moviePtr->ximage != NULL)
	    {
	      /* XDestroyImage frees image AND the data for that image (pixels) */
	      /* this is mainly needed when changing from or to 2x2 dither */
	      XDestroyImage (moviePtr->ximage);
	      moviePtr->pixels = NULL;
	      moviePtr->ximage = NULL;
	    }

	  SetMPEGDither (Tk_Display (moviePtr->tkwin),
			 Tk_WindowId (moviePtr->tkwin),
			 &moviePtr->img,
			 moviePtr->img.ditherType);
	  /* get a new frame so dither is displayed immediately*/
	  moviePtr->moreframes = 1;
	  moviePtr->sample++;
	}
      moviePtr->dither = NULL;
    }


  /*
    * Create the movie by attaching it to a mpeg stream
    */

  if (moviePtr->newFile != NULL)
    {
      if (moviePtr->mpeg != NULL)
	{
	  if (MDEBUG)
	    printf ("Closing \n");
	  CloseMPEG (&moviePtr->img);
	}
      /* open the mpeg stream */
      if (MDEBUG)
	printf ("Opening Mpeg \n");

      moviePtr->mpeg = fopen (moviePtr->newFile, "r");
      if (!moviePtr->mpeg)
	{
	   Tcl_ResetResult(moviePtr->interp);
           Tcl_AppendResult(moviePtr->interp, "couldn't open mpeg file \"",
                moviePtr->newFile, "\" : ", 
                Tcl_PosixError(moviePtr->interp),(char *) NULL);
          moviePtr->newFile = NULL ; 
	  moviePtr->mpeg = NULL ;
	  return TCL_ERROR;
	}
      if (!OpenMPEG (Tk_Display (moviePtr->tkwin), Tk_WindowId (moviePtr->tkwin),
		     moviePtr->mpeg, &moviePtr->img))
	{
        Tcl_AppendResult(moviePtr->interp, "This is not an mpeg file \"",
                moviePtr->newFile, "\": ", Tcl_PosixError(moviePtr->interp),
                (char *) NULL);
	  moviePtr->newFile = NULL ;
	  return TCL_ERROR;
	}

      /* stream opening finished */
      moviePtr->newFile = NULL;
    }

  if (moviePtr->pixels == NULL)
    {
      if (MDEBUG)
	printf ("Making pixels\n");
      moviePtr->pixels = (char *) malloc (moviePtr->img.Size * sizeof (char));
    }


  /*
     * First time round create the x image
     */

  if (moviePtr->ximage == NULL)
    {
      if (MDEBUG)
	printf ("Creating ximage that was null\n");
      moviePtr->ximage = XCreateImage (Tk_Display (moviePtr->tkwin), None, 8, ZPixmap, 0,
				       &dummy, moviePtr->img.Width,
				       moviePtr->img.Height,
				       8, 0);
    }

  /*
     * Register the desired geometry for the window.  Then arrange for the
     * window to be redisplayed.
     */
  width = moviePtr->img.Width + 2 * moviePtr->borderWidth;
  height = moviePtr->img.Height + 2 * moviePtr->borderWidth;


  Tk_GeometryRequest (moviePtr->tkwin, width, height);



  Tk_SetInternalBorder (moviePtr->tkwin, moviePtr->borderWidth);
  if (!moviePtr->updatePending)
    {
      Tk_DoWhenIdle (MovieDisplay, (ClientData) moviePtr);
      moviePtr->updatePending = 1;
    }
  if (MDEBUG)
    printf ("Leaving Movie configure...\n");
  return TCL_OK;
}

/*
 * --------------------------------------------------------------
 *
 * MovieEventProc --
 *
 * This procedure is invoked by the Tk dispatcher for various events on
 * movies.
 *
 * Results: None.
 *
 * Side effects: When the window gets deleted, internal structures get
 * cleaned up.  When it gets exposed, it is redisplayed.
 *
 * --------------------------------------------------------------
 */

static void
MovieEventProc (clientData, eventPtr)
     ClientData clientData;	/* Information about window. */
     XEvent *eventPtr;		/* Information about event. */
{
  Movie *moviePtr = (Movie *) clientData;

  if (MDEBUG)
    printf ("Entering Movie Event proc ...\n");

  if (eventPtr->type == Expose)
    {
      if (!moviePtr->updatePending)
	{
	  Tk_DoWhenIdle (MovieDisplay, (ClientData) moviePtr);
	  moviePtr->updatePending = 1;
	}
    }
  else if (eventPtr->type == ConfigureNotify)
    {

      /*      if (!moviePtr->updatePending)
	{
	  Tk_DoWhenIdle (MovieDisplay, (ClientData) moviePtr);
	  moviePtr->updatePending = 1;
	}*/
    }
  else if (eventPtr->type == DestroyNotify)
    {
      Tcl_DeleteCommand (moviePtr->interp, Tk_PathName (moviePtr->tkwin));
      moviePtr->tkwin = NULL;
      if (moviePtr->updatePending)
	{
	  Tk_CancelIdleCall (MovieDisplay, (ClientData) moviePtr);
	}
      Tk_EventuallyFree ((ClientData) moviePtr, MovieDestroy);
    }
  if (MDEBUG)
    printf ("Leaving Movie EventProc...\n");
}

/*
 * --------------------------------------------------------------
 *
 * MovieDisplay --
 *
 * This procedure redraws the contents of a movie window. It is invoked as a
 * do-when-idle handler, so it only runs when there's nothing else for the
 * application to do.
 *
 * Results: None.
 *
 * Side effects: Information appears on the screen.
 *
 * --------------------------------------------------------------
 */

static void
MovieDisplay (clientData)
     ClientData clientData;	/* Information about window. */
{
  Movie *moviePtr = (Movie *) clientData;
  Tk_Window tkwin = moviePtr->tkwin;
  Pixmap pm = None;
  Drawable d;

  if (MDEBUG)
    printf ("Entering Movie Display proc...\n");
  moviePtr->updatePending = 0;
  if (!Tk_IsMapped (tkwin))
    {
      if (MDEBUG)
	printf ("tkwin is not mapped returning...\n");
      return;
    }
    /*
     * Create a pixmap to hold ximage which we will blast to screen
     */

  if (MDEBUG)
    printf ("call to XCreatePixmap...\n");


  d = pm = XCreatePixmap (Tk_Display (tkwin), Tk_WindowId (tkwin),
			  Tk_Width (tkwin), Tk_Height (tkwin),
			  DefaultDepthOfScreen (Tk_Screen (tkwin)));
    /*
     * Draw widget border
     */
  Tk_Fill3DRectangle (Tk_Display (tkwin), d,
		      moviePtr->bgBorder, 0, 0, Tk_Width (tkwin),
		      Tk_Height (tkwin), moviePtr->borderWidth,
		      moviePtr->relief);



  if (moviePtr->sample)
    {
      while (moviePtr->sample--)
	{
	  if (moviePtr->moreframes)
	    {
	      if (MDEBUG)
		printf ("GetMPEGFrame call\n");

	      moviePtr->moreframes = GetMPEGFrame (&moviePtr->img, moviePtr->pixels);

	      if (MDEBUG)
		{
		  printf ("Got pixels\n");
		}

	      moviePtr->ximage->data = (char *) moviePtr->pixels;
	      if (MDEBUG)
		printf ("Copying image\n");
	      XPutImage (Tk_Display (tkwin), pm,
		moviePtr->gc, moviePtr->ximage, 0, 0, moviePtr->borderWidth,
			 moviePtr->borderWidth, moviePtr->img.Width,
			 moviePtr->img.Height);
	      XCopyArea (Tk_Display (tkwin), pm, Tk_WindowId (tkwin), moviePtr->gc,
			 0, 0, Tk_Width (tkwin), Tk_Height (tkwin), 0, 0);
	      if (MDEBUG)
		printf ("You see XMovies\n");
	    }
	  else if (moviePtr->loopFlag)
	    {
	      RewindMPEG (&moviePtr->img);
	      moviePtr->moreframes = TRUE;
	      moviePtr->sample++;
	      if (MDEBUG)
		printf ("Opening again cos of loopFlag\n");

	    }
	}
      moviePtr->sample = 0;
    }
  else
    {
      if (MDEBUG)
	printf ("Putting image to screen with no call...\n");

      XPutImage (Tk_Display (tkwin), pm,
		 moviePtr->gc, moviePtr->ximage, 0, 0, moviePtr->borderWidth,
		 moviePtr->borderWidth, moviePtr->img.Width,
		 moviePtr->img.Height);
      XCopyArea (Tk_Display (tkwin), pm, Tk_WindowId (tkwin), moviePtr->gc,
		 0, 0, Tk_Width (tkwin), Tk_Height (tkwin), 0, 0);
    }

  XFreePixmap (Tk_Display (tkwin), pm);
  if (MDEBUG)
    printf ("Leaving Display\n");

}

/*
 * ----------------------------------------------------------------------
 *
 * MovieDestroy --
 *
 * This procedure is invoked by Tk_EventuallyFree or Tk_Release to clean up
 * the internal structure of a movie at a safe time (when no-one is using
 * it anymore).
 *
 * Results: None.
 *
 * Side effects: Everything associated with the movie is freed up.
 *
 * ----------------------------------------------------------------------
 */

static void
MovieDestroy (clientData)
     ClientData clientData;	/* Info about movie widget. */
{
  Movie *moviePtr = (Movie *) clientData;

  /*
     * Free Mpeg stuff
     */

  if(moviePtr->mpeg != NULL) CloseMPEG (&moviePtr->img);

  Tk_FreeOptions (configSpecs, (char *) moviePtr, moviePtr->display, 0);
  if (moviePtr->gc != None)
    {
      Tk_FreeGC (moviePtr->display, moviePtr->gc);
    }
  ckfree ((char *) moviePtr);
}



/*
 * Movies Init procedure This procedure is invocked from tkAppInit.c In
 * the stanard fashion for a tk package
 *
 */

extern int
Movie_Init (interp)
     Tcl_Interp *interp;	/* pointer to tcl interpretor */
{
  Tcl_CreateCommand (interp, "movie", MovieCmd,
		     (ClientData) Tk_MainWindow (interp),
		     (Tcl_CmdDeleteProc *) NULL);
  return TCL_OK;
}
