%{
/****************************************************************************

                   PARSER for  'C C O U N T'  PROGRAM

                       Y A C C  SPECIFICATION FILE

*****************************************************************************

  Description:

  'ccount.y' is the yacc specification file of 'ccount' program that
  supplies results of the structure of a C source file. The command 
  'yacc [-d] ccount.y' produces the C source code for this file 
  (--> y.tab.c, the real parser of 'ccount'). Finally you get 'ccount' 
  with 
         'cc -o ccount y.tab.c -ll -lm'.

  See also 'README.1'!

*****************************************************************************

  Author: Joerg Lawrenz, Universitaet Karlsruhe
  Date:   12/1/93

  Portions Copyright (c)  1989,  1990  James  A.  Roskind

****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <search.h>
#include <math.h>

#define NUM_EMPL      557   /* size of hash table */
#define STR_LEN      1000   /* max. string length */
#define MAX_FUNCTION  200   /* max. number of functions */
#define MAX_BLOCKS    500   /* max. number of blocks in a function */
#define MAX_INBLOCKS  500   /* max. number of innermost blocks in a function */
#define MAX(x,y)  ((x) > (y) ? (x) : (y))
#define SQR(x)    ((x) * (x))
#define BOOL      int
#define EMPTY     -1

/* Type of the result of $hold_info, a rule that saves the current byte,
   comment byte and line number within the module before the analysis
   of an object. With this informations we can determine e.g. the length 
   of an object at the end of it: */

typedef struct info_val {
     long byte;
     long comment;
     int line;     
   }  INFO_VAL; 

/* Type of the result of a statement or a function rule: */

typedef struct statement_val {    
     int num_if;             /* number of 'if' statements
                                  (only relev. for functions) */
     int num_switch;         /* number of 'switch' statements
                                  (only relev. for functions) */
     int num_while;          /* number of 'while' statements 
                                  (only relev. for functions) */
     int num_for;            /* number of 'for' statements
                                  (only relev. for functions) */
     int num_gotos;          /* number of 'goto' statements 
                                  (only relev. for functions) */
     int num_labels;         /* number of 'label' statements
                                  (only relev. for functions) */
     int num_case;           /* number of 'case' regarding this statement
                                  (only relev. for 'switch' statements) */
     int num_break;          /* number of 'break's regarding this statement
                                  (only relev. for 'while','for' and 'switch'
                                  statements) */
     int num_continue;       /* number of 'continue's regarding this statement
                                  (only relev. for 'while' statements) */
     int num_statements;     /* number of statements */
     int num_decls;          /* number of declarations */
     int num_blocks;         /* number of enclosed blocks 
                                  (only relev. for functions and blocks */
     int max_depth;          /* max. nesting depth of enclosed blocks 
                                  (only relev. for functions and blocks */
     int max_Inblock_length; /* max. innermost block length in ';' 
                                  (only relev. for functions and blocks */
     int length;             /* length in line */
     int last_line;          /* last line number */
   }  STATEMENT_VAL;


STATEMENT_VAL init = {0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0};
          /* initial result of a statement rule */

char string_space [NUM_EMPL * 100];     /* memory for hash table */

FILE *in, *out, *tppfile;
char c_name[100], out_name[100], tpp_name[100];

int t = 1;              /* current nesting depth */

BOOL TypeFlag = 0;      /* is set to 1 if 'typedef' has been read */
BOOL DeclFlag = 1;      /* indicates, if actually declarations are read */
BOOL ElabTypeFlag = 0;  /* indicates, if 'struct','union' or 'enum' has been
                           read (set to 1 by the scanner, set to 0 by the 
                           parser), important for type name decisions in the
                           scanner (rule 1-4) */ 
BOOL FatalError = 0;    /* indicates if the maximums of number of functions
                           or blocks has been passed */
BOOL Debug = 0;         /* is set by the -v option to get more informations
                           about the lexical analysis */

int num_statements_total = 0;        /* total number of ';' in the module */


/************ Memories of values for statistical processing:  **************/

/* For one function: */
double Depth [MAX_BLOCKS];        /* nesting depth per block */
int blockindex = 0;               /* index, number of blocks in a function */

/* For one function: */
double InblockLength [MAX_INBLOCKS];  /* length in ';' per innermost block */ 
int inblockindex = 0;   /* index, number of innermost blocks in a function */

/* For module: */
double InblockLengthTotal [MAX_INBLOCKS * MAX_FUNCTION];
                                 /* lengths in ';' of all innermost blocks */
int inblockindex_total = 0;       /* index, number of all innermost blocks */
int max_inblock_length_total = 0; /* maximum of all innermost block length */

/* For module: */
double AveDepth [MAX_FUNCTION];   /* average nesting depth per function */
int num_functions = 0;            /* index, number of functions */
double max_ave_depth_total = 0;   /* module max. average depth per function */


/*** The values of the following variables are supplied by the scanner: *****/

int lines = 0;       /* number of lines already read */
long bytes = 0;      /* number of bytes without comments */
long comments = 0;   /* number of bytes of comment */
long Pcomments = 0;  /* number of bytes of preprocessor comment,
                           see 'README.1', 2.2 */

int num_identifier = 0;       /* number of identifier uses */
int num_symbols = 0;          /* number of symbols and keywords uses */
int num_ppdirectives = 0;     /* number of preprocessor directives */
long bytes_symbols = 0;       
              /* accumulated length of keyword/symbol uses in bytes */
long bytes_identifier = 0;
              /* accumulated length of identifier uses in bytes */
long bytes_ppdirectives = 0;
              /* accumulated length of preprocessor directives uses in bytes */

char lexem[STR_LEN] = {""};       /* current lexeme */
char last_identifier[STR_LEN];    /* last identifier */

%}

/********************** Define YACC value stack ****************************/

%union   {
   int ival;
   INFO_VAL info;
   STATEMENT_VAL sval;
 }

/********************** Define terminal tokens *****************************/

/* keywords */

%token AUTO            DOUBLE          INT             STRUCT
%token BREAK           ELSE            LONG            SWITCH
%token CASE            ENUM            REGISTER        TYPEDEF
%token CHAR            EXTERN          RETURN          UNION
%token CONST           FLOAT           SHORT           UNSIGNED
%token CONTINUE        FOR             SIGNED          VOID
%token DEFAULT         GOTO            SIZEOF          VOLATILE
%token DO              IF              STATIC          WHILE

/* ANSI Grammar suggestions */

%token IDENTIFIER              STRINGliteral
%token FLOATINGconstant        INTEGERconstant        CHARACTERconstant
%token OCTALconstant           HEXconstant

/* New Lexical element, whereas ANSI suggested non-terminal */

%token TYPEDEFname

%token MACRODEF

/* Multi-Character operators */

%token  ARROW            /*    ->                              */
%token  ICR DECR         /*    ++      --                      */
%token  LS RS            /*    <<      >>                      */
%token  LE GE EQ NE      /*    <=      >=      ==      !=      */
%token  ANDAND OROR      /*    &&      ||                      */
%token  ELLIPSIS         /*    ...                             */

/* modifying assignment operators */

%token MULTassign  DIVassign    MODassign   /*   *=      /=      %=      */
%token PLUSassign  MINUSassign              /*   +=      -=              */
%token LSassign    RSassign                 /*   <<=     >>=             */
%token ANDassign   ERassign     ORassign    /*   &=      ^=      |=      */


/********************** Define types ****************************************/

%type <sval> statement statement_list labeled_statement compound_statement 
%type <sval> selection_statement iteration_statement jump_statement 
%type <sval> expression_statement function_definition

%type <ival> declaration_list
%type <ival> primary_expression postfix_expression argument_expression_list
%type <ival> unary_argument
%type <ival> unary_expression cast_expression multiplicative_expression
%type <ival> additive_expression shift_expression relational_expression
%type <ival> equality_expression AND_expression exclusive_OR_expression
%type <ival> inclusive_OR_expression logical_AND_expression 
%type <ival> logical_OR_expression conditional_expression
%type <ival> assignment_expression comma_expression comma_expression_opt

%type <info> $hold_info 

/***************************************************************************/

%start translation_unit

/*************************************************************************/

%%

/* CONSTANTS */
constant:
        INTEGERconstant
        | FLOATINGconstant
        /* We are not including ENUMERATIONconstant here  because  we
        are  treating  it like a variable with a type of "enumeration
        constant".  */
        | OCTALconstant
        | HEXconstant
        | CHARACTERconstant
        ;

string_literal_list:
                STRINGliteral
                | string_literal_list STRINGliteral
                ;


/************************* EXPRESSIONS ********************************/
primary_expression:
        IDENTIFIER                    {$$ = 0;}
        | constant                    {$$ = 0;}
        | string_literal_list         {$$ = 0;}
        | '(' comma_expression ')'    {$$ = 1 + $2;}
        | '(' ')'                     {$$ = 1;}
        ;

postfix_expression:
        primary_expression                                    {$$ = $1;}
        | postfix_expression '[' comma_expression ']'         {$$ = $1+$3+1;}
        | postfix_expression '(' ')'                          {$$ = $1 + 1;}
        | postfix_expression '(' argument_expression_list ')' {$$ = $1+$3+1;}
        | postfix_expression {} '.'   member_name             {$$ = $1 + 1;}
        | postfix_expression {} ARROW member_name             {$$ = $1 + 1;}
        | postfix_expression ICR                              {$$ = $1 + 1;}
        | postfix_expression DECR                             {$$ = $1 + 1;}
        ;

member_name:
        IDENTIFIER
        | TYPEDEFname
        ;

opt_abstract_declarator:
        abstract_declarator
        | /* nothing */
        ;

unary_argument:
        assignment_expression                             {$$ = $1;}
        | TYPEDEFname opt_abstract_declarator             {$$ = 0;}
        | basic_type_specifier opt_abstract_declarator    {$$ = 0;}
        | elaborated_type_name opt_abstract_declarator    {$$ = 0;}
        ;

argument_expression_list:
        unary_argument
        | argument_expression_list ',' unary_argument   {$$ = $1+1+$3;}
        ;

unary_expression:
        postfix_expression                    {$$ = $1;}
        | ICR unary_expression                {$$ = 1 + $2;}             
        | DECR unary_expression               {$$ = 1 + $2;}             
        | unary_operator cast_expression      {$$ = 1 + $2;}             
        | SIZEOF unary_expression             {$$ = 1 + $2;}             
        | SIZEOF '(' type_name ')'            {$$ = 2;}             
        ;

unary_operator:
        '&'
        | '*'
        | '+'
        | '-'
        | '~'
        | '!'
        ;

cast_expression:
        unary_expression                                    {$$ = $1;}
        | '(' type_name ')' cast_expression                 {$$ = 1 + $4;}    
        ;

multiplicative_expression:
        cast_expression                                     {$$ = $1;}
        | multiplicative_expression '*' cast_expression     {$$ = $1 + 1 + $3;}
        | multiplicative_expression '/' cast_expression     {$$ = $1 + 1 + $3;}
        | multiplicative_expression '%' cast_expression     {$$ = $1 + 1 + $3;}
        ;

additive_expression:
        multiplicative_expression                           {$$ = $1;}
        | additive_expression '+' multiplicative_expression {$$ = $1 + 1 + $3;}
        | additive_expression '-' multiplicative_expression {$$ = $1 + 1 + $3;}
        ;

shift_expression:
        additive_expression                                 {$$ = $1;}
        | shift_expression LS additive_expression           {$$ = $1 + 1 + $3;}
        | shift_expression RS additive_expression           {$$ = $1 + 1 + $3;}
        ;

relational_expression:
        shift_expression                                    {$$ = $1;}
        | relational_expression '<' shift_expression        {$$ = $1 + 1 + $3;}
        | relational_expression '>' shift_expression        {$$ = $1 + 1 + $3;}
        | relational_expression LE shift_expression         {$$ = $1 + 1 + $3;}
        | relational_expression GE shift_expression         {$$ = $1 + 1 + $3;}
        ;

equality_expression:
        relational_expression                               {$$ = $1;}
        | equality_expression EQ relational_expression      {$$ = $1 + 1 + $3;}
        | equality_expression NE relational_expression      {$$ = $1 + 1 + $3;}
        ;

AND_expression:
        equality_expression                                 {$$ = $1;}
        | AND_expression '&' equality_expression            {$$ = $1 + 1 + $3;}
        ;

exclusive_OR_expression:
        AND_expression                                      {$$ = $1;}
        | exclusive_OR_expression '^' AND_expression        {$$ = $1 + 1 + $3;}
        ;

inclusive_OR_expression:
        exclusive_OR_expression                             {$$ = $1;}
        | inclusive_OR_expression '|' exclusive_OR_expression 
                                                            {$$ = $1 + 1 + $3;}
        ;

logical_AND_expression:
        inclusive_OR_expression                             {$$ = $1;}
        | logical_AND_expression ANDAND inclusive_OR_expression 
                                                            {$$ = $1 + 1 + $3;}
        ;

logical_OR_expression:
        logical_AND_expression                              {$$ = $1;}
        | logical_OR_expression OROR logical_AND_expression {$$ = $1 + 1 + $3;}
        ;

conditional_expression:
        logical_OR_expression                          {$$ = $1;}
        | logical_OR_expression '?' comma_expression ':'    
                conditional_expression                 {$$ = $1 + 1 + $3 + $5;}
        ;

assignment_expression:
        conditional_expression                              {$$ = $1;}
        | unary_expression assignment_operator assignment_expression
                                                            {$$ = $1 + 1 + $3;}
        ;

assignment_operator:
        '='             
        | MULTassign    
        | DIVassign     
        | MODassign     
        | PLUSassign    
        | MINUSassign   
        | LSassign      
        | RSassign      
        | ANDassign     
        | ERassign      
        | ORassign      
        ;

comma_expression:
        assignment_expression                             {$$ = $1;}
        | comma_expression ',' assignment_expression      {$$ = $1 + 1 + $3;} 
        ;

constant_expression:
        conditional_expression
        ;

    /* The following was used for clarity */
comma_expression_opt:
        /* Nothing */                                     {$$ = EMPTY;}
        | comma_expression                                {$$ = $1;}
        ;



/******************************* DECLARATIONS *********************************/

    /* The following is different from the ANSI C specified  grammar.
    The  changes  were  made  to  disambiguate  typedef's presence in
    declaration_specifiers (vs.  in the declarator for redefinition);
    to allow struct/union/enum tag declarations without  declarators,
    and  to  better  reflect the parsing of declarations (declarators
    must be combined with declaration_specifiers ASAP  so  that  they
    are visible in scope).

    Example  of  typedef  use  as either a declaration_specifier or a
    declarator:

      typedef int T;
      struct S { T T;}; /* redefinition of T as member name * /

    Example of legal and illegal statements detected by this grammar:

      int; /* syntax error: vacuous declaration * /
      struct S;  /* no error: tag is defined or elaborated * /

    Example of result of proper declaration binding:

        int a=sizeof(a); /* note that "a" is declared with a type  in
            the name space BEFORE parsing the initializer * /

        int b, c[sizeof(b)]; /* Note that the first declarator "b" is
             declared  with  a  type  BEFORE the second declarator is
             parsed * /

    */

declaration:
        sue_declaration_specifier ';'    { TypeFlag = 0; }
        | sue_type_specifier ';'         
        | declaring_list ';'             { TypeFlag = 0;
                                           DeclFlag = 1; 
                                          }
        | default_declaring_list ';'     { TypeFlag = 0; 
                                           DeclFlag = 1;
                                          }
        | MACRODEF                /* e.g. va_dcl */
        | MACRODEF '(' ')'
        | MACRODEF '(' ')' ';'
        | MACRODEF '(' argument_expression_list ')'
        | MACRODEF '(' argument_expression_list ')' ';'
          /* e.g. PROTO (boolean, aout_32_slurp_symbol_table); */
        ;

    /* Note that if a typedef were  redeclared,  then  a  declaration
    specifier must be supplied */


default_declaring_list:  /* Can't  redeclare typedef names */
        declaration_qualifier_list identifier_declarator 
                                      {DeclFlag = 0;} initializer_opt
        | type_qualifier_list identifier_declarator 
                                      {DeclFlag = 0;} initializer_opt
        | default_declaring_list ',' identifier_declarator {} initializer_opt
        ;

declaring_list:
        declaration_specifier declarator 
         {if (TypeFlag) henter(last_identifier); DeclFlag = 0;} initializer_opt
        | type_specifier declarator {DeclFlag = 0;} initializer_opt
        | declaring_list ',' declarator 
           {if (TypeFlag) henter(last_identifier);} initializer_opt  
        ;

declaration_specifier:
        basic_declaration_specifier          /* Arithmetic or void */          
        | sue_declaration_specifier          /* struct/union/enum */
        | typedef_declaration_specifier      /* typedef*/  
        ;

type_specifier:
        basic_type_specifier                 /* Arithmetic or void */
        | sue_type_specifier                 /* Struct/Union/Enum */
        | typedef_type_specifier             /* Typedef */
        ;

declaration_qualifier_list:  /* const/volatile, AND storage class */
        storage_class                    
        | type_qualifier_list storage_class  
        | declaration_qualifier_list declaration_qualifier 
        ;

type_qualifier_list:
        type_qualifier
        | type_qualifier_list type_qualifier
        ;

declaration_qualifier:
        storage_class
        | type_qualifier                  /* const or volatile */
        ;

type_qualifier:
        CONST          
        | VOLATILE 
        ;

basic_declaration_specifier:      /*Storage Class+Arithmetic or void*/
        declaration_qualifier_list basic_type_name
        | basic_type_specifier  storage_class                 
        | basic_declaration_specifier declaration_qualifier 
        | basic_declaration_specifier basic_type_name
        ;

basic_type_specifier:
        basic_type_name            /* Arithmetic or void */
        | type_qualifier_list  basic_type_name
        | basic_type_specifier type_qualifier
        | basic_type_specifier basic_type_name
        ;

sue_declaration_specifier:          /* Storage Class + struct/union/enum */
        declaration_qualifier_list elaborated_type_name   
        | sue_type_specifier        storage_class            
        | sue_declaration_specifier declaration_qualifier     
        ;

sue_type_specifier:
        elaborated_type_name              /* struct/union/enum */
        | type_qualifier_list elaborated_type_name
        | sue_type_specifier  type_qualifier
        ;

typedef_declaration_specifier:       /*Storage Class + typedef types */
        typedef_type_specifier          storage_class       
        | declaration_qualifier_list   TYPEDEFname 
        | typedef_declaration_specifier declaration_qualifier
        ;

typedef_type_specifier:              /* typedef types */
        TYPEDEFname
        | type_qualifier_list    TYPEDEFname   
        | basic_type_name TYPEDEFname             /* extra rule */    
        | typedef_type_specifier type_qualifier
        ;

storage_class:
        TYPEDEF         { TypeFlag = 1; }
        | EXTERN     
        | STATIC     
        | AUTO       
        | REGISTER
        ;

basic_type_name:
        INT          
        | CHAR       
        | SHORT      
        | LONG       
        | FLOAT      
        | DOUBLE     
        | SIGNED     
        | UNSIGNED   
        | VOID       
        ;

elaborated_type_name:
        aggregate_name
        | enum_name        { ElabTypeFlag = 0; }
        ;

aggregate_name:
        aggregate_key {ElabTypeFlag = 0;} '{' member_declaration_list '}'  
        | aggregate_key identifier_or_typedef_name {ElabTypeFlag = 0;} 
                '{' member_declaration_list '}'        
        | aggregate_key identifier_or_typedef_name {ElabTypeFlag = 0;} 
        ;

aggregate_key:
        STRUCT
        | UNION
        ;

member_declaration_list:
        member_declaration
        |  member_declaration_list member_declaration
        ;

member_declaration:
        member_declaring_list ';'             
        | member_default_declaring_list ';'   
        ;

member_default_declaring_list:        /* doesn't redeclare typedef*/
        type_qualifier_list member_identifier_declarator
        | member_default_declaring_list ',' member_identifier_declarator
        ;

member_declaring_list:
        type_specifier member_declarator
        | member_declaring_list ',' member_declarator  
        ;


member_declarator:
        declarator bit_field_size_opt
        | bit_field_size
        ;

member_identifier_declarator:
        identifier_declarator {} bit_field_size_opt
        | bit_field_size
        ;

bit_field_size_opt:
        /* nothing */
        | bit_field_size
        ;

bit_field_size:
        ':' constant_expression  
        ;

enum_name:
        ENUM '{' enumerator_list '}'             
        | ENUM identifier_or_typedef_name '{' enumerator_list '}'
        | ENUM identifier_or_typedef_name        
        ;

enumerator_list:
        identifier_or_typedef_name enumerator_value_opt
        | enumerator_list ',' identifier_or_typedef_name enumerator_value_opt
        ;

enumerator_value_opt:
        /* Nothing */
        | '=' constant_expression                
        ;

parameter_type_list:
        parameter_list
        | parameter_list ',' ELLIPSIS            
        ;

parameter_list:
        parameter_declaration
        | parameter_list ',' parameter_declaration   
        ;

parameter_declaration:
        declaration_specifier
        | declaration_specifier abstract_declarator
        | declaration_specifier identifier_declarator
        | declaration_specifier parameter_typedef_declarator
        | declaration_qualifier_list
        | declaration_qualifier_list abstract_declarator
        | declaration_qualifier_list identifier_declarator
        | type_specifier
        | type_specifier abstract_declarator
        | type_specifier identifier_declarator
        | type_specifier parameter_typedef_declarator
        | type_qualifier_list
        | type_qualifier_list abstract_declarator
        | type_qualifier_list identifier_declarator
        ;

    /*  ANSI  C  section  3.7.1  states  "An identifier declared as a
    typedef name shall not be redeclared as a parameter".  Hence  the
    following is based only on IDENTIFIERs */

identifier_list:
        IDENTIFIER
        | identifier_list ',' IDENTIFIER     
        ;

identifier_or_typedef_name:
        IDENTIFIER
        | TYPEDEFname    
        ;

type_name:
        type_specifier
        | type_specifier abstract_declarator
        | type_qualifier_list
        | type_qualifier_list abstract_declarator
        ;

initializer_opt:
        /* nothing */
        | '=' initializer     
        ;

initializer:
        '{' initializer_list '}'          
        | '{' initializer_list ',' '}'    
        | assignment_expression
        ;

initializer_list:
        initializer
        | /* nothing */
        | initializer_list ',' initializer   
        ;


/*************************** STATEMENTS *******************************/

$hold_info:   /* nothing */    {$$.byte = bytes - strlen(lexem);
                                 /* '-strlen(lexem)', because the first
                                    word of an object has already been
                                    read before this rule is activated */
                                $$.comment = comments;
                                $$.line = lines;
			        }
        ;


statement:
        $hold_info labeled_statement           {$$ = $2;
                                                }   
        | {DeclFlag = 1;  t++;} $hold_info compound_statement 
                {$$ = $3;
                 $$.length = lines+1 - $2.line;
                 $$.num_blocks++;
                                    
                 if ($3.num_blocks == 0) {
                   if (check(&inblockindex_total) ||
                       check(&inblockindex))  YYERROR;
                   $$.max_depth = t;
                   $$.max_Inblock_length = 
                       InblockLengthTotal[inblockindex_total++] =
                          InblockLength[inblockindex++] = $3.num_statements;
                   printf("I %d %d %d\n", $$.length, $3.num_statements, t);
		   } 
                  else
                   printf("B %d %d %d %d %d\n", $$.length, $3.num_statements, 
                                               t, $3.num_blocks, $3.max_depth);

                 if (check(&blockindex)) YYERROR;
                 Depth[blockindex++] = t--;
                 }
        | $hold_info expression_statement       {$$ = $2;
                                                 }
        | $hold_info selection_statement        {$$ = $2;
						 }
        | $hold_info iteration_statement        {$$ = $2;
                                                 }
        | $hold_info jump_statement             {$$ = $2;
                                                 $$.length = lines+1 - $1.line;
                                                 $$.last_line = lines+1;
                                                 }
        ;

labeled_statement:
        identifier_or_typedef_name ':' statement    
                                   {$$ = $3;
                                    $$.length = $3.last_line - $<info>0.line;
                                    $$.num_labels++;
                                    }
        | CASE constant_expression ':' statement    
                                   {$$ = $4;
                                    $$.length = $4.last_line - $<info>0.line;
                                    $$.num_case = 1;
                                    }
        | DEFAULT ':' statement    {$$ = $3;
                                    $$.length = $3.last_line - $<info>0.line;
                                    $$.num_case = 1;
				    }
        ;

compound_statement:
        '{' '}'                                    {$$ = init;
                                                    $$.last_line = lines+1;
                                                    DeclFlag = 0;
                                                    }
        | '{' declaration_list '}'                 {$$ = init;
                                                    $$.last_line = lines+1;
                                                    $$.num_statements = $2;
                                                    DeclFlag = 0;
                                                    }
        | '{' {DeclFlag = 0;} statement_list '}'   
                                                   {$$ = $3;
                                                    $$.last_line = lines+1;
                                                    }
        | '{' declaration_list {DeclFlag = 0;} statement_list '}' 
                                                   {$$ = $4;
                                                    $$.last_line = lines+1;
                                                    $$.num_statements += $2;
                                                    }
        ;

declaration_list:
        declaration                       {$$ = 1;}
        | declaration_list declaration    {$$ = $1 + 1;}
        ;

statement_list:
        statement                         {$$ = $1;}
        | statement_list statement        {$$ = add($1, $2);}
        ;

expression_statement:
        comma_expression_opt ';'          {$$ = init;
                                           $$.length = lines+1 - $<info>0.line;
                                           $$.last_line = lines+1;
                                           $$.num_statements = 1;
                                           if ($1 != EMPTY)
                                              printf("e %d %d\n", $1, t);
                                           }
        ;

selection_statement:
        IF '(' comma_expression ')' statement        
                                    {$$ = $5;
                                     $$.length = $5.last_line - $<info>0.line;
                                     $$.num_if++;
                                     printf("i %d %d %d %d 0 0 %d\n", 
                                      $$.length, $3, 
                                      $5.length, $5.num_statements, t); 
                                     }
        | IF '(' comma_expression ')' statement ELSE statement
                                    {$$ = add($5, $7);
                                     $$.length = $7.last_line - $<info>0.line;
                                     $$.num_if++;
                                     printf("i %d %d %d %d %d %d %d\n",
                                      $$.length, $3, 
                                      $5.length, $5.num_statements, 
                                      $7.length, $7.num_statements, t);
                                     }
        | SWITCH '(' comma_expression ')' statement    
                                    {$$ = $5;
                                     $$.length = $5.last_line - $<info>0.line;
                                     $$.num_break = 0;
                                     $$.num_switch++;
                                     printf("s %d %d %d %d %d %d\n", 
                                      $$.length, $5.num_statements, t, $3,
                                      $5.num_case, $5.num_break);
                                     }
        ;

iteration_statement:
        WHILE '(' comma_expression ')' statement       
                                    {$$ = $5;
                                     $$.length = $5.last_line - $<info>0.line;
                                     $$.num_break = 0;
                                     $$.num_continue = 0;
                                     $$.num_while++;
                                     printf("w %d %d %d %d %d\n", 
                                      $$.length, $5.num_statements, t, $3,
                                      $5.num_break + $5.num_continue); 
                                     }
        | DO statement WHILE '(' comma_expression ')' ';'   
                                    {$$ = $2;
                                     $$.length = lines+1 - $<info>0.line;
                                     $$.last_line = lines+1;
                                     $$.num_statements++;
                                     $$.num_break = 0;
                                     $$.num_continue = 0;
                                     $$.num_while++;
                                     printf("w %d %d %d %d %d\n", 
                                      $$.length, $2.num_statements, t, $5,
                                      $2.num_break + $2.num_continue);
                                     }
        | FOR '(' comma_expression_opt ';' comma_expression_opt ';'
                comma_expression_opt ')' statement     
                                    {$$ = $9;
                                     $$.length = $9.last_line - $<info>0.line;
                                     $$.num_statements += 2;
                                     $$.num_break = 0;
                                     $$.num_continue = 0;
                                     $$.num_for++;
                                     printf("f %d %d %d %d %d\n", 
                                      $$.length, $9.num_statements+2, t, 
                                      $5 == EMPTY ? 0 : $5, 
                                      $9.num_break + $9.num_continue);
                                     }
        ;

jump_statement:
        GOTO identifier_or_typedef_name ';'   {$$ = init;
                                               $$.num_statements = 1;
                                               $$.num_gotos = 1;
                                               }   
        | CONTINUE ';'                        {$$ = init;
                                               $$.num_statements = 1;
                                               $$.num_continue = 1;
                                               }
        | BREAK ';'                           {$$ = init;
                                               $$.num_statements = 1;
                                               $$.num_break = 1;
                                               }
        | RETURN comma_expression_opt ';'     {$$ = init;
                                               $$.num_statements = 1;
                                               if ($2 != EMPTY)
                                                 printf("e %d %d\n", $2, t);
                                               }
        ;



/***************************** EXTERNAL DEFINITIONS *****************************/
translation_unit:
        external_definition
        | translation_unit external_definition
        ;


external_definition:
        $hold_info function_definition

  {double ave_inblock_length, stdev_inblock_length,
          ave_depth, stdev_depth;

   Depth[blockindex++] = 1;
   if ($2.num_blocks == 0)
      $2.max_Inblock_length = 
           InblockLengthTotal[inblockindex_total++] =
                InblockLength[inblockindex++] = $2.num_statements;

   Statistic (InblockLength, inblockindex, &ave_inblock_length, 
                                             &stdev_inblock_length);
   Statistic (Depth, blockindex, &ave_depth, &stdev_depth);

   printf("F %ld %d %ld %d %d %d %d %d %d %d %d %.1f %.1f %d %.1f %.1f %d\n\n",
       bytes-$1.byte + comments-$1.comment, 
       lines+1 - $1.line, 
       comments - $1.comment, 
       $2.num_decls + $2.num_statements, 
       $2.num_if,    $2.num_while,  $2.num_for,   $2.num_switch, 
       $2.num_gotos, $2.num_labels, $2.num_blocks, 
       ave_inblock_length, stdev_inblock_length, $2.max_Inblock_length,
       ave_depth,          stdev_depth,          $2.max_depth);

   if (check(&num_functions)) YYERROR;

   AveDepth[num_functions++] = ave_depth;
   max_ave_depth_total = MAX(max_ave_depth_total, ave_depth);
   num_statements_total += ($2.num_statements + $2.num_decls);
   max_inblock_length_total = MAX(max_inblock_length_total, 
                                                   $2.max_Inblock_length);
   SetNull(InblockLength, &inblockindex);          
   SetNull(Depth, &blockindex);
   DeclFlag = 1;
   }
        | $hold_info declaration       {num_statements_total++;}
        | ';'
        ;


function_definition:
                                     identifier_declarator 
                              compound_statement              { $$ = $2; }
        | declaration_specifier      identifier_declarator 
                              compound_statement              { $$ = $3; }
        | type_specifier             identifier_declarator 
                              compound_statement              { $$ = $3; }
        | declaration_qualifier_list identifier_declarator 
                              compound_statement              { $$ = $3; }
        | type_qualifier_list        identifier_declarator 
                              compound_statement              { $$ = $3; }

        |                            old_function_declarator 
                              compound_statement              { $$ = $2; }
        | declaration_specifier      old_function_declarator 
                              compound_statement              { $$ = $3; }
        | type_specifier             old_function_declarator 
                              compound_statement              { $$ = $3; }
        | declaration_qualifier_list old_function_declarator 
                              compound_statement              { $$ = $3; }
        | type_qualifier_list        old_function_declarator 
                              compound_statement              { $$ = $3; }

        |                            old_function_declarator 
             declaration_list compound_statement   { $$ = $3;
                                                     $$.num_decls = $2; }
        | declaration_specifier      old_function_declarator 
             declaration_list compound_statement   { $$ = $4;
                                                     $$.num_decls = $3; }
        | type_specifier             old_function_declarator 
             declaration_list compound_statement   { $$ = $4;
                                                     $$.num_decls = $3; }
        | declaration_qualifier_list old_function_declarator 
             declaration_list compound_statement   { $$ = $4;
                                                     $$.num_decls = $3; }
        | type_qualifier_list        old_function_declarator 
             declaration_list compound_statement   { $$ = $4;
                                                     $$.num_decls = $3; }
        ;

declarator:
        identifier_declarator
        | typedef_declarator
        ;

typedef_declarator:
        paren_typedef_declarator          /* would be ambiguous as parameter*/
        | parameter_typedef_declarator    /* not ambiguous as param*/
        ;

parameter_typedef_declarator:
        TYPEDEFname
        | TYPEDEFname postfixing_abstract_declarator
        | clean_typedef_declarator
        ;

    /*  The  following have at least one '*'. There is no (redundant)
    '(' between the '*' and the TYPEDEFname. */

clean_typedef_declarator:
        clean_postfix_typedef_declarator
        | '*' parameter_typedef_declarator              
        | '*' type_qualifier_list parameter_typedef_declarator
        ;

clean_postfix_typedef_declarator:
        '(' clean_typedef_declarator ')'                
        | '(' clean_typedef_declarator ')' postfixing_abstract_declarator
        ;

    /* The following have a redundant '(' placed immediately  to  the
    left of the TYPEDEFname */

paren_typedef_declarator:
        paren_postfix_typedef_declarator
        | '*' '(' simple_paren_typedef_declarator ')' /* redundant paren */
        | '*' type_qualifier_list
                '(' simple_paren_typedef_declarator ')' /* redundant paren */
        | '*' paren_typedef_declarator                  
        | '*' type_qualifier_list paren_typedef_declarator
        ;

paren_postfix_typedef_declarator: /* redundant paren to left of tname*/
        '(' paren_typedef_declarator ')'                
        | '(' simple_paren_typedef_declarator postfixing_abstract_declarator ')' /* redundant paren */
        | '(' paren_typedef_declarator ')' postfixing_abstract_declarator
        ;

simple_paren_typedef_declarator:
        TYPEDEFname
        | '(' simple_paren_typedef_declarator ')'       
        ;

identifier_declarator:
        unary_identifier_declarator
        | paren_identifier_declarator
        ;

unary_identifier_declarator:
        postfix_identifier_declarator
        | '*' identifier_declarator                      
        | '*' type_qualifier_list identifier_declarator  
        ;

postfix_identifier_declarator:
        paren_identifier_declarator postfixing_abstract_declarator
        | '(' unary_identifier_declarator ')'            
        | '(' unary_identifier_declarator ')' postfixing_abstract_declarator
        ;

paren_identifier_declarator:
        IDENTIFIER  
        | '(' paren_identifier_declarator ')'            
        ;

old_function_declarator:
        postfix_old_function_declarator  
        | '*' old_function_declarator                     
        | '*' type_qualifier_list old_function_declarator 
        ;

postfix_old_function_declarator:
        paren_identifier_declarator '(' identifier_list ')'
        | '(' old_function_declarator ')'                
        | '(' old_function_declarator ')' postfixing_abstract_declarator
        ;

abstract_declarator:
        unary_abstract_declarator
        | postfix_abstract_declarator
        | postfixing_abstract_declarator
        ;

postfixing_abstract_declarator:
        array_abstract_declarator
        | '(' ')'                     
        | '(' parameter_type_list ')' 
        ;

array_abstract_declarator:
        '[' ']'                                          
        | '[' constant_expression ']'                    
        | array_abstract_declarator '[' constant_expression ']'
        ;

unary_abstract_declarator:
        '*'     
        | '*' type_qualifier_list                        
        | '*' abstract_declarator                        
        | '*' type_qualifier_list abstract_declarator    
        ;

postfix_abstract_declarator:
        '(' unary_abstract_declarator ')'           
        | '(' postfix_abstract_declarator ')'            
        | '(' postfixing_abstract_declarator ')'         
        | '(' unary_abstract_declarator ')' postfixing_abstract_declarator
        ;

%%
/************************** END OF GRAMMAR **********************************/


#include "lex.yy.c"        /* the scanner */


/***************************************************************************/
/*  The following function gives a message in case of a parser error. */
/***************************************************************************/

yyerror (string)
char *string;
{
    fprintf(stderr, "\n%s: line %d: parser error: %s at or near '%s'\n", 
                                      c_name, lines+1, string, lexem);
}


/***************************************************************************/
/* The following function checks if the index '*pnum' has passed the size
   of the respective memory array Depth, InblockLength, InblockLengthTotal
   or AveDepth and gives a message if necessary. */
/***************************************************************************/

BOOL check (pnum)
int *pnum;
{
  char *msg = "";

  if (pnum == &num_functions && *pnum >= MAX_FUNCTION)
           msg = "Too many functions!";
  else if (pnum == &blockindex && *pnum >= MAX_BLOCKS)
           msg = "Too many blocks in function!";
  else if (pnum == &inblockindex && *pnum >= MAX_INBLOCKS)
           msg = "Too many innermost blocks in function!";
  else if (pnum == &inblockindex_total && *pnum >= MAX_INBLOCKS*MAX_FUNCTION)
           msg = "Too many innermost blocks in module!";
	    
  if (*msg != '\0') {
    fprintf(stderr, "Error: %s\n", msg);
    FatalError = 1;
    return 1;
       }
  return 0;
}


/***************************************************************************/
/*  The following function adds the results of two statement rules. */
/***************************************************************************/

STATEMENT_VAL add (x, y)
STATEMENT_VAL x, y;
{
  STATEMENT_VAL v;

  v.num_if             = x.num_if         + y.num_if;
  v.num_switch         = x.num_switch     + y.num_switch;
  v.num_while          = x.num_while      + y.num_while;
  v.num_for            = x.num_for        + y.num_for;
  v.num_gotos          = x.num_gotos      + y.num_gotos;
  v.num_labels         = x.num_labels     + y.num_labels;
  v.num_case           = x.num_case       + y.num_case;
  v.num_break          = x.num_break      + y.num_break;
  v.num_continue       = x.num_continue   + y.num_continue;
  v.num_statements     = x.num_statements + y.num_statements;
  v.num_decls          = 0;   /* not relevant here */
  v.num_blocks         = x.num_blocks     + y.num_blocks;
  v.max_depth          = MAX(x.max_depth, y.max_depth);
  v.max_Inblock_length = MAX(x.max_Inblock_length, y.max_Inblock_length);
  v.length             = 0;   /* not relevant here */
  v.last_line          = y.last_line;    

  return(v);
}


/***************************************************************************/
/*  The following function enters a string in the hash table. */
/***************************************************************************/

void henter (string)
char *string;
{ 
  static char *str_ptr = string_space; 
  ENTRY item;

  strcpy(str_ptr, string);
  item.key = str_ptr;
  str_ptr += strlen(str_ptr) + 1;
  hsearch(item, ENTER);
}


/***************************************************************************/
/*  The following function looks for a string in the hash table. */
/***************************************************************************/

BOOL hfind (string)
char *string;
{
  ENTRY item;
  
  item.key = string;
  return (hsearch(item, FIND) != NULL);
}


/***************************************************************************/
/*  The following function supplies the average and the standard deviation
    of a list of values. */
/***************************************************************************/

void Statistic (a, n, ave_ptr, std_dev_ptr)
double a[];
int n;                             /* size of the list */
double *ave_ptr, *std_dev_ptr;
{
  int i;
  double sum = 0;

  for (i = 0; i < n; i++) 
     sum += a[i];
  *ave_ptr = ( n > 0 ? sum / n : 0 );

  sum = 0;
  for (i = 0; i < n; i++) 
     sum += SQR(a[i] - *ave_ptr);
  *std_dev_ptr = ( n > 1 ? sqrt( sum / (n - 1) ) : 0 );
}


/***************************************************************************/
/*  The following function sets each element of a list to 0. */
/***************************************************************************/

void SetNull (a, pn)
double a[];
int *pn;
{
  int i;

  for (i = 0; i < *pn; i++)
     a[i] = 0;
  *pn = 0;
}


/***************************************************************************/
/*  The following function shows the according context of the module in 
    case of a parser error, i.e. the error line with four additional lines
    before and after it. */
/***************************************************************************/

void ShowErrorContext()
{
  int i;
  char str[STR_LEN];

  rewind(in);
  printf("----------------------------------------------------------------------\n");
  for (i = 1; i <= lines+5 && gets(str) != NULL; i++)
       if (i >= lines-3)  {
          if (i == lines+1) printf(">> ");
                       else printf("   ");
          puts(str);
       }
  printf("----------------------------------------------------------------------");
}


/***************************************************************************/
/*  The following function is the main routine, that prepares the parsing 
    (e.g. putting all entries of the .tpp-file in a hash table) and 
    activates the parser. If the module has successfully been parsed the 
    results of the module are added to the output file in the module line
    'M'. If there is a parser error the present results (i.e. the output 
    file) are deleted, the error messages and the request are indicated, 
    and finally the program waits for your entry before proceeding 
    accordingly. 
    Note that there are 4 kinds of 'exit' that allow us to handle further 
    actions by 'ccounter':

      0: the module has successfully been parsed
         -> 'ccounter' action:  "Go to the next module if existing"
      1: the user gives up in case of parser error
         -> 'ccounter' action:  "Go to the next module if existing"
      2: there was a entry in the .tpp-file (in case of parser error)
         or the user demands a repetition of the run (perhaps after he has
         edited the module source file in another window in case of parser
         error)
         -> 'ccounter' action:  "Repeat the run of 'ccount'"
      3: the user want to repeat the run by following the lexical analysis
         -> 'ccounter' action:  "Repeat the run of 'ccount' with -v option" */
/***************************************************************************/

main (argc, argv)
int argc;
char *argv[];
{ 
  char str[STR_LEN], str2[STR_LEN];  

  if (argc < 2)  { 
    fprintf(stderr, "Usage: ccount [-v] <basename.c>\n");
    exit(1);
  } 

  if (!strcmp(*++argv, "-v")) {        /* -v option is set ? */
           Debug = 1;
           if (argc < 3) {
             fprintf(stderr, "Usage: ccount [-v] <basename.c>\n");
             exit(1);
           }
           ++argv;
	 }

  /* now form the names of the output file and the .tpp-file */

  strcpy(c_name, *argv);
  strncpy(out_name, c_name, strlen(c_name)-1);
  strncpy(tpp_name, c_name, strlen(c_name)-1);
  strcat(out_name, "out");
  strcat(tpp_name, "tpp");  

  if ((in = freopen(c_name, "r", stdin)) == NULL)  {
    fprintf(stderr, "Cannot open %s\n", c_name);
    exit(1);
  }
  out = freopen(out_name, "w", stdout);
  hcreate(NUM_EMPL);                       /* create hash table */

  /* now put all entries of the .tpp-file in the hash table */

  if ((tppfile = fopen(tpp_name, "r")) != NULL)  {
    while (fgets(str, STR_LEN, tppfile) != NULL) {
               sprintf(str2, "");
               strncat(str2, str, strlen(str)-1);
               henter(str2);
              }
    fclose(tppfile);
  }

  if (!yyparse()) {              /* activate the parser */

    /* the module has successfully been parsed */

    double ave_inblock_length_total, stdev_inblock_length_total,
           ave_ave_depth, stdev_ave_depth; 

    Statistic (InblockLengthTotal, inblockindex_total, 
                  &ave_inblock_length_total, &stdev_inblock_length_total);
    Statistic (AveDepth, num_functions, &ave_ave_depth, &stdev_ave_depth); 

    printf("M %ld %d %ld %ld %d %ld %d %ld %d %ld %d %d %.1f %.1f %d %.1f %.1f %.1f\n",
             bytes+comments+Pcomments, lines, comments, Pcomments,
             num_identifier, bytes_identifier,
             num_symbols, bytes_symbols, 
             num_ppdirectives, bytes_ppdirectives, 
             num_statements_total, num_functions, 
             ave_inblock_length_total, 
             stdev_inblock_length_total, 
             max_inblock_length_total,
             ave_ave_depth, 
             stdev_ave_depth, 
             max_ave_depth_total);
 
    out = freopen("/dev/tty", "w", out);
    in = freopen("/dev/tty", "r", in);
    printf("Success. Results in '%s'.\n", out_name); 
    exit(0);
  } 
  else  {

    /* parser error */

    out = freopen("/dev/tty", "w", out);
    unlink(out_name);  
    ShowErrorContext();
    in = freopen("/dev/tty", "r", in);
    if (FatalError) exit(1);

    ENTER:
    printf("\nPlease enter type name OR\n");
    printf("preprocessor conditional directive beginning with '#' OR\n");
    printf("name of macro declaration beginning with '&' OR\n");
    printf("identifier to be ignored beginning with '!' OR\n");
    printf("just RETURN to give up on this file!\n");
    printf("'@' at the end will repeat this request:\n");

    if (gets(str) != NULL)  
        switch (*str) {
            case '\0': printf("Good bye!\n");
                       exit(1);
            case '%' : exit(2);       /* -> 'ccounter' will repeat the run */
            case '*' : exit(3);       /* -> 'ccounter' will repeat the run
                                            with the -v option */
            default  : if (str[strlen(str)-1] == '@') 
                         goto ENTER;
                       else {
                         /* store the word in the .tpp-file */
                         tppfile = fopen(tpp_name, "a");
                         fprintf(tppfile, "%s\n", str);
                         fclose(tppfile);
                         printf("'%s' is saved in %s\n", str, tpp_name);
                         exit(2);     /* -> 'ccounter' will repeat the run */
                       }
          }
  }
}
