#
# Module for performing searches in Text widgets
#


# Finds next occurrence of string in w.
proc th_Text_find {w string {next 1} {direction "forward"} {type "string"}} {
  global TH
  set search $TH(Search,Select,[winfo class $w])

  if {($search != "sel") && ([lsearch [$w tag names] $search] == -1)} {
    eval $w tag configure $search $TH(Search,Text,Tag)
  }
  if {[lsearch [$w tag names] was_search] == -1} {
    eval $w tag configure was_search $TH(Search,Text,Oldtag)
    $w tag lower was_search $search
  }
  set was_search_area [$w tag nextrange $search 1.0 end]
  if {([llength $was_search_area] == 2) &&
      [$w compare [lindex $was_search_area 0] <= insert] &&
      [$w compare insert <= [lindex $was_search_area 1]]} {
    eval $w tag remove $search $was_search_area
    eval $w tag add was_search $was_search_area

    if $next {
      set border(forward) "[lindex $was_search_area 0] +1c"
      set border(backward) "[lindex $was_search_area 1] -1c"
    } else {
      set border(forward) [lindex $was_search_area 0]
      set border(backward) "[lindex $was_search_area 1] +1c"
  }} else {
    set border(forward) [$w index insert]
    set border(backward) [$w index insert]
  }
  if {[catch "set TH(Search,Failed,$w)"]} {set TH(Search,Failed,$w) 0}
  if $TH(Search,Failed,$w) {
    set border(forward) 1.0
    set border(backward) end
  }

  set length [string length $string]
  if {($length == 0)} {return 0}

  set slist [th_Text_[set type]_[set direction] $w $border($direction) $string]
  if {($slist != "")} {
    $w tag add $search [lindex $slist 0] [lindex $slist 1]
    $w mark set insert [lindex $slist 0]
    $w yview -pickplace insert
    set TH(Search,Failed,$w) 0
    return 1
  } else {
    set TH(Search,Failed,$w) 1
    return 0
}}


# Finds first occurrence of string after point in widget w
# Returns start and end of occurance or "" if unsuccessful.
proc th_Text_string_forward {w point string} {
  set answer [th_Text_string_first $w $string $point]
  if {($answer == "")} {return ""
  } else {return [list "$answer" "$answer +[string length $string] chars"]
}}

# Finds last occurrence of string before point in widget w
# Returns start and end of occurance or "" if unsuccessful.
proc th_Text_string_backward {w point string} {
  set answer [th_Text_string_last $w $string $point]
  if {($answer == "")} {return ""
  } else {return [list $answer "$answer +[string length $string] chars"]
}}


# The glob routines find first occurrence of string after point in widget
# w, using Tcl's string match command (which uses glob expressions).
# Returns start and end of occurance or "" if unsuccessful.

set TH(Glob,Maxlines) 2

proc th_Text_glob_forward {w point string} {return [th_Text_glob $w $point $string forward]}

proc th_Text_glob_backward {w point string} {return [th_Text_glob $w $point $string backward]}

proc th_Text_glob {w point string direction} {
  if {![regexp -- {([^][?*\]*)(.*)} $string dummy start]} {return}
  if {$string == $start} {
    return [th_Text_string_[set direction] $w $point $string]}
  set l [string length $string] ; incr l -1
  set end ""
  while {[regexp -- {[^][?*\]} [string index $string $l]]} {
    set char [string index $string $l]
    incr l -1
    if {[string index $string $l] == "\\"} {break}
    set end "$char$end"
  }

  set start_index $point
  set end_index $point
  set sl [string length $start]
  set el [string length $end]
  global TH
  if {[catch "set TH(Search,Case,$w)"]} {set TH(Search,Case,$w) 0}

  if {$direction == "forward"} {set fn "th_Text_string_first"
  } else {set fn "th_Text_string_last"}

  if {($start != "") && ($end != "")} {
    set end_flag 1
    while {[set start_index [$fn $w $start $start_index]] != ""} {
      set start_index [$w index $start_index]
      if {!(($direction == "forward") && [$w compare $start_index <= $point])} {
        set text [$w get $start_index end]
        if $TH(Search,Case,$w) {set text [string tolower $text]}
        if {[string match "[set string]*" $text]} {
          set end_index $start_index
          set end_flag 0
          while {[set end_index [th_Text_string_first $w $end \
  		"$end_index +1c"]] != ""} {
            set end_flag 1
            set end_index [$w index "$end_index + $el c"]
            if {[$w compare "$start_index +$TH(Glob,Maxlines) l" < $end_index]} {break}
            if {!(($direction == "backward") && [$w compare $end_index >= $point])} {
              set text [$w get $start_index $end_index]
              if $TH(Search,Case,$w) {set text [string tolower $text]}
              if {[string match $string $text]} {
                return [list $start_index $end_index]
      }}}}}
      if {$direction == "forward"} {
        set start_index [$w index "$start_index +1c"]
      } else {
        set start_index [$w index "$start_index +$sl c -1c"]
    }}
    if {!$end_flag} {return ""}

  } elseif {($start != "") && ($end == "")} {
    while {[set start_index [$fn $w $start $start_index]] != ""} {
      set start_index [$w index $start_index]
      set end_index "$start_index + $sl c -1c wordend"
      if {!(($direction == "forward") && [$w compare $start_index <= $point]) &&
          !(($direction == "backward") && [$w compare $end_index >= $point])} {
        set text [$w get $start_index $end_index]
        if $TH(Search,Case,$w) {set text [string tolower $text]}
        if {[string match $string $text]} {
            return [list $start_index $end_index]}}
      if {$direction == "forward"} {
        set start_index [$w index "$start_index +1c"]
      } else {set start_index [$w index "$start_index +$sl c -1c"]

  }}} elseif {$end != ""} {
    while {[set end_index [$fn $w $end $end_index]] != ""} {
      set end_index "[$w index $end_index] + $el c"
      set start_index [$w index "$end_index - $el c wordstart"]
      if {!(($direction == "forward") && [$w compare $start_index <= $point]) &&
          !(($direction == "backward") && [$w compare $end_index >= $point])} {
        set text [$w get $start_index $end_index]
        if $TH(Search,Case,$w) {set text [string tolower $text]}
        if {[string match $string $text]} {
          return [list $start_index $end_index]}}
      if {$direction == "forward"} {
        set end_index [$w index "$end_index +1c"]
      } else {
        set end_index [$w index "$end_index +$sl c -1c"]

  }}} else {
    while {1} {
      set end_index [$w index "$start_index wordend"]
      if {!(($direction == "forward") && [$w compare $start_index <= $point]) &&
          !(($direction == "backward") && [$w compare $end_index >= $point])} {
        set text [$w get $start_index $end_index]
        if $TH(Search,Case,$w) {set text [string tolower $text]}
        if {[string match $string $text]} {
          return [list $start_index $end_index]
      }}
      if {$direction == "forward"} {
        set new_index $end_index
      } else {set new_index "$start_index -1c wordstart"}
      if {[$w compare $new_index == $start_index]} {break}
      set start_index $new_index
  }}
  return ""
}


# Finds first occurrence of string after point in widget w using regexp.
# Returns start and end of occurance or "" if unsuccessful.
proc th_Text_regexp_forward {w point string} {
  global TH
  if {[catch "set TH(Search,Case,$w)"]} {set TH(Search,Case,$w) 0}

  if $TH(Search,Case,$w) {
    if {[catch {regexp -indices -nocase -- $string [$w get $point end] where} result]} {th_beep ; return}
  } else {if {[catch {regexp -indices -- $string [$w get $point end] where} result]} {th_beep ; return}
  }
  if {!$result} {return ""}

  return [list "$point +[lindex $where 0] chars" \
		"$point +[lindex $where 1] chars +1 chars"]
}

# Like regexp_forward, but searches from beginning to point.
proc th_Text_regexp_backward {w point string} {
  global TH
  if {[catch "set TH(Search,Case,$w)"]} {set TH(Search,Case,$w) 0}

  set text [$w get 1.0 $point]
  set text_range $text
  set offset 0
  set answer ""
  while {1} {
    if $TH(Search,Case,$w) {
      if {[catch {regexp -indices -nocase -- $string $text_range where} result]} {th_beep ; return}
    } else {if {[catch {regexp -indices -- $string $text_range where} result]} {th_beep ; return}
    }
    if $result {
      set answer [list "1.0 +$offset chars +[lindex $where 0] chars" \
			"1.0 +$offset chars +[lindex $where 1] chars +1 chars"]
    } else {break}
    incr offset [expr [lindex $where 0] +1]
    set text_range [string range $text $offset end]
  }
  return $answer
}


# Finds all future occurrences of string 'n highlights 'em.
proc th_Text_highlight_searches {w string} {
  set label "[th_frame $w].search.l"
  set text [lindex [$label configure -text] 4]
  set direction [lindex $text 0]
  set type [lindex $text 1]
  while {[th_Text_find $w $string 1 $direction $type]} {}
}


# Cleans up widget from searching.
proc th_Text_search_exit {w} {
  global TH
  if {[lsearch [$w tag names] $TH(Search,Select,[winfo class $w])] >= 0} {
    $w tag remove $TH(Search,Select,[winfo class $w]) 1.0 end
  }
  if {[lsearch [$w tag names] was_search] >= 0} {
    $w tag remove was_search 1.0 end
}}


# Functionally equivalent to saying [string first $string [$w get $start $end]]
# except the index of the found string is returned, not just a character offset.
# However, it runs faster when $start and $end are far apart and $string is
# close to $start.
proc th_Text_string_first {w string {start 1.0} {end end}} {
  global TH
  if {[catch "set TH(Search,Case,$w)"]} {set TH(Search,Case,$w) 0}

  set start [$w index $start]
  set end [$w index $end]
  if {[$w compare "$start lineend" == "$end lineend"]} {
    set first_end $end
  } else {set first_end "$start lineend"}
  set text [$w get $start $first_end]
  if $TH(Search,Case,$w) {set text [string tolower $text]}
  set result [string first "$string" $text]
  if {$result != -1} {return "$start +$result chars"}
# Not on line with start, start traversing down.
  scan $start "%d.%d" row dummy
  scan $end "%d.%d" end_row dummy
  if {$row == $end_row} {return ""}
  set incr_factor 1
  for {incr row} {[expr $row + $incr_factor] < $end_row} \
      {incr row $incr_factor
      set incr_factor [expr $incr_factor * 2]} {
    set text [$w get "$row.0" "$row.0 +$incr_factor lines -1c"]
    if $TH(Search,Case,$w) {set text [string tolower $text]}
    set result [string first "$string" $text]
    if {$result != -1} {return "$row.0 + $result chars"}
  }
# Not on any line, except maybe last.
  set text [$w get "$row.0" $end]
  if $TH(Search,Case,$w) {set text [string tolower $text]}
  set result [string first "$string" $text]
  if {$result != -1} {return "$row.0 + $result chars"
  } else {return ""}
}

# Like [string last [$w get $start $end]], but more efficient if "$string" is
# close to $end.
proc th_Text_string_last {w string {end end} {start 1.0}} {
  global TH
  if {[catch "set TH(Search,Case,$w)"]} {set TH(Search,Case,$w) 0}

  set start [$w index $start]
  set end [$w index $end]
  if {[$w compare "$start linestart" == "$end linestart"]} {
    set first_start $start
  } else {set first_start "$end linestart"}
  set text [$w get $first_start $end]
  if $TH(Search,Case,$w) {set text [string tolower $text]}
  set result [string last "$string" $text]
  if {$result != -1} {return "$first_start +$result chars"}
# Not on line with end, start traversing up
  scan $start "%d.%d" start_row dummy
  scan $end "%d.%d" row dummy
  if {$row == $start_row} {return ""}
  set incr_factor 1
  for {incr row -1} {[expr $row - $incr_factor] > $start_row} \
      {set incr_factor [expr $incr_factor * 2]
      incr row -$incr_factor} {
    set text [$w get "$row.0" "$row.0 +$incr_factor lines -1c"]
    if $TH(Search,Case,$w) {set text [string tolower $text]}
    set result [string last "$string" $text]
    if {$result != -1} {return "$row.0 + $result chars"}
  }
# Not on any line, except maybe first
  incr row $incr_factor
  set text [$w get $start "$row.0 -1c"]
  if $TH(Search,Case,$w) {set text [string tolower $text]}
  set result [string last "$string" $text]
  if {$result != -1} {return "$start + $result chars"
  } else {return ""}
}


