# text.tcl --
#
# This file defines the default bindings for Tk text widgets.
#
# @(#) text.tcl 1.20 95/01/08 16:10:01
#
# Copyright (c) 1992-1994 The Regents of the University of California.
# Copyright (c) 1994-1995 Sun Microsystems, Inc.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#

# tkTextBind --
# This procedure below invoked the first time the mouse enters a text
# widget or a text widget receives the input focus.  It creates all of
# the class bindings for texts.
#
# Arguments:
# event -	Indicates which event caused the procedure to be invoked
#		(Enter or FocusIn).  It is used so that we can carry out
#		the functions of that event in addition to setting up
#		bindings.

proc tkTextBind event {
    global tkPriv tk_strictMotif

    bind Text <FocusIn> {}

    # Standard Motif bindings:

    bind Text <Left> {
	tkTextSetCursor %W [%W index {insert - 1c}]
    }
    bind Text <Right> {
	tkTextSetCursor %W [%W index {insert + 1c}]
    }
    bind Text <Up> {
	tkTextSetCursor %W [tkTextUpDownLine %W -1]
    }
    bind Text <Down> {
	tkTextSetCursor %W [tkTextUpDownLine %W 1]
    }
    bind Text <Shift-Left> {
	tkTextKeySelect %W [%W index {insert - 1c}]
    }
    bind Text <Shift-Right> {
	tkTextKeySelect %W [%W index {insert + 1c}]
    }
    bind Text <Shift-Up> {
	tkTextKeySelect %W [tkTextUpDownLine %W -1]
    }
    bind Text <Shift-Down> {
	tkTextKeySelect %W [tkTextUpDownLine %W 1]
    }
    bind Text <Control-Left> {
	tkTextSetCursor %W [%W index {insert - 1c wordstart}]
    }
    bind Text <Control-Right> {
	tkTextSetCursor %W [%W index {insert wordend}]
    }
    bind Text <Control-Up> {
	tkTextSetCursor %W [tkTextPrevPara %W insert]
    }
    bind Text <Control-Down> {
	tkTextSetCursor %W [tkTextNextPara %W insert]
    }
    bind Text <Shift-Control-Left> {
	tkTextKeySelect %W [%W index {insert - 1c wordstart}]
    }
    bind Text <Shift-Control-Right> {
	tkTextKeySelect %W [%W index {insert wordend}]
    }
    bind Text <Shift-Control-Up> {
	tkTextKeySelect %W [tkTextPrevPara %W insert]
    }
    bind Text <Shift-Control-Down> {
	tkTextKeySelect %W [tkTextNextPara %W insert]
    }
    bind Text <Prior> {
	tkTextSetCursor %W [tkTextScrollPages %W -1]
    }
    bind Text <Shift-Prior> {
	tkTextKeySelect %W [tkTextScrollPages %W -1]
    }
    bind Text <Next> {
	tkTextSetCursor %W [tkTextScrollPages %W 1]
    }
    bind Text <Shift-Next> {
	tkTextKeySelect %W [tkTextScrollPages %W 1]
    }
    bind Text <Control-Prior> {
	%W xview scroll -1 page
    }
    bind Text <Control-Next> {
	%W xview scroll 1 page
    }

    bind Text <Home> {
	tkTextSetCursor %W {insert linestart}
    }
    bind Text <Shift-Home> {
	tkTextKeySelect %W {insert linestart}
    }
    bind Text <End> {
	tkTextSetCursor %W {insert lineend}
    }
    bind Text <Shift-End> {
	tkTextKeySelect %W {insert lineend}
    }
    bind Text <Control-Home> {
	tkTextSetCursor %W 1.0
    }
    bind Text <Control-Shift-Home> {
	tkTextKeySelect %W 1.0
    }
    bind Text <Control-End> {
	tkTextSetCursor %W {end - 1 char}
    }
    bind Text <Control-Shift-End> {
	tkTextKeySelect %W {end - 1 char}
    }

#    bind Text <Tab> {
#	tkTextInsert %W \t
#	focus %W
#	break
#    }
#    bind Text <Shift-Tab> {
#	# Needed only to keep <Tab> binding from triggering;  doesn't
#	# have to actually do anything.
#    }
#    bind Text <Control-Tab> {
#	tk_focusNext %W
#    }
#    bind Text <Control-Shift-Tab> {
#	tk_focusPrev %W
#    }
    bind Text <Control-i> {
	tkTextInsert %W \t
    }
    bind Text <Return> {
	tkTextInsert %W \n
    }
    bind Text <Delete> {
	if {[%W tag nextrange sel 1.0 end] != ""} {
	    %W delete sel.first sel.last
	} else {
	    %W delete insert
	    %W see insert
	}
    }
    bind Text <BackSpace> {
	if {[%W tag nextrange sel 1.0 end] != ""} {
	    %W delete sel.first sel.last
	} elseif [%W compare insert != 1.0] {
	    %W delete insert-1c
	    %W see insert
	}
    }

    bind Text <Control-space> {
	%W mark set anchor insert
    }
    bind Text <Select> {
	%W mark set anchor insert
    }
    bind Text <Control-Shift-space> {
	set tkPriv(selectMode) char
	tkTextSelectTo %W insert
    }
    bind Text <Shift-Select> {
	set tkPriv(selectMode) char
	tkTextSelectTo %W insert
    }
    bind Text <Control-slash> {
	%W tag add sel 1.0 end
    }
    bind Text <Control-backslash> {
	%W tag remove sel 1.0 end
    }
    bind Text <Insert> {
	catch {tkTextInsert %W [selection get -displayof %W]}
    }
    bind Text <KeyPress> {
	tkTextInsert %W %A
    }

    # Additional emacs-like bindings:

    if !$tk_strictMotif {
	bind Text <Control-a> {
	    tkTextSetCursor %W {insert linestart}
	}
	bind Text <Control-b> {
	    tkTextSetCursor %W insert-1c
	}
	bind Text <Control-d> {
	    %W delete insert
	}
	bind Text <Control-e> {
	    tkTextSetCursor %W {insert lineend}
	}
	bind Text <Control-f> {
	    tkTextSetCursor %W insert+1c
	}
	bind Text <Control-k> {
	    if [%W compare insert == {insert lineend}] {
		%W delete insert
	    } else {
		%W delete insert {insert lineend}
	    }
	}
	bind Text <Control-n> {
	    tkTextSetCursor %W [tkTextUpDownLine %W 1]
	}
	bind Text <Control-o> {
	    %W insert insert \n
	    %W mark set insert insert-1c
	}
	bind Text <Control-p> {
	    tkTextSetCursor %W [tkTextUpDownLine %W -1]
	}
	bind Text <Control-t> {
	    tkTextTranspose %W
	}
	bind Text <Meta-b> {
	    tkTextSetCursor %W {insert - 1c wordstart}
	}
	bind Text <Meta-d> {
	    %W delete insert {insert wordend}
	}
	bind Text <Meta-f> {
	    tkTextSetCursor %W {insert wordend}
	}
	bind Text <Meta-less> {
	    tkTextSetCursor %W 1.0
	}
	bind Text <Meta-greater> {
	    tkTextSetCursor %W end-1c
	}
	bind Text <Meta-BackSpace> {
	    %W delete {insert -1c wordstart} insert
	}

	# A few additional bindings of my own.
    
	bind Text <Control-h> {
	    if [%W compare insert != 1.0] {
		%W delete insert-1c
		%W see insert
	    }
	}
    }

    rename tkTextBind {}
    set tkPriv(prevPos) {}
}


# tkTextSelectTo --
# This procedure is invoked to extend the selection, typically when
# dragging it with the mouse.  Depending on the selection mode (character,
# word, line) it selects in different-sized units.  This procedure
# ignores mouse motions initially until the mouse has moved from
# one character to another or until there have been multiple clicks.
#
# Arguments:
# w -		The text window in which the button was pressed.
# index -	Index of character at which the mouse button was pressed.

proc tkTextSelectTo {w index} {
    global tkPriv

    set cur [$w index $index]
    if [catch {$w index anchor}] {
	$w mark set anchor $cur
    }
    set anchor [$w index anchor]
    if [$w compare $cur != $anchor] {
	set tkPriv(mouseMoved) 1
    }
    switch $tkPriv(selectMode) {
	char {
	    if [$w compare $cur < anchor] {
		set first $cur
		set last anchor
	    } else {
		set first anchor
		set last $cur
	    }
	}
	word {
	    if [$w compare $cur < anchor] {
		set first [$w index "$cur wordstart"]
		set last [$w index "anchor - 1c wordend"]
	    } else {
		set first [$w index "anchor wordstart"]
		set last [$w index "$cur wordend"]
	    }
	}
	line {
	    if [$w compare $cur < anchor] {
		set first [$w index "$cur linestart"]
		set last [$w index "anchor - 1c lineend + 1c"]
	    } else {
		set first [$w index "anchor linestart"]
		set last [$w index "$cur lineend + 1c"]
	    }
	}
    }
    if {$tkPriv(mouseMoved) || ($tkPriv(selectMode) != "char")} {
	$w tag remove sel 0.0 $first
	$w tag add sel $first $last
	$w tag remove sel $last end
	update idletasks
    }
}

# tkTextSetCursor
# Move the insertion cursor to a given position in a text.  Also
# clears the selection, if there is one in the text, and makes sure
# that the insertion cursor is visible.  Also, don't let the insertion
# cursor appear on the dummy last line of the text.
#
# Arguments:
# w -		The text window.
# pos -		The desired new position for the cursor in the window.

proc tkTextSetCursor {w pos} {
    global tkPriv

    if [$w compare $pos == end] {
	set pos {end - 1 chars}
    }
    $w mark set insert $pos
    $w tag remove sel 1.0 end
    $w see insert
}

# tkTextKeySelect
# This procedure is invoked when stroking out selections using the
# keyboard.  It moves the cursor to a new position, then extends
# the selection to that position.
#
# Arguments:
# w -		The text window.
# new -		A new position for the insertion cursor (the cursor hasn't
#		actually been moved to this position yet).

proc tkTextKeySelect {w new} {
    global tkPriv

    if {[$w tag nextrange sel 1.0 end] == ""} {
	if [$w compare $new < insert] {
	    $w tag add sel $new insert
	} else {
	    $w tag add sel insert $new
	}
    } else {
	if [$w compare $new < anchor] {
	    set first $new
	    set last anchor
	} else {
	    set first anchor
	    set last $new
	}
	$w tag remove sel 1.0 $first
	$w tag add sel $first $last
	$w tag remove sel $last end
    }
    $w mark set insert $new
    $w see insert
    update idletasks
}

# tkTextInsert --
# Insert a string into a text at the point of the insertion cursor.
# If there is a selection in the text, and it covers the point of the
# insertion cursor, then delete the selection before inserting.
#
# Arguments:
# w -		The text window in which to insert the string
# s -		The string to insert (usually just a single character)

proc tkTextInsert {w s} {
    if {$s == ""} {
	return
    }
    catch {
	if {[$w compare sel.first <= insert]
		&& [$w compare sel.last >= insert]} {
	    $w delete sel.first sel.last
	}
    }
    $w insert insert $s
    $w see insert
}

# tkTextUpDownLine --
# Returns the index of the character one line above or below the
# insertion cursor.  There are two tricky things here.  First,
# we want to maintain the original column across repeated operations,
# even though some lines that will get passed through don't have
# enough characters to cover the original column.  Second, don't
# try to scroll past the beginning or end of the text.
#
# Arguments:
# w -		The text window in which the cursor is to move.
# n -		The number of lines to move: -1 for up one line,
#		+1 for down one line.

proc tkTextUpDownLine {w n} {
    global tkPriv

    set i [$w index insert]
    scan $i "%d.%d" line char
    if {[string compare $tkPriv(prevPos) $i] != 0} {
	set tkPriv(char) $char
    }
    set new [$w index [expr $line + $n].$tkPriv(char)]
    if {[$w compare $new == end] || [$w compare $new == "insert linestart"]} {
	set new $i
    }
    set tkPriv(prevPos) $new
    return $new
}

# tkTextPrevPara --
# Returns the index of the beginning of the paragraph just before a given
# position in the text (the beginning of a paragraph is the first non-blank
# character after a blank line).
#
# Arguments:
# w -		The text window in which the cursor is to move.
# pos -		Position at which to start search.

proc tkTextPrevPara {w pos} {
    set pos [$w index "$pos linestart"]
    while 1 {
	if {(([$w get "$pos - 1 line"] == "\n") && ([$w get $pos] != "\n"))
		|| ($pos == "1.0")} {
	    if [regexp -indices {^[ 	]+(.)} [$w get $pos "$pos lineend"] \
		    dummy index] {
		set pos [$w index "$pos + [lindex $index 0] chars"]
	    }
	    if {[$w compare $pos != insert] || ($pos == "1.0")} {
		return $pos
	    }
	}
	set pos [$w index "$pos - 1 line"]
    }
}

# tkTextNextPara --
# Returns the index of the beginning of the paragraph just after a given
# position in the text (the beginning of a paragraph is the first non-blank
# character after a blank line).
#
# Arguments:
# w -		The text window in which the cursor is to move.
# start -	Position at which to start search.

proc tkTextNextPara {w start} {
    set pos [$w index "$start linestart + 1 line"]
    while {[$w get $pos] != "\n"} {
	if [$w compare $pos == end] {
	    return [$w index "end - 1c"]
	}
	set pos [$w index "$pos + 1 line"]
    }
    while {[$w get $pos] == "\n"} {
	set pos [$w index "$pos + 1 line"]
	if [$w compare $pos == end] {
	    return [$w index "end - 1c"]
	}
    }
    if [regexp -indices {^[ 	]+(.)} [$w get $pos "$pos lineend"] \
	    dummy index] {
	return [$w index "$pos + [lindex $index 0] chars"]
    }
    return $pos
}

# tkTextScrollPages --
# This is a utility procedure used in bindings for moving up and down
# pages and possibly extending the selection along the way.  It scrolls
# the view in the widget by the number of pages, and it returns the
# index of the character that is at the same position in the new view
# as the insertion cursor used to be in the old view.
#
# Arguments:
# w -		The text window in which the cursor is to move.
# count -	Number of pages forward to scroll;  may be negative
#		to scroll backwards.

proc tkTextScrollPages {w count} {
    set bbox [$w bbox insert]
    $w yview scroll $count pages
    if {$bbox == ""} {
	return [$w index @[expr [winfo height $w]/2],0]
    }
    set x [expr [lindex $bbox 0] + [lindex $bbox 2]/2]
    set y [expr [lindex $bbox 1] + [lindex $bbox 3]/2]
    return [$w index @$x,$y]
}

# tkTextTranspose --
# This procedure implements the "transpose" function for text widgets.
# It tranposes the characters on either side of the insertion cursor,
# unless the cursor is at the end of the line.  In this case it
# transposes the two characters to the left of the cursor.  In either
# case, the cursor ends up to the right of the transposed characters.
#
# Arguments:
# w -		Text window in which to transpose.

proc tkTextTranspose w {
    set pos insert
    if [$w compare $pos != "$pos lineend"] {
	set pos [$w index "$pos + 1 char"]
    }
    set new [$w get "$pos - 1 char"][$w get  "$pos - 2 char"]
    if [$w compare "$pos - 1 char" == 1.0] {
	return
    }
    puts "new is $new"
    $w delete "$pos - 2 char" $pos
    $w insert insert $new
    $w see insert
}
