# entry.tcl --
#
# This file defines the default bindings for Tk entry widgets.
#
# @(#) entry.tcl 1.23 95/01/08 16:10:00
#
# Copyright (c) 1992-1994 The Regents of the University of California.
# Copyright (c) 1994-1995 Sun Microsystems, Inc.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#

# tkEntryBind --
# This procedure is invoked the first time the mouse enters an
# entry widget or an entry widget receives the input focus.  It creates
# all of the class bindings for entries.
#
# Arguments:
# event -	Indicates which event caused the procedure to be invoked
#		(Enter or FocusIn).  It is used so that we can carry out
#		the functions of that event in addition to setting up
#		bindings.

proc tkEntryBind event {
    global tk_strictMotif

    bind Entry <FocusIn> {}

    # Standard Motif bindings:

    bind Entry <Left> {
	tkEntrySetCursor %W [expr [%W index insert] - 1]
    }
    bind Entry <Right> {
	tkEntrySetCursor %W [expr [%W index insert] + 1]
    }
    bind Entry <Shift-Left> {
	tkEntryKeySelect %W [expr [%W index insert] - 1]
	tkEntrySeeInsert %W
    }
    bind Entry <Shift-Right> {
	tkEntryKeySelect %W [expr [%W index insert] + 1]
	tkEntrySeeInsert %W
    }
    bind Entry <Control-Left> {
	tkEntrySetCursor %W \
		[string wordstart [%W get] [expr [%W index insert] - 1]]
    }
    bind Entry <Control-Right> {
	tkEntrySetCursor %W [string wordend [%W get] [%W index insert]]
    }
    bind Entry <Shift-Control-Left> {
	tkEntryKeySelect %W \
		[string wordstart [%W get] [expr [%W index insert] - 1]]
	tkEntrySeeInsert %W
    }
    bind Entry <Shift-Control-Right> {
	tkEntryKeySelect %W [string wordend [%W get] [%W index insert]]
	tkEntrySeeInsert %W
    }
    bind Entry <Home> {
	tkEntrySetCursor %W 0
    }
    bind Entry <Shift-Home> {
	tkEntryKeySelect %W 0
	tkEntrySeeInsert %W
    }
    bind Entry <End> {
	tkEntrySetCursor %W end
    }
    bind Entry <Shift-End> {
	tkEntryKeySelect %W end
	tkEntrySeeInsert %W
    }

    bind Entry <Delete> {
	if [%W selection present] {
	    %W delete sel.first sel.last
	} else {
	    %W delete insert
	}
    }
    bind Entry <BackSpace> {
	tkEntryBackspace %W
    }

    bind Entry <Control-space> {
	%W select from insert
    }
    bind Entry <Select> {
	%W select from insert
    }
    bind Entry <Control-Shift-space> {
	%W select adjust insert
    }
    bind Entry <Shift-Select> {
	%W select adjust insert
    }
    bind Entry <Control-slash> {
	%W select range 0 end
    }
    bind Entry <Control-backslash> {
	%W select clear
    }

    bind Entry <KeyPress> {
	tkEntryInsert %W %A
    }
    bind Entry <Insert> {
	catch {tkEntryInsert %W [selection get -displayof %W]}
    }

    # Additional emacs-like bindings:

    if !$tk_strictMotif {
	bind Entry <Control-a> {
	    tkEntrySetCursor %W 0
	}
	bind Entry <Control-b> {
	    tkEntrySetCursor %W [expr [%W index insert] - 1]
	}
	bind Entry <Control-d> {
	    %W delete insert
	}
	bind Entry <Control-e> {
	    tkEntrySetCursor %W end
	}
	bind Entry <Control-f> {
	    tkEntrySetCursor %W [expr [%W index insert] + 1]
	}
	bind Entry <Control-h> {
	    tkEntryBackspace %W
	}
	bind Entry <Control-k> {
	    %W delete insert end
	}
	bind Entry <Control-t> {
	    tkEntryTranspose %W
	}
	bind Entry <Meta-b> {
	    tkEntrySetCursor %W \
		    [string wordstart [%W get] [expr [%W index insert] - 1]]
	}
	bind Entry <Meta-d> {
	    %W delete insert [string wordend [%W get] [%W index insert]]
	}
	bind Entry <Meta-f> {
	    tkEntrySetCursor %W [string wordend [%W get] [%W index insert]]
	}
	bind Entry <Meta-BackSpace> {
	    %W delete [string wordstart [%W get] [expr [%W index insert] - 1]] \
		    insert
	}
    
	# A few additional bindings of my own.
    
	bind Entry <Control-v> {
	    catch {
		%W insert insert [selection get -displayof %W]
		tkEntrySeeInsert %W
	    }
	}
	bind Entry <Control-w> {
	    %W delete [string wordstart [%W get] [expr [%W index insert] - 1]] \
		    insert
	}
    }
    rename tkEntryBind {}
}

# tkEntryKeySelect --
# This procedure is invoked when stroking out selections using the
# keyboard.  It moves the cursor to a new position, then extends
# the selection to that position.
#
# Arguments:
# w -		The entry window.
# new -		A new position for the insertion cursor (the cursor hasn't
#		actually been moved to this position yet).

proc tkEntryKeySelect {w new} {
    if ![$w selection present] {
	$w select from insert
	$w select to $new
    } else {
	$w select adjust $new
    }
    $w icursor $new
}

# tkEntryInsert --
# Insert a string into an entry at the point of the insertion cursor.
# If there is a selection in the entry, and it covers the point of the
# insertion cursor, then delete the selection before inserting.
#
# Arguments:
# w -		The entry window in which to insert the string
# s -		The string to insert (usually just a single character)

proc tkEntryInsert {w s} {
    if {$s == ""} {
	return
    }
    catch {
	set insert [$w index insert]
	if {([$w index sel.first] <= $insert)
		&& ([$w index sel.last] >= $insert)} {
	    $w delete sel.first sel.last
	}
    }
    $w insert insert $s
    tkEntrySeeInsert $w
}

# tkEntryBackspace --
# Backspace over the character just before the insertion cursor.
#
# Arguments:
# w -		The entry window in which to backspace.

proc tkEntryBackspace w {
    if [$w selection present] {
	$w delete sel.first sel.last
    } else {
	set x [expr {[$w index insert] - 1}]
	if {$x >= 0} {$w delete $x}
    }
}

# tkEntrySeeInsert --
# Make sure that the insertion cursor is visible in the entry window.
# If not, adjust the view so that it is.
#
# Arguments:
# w -		The entry window.

proc tkEntrySeeInsert w {
    set c [$w index insert]
    set left [$w index @0]
    if {$left > $c} {
	$w xview $c
	return
    }
    set x [winfo width $w]
    while {([$w index @$x] <= $c) && ($left < $c)} {
	incr left
	$w xview $left
    }
}

# tkEntrySetCursor -
# Move the insertion cursor to a given position in an entry.  Also
# clears the selection, if there is one in the entry, and makes sure
# that the insertion cursor is visible.
#
# Arguments:
# w -		The entry window.
# pos -		The desired new position for the cursor in the window.

proc tkEntrySetCursor {w pos} {
    $w icursor $pos
    $w select clear
    tkEntrySeeInsert $w
}

# tkEntryTranspose -
# This procedure implements the "transpose" function for entry widgets.
# It tranposes the characters on either side of the insertion cursor,
# unless the cursor is at the end of the line.  In this case it
# transposes the two characters to the left of the cursor.  In either
# case, the cursor ends up to the right of the transposed characters.
#
# Arguments:
# w -		The entry window.

proc tkEntryTranspose w {
    set i [$w index insert]
    if {$i < [$w index end]} {
	incr i
    }
    set first [expr $i-2]
    if {$first < 0} {
	return
    }
    set new [string index [$w get] [expr $i-1]][string index [$w get] $first]
    $w delete $first $i
    $w insert insert $new
    tkEntrySeeInsert $w
}
