/*
 * TmWidget.c --
 *	This module contains the main set of functions
 *	common to all widget types. ie it implements the
 *	Tm Core widget stuff.
 *
 * Copyright 1993 Jan Newmarch, University of Canberra.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The author
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include "tm.h"
#include "tmFuncs.h"
#include <Xm/List.h>
#include <Xm/Xm.h>
#include <Xm/AtomMgr.h>
#if XmVERSION >= 2
#include <Xm/CSText.h>
#endif

XEvent *Tm_HackXEvent;	/* needed for D&D to pass X event into XDragStart */

/*
 *--------------------------------------------------------------
 *
 * Tm_ParentWidgetFromPath --
 *
 *	Given a Tm widget pathname, finds the parent Xt widget.
 *
 * Results:
 *
 *	returns the Xt parent
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

Widget Tm_ParentWidgetFromPath (interp, pathName)
    Tcl_Interp *interp;
    char *pathName;
{
    char *p;
    int numChars;
    Tcl_CmdInfo cmdInfo;

    /*
     * Strip the parent's name out of pathName (it's everything up
     * to the last dot).  There are two tricky parts: (a) must
     * copy the parent's name somewhere else to avoid modifying
     * the pathName string (for large names, space for the copy
     * will have to be malloc'ed);  (b) must special-case the
     * situation where the parent is ".".
     */

    p = strrchr(pathName, '.');
    if (p == NULL) {
        Tcl_AppendResult(interp, "bad window path name \"", pathName,
                "\"", (char *) NULL);
        return NULL;
    }

    numChars = p-pathName;

    p = (char *) XtMalloc((unsigned) (numChars+2));
    if (numChars == 0) {
	*p = '.';
	p[1] = '\0';
    } else {
	strncpy(p, pathName, numChars);
	p[numChars] = '\0';
    }

    if (Tcl_GetCommandInfo(interp, p, &cmdInfo) == 0) {
        Tcl_AppendResult(interp, "no such widget \"", pathName,
                "\"", (char *) NULL);
        return NULL;
    }
    XtFree(p);
    return ( ((Tm_Widget *) (cmdInfo.clientData))->widget);
}


/*
 *--------------------------------------------------------------
 *
 * Tm_WidgetInfoFromPath --
 *
 *	looks up the hash table to find the info about the widget
 *
 * Results:
 *
 *	returns the widget info record.
 *
 * Side effects:
 *
 *	none
 *--------------------------------------------------------------
 */

Tm_Widget * 
Tm_WidgetInfoFromPath (interp, pathName)
    Tcl_Interp *interp;
    char *pathName;
{
    Tcl_CmdInfo cmdInfo;

    if (Tcl_GetCommandInfo(interp, pathName, &cmdInfo) == 0) {
        Tcl_AppendResult(interp, "no such widget \"", pathName,
                "\"", (char *) NULL);
        return NULL;
    }
   return (Tm_Widget *) (cmdInfo.clientData);
}

/*
 *--------------------------------------------------------------
 *
 * Tm_ActionsHandler --
 *
 *	All actions are vectored through here.
 *	It calls the Tcl command contained in the args
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_ActionsHandler(w, event, argv, argc)
    Widget w;
    XEvent *event;
    char **argv;
    Cardinal *argc;
{
    Tm_Widget *wPtr;
    Tcl_Interp *interp;
    char *orig_command, *new_command;
    char *p_orig, *msg;
    int size, n;

    /* can only call actions on Motif widgets */
    if ( ! XtIsSubclass(w, xmPrimitiveWidgetClass) &&
	 ! XtIsSubclass(w, xmManagerWidgetClass)) {
	fprintf(stderr, 
	    "Can only set actions on Motif widgets with a userData field\n");
	return;
    }

    XtVaGetValues(w, XmNuserData, &wPtr, NULL);
    interp = wPtr->interp;

    if (*argc < 1) {
	fprintf(stderr, "action must have an arg\n");
	return;
    }

    Tm_HackXEvent = event; /* hack to get value into XmDragStart */

    size = 128;
    orig_command = XtMalloc(size);
    *orig_command = '\0';

    for (n = 0; n < *argc; n++) {
	if (strlen(orig_command) + strlen(argv[n]) + 2 > size) {
	    size = 2*size + strlen(argv[n]);
	    orig_command = XtRealloc(orig_command, size);
	}
	strcat(orig_command, argv[n]);
	strcat(orig_command, " ");
    }
    p_orig = orig_command;

    new_command = Tm_ExpandPercents(wPtr->pathName, w, event,
				NULL, orig_command);

    if (Tcl_GlobalEval(interp, new_command) != TCL_OK) {
        msg = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
        if (msg == NULL) {
            msg = interp->result;
        }
        XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
    }

    /* record result in case callActionProc invoked this */
    if (Tm_SaveResult(interp))
        Tm_AppendResult(interp, interp->result);

    XtFree(orig_command);
    XtFree(new_command);
}

/*
 *--------------------------------------------------------------
 *
 * Tm_WidgetCallbackHandler --
 *
 *	nearly all callbacks are vectored through here.
 *	It calls the appropriate callback with right
 *	Tcl command
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_WidgetCallbackHandler(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    Tm_ClientData *c_data = (Tm_ClientData *) client_data;
    Tcl_Interp *interp;
    char *command;
    char *msg;

    interp = c_data->widget_info->interp;
#   ifdef DEBUG
    fprintf(stderr, "%s\n", (char *) c_data->callback_func);
#   endif

    /* some callbacks (XmNdestroyCallback) come from non-Motif
       widgets and have a NULL call_data
     */
    if (call_data == NULL) {
        command = Tm_ExpandPercents(c_data->widget_info->pathName,
		c_data->widget_info->widget,
		NULL, NULL, 
		(char *) c_data->callback_func);
    } else {
        command = Tm_ExpandPercents(c_data->widget_info->pathName,
		c_data->widget_info->widget,
		((XmAnyCallbackStruct *) call_data)->event, call_data, 
		(char *) c_data->callback_func);
    }
#   ifdef DEBUG
    fprintf(stderr, "%% expanded command: %s\n", command);
#   endif

    if (Tcl_GlobalEval(interp, command) != TCL_OK) {
	msg = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
	if (msg == NULL) {
	    msg = interp->result;
	}
	XtAppWarningMsg(XtWidgetToApplicationContext(w),
		"TclError", "TclError", "TclError", msg, NULL, 0);
    }

    if (Tm_SaveResult(interp))
	Tm_AppendResult(interp, interp->result);

    XtFree(command);
}


/*
 *--------------------------------------------------------------
 *
 * DestroyWidgetInfo -
 *
 *	destroy widget info when it is safe to do so 
 *
 *--------------------------------------------------------------
 */

static void
DestroyWidgetInfo(client_data, timer)
    XtPointer client_data;
    XtIntervalId *timer;
{
    Tm_Widget *c_data = (Tm_Widget *) client_data;
    Tcl_Interp *interp;
    char *path;
    char *parent;

    interp = c_data->interp;
    path = c_data->pathName;
    parent = c_data->parent;

    if(XtIsShell(c_data->widget))
	Tm_DisplayRemoveShell(c_data->displayInfo,c_data);
    XtFree(parent);
    XtFree(path);
    XtFree((char *) c_data);
}

/*
 *--------------------------------------------------------------
 *
 * Tm_DestroyWidgetHandler --
 *
 *	the widget is being destroyed, so call this callback.
 *	Alas: there may be other destroy callbacks, and we
 *	have no control over order of execution. So wait till
 *	all callbacks are over and then reclaim space, destroy
 *	commands, etc.
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_DestroyWidgetHandler(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    XtAppAddTimeOut(XtWidgetToApplicationContext(w),0,DestroyWidgetInfo,client_data);
}


/*
 *--------------------------------------------------------------
 *
 * Tm_DestroyReclaimHandler --
 *
 *	reclaim space in callback client data when widget 
 *	is destroyed
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_DestroyReclaimHandler(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    Tm_ClientData *c_data = (Tm_ClientData *) client_data;

    XtFree(c_data->callback_func);
    XtFree((char *) c_data);
}

/*
 *--------------------------------------------------------------
 *
 * Tm_TextVerifyCallbackHandler --
 *
 *	special case callback handler for Text Verify callbacks.
 *	It calls the appropriate callback with right
 *	Tcl command, then sets fields as needed by Text
 *	(or will do)
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_TextVerifyCallbackHandler(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    Tm_ClientData *c_data = (Tm_ClientData *) client_data;
    XmTextVerifyCallbackStruct *verify_data = 
			(XmTextVerifyCallbackStruct *) call_data;
    Tcl_Interp *interp;
    char *path;
    char *msg;
    char *command;
    int doit;
    XmTextPosition startPos, endPos;
    int length;
    char buf_startPos[128];
    char buf_endPos[128];
    char buf_length[128];
    char *buf_ptr;

    interp = c_data->widget_info->interp;
    path = c_data->widget_info->pathName;

/* in here we have to set tcl vbls to the values of the callback fields
   and afterwards get their values and set them in the callback data
*/
    if (verify_data->doit)
        Tcl_SetVar(interp, TM_TEXT_DOIT, "true", TCL_GLOBAL_ONLY);
    else
	Tcl_SetVar(interp, TM_TEXT_DOIT, "false", TCL_GLOBAL_ONLY);

    sprintf(buf_startPos, "%ld", verify_data->startPos);
    Tcl_SetVar(interp, TM_TEXT_STARTPOS, buf_startPos, TCL_GLOBAL_ONLY);

    sprintf(buf_endPos, "%ld", verify_data->endPos);
    Tcl_SetVar(interp, TM_TEXT_ENDPOS, buf_endPos, TCL_GLOBAL_ONLY);

    if (verify_data->reason == XmCR_MODIFYING_TEXT_VALUE) {
        length = verify_data->text->length;
        buf_ptr = XtMalloc(length + 1);
        strncpy(buf_ptr, verify_data->text->ptr, length);
        buf_ptr[length] = '\0';
        Tcl_SetVar(interp, TM_TEXT_PTR, buf_ptr, TCL_GLOBAL_ONLY);
    
        sprintf(buf_length, "%d", length);
        Tcl_SetVar(interp, TM_TEXT_LENGTH, buf_length, TCL_GLOBAL_ONLY);
    } else {
	Tcl_SetVar(interp, TM_TEXT_PTR, "", TCL_GLOBAL_ONLY);
	Tcl_SetVar(interp, TM_TEXT_LENGTH, "0", TCL_GLOBAL_ONLY);
	buf_ptr = NULL;
    }
	

    command = Tm_ExpandPercents(c_data->widget_info->pathName,
		c_data->widget_info->widget,
		((XmAnyCallbackStruct *) call_data)->event, call_data, 
		(char *) c_data->callback_func);
    if (Tcl_GlobalEval(interp, command) != TCL_OK) {
        msg = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
        if (msg == NULL) {
            msg = interp->result;
        }
        XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
	XtFree(command);
        XtFree(buf_ptr);
	return;
    }
    XtFree(command);

    if (Tm_SaveResult(interp))
	Tm_AppendResult(interp, interp->result);

    /* now set results back into callback struct for Text */
    msg = Tcl_GetVar(interp, TM_TEXT_DOIT, TCL_GLOBAL_ONLY);
    if (Tcl_GetBoolean(interp, msg, &doit) == TCL_ERROR) {
        XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
        XtFree(buf_ptr);
        return;
    }
    verify_data->doit = doit;

    if (verify_data->reason != XmCR_MODIFYING_TEXT_VALUE) {
	return;
    }

    msg = Tcl_GetVar(interp, TM_TEXT_STARTPOS, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_startPos) != 0) {
	/* no error checks here - need Tcl_GetLong */
        startPos = strtol(msg, NULL, 0);
        verify_data->startPos = startPos;
    }

    msg = Tcl_GetVar(interp, TM_TEXT_ENDPOS, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_endPos) != 0) {
	/* no error checks here - need Tcl_GetLong */
        endPos = strtol(msg, NULL, 0);
        verify_data->endPos = endPos;
    }
    msg = Tcl_GetVar(interp, TM_TEXT_PTR, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_ptr) != 0) {
	XtFree(verify_data->text->ptr);
        verify_data->text->ptr = XtNewString(msg);
    }
    msg = Tcl_GetVar(interp, TM_TEXT_LENGTH, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_length) != 0) {
        if (Tcl_GetInt(interp, msg, &length) == TCL_ERROR) {
            XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
	    XtFree(buf_ptr);
            return;
        }
        verify_data->text->length = length;
    }
    XtFree(buf_ptr);
}

#if XmVERSION >= 2
/*
 *--------------------------------------------------------------
 *
 * Tm_CSTextVerifyCallbackHandler --
 *
 *	special case callback handler for CSText Verify callbacks.
 *	It calls the appropriate callback with right
 *	Tcl command, then sets fields as needed by CSText
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_CSTextVerifyCallbackHandler(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    Tm_ClientData *c_data = (Tm_ClientData *) client_data;
    XmCSTextVerifyCallbackStruct *verify_data = 
			(XmCSTextVerifyCallbackStruct *) call_data;
    Tcl_Interp *interp;
    char *path;
    char *msg;
    char *command;
    int doit;
    XmTextPosition startPos, endPos;
    char *ptr;
    int length;
    char buf_startPos[128];
    char buf_endPos[128];
    char buf_length[128];
    char buf[128];
    char *buf_ptr;

    interp = c_data->widget_info->interp;
    path = c_data->widget_info->pathName;

/* in here we have to set tcl vbls to the values of the callback fields
   and afterwards get their values and set them in the callback data
*/
    if (verify_data->doit)
        Tcl_SetVar(interp, TM_TEXT_DOIT, "true", TCL_GLOBAL_ONLY);
    else
	Tcl_SetVar(interp, TM_TEXT_DOIT, "false", TCL_GLOBAL_ONLY);

    sprintf(buf_startPos, "%ld", verify_data->startPos);
    Tcl_SetVar(interp, TM_TEXT_STARTPOS, buf_startPos, TCL_GLOBAL_ONLY);

    sprintf(buf_endPos, "%ld", verify_data->endPos);
    Tcl_SetVar(interp, TM_TEXT_ENDPOS, buf_endPos, TCL_GLOBAL_ONLY);

    if (verify_data->reason == XmCR_MODIFYING_TEXT_VALUE) {
	buf_ptr = Tm_XmStringToString(verify_data->text);
        Tcl_SetVar(interp, TM_TEXT_PTR, buf_ptr, TCL_GLOBAL_ONLY);
    } else {
	Tcl_SetVar(interp, TM_TEXT_PTR, "", TCL_GLOBAL_ONLY);
	buf_ptr = NULL;
    }

    command = Tm_ExpandPercents(c_data->widget_info->pathName,
		c_data->widget_info->widget,
		((XmAnyCallbackStruct *) call_data)->event, call_data, 
		(char *) c_data->callback_func);
    if (Tcl_GlobalEval(interp, command) != TCL_OK) {
        msg = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
        if (msg == NULL) {
            msg = interp->result;
        }
        XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
	XtFree(command);
        XtFree(buf_ptr);
	return;
    }
    XtFree(command);

    if (Tm_SaveResult(interp))
	Tm_AppendResult(interp, interp->result);

    /* now set results back into callback struct for Text */
    msg = Tcl_GetVar(interp, TM_TEXT_DOIT, TCL_GLOBAL_ONLY);
    if (Tcl_GetBoolean(interp, msg, &doit) == TCL_ERROR) {
        XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
        XtFree(buf_ptr);
        return;
    }
    verify_data->doit = doit;

    if (verify_data->reason != XmCR_MODIFYING_TEXT_VALUE) {
	return;
    }

    msg = Tcl_GetVar(interp, TM_TEXT_STARTPOS, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_startPos) != 0) {
	/* no error checks here - need Tcl_GetLong */
        startPos = strtol(msg, NULL, 0);
        verify_data->startPos = startPos;
    }

    msg = Tcl_GetVar(interp, TM_TEXT_ENDPOS, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_endPos) != 0) {
	/* no error checks here - need Tcl_GetLong */
        endPos = strtol(msg, NULL, 0);
        verify_data->endPos = endPos;
    }
    msg = Tcl_GetVar(interp, TM_TEXT_PTR, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_ptr) != 0) {
	XmStringFree(verify_data->text);
        verify_data->text = Tm_StringToXmString(interp, msg);
    }
    XtFree(buf_ptr);
}
#endif /* XmVERSION */

#if XmVERSION >= 2
/*
 *--------------------------------------------------------------
 *
 * Tm_NotebookVerifyCallbackHandler --
 *
 *	special case callback handler for Notebook Verify callbacks.
 *	It calls the appropriate callback with right
 *	Tcl command, then sets fields as needed by Text
 *	(or will do)
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

void
Tm_NotebookVerifyCallbackHandler(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{

/* THIS IS STILL TEXT VERSION */
    Tm_ClientData *c_data = (Tm_ClientData *) client_data;
    XmTextVerifyCallbackStruct *verify_data = 
			(XmTextVerifyCallbackStruct *) call_data;
    Tcl_Interp *interp;
    char *path;
    char *msg;
    char *command;
    int doit;
    XmTextPosition startPos, endPos;
    char *ptr;
    int length;
    char buf_startPos[128];
    char buf_endPos[128];
    char buf_length[128];
    char buf[128];
    char *buf_ptr;

    interp = c_data->widget_info->interp;
    path = c_data->widget_info->pathName;

/* in here we have to set tcl vbls to the values of the callback fields
   and afterwards get their values and set them in the callback data
*/
    if (verify_data->doit)
        Tcl_SetVar(interp, TM_TEXT_DOIT, "true", TCL_GLOBAL_ONLY);
    else
	Tcl_SetVar(interp, TM_TEXT_DOIT, "false", TCL_GLOBAL_ONLY);

    sprintf(buf_startPos, "%ld", verify_data->startPos);
    Tcl_SetVar(interp, TM_TEXT_STARTPOS, buf_startPos, TCL_GLOBAL_ONLY);

    sprintf(buf_endPos, "%ld", verify_data->endPos);
    Tcl_SetVar(interp, TM_TEXT_ENDPOS, buf_endPos, TCL_GLOBAL_ONLY);

    if (verify_data->reason == XmCR_MODIFYING_TEXT_VALUE) {
        length = verify_data->text->length;
        buf_ptr = XtMalloc(length + 1);
        strncpy(buf_ptr, verify_data->text->ptr, length);
        buf_ptr[length] = '\0';
        Tcl_SetVar(interp, TM_TEXT_PTR, buf_ptr, TCL_GLOBAL_ONLY);
    
        sprintf(buf_length, "%d", length);
        Tcl_SetVar(interp, TM_TEXT_LENGTH, buf_length, TCL_GLOBAL_ONLY);
    } else {
	Tcl_SetVar(interp, TM_TEXT_PTR, "", TCL_GLOBAL_ONLY);
	Tcl_SetVar(interp, TM_TEXT_LENGTH, "0", TCL_GLOBAL_ONLY);
	buf_ptr = NULL;
    }
	

    command = Tm_ExpandPercents(c_data->widget_info->pathName,
		c_data->widget_info->widget,
		((XmAnyCallbackStruct *) call_data)->event, call_data, 
		(char *) c_data->callback_func);
    if (Tcl_GlobalEval(interp, command) != TCL_OK) {
        msg = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
        if (msg == NULL) {
            msg = interp->result;
        }
        XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
	XtFree(command);
        XtFree(buf_ptr);
	return;
    }
    XtFree(command);

    if (Tm_SaveResult(interp))
	Tm_AppendResult(interp, interp->result);

    /* now set results back into callback struct for Text */
    msg = Tcl_GetVar(interp, TM_TEXT_DOIT, TCL_GLOBAL_ONLY);
    if (Tcl_GetBoolean(interp, msg, &doit) == TCL_ERROR) {
        XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
        XtFree(buf_ptr);
        return;
    }
    verify_data->doit = doit;

    if (verify_data->reason != XmCR_MODIFYING_TEXT_VALUE) {
	return;
    }

    msg = Tcl_GetVar(interp, TM_TEXT_STARTPOS, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_startPos) != 0) {
	/* no error checks here - need Tcl_GetLong */
        startPos = strtol(msg, NULL, 0);
        verify_data->startPos = startPos;
    }

    msg = Tcl_GetVar(interp, TM_TEXT_ENDPOS, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_endPos) != 0) {
	/* no error checks here - need Tcl_GetLong */
        endPos = strtol(msg, NULL, 0);
        verify_data->endPos = endPos;
    }
    msg = Tcl_GetVar(interp, TM_TEXT_PTR, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_ptr) != 0) {
	XtFree(verify_data->text->ptr);
        verify_data->text->ptr = XtNewString(msg);
    }
    msg = Tcl_GetVar(interp, TM_TEXT_LENGTH, TCL_GLOBAL_ONLY);
    if (strcmp(msg, buf_length) != 0) {
        if (Tcl_GetInt(interp, msg, &length) == TCL_ERROR) {
            XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
	    XtFree(buf_ptr);
            return;
        }
        verify_data->text->length = length;
    }
    XtFree(buf_ptr);
}
#endif /* XmVERSION */

/*
 *--------------------------------------------------------------
 *
 * Tm_InputHandler --
 *
 * Results:
 * 	none
 *
 * Side effects:
 * 	could be any - this handles any Xt input
 *
 *--------------------------------------------------------------
 */

/* ARGSUSED */
void
Tm_InputHandler(clientData, source, id)
    XtPointer clientData;
    int *source;
    XtInputId *id;
{
    Tm_InputData *i_data = (Tm_InputData *) clientData;
    Tcl_Interp *interp = i_data->interp;
    char *command = i_data->command;
    char *message;

    /* should "expand percents" first */
    if (Tcl_Eval(interp, command) != TCL_OK) {
	message = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
	if (message == NULL) {
	    message = interp->result;
	}
	/* we don't have an AppContext for an XtAppWarningMessage! */
	fprintf(stderr, "%s\n", message);
    }
}

/*
 *--------------------------------------------------------------
 *
 * Tm_TimerHandler --
 *
 * Results:
 * 	none
 *
 * Side effects:
 * 	could be any - this handles any Xt timer
 *
 *--------------------------------------------------------------
 */

void
Tm_TimerHandler(clientData, id)
    XtPointer clientData;
    XtIntervalId *id;
{
    Tm_TimerData *t_data = (Tm_TimerData *) clientData;
    Tcl_Interp *interp = t_data->interp;
    char *command = t_data->command;
    char *message;

    /* should "expand percents" first */
    if (Tcl_Eval(interp, command) != TCL_OK) {
	message = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
	if (message == NULL) {
	    message = interp->result;
	}
	/* we don't have an AppContext for an XtAppWarningMessage! */
	fprintf(stderr, "%s\n", message);
    }
    XtFree(command);
    XtFree((char *) clientData);
}


/*
 *--------------------------------------------------------------
 *
 * Tm_SelectionCallbackHandler --
 *
 * Results:
 *      none
 *
 * Side effects:
 *      could be any - this handles the pasting of a selection
 *
 *--------------------------------------------------------------
 */

void
Tm_SelectionCallbackHandler(w, client_data, selection, 
		type, value, length, format)
    Widget w;
    XtPointer client_data;
    Atom *selection;
    Atom *type;
    XtPointer value;
    unsigned long *length;
    int *format;
{
    Tm_ClientData *c_data = (Tm_ClientData *) client_data;
    Tcl_Interp *interp = c_data->widget_info->interp;
    char *command = c_data->callback_func;
    char *new_command;
    char *new_value;
    Atom TEXT = XmInternAtom(XtDisplayOfObject(w), "TEXT", False);

    if ((* (char **) value == NULL) && (*length == 0)) {
	/* no slection, or timed out */
	return;
    }

    /* right now, will only handle XA_STRING or TEXT :-( */
    if (  (*type != XA_STRING && *type != TEXT) ||
	*format != 8 || value == NULL) {
	fprintf(stderr, "bad selection type\n");
	return;
    }

    /* make sure we are null terminated */
    new_value = XtMalloc(*length + 1);
    strncpy(new_value, value, *length);
    new_value[*length] = '\0';

    /* we only recognise the %selection substitution,
       do this (be lazy) in place by calling the tcl command
	regsub %selection {command} {new_value} _TM_SELECT;
	eval $_TM_SELECT
     */
    new_command = XtMalloc(strlen(command) + strlen(value) + 1024);
    strcpy(new_command, "regsub %selection {");
    strcat(new_command, command);
    strcat(new_command, "} {");
    strcat(new_command, new_value);
    strcat(new_command, "} _TM_SELECT; eval $_TM_SELECT");

    if (Tcl_GlobalEval(interp, new_command) == TCL_ERROR) {
	char *msg;

        msg = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
        if (msg == NULL) {
            msg = interp->result;
        }
        XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
    }
    XtFree(new_command);
    XtFree(new_value);
    XtFree(value);
}



/*
 *--------------------------------------------------------------
 *
 * Tm_GetGC --
 *
 *	get a graphics context attached to a widget
 *
 * Results:
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

char *
Tm_GetGC(pathName, interp, w, class, argv, argc)
    char *pathName;
    Tcl_Interp *interp;
    Widget w;
    WidgetClass class;
    char **argv;
    int argc;
{
    char *resource;
    XGCValues gc_value;
    XtGCMask mask = 0;
    GC gc;
    char *buf;

    while (argc >= 2) {
	if (argv[0][0] != '-') {
	    fprintf(stderr, "Skipping argument %s\n", argv[0]);
	    argc -= 2; argv += 2;
	    continue;
	}
	resource = argv[0]+1;

	if (strcmp(resource, XmNforeground) == 0) {	
            if (Tm_ConvertValue(w, XmRString, argv[1], strlen(argv[1]),
		XmRPixel, &gc_value.foreground, sizeof(unsigned long))) {
		mask |= GCForeground;
	    }
	} else

	if (strcmp(resource, XmNbackground) == 0) {	
            if (Tm_ConvertValue(w, XmRString, argv[1], strlen(argv[1]),
		XmRPixel, &gc_value.background, sizeof(unsigned long))) {
		mask |= GCBackground;
	    }
	} else

	if (strcmp(resource, XmNfont) == 0) {	
            if (Tm_ConvertValue(w, XmRString, argv[1], strlen(argv[1]),
		XmRFont, &gc_value.font, sizeof(unsigned long))) {
		mask |= GCFont;
	    }
	}
	argc -= 2;
	argv += 2;
    }

    buf = XtMalloc(16);
    gc = XtGetGC(w, mask, &gc_value);
/*	%p may be broken on the Sun, so fit into an XtArgVal
    sprintf(buf, "%p", (void *) gc);
*/
    /* allow simple type checking: prefix value with "gc-" */
    sprintf(buf, "gc-%lu", (long) gc);
    return buf;
}

#if USE_UIL

static int
ReconnectParent(w)
    Widget w;
{
    Tm_Widget *wPtr = NULL;
    Widget parent;
    char *name;
    char *path;
    Arg arg;

    XtSetArg(arg, XmNuserData, &wPtr);
    
    parent = w;
    while (NULL != (parent = XtParent(parent))) {
	XtGetValues(parent, &arg, 1);
	if (wPtr != NULL) {
	    /* rebuild the widget path from here on down */
	    name = XtName(w);
	    path = XtMalloc(strlen(name) + strlen(wPtr->pathName) + 2);
	    
	    if (strcmp(wPtr->pathName, ".") == 0) {
		strcpy(path, ".");
	    } else {
		strcpy(path,wPtr->pathName);
		strcat(path, ".");
	    }
	    strcat(path, name);

	    Tm_CreateTmInfo(wPtr->interp, w, path, name, wPtr->displayInfo);
	    if (XtIsComposite(w)) {
		Tm_CreateTmInfoInChildren(wPtr->interp, w, path,
			wPtr->displayInfo);
	    }

	    XtFree(path);
	    return TRUE;
	}
    }
    return FALSE;
}

void
Tm_UILCallbackHandler(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    Tm_Widget *wPtr;
    Arg arg;
    Tcl_Interp *interp;
    String command = (String) client_data;
    String ncmd = NULL;
    String msg;

    /* UIL callbacks use the client_data much like tclMotif does,
       so there is UIL stuff instead of tclMotif stuff in client_data.
       We have to use the userData field instead, where we put info
     */
    XtSetArg(arg, XmNuserData, &wPtr);
    XtGetValues(w, &arg, 1);

    if (wPtr == NULL) {
	if (!ReconnectParent(w)) {
            msg = "can't find  userData in UIL callback";
            XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
            return;
        }
	XtGetValues(w, &arg, 1);

	if (wPtr == NULL) {
            msg = "reconnection to parent widget failed";
            XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
            return;
	}
    }

    interp = wPtr->interp;
    ncmd = Tm_ExpandPercents(wPtr->pathName, wPtr->widget,
			((XmAnyCallbackStruct *)call_data)->event,
			call_data, command);

    if (Tcl_GlobalEval(interp, ncmd) != TCL_OK) {
        msg = Tcl_GetVar(interp, "errorInfo", TCL_GLOBAL_ONLY);
        if (msg == NULL)
            msg = interp->result;
        XtAppWarningMsg(XtWidgetToApplicationContext(w),
                "TclError", "TclError", "TclError", msg, NULL, 0);
    }
    XtFree(ncmd);

}
#endif /* USE_UIL */


