#
# MakeReplayClassBindings: changes all the bindings so that the replay
#    system gets control before each event.
#
proc MakeReplayClassBindings {} {
    global thisApp targetApp ReplayData
    #
    # do the rebinding for each class of widget
    #
    foreach class {Button Checkbutton Entry Listbox Menu Menubutton
            Radiobutton Scale Scrollbar Text} {
        MakeComment 0 "Rebinding Class $class"
        ReplayRebind Class $class
    }
    #
    # rebind the All binding
    #
    ReplayRebind All all
    #
    # rebind all the existing widgets
    #
    MakeComment 0 "Rebinding individual widgets"
    RebindAllWidgets
    #
    # add bindings for Toplevels and Tk (the first window) so that we
    # get called to rebind the individual widget binding as we enter
    # the window for the first time
    #
    MakeComment 0 "Rebinding so we can rebind newly created toplevels"
    foreach event {Expose FocusIn Map Visibility} {
        foreach classname {Toplevel Tk} {
            if {$ReplayData(AppIsSeparate)} {
                set retcmd [format {send %s NewToplevel %%W} $thisApp]
                set cmd [format {bind %s <%s> {%s}} $classname $event $retcmd]
                SendToApp $cmd
            } else {
                bind $classname <$event> {NewToplevel %W}
            }
        }
    }
    MakeComment 0 "Done rebinding.  Now you can record or replay scripts."
}
#
# NewToplevel -- check to see if the widgets have been remapped.
#
proc NewToplevel {w} {
    global ReplayData
    global thisApp targetApp
    #
    # only rebind the widgets the first time we enter the Toplevel
    #
    if ![info exists ReplayData(Rebound,$w)] {
        set ReplayData(Rebound,$w) 1
        RebindWidgetAndChildren $w
        #
        # Make sure we are notified when the widget is destroyed so we
        # can unset the mark that says it was already rebound.  This is
        # so a window that is created, destroyed and then recreated will
        # have the widgets of the second version rebound correctly.
        #
        if {$ReplayData(AppIsSeparate)} {
            SendToApp [list bind $w <Destroy> "send $thisApp DestroyNotify $w"]
        } else {
            bind $w <Destroy> "DestroyNotify $w"
        }
    }    
}
#
# DestroyNotify: called by the target application when a window we have
#     rebound is destroyed.
#
proc DestroyNotify {w} {
    global ReplayData
    catch "unset ReplayData(Rebound,$w)"
}
#
# RebindAllWidgets: goes through all the widgets in the application
#         and rebinds them so we can record all events.
#
proc RebindAllWidgets {} {
    global ReplayData
    RebindWidgetAndChildren .
}
#
# RebindWidgetAndChildren: rebind one widget and then all its children
#
proc RebindWidgetAndChildren {w} {
    global ReplayData
    global thisApp targetApp
    ReplayRebind Widget $w
    set children [SendToApp "winfo children $w"]
    foreach child $children {
        RebindWidgetAndChildren $child
    }
}
#
# ReplayRebind: rebinds the class bindings for one class of widgets.
#    We rebind each event for the widget.
#    We save the original binding for the event in the ReplayData
#        global array.
#    Then we change the binding to call ReplayCapture which will
#        record the event and then execute the original binding.
#    Note that we have to capture all the event fields (%X fields)
#        so that we can pass them on to the original binding.
#
proc ReplayRebind {what which} {
    global ReplayData
    global thisApp targetApp
    if {!$ReplayData(AppIsSeparate) && [string match ".replay.*" $which]} {
        return
    }
    set cbn $ReplayData(CallbackName)
    if {[string compare $what "Widget"] == 0} {
        set class [SendToApp "winfo class $which"]
        switch $class \
            "Scale" - \
            "Scrollbar" {
                set conf [SendToApp "$which configure -command"]
                set binding [lindex $conf 4]
                if {[string compare $cbn [lindex $binding 0]] != 0} {
                    set subscript [format "%s,%s,%s" $class $which Command]
                    set ReplayData($subscript) $binding
                    set alist [list [list Widget $class] [list W $which]]
                    set remoteCmd [format {%s %s %s {%s}} \
                        $cbn $thisApp $subscript $alist]
                    set cmd [format {%s configure -command {%s}} \
                        $which $remoteCmd]
                    SendToApp $cmd
                }
            } \
            "Text" {
                foreach tag [SendToApp "$which tag names"] {
                    foreach seq [SendToApp "$which tag bind $tag"] {
                        set binding [SendToApp "$which tag bind $tag $seq"]
                        if {[string compare $cbn [lindex $binding 0]] != 0} {
                            set subscript\
                                [format "%s,%s,%s:%s" Text $which $tag $seq]
                            set ReplayData($subscript) $binding
                            set alist [GetPercentFields $binding]
                            set alist [lreplace $alist 0 0 \
                                [list W $which] [list Widget Text] \
                                [list x {%x}] [list y {%y}] ]
                            set remoteCmd [format {%s %s %s {%s}} \
                                $cbn $thisApp $subscript $alist]
                            set cmd [format {%s tag bind %s %s {%s}} \
                                $which $tag $seq $remoteCmd]
                            SendToApp $cmd
                        }
                    }
                }
            } \
            "Canvas" {
                # go through each item on the canvas
                set tags {}
                foreach item [SendToApp "$which find all"] {
                    foreach tag [SendToApp "$which gettags $item"] {
                        if {[lsearch $tags $tag] < 0} {
                            lappend tags $tag
                        }
                    }
                    # go through each sequence bound for that item
                    foreach seq [SendToApp "$which bind $item"] {
                        set binding [SendToApp "$which bind $item $seq"]
                        if {[string compare $cbn [lindex $binding 0]] != 0} {
                            set subscript \
                                [format "%s,%s,%s:%s" CItem $which $item $seq]
                            set ReplayData($subscript) $binding
                            set alist [GetPercentFields $binding]
                            set alist [lreplace $alist 0 0 \
                                [list W $which] [list Widget Canvas] \
                                [list x {%x}] [list y {%y}] ]
                            set remoteCmd [format {%s %s %s {%s}} \
                                $cbn $thisApp $subscript $alist]
                            set cmd [format {%s bind %s %s {%s}} \
                                $which $item $seq $remoteCmd]
                            SendToApp $cmd
                        }
                    }
                }
                foreach tag $tags {
                    foreach seq [SendToApp "$which bind $tag"] {
                        set binding [SendToApp "$which bind $tag $seq"]
                        if {[string compare $cbn [lindex $binding 0]] != 0} {
                            set subscript \
                                [format "%s,%s,%s:%s" CTag $which $tag $seq]
                            set ReplayData($subscript) $binding
                            set alist [GetPercentFields $binding]
                            set alist [lreplace $alist 0 0 \
                                [list W $which] [list Widget Canvas] \
                                [list x {%x}] [list y {%y}] ]
                            set remoteCmd [format {%s %s %s {%s}} \
                                $cbn $thisApp $subscript $alist]
                            set cmd [format {%s bind %s %s {%s}} \
                                $which $tag $seq $remoteCmd]
                            SendToApp $cmd
                        }
                    }
                }
            }
    }
    #
    # rebind each event defined for this class or widget
    #
    set events [SendToApp "bind $which"]
    foreach event $events {
        #
        # Construct the subscript to save the the original binding under
        # and then save it for later replay.
        #
        set binding [SendToApp "bind $which $event"]
        #
        # do not remap again if it has already been remapped
        #
        if {[string compare $cbn [lindex $binding 0]] != 0} {
            set subscript [format "%s,%s,%s" $what $which $event]
            set ReplayData($subscript) $binding
            #
            # find out what %-fields it requires
            # (but always include %W because we use it)
            #
            set percentFields [GetPercentFields $binding]
            lappend percentFields [list x {%x}] [list y {%y}]
            #
            # Now bind the event to call ReplayCapture.  Pass it the subscript
            # so it can look up the original binding.  We also pass it a list
            # of %-substitutions so it can simulate the %-substitutions done
            # by Tk event handling.
            #
            set remotecmd [format {RebindCallback %s %s {%s}} \
                $thisApp $subscript $percentFields]
            set cmd [format {bind %s %s {%s}} $which $event $remotecmd]
            SendToApp $cmd
        }
    }
}
#
# UnbindAllWidgets: undoes the replay bindings.
#
proc UnbindAllWidgets {} {
    global ReplayData
    foreach subscript [array names ReplayData] {
        if [regexp {([^,]*),([^,]*),(.*)} $subscript junk what which event] {
            set binding $ReplayData($subscript)
            switch $what \
                "Widget" {
                    SendToApp "bind $which $event $binding"
                    unset ReplayData($subscript)
                } \
                "Command" {
                    SendToApp "$which configure -command $binding"
                    unset ReplayData($subscript)
                } \
                "Text" {
                    regexp {(.*):(.*)} $event junk tag seq
                    SendToApp "$which tag bind $tag $seq $binding"
                    unset ReplayData($subscript)
                } \
                "CItem" {
                    regexp {(.*):(.*)} $event junk item seq
                    SendToApp "$which bind $item $seq $binding"
                    unset ReplayData($subscript)
                } \
                "CTag" {
                    regexp {(.*):(.*)} $event junk tag seq
                    SendToApp "$which bind $tag $seq $binding"
                    unset ReplayData($subscript)
                }
        }
    }
}
proc GetPercentFields {action} {
    set fields [list [list W {%W}]]
    set len [string length $action]
    #
    # go through each character of the script looking for %s
    #
    for {set i 0} {$i < $len} {incr i} {
        set ch [string index $action $i]
        if {[string compare $ch "%"]==0} {
            # we found a %, see what the letter is
            incr i
            set ch [string index $action $i]
            # skip %% which is an escaped %
            if {[string compare $ch "%"]!=0} {
                # we found a %? field, see if we already have it
                if {[lookup $ch $fields]==""} {
                    set x [format "%s {%%%s}" $ch $ch]
                    lappend fields $x
                }
            }
        }
    }
    return $fields
}











