/*
 * vms.c
 *
 * Call vms_init() to create the Tcl commands
 *  `vms getjpi', `vms setprn', and `vms trnlog'.
 *
 * scott snyder <snyder@fnald0.fnal.gov>
 */

#include <tcl.h>
#include <jpidef.h>
#include <descrip.h>
#include <string.h>
#include <uaidef.h>
#include <ssdef.h>

/* Initialize a string descriptor (struct dsc$descriptor_s) for an
   arbitrary string.   ADDR is a pointer to the first character
   of the string, and LEN is the length of the string. */

#define INIT_DSC_S(dsc, addr, len) do { \
  (dsc).dsc$b_dtype = DSC$K_DTYPE_T;    \
  (dsc).dsc$b_class = DSC$K_CLASS_S;    \
  (dsc).dsc$w_length = (len);           \
  (dsc).dsc$a_pointer = (addr);         \
} while (0)


/* Initialize a string descriptor (struct dsc$descriptor_s) for a
   NUL-terminated string.  S is a pointer to the string; the length
   is determined by calling strlen(). */

#define INIT_DSC_CSTRING(dsc, s) INIT_DSC_S(dsc, s, strlen(s))


/* these codes aren't defined in jpidef.h */
#define JPI$_CPU_ID          807
#define JPI$_FAST_VP_SWITCH  529
#define JPI$_LOGIN_FAILURES  533
#define JPI$_LOGIN_FLAGS     534
#define JPI$_NODE_CSID       810
#define JPI$_RIGHTS_SIZE     817
#define JPI$_SLOW_VP_SWITCH  530
#define JPI$_STS2            808
#define JPI$_VP_CPUTIM       1052
#define JPI$_NODENAME        809
#define JPI$_NODE_VERSION    811
#define JPI$_TT_ACCPORNAM    813
#define JPI$_TT_PHYDEVNAM    812
#define JPI$_LAST_LOGIN_I    531
#define JPI$_LAST_LOGIN_N    532
#define JPI$_PROCESS_RIGHTS  814
#define JPI$_RIGHTSLIST      806
#define JPI$_SYSTEM_RIGHTS   815

#define JPI$V_NEW_MAIL_AT_LOGIN 0
#define JPI$V_PASSWORD_CHANGED  1
#define JPI$V_PASSWORD_EXPIRED  2
#define JPI$V_PASSWORD_WARNING  3
#define JPI$V_PASSWORD2_CHANGED 4
#define JPI$V_PASSWORD2_EXPIRED 5
#define JPI$V_PASSWORD2_WARNING 6


/* these codes aren't defined in uaidef.h */
#define UAI$V_DISIMAGE   17
#define UAI$V_DISPWDDIC  18
#define UAI$V_DISPWDHIS  19
#define UAI$V_DISWELCOME 5
#define UAI$V_RESTRICTED 3


/* $pcbdef */
#define PCB$V_RES       0
#define PCB$V_DELPEN    1
#define PCB$V_FORCPEN   2
#define PCB$V_INQUAN    3
#define PCB$V_PSWAPM    4
#define PCB$V_RESPEN    5
#define PCB$V_SSFEXC    6
#define PCB$V_SSFEXCE   7
#define PCB$V_SSFEXCS   8
#define PCB$V_SSFEXCU   9
#define PCB$V_SSRWAIT   10
#define PCB$V_SUSPEN    11
#define PCB$V_WAKEPEN   12
#define PCB$V_WALL      13
#define PCB$V_BATCH     14
#define PCB$V_NOACNT    15
#define PCB$V_SWPVBN    16
#define PCB$V_ASTPEN    17
#define PCB$V_PHDRES    18
#define PCB$V_HIBER     19
#define PCB$V_LOGIN     20
#define PCB$V_NETWRK    21
#define PCB$V_PWRAST    22
#define PCB$V_NODELET   23
#define PCB$V_DISAWS    24
#define PCB$V_INTER     25
#define PCB$V_RECOVER   26
#define PCB$V_SECAUDIT  27
#define PCB$V_HARDAFF   28
#define PCB$V_ERDACT    29
#define PCB$V_SOFTSUSP  30
#define PCB$V_PREEMPTED 31

#define PCB$V_QUANTUM_RESCHED 0
#define PCB$V_FORK_RESPEN     1
#define PCB$V_POSIX_SIGNAL    2

#ifdef __GNUC__
#define INITLABEL(l) [l]
#else
#define INITLABEL(l)
#endif

#define ARY_SIZE(a) ((sizeof(a) / sizeof(a[0])))

static void
append_vms_error (Tcl_Interp *interp, int stat)
{
  short msglen;
  char msg[256];
  $DESCRIPTOR (msg_dsc, msg);
  int getmsg_stat;

  getmsg_stat = sys$getmsg (stat, &msglen, &msg_dsc, 15, 0);
  if ((getmsg_stat&1) == 0) lib$signal (getmsg_stat);

  if (msglen >= sizeof (msg))
    msglen = sizeof (msg) - 1;
  msg[msglen] = '\0';

  Tcl_AppendResult (interp, msg, NULL);
}


static
void jpi_int (Tcl_Interp *interp, int num_result[2],
	      char *str_result, int rsltlen)
{
  char buf[20];

  sprintf (buf, "%d", num_result[0]);
  Tcl_SetResult (interp, buf, TCL_VOLATILE);
}


static
void jpi_quad (Tcl_Interp *interp, int num_result[2],
	       char *str_result, int rsltlen)
{
  char buf[20];

  sprintf (buf, "%d", num_result[0]);
  Tcl_AppendElement (interp, buf);

  sprintf (buf, "%d", num_result[1]);
  Tcl_AppendElement (interp, buf);
}


static
void jpi_string (Tcl_Interp *interp, int num_result[2],
	      char *str_result, int rsltlen)
{
  str_result[rsltlen] = '\0';
  Tcl_SetResult (interp, str_result, TCL_VOLATILE);
}


static
void jpi_uncomma (Tcl_Interp *interp, int num_result[2],
		  char *str_result, int rsltlen)
{
  char *p;

  for (p = str_result; p < str_result + rsltlen; p++)
    if (*p == ',') *p = ' ';

  jpi_string (interp, num_result, str_result, rsltlen);
}


#if 0
static
void jpi_rights (Tcl_Interp *interp, int num_result[2],
		 char *str_result, int rsltlen)
{
  int *id_p;
  int nent;
  int i, stat;

printf ("%d %x %d\n", rsltlen, *(int *)str_result, *(int *)(str_result+4));

  nent = rsltlen / sizeof (int) / 2;
  id_p = (int *)str_result;
  for (i=0; i<nent; i++) {
    char nam[512];
    int namlen;
    $DESCRIPTOR (nam_dsc, nam);
    stat = sys$idtoasc (id_p, &namlen, &nam_dsc, 0, 0, 0);
    if ((stat&1) == 0) lib$signal (stat);
    Tcl_AppendElement (interp, nam, 0);
    id_p += 2;
  }
}
#endif


static
void jpi_from_table (Tcl_Interp *interp, int index,
		     char **table, int table_len, char *invalidstr)
{
  char *s;

  if (index > table_len)
    s = invalidstr;
  else
    s = table[index];

  Tcl_SetResult (interp, s, TCL_STATIC);
}


static
void jpi_bitfield (Tcl_Interp *interp, int *field,
		   char **table, int table_len)
{
  int bit, lword, i;

  bit = 0;
  lword = 0;
  for (i=0; i<table_len; i++) {
    if (field[lword] & (1<<bit))
      Tcl_AppendElement (interp, table[i]);
    ++bit;
    if (bit >= 32) {
      bit = 0;
      ++lword;
    }
  }
}


static char *jpi_jobtype_names[] = {
  INITLABEL(JPI$K_DETACHED)  "detached",
  INITLABEL(JPI$K_NETWORK)   "network",
  INITLABEL(JPI$K_BATCH)     "batch",
  INITLABEL(JPI$K_LOCAL)     "local",
  INITLABEL(JPI$K_DIALUP)    "dialup",
  INITLABEL(JPI$K_REMOTE)    "remote",
};


static
void jpi_jobtype (Tcl_Interp *interp, int num_result[2],
	       char *str_result, int rsltlen)
{
  jpi_from_table (interp, num_result[0],
		  jpi_jobtype_names, ARY_SIZE (jpi_jobtype_names),
		  "unknown-jobtype");
}


static char *jpi_sts_names[] = {
  INITLABEL(PCB$V_RES)        "res",
  INITLABEL(PCB$V_DELPEN)     "delpen",
  INITLABEL(PCB$V_FORCPEN)    "forcpen",
  INITLABEL(PCB$V_INQUAN)     "inquan",
  INITLABEL(PCB$V_PSWAPM)     "pswapm",
  INITLABEL(PCB$V_RESPEN)     "respen",
  INITLABEL(PCB$V_SSFEXC)     "ssfexc",
  INITLABEL(PCB$V_SSFEXCE)    "ssfexce",
  INITLABEL(PCB$V_SSFEXCS)    "ssfexcs",
  INITLABEL(PCB$V_SSFEXCU)    "ssfexcu",
  INITLABEL(PCB$V_SSRWAIT)    "ssrwait",
  INITLABEL(PCB$V_SUSPEN)     "suspen",
  INITLABEL(PCB$V_WAKEPEN)    "wakepen",
  INITLABEL(PCB$V_WALL)       "wall",
  INITLABEL(PCB$V_BATCH)      "batch",
  INITLABEL(PCB$V_NOACNT)     "noacnt",
  INITLABEL(PCB$V_SWPVBN)     "swpvbn",
  INITLABEL(PCB$V_ASTPEN)     "astpen",
  INITLABEL(PCB$V_PHDRES)     "phdres",
  INITLABEL(PCB$V_HIBER)      "hiber",
  INITLABEL(PCB$V_LOGIN)      "login",
  INITLABEL(PCB$V_NETWRK)     "netwrk",
  INITLABEL(PCB$V_PWRAST)     "pwrast",
  INITLABEL(PCB$V_NODELET)    "nodelet",
  INITLABEL(PCB$V_DISAWS)     "disaws",
  INITLABEL(PCB$V_INTER)      "inter",
  INITLABEL(PCB$V_RECOVER)    "recover",
  INITLABEL(PCB$V_SECAUDIT)   "secaudit",
  INITLABEL(PCB$V_HARDAFF)    "hardaff",
  INITLABEL(PCB$V_ERDACT)     "erdact",
  INITLABEL(PCB$V_SOFTSUSP)   "softsusp",
  INITLABEL(PCB$V_PREEMPTED)  "preempted",
};


static
void jpi_sts (Tcl_Interp *interp, int num_result[2],
	      char *str_result, int rsltlen)
{
  jpi_bitfield (interp, num_result, jpi_sts_names, ARY_SIZE (jpi_sts_names));
}


static char *jpi_sts2_names[] = {
  INITLABEL(PCB$V_QUANTUM_RESCHED)  "quantum_resched",
  INITLABEL(PCB$V_FORK_RESPEN)      "fork_respen",
  INITLABEL(PCB$V_POSIX_SIGNAL)     "posix_signal",
};


static
void jpi_sts2 (Tcl_Interp *interp, int num_result[2],
	       char *str_result, int rsltlen)
{
  jpi_bitfield (interp, num_result, jpi_sts2_names, ARY_SIZE (jpi_sts2_names));
}


static char *jpi_login_flags_names[] = {
  INITLABEL(JPI$V_NEW_MAIL_AT_LOGIN)   "new_mail_at_login",
  INITLABEL(JPI$V_PASSWORD_CHANGED)    "password_changed",
  INITLABEL(JPI$V_PASSWORD_EXPIRED)    "password_expired",
  INITLABEL(JPI$V_PASSWORD_WARNING)    "password_warning",
  INITLABEL(JPI$V_PASSWORD2_CHANGED)   "password2_changed",
  INITLABEL(JPI$V_PASSWORD2_EXPIRED)   "password2_expired",
  INITLABEL(JPI$V_PASSWORD2_WARNING)   "password2_warning",
};


static
void jpi_login_flags (Tcl_Interp *interp, int num_result[2],
		      char *str_result, int rsltlen)
{
  jpi_bitfield (interp, num_result,
		jpi_login_flags_names, ARY_SIZE (jpi_login_flags_names));
}


static char *jpi_uaf_flags_names[] = {
  INITLABEL(UAI$V_DISCTLY)               "disctly",
  INITLABEL(UAI$V_DEFCLI)                "defcli",
  INITLABEL(UAI$V_LOCKPWD)               "lockpwd",
  INITLABEL(UAI$V_RESTRICTED)            "restricted",
  INITLABEL(UAI$V_DISACNT)               "disacnt",
  INITLABEL(UAI$V_DISWELCOME)            "diswelcome",
  INITLABEL(UAI$V_DISMAIL)               "dismail",
  INITLABEL(UAI$V_NOMAIL)                "nomail",
  INITLABEL(UAI$V_GENPWD)                "genpwd",
  INITLABEL(UAI$V_PWD_EXPIRED)           "pwd_expired",
  INITLABEL(UAI$V_PWD2_EXPIRED)          "pwd2_expired",
  INITLABEL(UAI$V_AUDIT)                 "audit",
  INITLABEL(UAI$V_DISREPORT)             "disreport",
  INITLABEL(UAI$V_DISRECONNECT)          "disreconnect",
  INITLABEL(UAI$V_AUTOLOGIN)             "autologin",
  INITLABEL(UAI$V_DISFORCE_PWD_CHANGE)   "disforce_pwd_change",
  INITLABEL(16)                          "uai??16",
  INITLABEL(UAI$V_DISIMAGE)              "disimage",
  INITLABEL(UAI$V_DISPWDDIC)             "dispwddic",
  INITLABEL(UAI$V_DISPWDHIS)             "dispwdhis",
};


static
void jpi_uaf_flags (Tcl_Interp *interp, int num_result[2],
		    char *str_result, int rsltlen)
{
  jpi_bitfield (interp, num_result,
		jpi_uaf_flags_names, ARY_SIZE (jpi_uaf_flags_names));
}


struct jpi_info_s {
  char *name;
  int  itmcode;
  void (*converter) (Tcl_Interp *, int *, char *, int);
  void (*plain_converter) (Tcl_Interp *, int *, char *, int);
} jpi_info[] = {
  "account",             JPI$_ACCOUNT,        jpi_string,      jpi_string,
  "aptcnt",              JPI$_APTCNT,         jpi_int,	       jpi_int,
  "astact",              JPI$_ASTACT,         jpi_int,	       jpi_int,
  "astcnt",              JPI$_ASTCNT,         jpi_int,	       jpi_int,
  "asten",               JPI$_ASTEN,          jpi_int,	       jpi_int,
  "astlm",               JPI$_ASTLM,          jpi_int,	       jpi_int,
  "authpri",             JPI$_AUTHPRI,        jpi_int,	       jpi_int,
  "authpriv",            JPI$_AUTHPRIV,       jpi_uncomma,     jpi_quad,
  "biocnt",              JPI$_BIOCNT,         jpi_int,	       jpi_int,
  "biolm",               JPI$_BIOLM,          jpi_int,	       jpi_int,
  "bufio",               JPI$_BUFIO,          jpi_int,	       jpi_int,
  "bytcnt",              JPI$_BYTCNT,         jpi_int,	       jpi_int,
  "bytlm",               JPI$_BYTLM,          jpi_int,	       jpi_int,
  "cliname",             JPI$_CLINAME,        jpi_string,      jpi_string,
  "cpu_id",              JPI$_CPU_ID,         jpi_int,	       jpi_int,
  "cpulim",              JPI$_CPULIM,         jpi_int,	       jpi_int,
  "cputim",              JPI$_CPUTIM,         jpi_int,	       jpi_int,
  "creprc_flags",        JPI$_CREPRC_FLAGS,   jpi_int,	       jpi_int,
  "curpriv",             JPI$_CURPRIV,        jpi_uncomma,     jpi_quad,
  "dfpfc",               JPI$_DFPFC,          jpi_int,	       jpi_int,
  "dfwscnt",             JPI$_DFWSCNT,        jpi_int,	       jpi_int,
  "diocnt",              JPI$_DIOCNT,         jpi_int,	       jpi_int,
  "diolm",               JPI$_DIOLM,          jpi_int,	       jpi_int,
  "dirio",               JPI$_DIRIO,          jpi_int,	       jpi_int,
  "efcs",                JPI$_EFCS,           jpi_int,	       jpi_int,
  "efcu",                JPI$_EFCU,           jpi_int,	       jpi_int,
  "efwm",                JPI$_EFWM,           jpi_int,	       jpi_int,
  "enqcnt",              JPI$_ENQCNT,         jpi_int,	       jpi_int,
  "enqlm",               JPI$_ENQLM,          jpi_int,	       jpi_int,
  "excvec",              JPI$_EXCVEC,         jpi_int,	       jpi_int,
  "fast_vp_switch",      JPI$_FAST_VP_SWITCH, jpi_int,	       jpi_int,
  "filcnt",              JPI$_FILCNT,         jpi_int,	       jpi_int,
  "fillm",               JPI$_FILLM,          jpi_int,	       jpi_int,
  "finalexc",            JPI$_FINALEXC,       jpi_int,	       jpi_int,
  "frep0va",             JPI$_FREP0VA,        jpi_int,	       jpi_int,
  "frep1va",             JPI$_FREP1VA,        jpi_int,	       jpi_int,
  "freptecnt",           JPI$_FREPTECNT,      jpi_int,	       jpi_int,
  "gpgcnt",              JPI$_GPGCNT,         jpi_int,	       jpi_int,
  "grp",                 JPI$_GRP,            jpi_int,	       jpi_int,
  "imagecount",          JPI$_IMAGECOUNT,     jpi_int,	       jpi_int,
  "imagname",            JPI$_IMAGNAME,       jpi_string,      jpi_string,
  "imagpriv",            JPI$_IMAGPRIV,       jpi_uncomma,     jpi_quad,
  "jobprccnt",           JPI$_JOBPRCCNT,      jpi_int,	       jpi_int,
  "jobtype",             JPI$_JOBTYPE,        jpi_jobtype,     jpi_int,
  "last_login_i",        JPI$_LAST_LOGIN_I,   jpi_string,      jpi_quad,
  "last_login_n",        JPI$_LAST_LOGIN_N,   jpi_string,      jpi_quad,
  "login_failures",      JPI$_LOGIN_FAILURES, jpi_int,	       jpi_int,
  "login_flags",         JPI$_LOGIN_FLAGS,    jpi_login_flags, jpi_int,
  "logintim",            JPI$_LOGINTIM,       jpi_string,      jpi_quad,
  "master_pid",          JPI$_MASTER_PID,     jpi_int,	       jpi_int,
  "maxdetach",           JPI$_MAXDETACH,      jpi_int,	       jpi_int,
  "maxjobs",             JPI$_MAXJOBS,        jpi_int,	       jpi_int,
  "mem",                 JPI$_MEM,            jpi_int,	       jpi_int,
  "mode",                JPI$_MODE,           jpi_string,      jpi_int,
  "msgmask",             JPI$_MSGMASK,        jpi_int,	       jpi_int,
  "node_csid",           JPI$_NODE_CSID,      jpi_int,	       jpi_int,
  "node_version",        JPI$_NODE_VERSION,   jpi_string,      jpi_string,
  "nodename",            JPI$_NODENAME,       jpi_string,      jpi_string,
  "owner",               JPI$_OWNER,          jpi_int,	       jpi_int,
  "pageflts",            JPI$_PAGEFLTS,       jpi_int,	       jpi_int,
  "pagfilcnt",           JPI$_PAGFILCNT,      jpi_int,	       jpi_int,
  "pagfilloc",           JPI$_PAGFILLOC,      jpi_int,	       jpi_int,
  "pgflquota",           JPI$_PGFLQUOTA,      jpi_int,	       jpi_int,
  "phdflags",            JPI$_PHDFLAGS,       jpi_int,	       jpi_int,
  "pid",                 JPI$_PID,            jpi_int,	       jpi_int,
  "ppgcnt",              JPI$_PPGCNT,         jpi_int,	       jpi_int,
  "prccnt",              JPI$_PRCCNT,         jpi_int,	       jpi_int,
  "prclm",               JPI$_PRCLM,          jpi_int,	       jpi_int,
  "prcnam",              JPI$_PRCNAM,         jpi_string,      jpi_string,
  "pri",                 JPI$_PRI,            jpi_int,	       jpi_int,
  "prib",                JPI$_PRIB,           jpi_int,	       jpi_int,
/*  "process_rights",      JPI$_PROCESS_RIGHTS, jpi_rights,      jpi_string,*/
  "proc_index",          JPI$_PROC_INDEX,     jpi_int,	       jpi_int,
  "procpriv",            JPI$_PROCPRIV,       jpi_uncomma,     jpi_quad,
/*  "rightslist",          JPI$_RIGHTSLIST,     jpi_rights,      jpi_string,*/
  "rights_size",         JPI$_RIGHTS_SIZE,    jpi_int,	       jpi_int,
  "shrfillm",            JPI$_SHRFILLM,       jpi_int,	       jpi_int,
  "sitespec",            JPI$_SITESPEC,       jpi_int,	       jpi_int,
  "slow_vp_switch",      JPI$_SLOW_VP_SWITCH, jpi_int,	       jpi_int,
  "state",               JPI$_STATE,          jpi_string,      jpi_int,
  "sts",                 JPI$_STS,            jpi_sts,	       jpi_int,
  "sts2",                JPI$_STS2,           jpi_sts2,	       jpi_int,
  "swpfilloc",           JPI$_SWPFILLOC,      jpi_int,	       jpi_int,
/*  "system_rights",       JPI$_SYSTEM_RIGHTS,  jpi_rights,      jpi_string,*/
  "tablename",           JPI$_TABLENAME,      jpi_string,      jpi_string,
  "terminal",            JPI$_TERMINAL,       jpi_string,      jpi_string,
  "tmbu",                JPI$_TMBU,           jpi_int,	       jpi_int,
  "tqcnt",               JPI$_TQCNT,          jpi_int,	       jpi_int,
  "tqlm",                JPI$_TQLM,           jpi_int,	       jpi_int,
  "tt_accpornam",        JPI$_TT_ACCPORNAM,   jpi_string,      jpi_string,
  "tt_phydevnam",        JPI$_TT_PHYDEVNAM,   jpi_string,      jpi_string,
  "uaf_flags",           JPI$_UAF_FLAGS,      jpi_uaf_flags,   jpi_int,
  "uic",                 JPI$_UIC,            jpi_string,      jpi_int,
  "username",            JPI$_USERNAME,       jpi_string,      jpi_string,
  "virtpeak",            JPI$_VIRTPEAK,       jpi_int,	       jpi_int,
  "volumes",             JPI$_VOLUMES,        jpi_int,	       jpi_int,
  "vp_cputim",           JPI$_VP_CPUTIM,      jpi_int,	       jpi_int,
  "wsauth",              JPI$_WSAUTH,         jpi_int,	       jpi_int,
  "wsauthext",           JPI$_WSAUTHEXT,      jpi_int,	       jpi_int,
  "wsextent",            JPI$_WSEXTENT,       jpi_int,	       jpi_int,
  "wspeak",              JPI$_WSPEAK,         jpi_int,	       jpi_int,
  "wsquota",             JPI$_WSQUOTA,        jpi_int,	       jpi_int,
  "wssize",              JPI$_WSSIZE,         jpi_int,         jpi_int,
};

#define JPI_INFO_LENGTH (sizeof (jpi_info) / sizeof (jpi_info[0]))


int find_jpi_index (Tcl_Interp *interp, char *key)
{
  struct jpi_info_s p;
  int result;
  int len = strlen (key);
  int hi, mid, lo, comp;

  hi = JPI_INFO_LENGTH;
  lo = 0;

  while (lo < hi-1) {
    mid = (lo + hi) / 2;
    comp = strcmp (key, jpi_info[mid].name);
    if (comp == 0)
      return mid;
    else if (comp < 0)
      hi = mid;
    else
      lo = mid;
  }

  result = lo;
  if (result > 0 || strcmp (key, jpi_info[0].name) > 0)
    ++result;
  if (result >= JPI_INFO_LENGTH ||
      strncmp (key, jpi_info[result].name, len) != 0) {
    Tcl_AppendResult (interp, "bad getjpi option \"", key, (char *)NULL);
    result = -1;
  }
  else {
    if (mid < JPI_INFO_LENGTH-1 &&
	strncmp (key, jpi_info[result+1].name, len) == 0) {
      Tcl_AppendResult (interp, "ambiguous getjpi option \"", key,
			(char *)NULL);
      result = -1;
    }
  }

  if (result < 0) {
    int i, j;
    Tcl_AppendResult (interp, "\": should be one of:\n", (char *)NULL);
    j = 0;
    for (i=0; i<JPI_INFO_LENGTH; i++) {
      Tcl_AppendResult (interp, jpi_info[i].name, " ", (char *)NULL);
      if (++j > 5) {
	Tcl_AppendResult (interp, "\n", (char *)NULL);
	j = 0;
      }
    }
  }

  return result;
}

static int
do_getjpi (ClientData notused, Tcl_Interp *interp,
	   int argc, char **argv)
{
  char c;
  int length;
  int i, stat;
  int num_result[2];
  char str_result[65536];
  $DESCRIPTOR (str_result_dsc, str_result);
  short rsltlen;
  int my_argc, translate_p;
  char **my_argv;
  int pid, *pid_ptr = 0;
  struct dsc$descriptor_s pname_dsc, *pname_ptr = 0;

  my_argc = argc - 2;
  my_argv = argv + 2;

  translate_p = 1;
  if (my_argc > 0 && strncmp (my_argv[0], "-not", 4) == 0) {
    translate_p = 0;
    --my_argc;
    ++my_argv;
  }

  if (my_argc >= 3 && strncmp (my_argv[0], "-pid", 4) == 0) {
    pid_ptr = &pid;
    if (Tcl_GetInt (interp, my_argv[1], pid_ptr) != TCL_OK)
      return TCL_ERROR;
    my_argc -= 2;
    my_argv += 2;
  }

  if (my_argc > 1) {
    INIT_DSC_CSTRING (pname_dsc, my_argv[0]);
    pname_ptr = &pname_dsc;
    --my_argc;
    ++my_argv;
  }

  if (my_argc != 1) {
    Tcl_AppendResult (interp, "wrong # args; should be \"",
		      argv[0], " ", argv[1],
		      " ?-notranslate? ?-pid pid? ?process-name? option\"",
		      (char *)NULL);
    return TCL_ERROR;
  }

  if ((i = find_jpi_index (interp, my_argv[0])) < 0)
    return TCL_ERROR;

  stat = lib$getjpi (&jpi_info[i].itmcode, pid_ptr, pname_ptr,
		     &num_result, &str_result_dsc, &rsltlen);
  if ((stat&1) == 0) {
    append_vms_error (interp, stat);
    return TCL_ERROR;
  }

  (*(translate_p ? jpi_info[i].converter : jpi_info[i].plain_converter))
   (interp, num_result, str_result, (int)rsltlen);

  return TCL_OK;
}


static int
do_trnlog (ClientData notused, Tcl_Interp *interp,
	   int argc, char **argv)
{
  int stat;
  char reslt[1024];
  $DESCRIPTOR (reslt_dsc, reslt);
  short resltlen;
  struct dsc$descriptor_s name_dsc;

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # args; should be \"",
		      argv[0], " ", argv[1],
		      " name\"", (char *)NULL);
    return TCL_ERROR;
  }

  INIT_DSC_CSTRING (name_dsc, argv[2]);
  stat = lib$sys_trnlog (&name_dsc, &resltlen, &reslt_dsc);
  if ((stat&1) == 0) {
    append_vms_error (interp, stat);
    return TCL_ERROR;
  }
  if (stat == SS$_NOTRAN) return TCL_OK;

  reslt[resltlen] = '\0';
  Tcl_SetResult (interp, reslt, TCL_VOLATILE);
  return TCL_OK;
}


static int
do_setprn (ClientData notused, Tcl_Interp *interp,
	   int argc, char **argv)
{
  int stat;
  struct dsc$descriptor_s name_dsc;

  if (argc != 3) {
    Tcl_AppendResult (interp, "wrong # args; should be \"",
		      argv[0], " ", argv[1],
		      " name\"", (char *)NULL);
    return TCL_ERROR;
  }

  INIT_DSC_CSTRING (name_dsc, argv[2]);
  stat = sys$setprn (&name_dsc);
  if ((stat&1) == 0) {
    append_vms_error (interp, stat);
    return TCL_ERROR;
  }
  return TCL_OK;
}


static int
do_vms (ClientData notused, Tcl_Interp *interp,
	int argc, char **argv)
{
  char c;
  int length;

  if (argc < 2) {
    Tcl_AppendResult (interp, "wrong # args; should be \"", argv[0],
		      " option ?arg...?\"", (char *)NULL);
    return TCL_ERROR;
  }

  c = argv[1][0];
  length = strlen (argv[1]);

  if ((c == 'g') && (strncmp (argv[1], "getjpi", length) == 0)) {
    return do_getjpi (notused, interp, argc, argv);
  }
  else if ((c == 't') && (strncmp (argv[1], "trnlog", length) == 0)) {
    return do_trnlog (notused, interp, argc, argv);
  }
  else if ((c == 's') && (strncmp (argv[1], "setprn", length) == 0)) {
    return do_setprn (notused, interp, argc, argv);
  }

  Tcl_AppendResult (interp, "bad option \"", argv[1],
		    "\": should be getjpi, trnlog, or setprn",
		    (char *)NULL);
  return TCL_ERROR;
}


int vms_init (Tcl_Interp *interp)
{
  Tcl_CreateCommand (interp, "vms", do_vms, 0, NULL);
  return TCL_OK;
}


/*
* Date: Fri, 17 Sep 1993 0:56:42 -0500 (CDT)
* From: scott snyder <SNYDER@D0SB10.FNAL.GOV>
* To: vms-tcl@src.honeywell.com
* Subject: $getjpi, $setprn, trnlog for Tcl on VMS
* 
* 
* Here's a start at providing some VMS system services to Tcl programs.
* It's not very polished, but maybe someone will find it useful.
* 
* Call vms_init() during from Tcl_AppInit.  This will define the following
* Tcl commands:
* 
* vms getjpi ?-notranslate?  ?-pid <pid>?  ?<process-name>?  <item>
* 
*   This is an interface to $getjpi.  <item> is the name of the JPI
*   item code; all the standard item codes in VMS 5.5 are supported
*   except for those which return a rights list.  By default, results
*   are translated to a human-readable string, if that makes sense
*   for the item in question.  To suppress this and return the result
*   in a numeric form, use the -notranslate switch.  (With -notranslate,
*   quadword results are returned as a list of two integers.)
* 
*   By default, getjpi returns information for the current process.
*   To specify another process, insert either `-pid <pid>' (where pid
*   is the VMS EPID expressed as an integer) or the process name
*   before the item code.
* 
*   Examples:
* 
*     vms getjpi pgflquota
*     150000
*     % vms getjpi account
*     CALORIM
*     % vms getjpi curpriv
*     TMPMBX NETMBX
*     % vms getjpi -notranslate curpriv
*     1081344 0
*     vms getjpi epoch imagname
*     D0GSG$DKB200:[PRJ19.FREEWARE.EPOCH-4_0P2.][000000]EPOCH.EXE;10
*     % vms getjpi -pid 570425409 imagname
*      SYSTEM-F-NOPRIV, no privilege for attempted operation
* 
* 
* vms trnlog <name>
* 
*   Translates the logical name <name>.  There is currently no way to control
*   what tables are used, and only the first item in a search list is returned.
* 
*   Examples:
* 
*     % vms trnlog {SYS$LOGIN}
*     USR$ROOT2:[SNYDER]
* 
* 
* vms setprn <name>
* 
*   Changes the name of the current process to <name>.
* 
*   Example:
* 
*     % vms getjpi prcnam
*     _FTA21:
*     % vms setprn "Flooby"
*     % vms getjpi prcnam
*     Flooby
* 
* 
* sss
*/
