# Listbox.tcl - itcl widget for scrolled lists, 
#                based on the Tk listbox widget. 
#
# Note: the primitive operations of listboxes are not duplicated here,
# but you can access them through $this.listbox
#
# Copyright (C) 1994 Allan Brighton (allan@piano.sta.sub.org)


itcl_class Listbox {
    inherit FrameWidget


    # set the contents of the listbox from the list

    method set_contents {list} {
	$listbox_ delete 0 end
	foreach i $list {
	    $listbox_ insert end $i
	}
    }


    # return the contents of the listbox as a tcl list

    method get_contents {} {
	set list {}
	set n [$listbox_ size]
	for {set i 0} {$i < $n} {incr i} {
	    lappend list [$listbox_ get $i]
	}
	return $list
    }


    # return a list of the selected items

    method get_selected {} {
	set list {}
	foreach i [$listbox_ curselection] {
	    lappend list [$listbox_ get $i]
	}
	return $list
    }


    # remove the selected items from the listbox
    # and return them as a list

    method remove_selected {} {
	set list [get_selected]
	set n 0
	foreach i [lsort -decreasing [$listbox_ curselection]] {
	    $listbox_ delete $i
	    incr n
	}
	if {$n} {
	    $listbox_ select from $i
	    $listbox_ select to $i
	}
	return $list
    }


    # append a line to the list

    method append {line} {
	$listbox_ insert end $line
    }

    
    # append a list of items to the list

    method append_list {list} {
	foreach i $list {
	    $listbox_ insert end $i
	}
    }

    
    # move the selected row down 1 row

    method move_down {} {
	set list [$listbox_ curselection]
	if {[lempty $list]} {
	    return
	}
	set rlist [lsort -decreasing $list]
	foreach i $rlist {
	    set s [$listbox_ get $i]
	    $listbox_ delete $i
	    $listbox_ insert [expr $i+1] $s
	}
	$listbox_ select from [expr [lindex $list 0]+1]
	$listbox_ select to [expr [lindex $rlist 0]+1]
    }

    
    # move the selected row up 1 row

    method move_up {} {
	set list [$listbox_ curselection]
	if {[lempty $list]} {
	    return
	}
	foreach i $list {
	    set s [$listbox_ get $i]
	    $listbox_ delete $i
	    $listbox_ insert [expr $i-1] $s
	}
	$listbox_ select from [expr [lindex $list 0]-1]
	$listbox_ select to [expr $i-1]
    }


    # make the list empty

    method clear {} {
	$listbox_ delete 0 end
    }


    #  create a new object of this class

    constructor {config} {
	FrameWidget::constructor

	if {"$title" != ""} {
	    pack [label $this.title -text $title] \
		    -side top -fill x
	}
	set listbox_ [listbox $this.listbox \
		-relief sunken \
		-bd 3 \
		-exportselection $exportselection]

	if {$vscroll} {
	    scrollbar $this.vscroll -relief sunken \
		    -command "$listbox_ yview"
	    pack $this.vscroll -side right -fill y
	    $listbox_ config -yscroll "$this.vscroll set"
	}

	if {$hscroll} {
	    scrollbar $this.hscroll -relief sunken \
		    -command "$listbox_ xview" \
		    -orient horizontal
	    pack $this.hscroll -side bottom -fill x
	    $listbox_ config -xscroll "$this.hscroll set"
	}

	pack $listbox_ -side top -fill both -expand 1

	if {$single_select} {
	    tk_listboxSingleSelect $listbox_
	}
    }

    
    # -- public member variables --

    # title string for listbox
    public title {}

    # flag: if true, list will have a vertical scrollbar
    public vscroll {1}

    # flag: if true, list will have a horizontal scrollbar
    public hscroll {0}

    # flag: single selection mode
    public single_select {1}

    # flag: export selection ?
    public exportselection {1}

    
    # -- protected members --

    # listbox widget
    protected listbox_
}


