#-------------------------------------------------------------------------
# MODULE mkSize.tcl
#
# This module is a part of the Visual Ising Machine
#-------------------------------------------------------------------------


#-------------------------------------------------------------------------
#
# mkSize w
#
# Builds a toplevel dialog box that asks the user for 
# the x- and y-resolution of the ising widget
#
# Arguments:
#	w -	window
#-------------------------------------------------------------------------
proc mkSize {w} {

	catch {destroy $w}
	toplevel $w
	wm title $w "VIM \[Size\]"
	wm iconname $w "Size"

	set radiosF [mkSzRadios $w]
	set entryF  [mkSzEntry $w]
	set buttonF [mkSzDlgButtons $w]
	
	pack $buttonF -side bottom -fill both 
	pack $radiosF -side left -fill both
	pack $entryF -fill both
}


#-------------------------------------------------------------------------
#
# mkSzRadios top
#
# Builds radiobuttons in a frame to be placed in window <top>
#
# Arguments:
#	top -	window
#-------------------------------------------------------------------------
proc mkSzRadios {top} {
	global resolution

	set w [frame $top.f1 -relief raised -bd 1]
	set radioF [frame $w.radioF]

	radiobutton $radioF.r1 -text "32 x 32" -width 10 \
		-variable resolution -value 1 \
		-relief groove
	radiobutton $radioF.r2 -text "64 x 64" -width 10 \
		-variable resolution -value 2 \
		-relief groove	
	radiobutton $radioF.r3 -text "100 x 100" -width 10 \
		-variable resolution -value 3 \
		-relief groove
	radiobutton $radioF.r4 -text "128 x 128" -width 10 \
		-variable resolution -value 4 \
		-relief groove
	radiobutton $radioF.r5 -text "256 x 256" -width 10  \
		-variable resolution -value 5 \
		-relief groove
	radiobutton $radioF.r0 -text "Custom  >>" -width 10 \
		-variable resolution -value 0 \
		-relief groove

	pack $radioF.r1 $radioF.r2 $radioF.r3  \
		$radioF.r4 $radioF.r5 $radioF.r0
	pack $radioF -padx 20 -pady 10 -side bottom

	return $w
}


#-------------------------------------------------------------------------
#
# mkSzEntry top
#
# Builds two (TIX-)entries used to input a custom resolution
#
# Arguments:
#	top -	window
#-------------------------------------------------------------------------
proc mkSzEntry {top} {
	set w [frame $top.f2 -relief raised -bd 1]
	set xlbF [tixLabelFrame $w.xlbF -label "x size"]
	set ylbF [tixLabelFrame $w.ylbF -label "y size"]

	set xCtrl [tixControl $xlbF.ctrl -llimit 1 -value 1]
	set yCtrl [tixControl $ylbF.ctrl -llimit 1 -value 1]

	pack $xCtrl -padx 10 -pady 10
	pack $yCtrl -padx 10 -pady 10
	pack $w.xlbF $w.ylbF -fill x -padx 20 -pady 10 

	return $w
}


#-------------------------------------------------------------------------
#
# mkSzDlgButtons top
#
# Builds some (TIX)-dialogbuttons
#
# Arguments:
#	top -	window
#-------------------------------------------------------------------------
proc mkSzDlgButtons {top} {
	set btns [tixDlgBtns $top.btns]

	$btns add apply -text Apply -width 7
	$btns add close -text Close -width 7
	$btns add help -text Help -width 7 -state disabled
	$btns button close configure -command "destroy $top"
	$btns button apply configure -command "szApply $top"

 	return $btns
}


#-------------------------------------------------------------------------
#
# mkSzApply top
#
# This procedure is invoked when the Apply-button is pressed.
# It sends the resolution values to the remote application
#
# Arguments:
#	top -	window
#-------------------------------------------------------------------------
proc szApply {top} {
	global resolution
	
	set path_to_xentry $top.f2.xlbF.f2.ctrl
	set path_to_yentry $top.f2.ylbF.f2.ctrl
		
	$path_to_xentry invoke
	$path_to_yentry invoke

	switch $resolution {
	{0} { set xres [$path_to_xentry get]
	      set yres [$path_to_yentry get] }
	{1} { set xres 32; set yres 32 }
	{2} { set xres 64; set yres 64 }
	{3} { set xres 100; set yres 100 }
	{4} { set xres 128; set yres 128 }
	{5} { set xres 256; set yres 256 }
	}

	remote "stop"
	remote "changeSize $xres $yres"
}
