 #########################################################################
 #                                                                       #
 # Copyright (C) 1993 by General Electric company.  All rights reserved. #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################


# File:	msort.tcl
#
# Description:
#	Natural merge sort for Tcl

 # $Id: msort.tcl,v 1.10 1994/10/27 18:29:42 kennykb Exp $
 # $Source: /tmp_mnt/projects/cliff/iam/all/src/tkauxlib/RCS/msort.tcl,v $
 # $Log: msort.tcl,v $
 # Revision 1.10  1994/10/27  18:29:42  kennykb
 # Release 2.0 -- 10-27-94.  To be uploaded to archive sites.
 #
 # Revision 1.9  1993/11/01  18:20:46  kennykb
 # Beta release to be announced on comp.lang.tcl
 #
 # Revision 1.8  1993/10/27  15:52:49  kennykb
 # Package for alpha release to the Net, and for MMACE 931101 release.
 #
 # Revision 1.7  1993/10/20  19:10:47  kennykb
 # Alpha release #1 was thawed for bug fixes in tk 3.3.  Now frozen again at this
 # point.
 #
 # Revision 1.6  1993/10/20  19:04:11  kennykb
 # Repaired copyright notice so that it doesn't look like structured commentary.
 #
 # Revision 1.5  1993/10/14  18:15:42  kennykb
 # Cleaned up alignment of log messages, to avoid problems extracting
 # structured commentary.
 #
 # Revision 1.4  1993/10/14  18:06:59  kennykb
 # Added GE legal notice to head of file in preparation for release.
 #
 # Revision 1.3  1993/10/14  14:02:02  kennykb
 # Alpha release #1 frozen at this point.
 #
 # Revision 1.2  1993/07/21  19:44:36  kennykb
 # Finished cleaning up structured commentary.
 #
 # Revision 1.1  1993/06/03  15:30:47  kennykb
 # Initial revision
 #

# Procedure:	msort
#
# Synopsis:
#	Natural merge-sort for Tcl.
#
# Usage:
#c	msort list ?comparator?
#
# Parameters:
#c	list
#		List of items to sort
#c	comparator
#		Function to compare items.  The function is expected to
#		accept as parameters two elements from `list' and return
#		a result that is less than zero if the first element collates
#		before the second, zero if the first element is equal to the
#		second, and greater than zero if the first element collates
#		after the second.  Default is `mcomp' (vide infra), which
#		uses the `<' and `>' operators from Tcl to do the comparison.
#
# Return value:
#	List sorted in ascending order.
#
# Description:
#	`msort' sorts a Tcl list into ascending order using an arbitrary
#	comparison procedure (unlike `lsort', which uses a fixed rule for
#	comparison).  It uses the stupidest imaginable natural merge sort.
#
# Potential improvement:
#	Change the initial run formation to create runs longer than singletons.
#
# Notes:
#	This procedure will become obsolete in Tcl 7.0; instead, one will
#	use
#
#c		lsort -command comparator $list

proc msort {list {compar mcomp}} {

	# Distribute the list into singleton runs

	set runs {}
	foreach elt $list {
		lappend runs [list $elt]
	}

	# Merge runs in pairs until only one remains.

	while {[llength $runs] > 1} {
		set run1 [lindex $runs 0]
		set run2 [lindex $runs 1]
		set runs [lrange $runs 2 end]
		lappend runs [merge $run1 $run2 $compar]
	}
	return [lindex $runs 0]
}

# Procedure:	merge
#
# Synopsis:
#	Merge two sorted lists
#
# Usage:
#c	merge list1 list2 ?comparator?
#
# Parameters:
#c	list1
#
#c	list2
#		Lists to merge.
#c	comparator
#		Function to compare items.  The function is expected to
#		accept as parameters two elements from `list' and return
#		a result that is less than zero if the first element collates
#		before the second, zero if the first element is equal to the
#		second, and greater than zero if the first element collates
#		after the second.  Default is `mcomp' (q.v.), which
#		uses the `<' and `>' operators from Tcl to do the comparison.
#
# Return value:
#	A single list formed by merging the two lists in ascending order
#	according to `comparator.'
#
# Description:
#	`merge' merges two lists that are in ascending order into a single,
#	longer list.  The user has the choice of a procedure that defines the
#	ordering among elements.

proc merge {l1 l2 {compar mcomp}} {
	set result {}
	if {[llength $l1] == 0} {
		return $l2
	}
	if {[llength $l2] == 0} {
		return $l1
	}
	set elt1 [lindex $l1 0]
	set l1 [lrange $l1 1 end]
	set elt2 [lindex $l2 0]
	set l2 [lrange $l2 1 end]
	while 1 {
		if {[$compar $elt1 $elt2] < 0} {
			lappend result $elt1
			if {[llength $l1] == 0} {
				lappend result $elt2
				return [concat $result $l2]
			} else {
				set elt1 [lindex $l1 0]
				set l1 [lrange $l1 1 end]
			}
		} else {
			lappend result $elt2
			if {[llength $l2] == 0} {
				lappend result $elt1
				return [concat $result $l1]
			} else {
				set elt2 [lindex $l2 0]
				set l2 [lrange $l2 1 end]
			}
		}
	}
}

# Procedure:	mcomp
#
# Synopsis:
#	Default comparator for `msort' and `merge'
#
# Usage:
#c	mcomp item1 item2
#
# Parameters:
#c	item1
#
#c	item2
#		Items to be compared
#
# Return value:
#	-1 if item1 < item2
#
#	0 if item1 == item2
#
#	1 if item1 > item2
#
# Description:
#	`mcomp' is the default comparison procedure used by `msort' and
#	`merge'.  It compares list items according to the rule used by the
#	`<' and `>' operators of Tcl.

proc mcomp {e1 e2} {
	if {$e1 < $e2} {
		return -1
	}
	if {$e1 > $e2} {
		return 1
	}
	return 0
}
