set RCSID([file tail [info script]]) \
 "$Id: composite.tcl,v 1.4 1994/10/27 18:29:42 kennykb Exp $"

 #########################################################################
 #                                                                       #
 # Copyright (C) 1994 by General Electric Company.		 	 #
 # All rights reserved. 						 #
 #                                                                       #
 # Permission to use, copy, modify, and distribute this                  #
 # software and its documentation for any purpose and without            #
 # fee is hereby granted, provided that the above copyright              #
 # notice appear in all copies and that both that copyright              #
 # notice and this permission notice appear in supporting                #
 # documentation, and that the name of General Electric not be used in   #
 # advertising or publicity pertaining to distribution of the            #
 # software without specific, written prior permission.                  #
 #                                                                       #
 # General Electric makes no representations about the suitability of    #
 # this software for any purpose.  It is provided ``as is''              #
 # without express or implied warranty.                                  #
 #                                                                       #
 # This work was supported in part by the DARPA Initiative in Concurrent #
 # Engineering (DICE) through DARPA Contracts MDA972-88-C-0047 and       #
 # MDA972-92-C-0027.                                                     #
 #                                                                       #
 # This work was supported in part by the Tri-Services Microwave and     #
 # Millimeter-Wave Advanced Computational Environment (MMACE) program    #
 # under Naval Research Laboratory contract N00014-92-C-2044.            #
 #                                                                       #
 #########################################################################

 # $Id: composite.tcl,v 1.4 1994/10/27 18:29:42 kennykb Exp $
 # $Source: /tmp_mnt/projects/cliff/iam/all/src/tkauxlib/RCS/composite.tcl,v $
 # $Log: composite.tcl,v $
 # Revision 1.4  1994/10/27  18:29:42  kennykb
 # Release 2.0 -- 10-27-94.  To be uploaded to archive sites.
 #
 # Revision 1.3  1994/10/27  18:27:39  kennykb
 # Updated legal notices prior to release.
 #

# File:	composite.tcl
#
# Description:
#	Procedures, functions, and variables for managing composite widgets.
#
# Global variables:
#
#c	${class}_commands
#		Array whose indices are the commands accepted by the
#		specified widget class and whose values are the names
#		of Tcl procedures that handle the commands.
#c	${class}_configAction
#		Array whose indices are X resources corresponding to the
#		configuration options of the specified widget class and
#		whose values are the names of the procedures that handle
#		the options.
#c	${class}_configDefault
#		Array whose indices are X resources corresponding to the
#		configuration options of the specified widget class and
#		whose values are the default values for the options.
#c	${class}_configResource
#		Array whose indices are command-line configuration switches
#		accepted by the specified widget class and whose values are
#		the names of X resources corresponding to the switches.
#c	${class}_configResourceClass
#		Array whose indices are X resources corresponding to the
#		configuration options of the specified widget class and
#		whose values are the corresponding resource classes.
#c	${class}_configSwitch
#		Array whose indices are X resources corresponding to the
#		configuration options of the specified widget class and whose
#		values are the command-line configuration switches
#		corresponding to the options.
#c	${class}_initProc
#		Name of the initialization procedure for widgets of the
#		specified class.
#c	composite_config${pathName}
#		Array whose indices are the X resources corresponding to the
#		configuration options associated with the specified widget
#		and whose values are the current values of those options.
#
# Transient procedures:
#c	${class}_alias${pathName}
#		The Tk widget command ${pathName}, renamed so that the
#		composite widget system can define its own widget command.
#c	[string tolower $class]
#		The widget creation procedure for widgets of class $class.

 # $Id: composite.tcl,v 1.4 1994/10/27 18:29:42 kennykb Exp $
 # $Source: /tmp_mnt/projects/cliff/iam/all/src/tkauxlib/RCS/composite.tcl,v $
 # $Log: composite.tcl,v $
 # Revision 1.4  1994/10/27  18:29:42  kennykb
 # Release 2.0 -- 10-27-94.  To be uploaded to archive sites.
 #
 # Revision 1.3  1994/10/27  18:27:39  kennykb
 # Updated legal notices prior to release.
 #
 # Revision 1.2  1994/09/15  14:01:51  kennykb
 # Fixed several typos in applying child configuration.
 #
 # Revision 1.1  1993/11/15  18:13:09  kennykb
 # Initial revision
 #


#
		 ###################################
		 # Composite widget class creation #
		 ###################################

# Procedure:	composite_define
#
# Synopsis:
#	Define a new type of composite widget.
#
# Usage:
#c	composite_define class {config...} initProc
#
# Parameters:
#c	classes
#		A (possibly multi-element) list of widget class names.
#		The first one of these is the capitalized name of the
#		X resource class which locates the widget's defaults.
#		The widget creation command will be the lowercase
#		version of `class'.
#
#		The remaining class names are classes from which the
#		widget inherits configuration flags and widget commands.
#		Classes appearing later in the list override classes that
#		appear earlier in the list.  The `Composite' widget class
#		is always first.
#
#c	config
#
#		A list of lists giving the widget's configuration
#		parameters.
#
#		A singleton indicates that the parameter is not accepted.
#		(This notation is used to delete an inherited configuration
#		flag).
#
#		A two-element list indicates that the parameter is
#		a synonym for another parameter; an example is:
#		{-bg background}.
#
#		A three-, four- or five-element list gives the following:
#
#			+ The command-line flag for the option.
#
#			+ The name (X resource) of the option.
#
#			+ The class (X resource) of the option.
#
#			+ The default value of the option.
#			If not specified, the null string is used.
#
#			+ The name of a command that processes
#			the option.  The command accepts the window name,
#			the option name, and the option value.  Default is
#			`Composite_config', which causes the flag to be
#			recursively applied to all widgets in the composite's
#			subtree that support it.
#
#c	initProc
#		The name of a procedure that constructs the widget.  The
#		procedure is passed a single argument -- the name a frame
#		widget in which the composite will be constructed.  It is
#		expected to return the same name.
#
# Return value:
#	None.
#
# Description:
#	composite_define defines a procedure, which in turn is called to 
#	create widgets.  It is the fundamental procedure that is used to
#	make a composite widget.

proc composite_define {classes config initproc} {

	set class [lindex $classes 0]

	upvar #0 ${class}_commands commands
	upvar #0 ${class}_configResource configResource
	upvar #0 ${class}_configResourceClass configResourceClass
	upvar #0 ${class}_configDefault configDefault
	upvar #0 ${class}_configSwitch configSwitch
	upvar #0 ${class}_configAction configAction
	upvar #0 ${class}_initProc initProc

	if {$class != "Composite"} {
		composite:inherit $class Composite
	}
	foreach parent [lrange $classes 1 end] {
		require [string tolower $parent]
		composite:inherit $class $parent
	}

	foreach item $config {
		set flag [lindex $item 0]
		if {[llength $item] == 1} {
			set resource $configResource($flag)
			unset configResource($flag)
			unset configResourceClass($resource)
			unset configDefault($resource)
			catch {unset configSwitch($resource)}
			unset configAction($resource)
		}

		if {[llength $item] >= 2} {
			set resource [lindex $item 1]
			set configResource($flag) $resource
		}
		if {[llength $item] >= 3} {
			set resourceClass [lindex $item 2]
			if {[llength $item] >= 4} {
				set default [lindex $item 3]
			} else {
				set default {}
			}
			if {[llength $item] >= 5} {
				set action [lindex $item 4]
			} else {
				set action composite_config
			}
			set configResourceClass($resource) $resourceClass
			set configDefault($resource) $default
			set configAction($resource) $action
			set configSwitch($resource) $flag
		}
	}

	set initProc $initproc

	composite:createClassProc $class
}

# Procedure: composite:inherit
#
# Synopsis:
#	Internal procedure to establish an inheritance relationship
#	between two composite widget classes.
#
# Usage:
#c	composite:inherit childClass parentClass
#
# Parameters:
#c	childClass
#		Name of the child class (the one that is inheriting
#		widget commands and configuration switches)
#c	parentClass
#		Name of the parent class (the one from which
#		widget commands and configuration switches are being
#		inherited
#
# Return value:
#	None specified.
#
# Description:
#	composite:ingerit arranges the state of the global variables so that
#	widgets of class `childClass' accept the same configuration switches
#	and widget commands as do those of `parentClass'.

proc composite:inherit {class parent} {
	upvar #0 ${class}_commands commands
	upvar #0 ${class}_configResource configResource
	upvar #0 ${class}_configResourceClass configResourceClass
	upvar #0 ${class}_configDefault configDefault
	upvar #0 ${class}_configSwitch configSwitch
	upvar #0 ${class}_configAction configAction
	upvar #0 ${parent}_commands parentCommands
	upvar #0 ${parent}_configResource parentConfigResource
	upvar #0 ${parent}_configResourceClass parentConfigResourceClass
	upvar #0 ${parent}_configDefault parentConfigDefault
	upvar #0 ${parent}_configSwitch parentConfigSwitch
	upvar #0 ${parent}_configAction parentConfigAction

	foreach key [array names parentCommands] {
		set commands($key) $parentCommands($key)
	}
	foreach key [array names parentConfigResource] {
		set configResource($key) $parentConfigResource($key)
	}
	foreach key [array names parentConfigResourceClass] {
		set configResourceClass($key) $parentConfigResourceClass($key)
	}
	foreach key [array names parentConfigDefault] {
		set configDefault($key) $parentConfigDefault($key)
	}
	foreach key [array names parentConfigSwitch] {
		set configSwitch($key) $parentConfigSwitch($key)
	}
	foreach key [array names parentConfigAction] {
		set configAction($key) $parentConfigAction($key)
	}
}

# Procedure:	composite:createClassProc
#
# Synopsis:
#	Create the procedure that is named after a composite widget class
#	and in turn is used to create widgets of that class.
#
# Usage:
#c	composite:createClassProc className
#
# Parameters:
#c	className
#		Name of the widget class
#
# Return value:
#	None specified.

proc composite:createClassProc class {
	proc [string tolower $class] {w args} "
		composite:create \$w $class \$args
	"
}

#
		    #############################
		    # Composite widget creation #
		    #############################

# Procedure:	composite:create
#
# Synopsis:
#	Internal procedure to create a composite widget defined by
#	composite_define.
#
# Usage:
#c	composite:create pathName class params
#
# Parameters:
#c	pathName
#		Path name of the widget being created.
#c	class
#		Class of the widget being created.
#c	params
#		Configuration options supplied at widget creation time.
#
# Return value:
#	Name of the widget created.
#
# Description: 
#	composite:create does the dirty work of a widget command
#	defined by composite_define.  It accepts the widget name, the
#	widget's class, and the configuration parameters.  It creates
#	the frame in which the widget will be constructed, parses
#	configuration parameters (and the corresponding options from
#	the option database), calls the widget initialization
#	procedure, renames the frame command, and puts in its place
#	the composite widget command.  Finally, it calls
#	composite:applyInitialConfig to make the configuration options
#	take effect.

proc composite:create {w class params} {
	upvar #0 ${class}_initProc initproc
	frame $w -class $class
	set status [catch {
		composite:getInitialConfig $w $params
		widget_addBinding $w Destroy "composite:deleteConfig $w"
		$initproc $w
		rename $w ${class}_alias$w
		proc $w {command args} "
			eval \[list composite:command $w \$command\] \$args
		"
		widget_addBinding $w Destroy "composite:deleteAlias $w"
		update idletasks
		composite:applyInitialConfig $w
	} message]
	if {$status != 0} {
		global errorInfo
		global errorCode
		set info $errorInfo
		set code $errorCode
		catch {destroy $w}
		error $message $info $code
	}
	return $w
}

# Procedure: composite:getInitialConfig
#
# Synopsis:
#	Internal procedure to determine a composite widget's
#	initial configuration parameters.
#
# Usage:
#c	composite_getInitialConfig pathName ?-flag value?...
#
# Parameters:
#c	pathName
#		Path name of the widget being created.
#c	flag, value
#		Configuration flags that apply to the widget.
#
# Description:
#	composite:getInitialConfig determines the full set of configuration
#	parameters when creating a composite widget.  It scans the defaults
#	for the widget, and overrides any that are specified with X resources
#	or the `option' command.  It then further overrides any parameters
#	than are specified on the command line.  The resulting set of
#	parameters is stashed in the `composite_config$pathName' array
#	for further processing in composite:applyInitialConfig.

proc composite:getInitialConfig {w params} {
	set class [winfo class $w]
	upvar #0 ${class}_configDefault default
	upvar #0 ${class}_configResourceClass resourceClass
	upvar #0 composite_config$w config
	foreach r [array names default] {
		set value [option get $w $r $resourceClass($r)]
		if {$value == ""} {
			set value $default($r)
		}
		set config($r) $value
	}
	while {[llength $params] >= 2} {
		set r [composite:findConfigFlag $w [lindex $params 0]]
		set value [lindex $params 1]
		set params [lrange $params 2 end]
		set config($r) $value
	}
}

# Procedure:	composite:applyInitialConfig
#
# Synopsis:
#	Apply the initial configuration to a composite widget.
#
# Usage:
#c	composite:applyInitialConfig pathName
#
# Parameters:
#c	pathName
#		Path name of the widget being created
#
# Description:
#	`composite:applyInitialConfig' constructs a `pathName configure'
#	command that contains all the initial options for a composite widget.
#	It then executes that command in the caller's lexical scope to
#	configure the widget.

proc composite:applyInitialConfig w {
	set class [winfo class $w]
	upvar #0 composite_config$w config
	upvar #0 ${class}_configSwitch configSwitch
	set command [list $w configure]
	foreach r [lsort [array names config]] {
		lappend command $configSwitch($r) $config($r)
	}
	uplevel 1 $command
}

#
		   ################################
		   # Composite widget destruction #
		   ################################

# Procedure:	composite:deleteConfig
#
# Synopsis:
#	Delete the global array describing a composite widget's
#	configuration as the widget is being destroyed.
#
# Usage:
#c	composite:deleteConfig pathName
#
# Parameters:
#	pathName
#		Path name of the widget being destroyed.
#
# Return value:
#	None specified.

proc composite:deleteConfig w {
	upvar #0 composite_config$w config
	unset config
}

# Procedure:	composite:deleteAlias
#
# Synopsis:
#	Delete the alias procedure that handles the widget command
#	of a composite widget.
#
# Usage:
#c	composite:deleteAlias pathName
#
# Parameters:
#c	pathName
#		Path name of the widget being destroyed.
#
# Return value:
#	None specified.

proc composite:deleteAlias w {
	set class [winfo class $w]
	catch {rename ${class}_alias$w $w}
}

#
	 ####################################################
	 # Composite widget configuration option processing #
	 ####################################################

# Procedure:	composite_configFlag
#
# Synopsis:
#	Define a configuration option for a composite widget.
#
# Usage:
#c	composite_configFlag className option paramList body
#
# Parameters:
#c	className
#		Widget class for which the option is being defined.
#c	option
#		Name of the option being defined.  This is NOT its
#		command-line switch, but rather its X resource name.
#c	paramList
#		Parameters to the function that handles the option.  The
#		function is expected to accept three parameters -- the name
#		of the widget, the name of the configuration option, and
#		the value of the configuration option.
#c	body
#		Body of the procedure that handles the option.  In addition
#		to doing any local option processing, the procedure may call
#		one of the following procedures to apply an option recursively:
#
#			+ composite_config, which causes the option to
#			be applied to the Tk widget and all its descendants.
#
#			+ composite_configLocal, which causes the option to
#			be applied to the Tk widget only, and
#
#			+ composite_configChild, which causes the option to
#			be applied to the descendant widgets only.
#
# Return value:
#	The name of the function that handles the option.  The function will
#	always have the name
#
#c		${className}_config_XXX
#
#	where _XXX is the configuration flag being processed.
#
# Description:
#	composite_configFlag defines the function that handles a single
#	configuration option for a widget.  It is responsible for taking
#	any necessary action when the configuration changes.
#
# Example:
#c	composite_configFlag myWidget textVariable {w flag vname} {
#
#c		$w.entry.box configure -textvariable $vname
#
#c	}

proc composite_configFlag {class resource params body} {
	proc ${class}_config_${resource} $params $body
	composite_configFlagProc $class $resource ${class}_config_${resource}
}

# Procedure:	composite_configFlagProc
#
# Synopsis:
#	Define a configuration option for a composite widget.
#
# Usage:
#c	composite_configFlagProc className option procName
#
# Parameters:
#c	className
#		Widget class for which the option is being defined.
#c	option
#		Name of the option being defined.  This is NOT its
#		command-line switch, but rather its X resource name.
#c	procName
#		Name of the procedure that handles the option.  The
#		procedure is expected to accept three parameters:
#			+ the name of the widget.
#
#			+ the name of the option.
#
#			+ the value of the option.
#
#		Three choices for `procName' are provided by the system:
#
#			+ composite_config, which causes the option to
#			be applied to the Tk widget and all its descendants.
#
#			+ composite_configLocal, which causes the option to
#			be applied to the Tk widget only, and
#
#			+ composite_configChild, which causes the option to
#			be applied to the descendant widgets only.
#
#		These procedures may also be called from within a user's
#		configuration procedure.
#
# Return value:
#	The name of the procedure that handles the option.
#
# Description:
#	composite_configFlag associates the procedure that handles a single
#	configuration option for a widget.  It is responsible for taking
#	any necessary action when the configuration changes.
#
# Example:
#c	composite_configFlag myWidget relief composite_configLocal

proc composite_configFlagProc {class resource procName} {
	upvar #0 ${class}_configAction configAction
	set configAction($resource) $procName
}

# Procedure:	composite:applyConfig
#
# Synopsis:
#	Internal procedure that applies configuration flags to a composite
#	widget.
#
# Usage:
#c	composite:applyConfig pathName ?-flag value?...
#
# Parameters:
#c	pathName
#		Path name of the widget being configured
#c	-flag, value
#		Any number of flag-value pairs giving the widget's
#		desired configuration.  Flags may be abbreviated to any
#		unique prefix.  These flags are the command-line
#		switches, not the resource names.
#
# Return value:
#	Not specified.
#
# Description:
#	The `composite:applyConfig' procedure is called from a
#	composite widget's `configure' subcommand once it is determined that
#	the caller has supplied name-value pairs for the configuration
#	options.  It looks up each configuration option in the table
#	of flags that the widget accepts, and calls the action procedure
#	for each.
#
#	The action procedure is expected to call `composite_config',
#	`composite_configLocal', or `composite_configChild' for every
#	option it finds that must be applied recursively.  These procedures
#	import the LOCALFLAGS and CHILDFLAGS arrays from the lexical
#	scope of composite:applyConfig, and install the appropriate
#	flags therein.
#
#	After all the configuration procedures have run, the
#	`composite:applyLocalConfig' procedure is called with all the
#	local configuration flags, passing the name of the alias.
#	command that actually enters Tk for the widget.  It applies all
#	the local configuration flags to the widget.
#
#	Finally, `composite:applyChildConfig' is applied to all the
#	children of the widget, passing the child configuration flags.
#	This call recursively applies all the propagated configuration options.

proc composite:applyConfig {w args} {

	# Determine widget class

	set class [winfo class $w]

	# Import the configuration action table for the class

	upvar #0 ${class}_configAction action
	upvar #0 composite_config${w} values

	# Parse the arguments.  Call a configuration procedure for each.

	set argv $args
	while {[llength $argv] >= 2} {
		set key [composite:findConfigFlag $w [lindex $argv 0]]
		set value [lindex $argv 1]
		set argv [lrange $argv 2 end]
		$action($key) $w $key $value
		set values($key) $value
	}

	# Apply any local configuration flags

	set command "composite:applyLocalConfig ${class}_alias$w"
	set status [catch {array names LOCALFLAGS} keys]
	if {$status == 0} {
		foreach key $keys {
			lappend command $key $LOCALFLAGS($key)
		}
		eval $command
	}

	# Propagate flags into the children.

	set propagate {}
	set status [catch {array names CHILDFLAGS} keys]
	if {$status == 0} {
		foreach key $keys {
			lappend propagate $key $CHILDFLAGS($key)
		}
		foreach child [winfo children $w] {
			eval [list composite:applyChildConfig $child] \
				$propagate
		}
	}
}

# Procedure:	composite_config
#
# Synopsis:
#	Cause a configuration option to be applied to a widget and all of
#	its descendants.
#
# Usage:
#c	composite_config pathName optionName value
#
# Parameters:
#c	pathName
#		Path name of the composite widget.
#c	optionName
#		Name of the option (NOT its command-line switch).
#c	value
#		Value of the option.
#
# Return value:
#	None specified.
#
# Description:
#	The `composite_config' procedure is the default configuration
#	procedure for configuration options that have none; it also
#	may be called from users' configuration procedures.  It accepts
#	a widget path name, a configuration option name, and the option's
#	value.  It calls `composite_configLocal' and `composite_configChild'
#	to propagate the option.

proc composite_config {w flag value} {
	composite_configLocal $w $flag $value
	composite_configChild $w $flag $value
}

# Procedure:	composite_configLocal
#
# Synopsis:
#	Cause a configuration option to be applied to a widget but not
#	its descendants.
#
# Usage:
#c	composite_configLocal pathName optionName value
#
# Parameters:
#c	pathName
#		Path name of the composite widget.
#c	optionName
#		Name of the option (NOT its command-line switch).
#c	value
#		Value of the option.
#
# Return value:
#	None specified.
#
# Description: 
#	The `composite_configLocal' procedure marks an option for
#	application to the local widget.  It works by importing the
#	LOCALFLAGS array from the most recent invocation of
#	composite:applyConfig, and installing the command-line switch
#	and value in the array.  The option name is translated to the
#	command-line switch at this point.

proc composite_configLocal {w flag value} {
	set class [winfo class $w]
	upvar #0 ${class}_configSwitch switch
	set lev -1
	for {set l [info level]} {$l > 0} {incr l -1} {
		set pname [lindex [info level $l] 0]
		if {$pname == "composite:applyConfig"
		    || $pname == "debug:aliascomposite:applyConfig"} {
			set lev $l
			break
		}
	}
	if {$lev < 0} {
		error \
"composite_configLocal invoked, but not configuring a widget."
	}
	upvar #$lev LOCALFLAGS flags
	set s $switch($flag)
	if [info exists flags($s)] {
		error "composite_configLocal: conflicting values for $flag."
	}
	set flags($s) $value
}

# Procedure:	composite_configChild
#
# Synopsis:
#	Cause a configuration option to be applied to a widget's
#	descendants.
#
# Usage:
#c	composite_configChild pathName optionName value
#
# Parameters:
#c	pathName
#		Path name of the composite widget.
#c	optionName
#		Name of the option (NOT its command-line switch).
#c	value
#		Value of the option.
#
# Return value:
#	None specified.
#
# Description: 
#	The `composite_configLocal' procedure marks an option for
#	application to a widget's descendants.  It works by importing the
#	CHILDFLAGS array from the most recent invocation of
#	composite:applyConfig, and installing the command-line switch
#	and value in the array.  The option name is translated to the
#	command-line switch at this point.

proc composite_configChild {w flag value} {
	set class [winfo class $w]
	upvar #0 ${class}_configSwitch switch
	set lev -1
	for {set l [info level]} {$l > 0} {incr l -1} {
		set pname [lindex [info level $l] 0]
		if {$pname == "composite:applyConfig"
		    || $pname == "debug:aliascomposite:applyConfig"} {
			set lev $l
			break
		}
	}
	if {$lev < 0} {
		error \
"composite_configChild invoked, but not configuring a widget."
	}
	upvar #$lev CHILDFLAGS flags
	set s $switch($flag)
	if [info exists flags($s)] {
		error "composite_configChild: conflicting values for $flag."
	}
	set flags($s) $value
}

# Procedure:	composite:applyLocalConfig
#
# Synopsis:
#	Internal procedure to apply configuration options to the
#	local widget.
#
# Usage:
#c	composite:applyLocalConfig pathName ?-flag value?...
#
# Parameters:
#c	pathName
#		Path name of the widget being configured, or an alias
#		procedure that identifies the widget.
#c	-flag, value
#		Switch-value pairs that give the widget's configuration.
#		Switches that the widget does not accept may be included,
#		and will be deleted.  The flags may not be abbreviated
#		at this point.
#
# Return value:
#	None specified.
#
# Description:
#	composite:applyLocalConfig selects the flag-value pairs from its
#	command line that apply to the specified widget, and applies them.

proc composite:applyLocalConfig {w args} {
	while {[llength $args] >= 2} {
		set key [lindex $args 0]
		set value [lindex $args 1]
		set args [lrange $args 2 end]
		set flag($key) $value
	}
	set command [list $w configure]
	foreach entry [$w config] {
		set key [lindex $entry 0]
		if [info exists flag($key)] {
			lappend command $key $flag($key)
		}
	}
	eval $command
}

# Procedure:	composite:applyChildConfig
#
# Synopsis:
#	Internal procedure to apply configuration options to a descendant
#	of the local widget.
#
# Usage:
#c	composite:applyChildConfig pathName ?-flag value?...
#
# Parameters:
#c	pathName
#		Path name of the child widget being configured.
#c	-flag, value
#		Switch-value pairs that give the widget's configuration.
#		Switches that the widget does not accept may be included,
#		and will be deleted.  The flags may not be abbreviated
#		at this point.
#
# Return value:
#	None specified.
#
# Description:
#	composite:applyChildConfig selects the flag-value pairs from its
#	command line that apply to the specified widget, and applies them.
#	Configuration is not propagated into top-level widgets.
#	If a widget is not a composite widget, the flag-value pairs are
#	also applied recursively to all its children, allowing the
#	use of `frame' widgets as artifacts of geometry management without
#	needing to define widget classes for all of them.

proc composite:applyChildConfig {w args} {

	# Don't propagate configuration into toplevel widgets.

	if {$w == [winfo toplevel $w]} {
		return
	}

	# Apply any local configuration options that apply

	eval [list composite:applyLocalConfig $w] $args

	# Let managed widgets handle their own configuration.

	if [composite_isManaged $w] {
		return
	}

	# Apply flags to children of unmanaged widgets.

	foreach child [winfo children $w] {
		composite:applyChildConfig $child
	}
}

# Procedure:	composite:findConfigFlag
#
# Synopsis:
#	Internal procedure to process unique-prefix abbreviation of
#	configuration switches.
#
# Usage:
#c	composite:findConfigFlag pathName -flag
#
# Parameters:
#c	pathName
#		Path name of the widget being configured.
#c	-flag
#		Flag to locate.
#
# Return value:
#	X resource name corresponding to the specified flag.
#
# Description:
#	composite:findConfigFlag calls composite:matchName to select
#	the configuration option that matches the given switch, and
#	returns the switch's resource name.

proc composite:findConfigFlag {w flag} {
	set class [winfo class $w]
	upvar #0 ${class}_configResource configResource
	set status [catch {composite:matchName $flag configResource} result]
	if {$status == 0} {
		return $configResource($result)
	} else {
		error "$class does not support a $flag flag" \
			"$class does not support a $flag flag\n\
\tavailable flags are [lsort [array names configResource]]"
	}
}

#
	       ########################################
	       # Composite widget command processing. #
	       ########################################

# Procedure:	composite_subcommand
#
# Synopsis:
#	Define a subcommand for a composite widget's widget command.
#
# Usage:
#c	composite_subcommand className command paramList body
#
# Parameters:
#c	className
#		The class of widget for which the subcommand is being defined.
#c	command
#		The name of the widget command being defined.
#c	paramList
#		The parameters that the widget command expects.  The
#		first parameter is always the widget path name; the remaining
#		parameters are taken from the call to the widget command.
#c	body
#		The body of the procedure that executes the command.
#
# Return value:
#	The name of the procedure that executes the command.  It will have
#	the form
#
#c		${className}_command_XXX
#
#	where XXX is the name of the command.
#
# Description:
#	composite_subcommand defines a new widget command for a widget.
#
# Example:
#c	composite_subcommand myWidget show {w string} {
#
#c		$w.label config -text $string
#
#c	}
#
#	mywidget foo
#
#c	.foo show "Hi there"

proc composite_subcommand {class command params body} {
	proc ${class}_command_${command} $params $body
	composite_subcommandProc $class $command ${class}_command_${command}
}

# Procedure:	composite_subcommand
#
# Synopsis:
#	Define a subcommand for a composite widget's widget command.
#
# Usage:
#c	composite_subcommand className command procName
#
# Parameters:
#c	type
#		The type of widget for which the subcommand is being defined.
#c	command
#		The name of the widget command being defined.
#c	procName
#		The procedure that handles the widget command.  The procedure's
#		first parameter is always the widget path name; the remaining
#		parameters are taken from the call to the widget command.
#
# Return value:
#	The name of the procedure that executes the command.
#
# Description:
#	composite_subcommand defines a new widget command for a widget.

proc composite_subcommandProc {class command procName} {
	upvar #0 ${class}_commands commands
	set commands($command) ${class}_command_${command}
}

# Procedure:	composite:command
#
# Synopsis:
#	Internal procedure to evaluate a widget command on a composite widget.
#
# Usage:
#c	composite:command pathName command args
#
# Parameters:
#c	pathName
#		Path name of a composite widget.
#c	command
#		Name of the command to execute.
#c	args
#		Parameters to the command
#
# Return value:
#	Specified by the user.
#
# Description:
#	The composite:command procedure executes a widget command on a
#	composite widget.  It includes unique-prefix matching for
#	subcommand names.

proc composite:command {w command args} {
	set class [winfo class $w]
	upvar #0 ${class}_commands commands
	set status [catch {composite:matchName $command commands} result]
	if {$status != 0} {
		error "$w: invalid command $command" \
"$w: invalid command $command; available commands are:
	[lsort [array names commands]]"
	}
	eval [list $commands($result) $w] $args
}

#
	     ############################################
	     # Service procedures for composite widgets #
	     ############################################

# Procedure:	composite:matchName
#
# Synopsis:
#c	Name matching for widget commands
#
# Usage:
#c	composite:matchName name array
#
# Parameters:
#c	name
#		Name of a subcommand, configuration option, etc.
#c	array
#		Array whose indices are the names of available subcommands,
#		configuration options, etc.
#
# Return value:
#	Name extracted from the available set.
#
# Description:
#	composite:matchName accepts a name, and an array that contains the
#	possible values for the name.  It attempts to match the name to
#	one of the indices in the array.  If the name does not mach exactly,
#	but only one element of the array has the name as a prefix, that
#	array element is returned; this allows for the same unique-prefix
#	rules that Tk uses to match widget commands and command-line flags.

proc composite:matchName {name arrayname} {
	upvar 1 $arrayname array
	if [info exists array($name)] {
		return $name
	}
	set names [array names array]
	set index [lsearch $names $name*]
	if {$index >= 0} {
		set index2 [lsearch [lrange $names [expr $index+1] end] $name*]
		if {$index2 < 0} {
			return [lindex $names $index]
		}
	}
	error "$name not found in $arrayname"
}

# Procedure:	composite_isManaged
#
# Synopsis:
#	Determine whether a widget is a managed composite widget.
#
# Usage:
#c	composite_isManaged pathName
#
# Parameters:
#c	pathName
#		Path name of a widget.
#
# Return value:
#	1 if the widget is a managed composite widget.
#
#	0 otherwise.

proc composite_isManaged {w} {
	set class [winfo class $w]
	expr {[info commands ${class}_alias$w] == "${class}_alias$w"}
}

#
		 ####################################
		 # Base class for composite widgets #
		 ####################################

# Widget:	composite
#
# Synopsis:
#	Base class for composite widgets.
#
# Usage:
#	The `composite' widget may not be instantiated directly.  Other
#	widget classes inherit its behavior.
#
# Options:
#	Name:			borderWidth
#	Class:			BorderWidth
#	Command-line Switch:	-borderwidth, -bd
#	Default:		0
#		Width of the widget's 3-D border.
#
#	Name:			geometry
#	Class:			Geometry
#	Command-line Switch:	-geometry
#	Default:		{}
#		Requested geometry for the widget.
#
#	Name:			height
#	Class:			Height
#	Command-line Switch:	-height
#	Default:		0
#		Requested height of the widget.
#
#	Name:			relief
#	Class:			Relief
#	Command-line Switch:	-relief
#	Default:		flat
#		Requested relief (flat, raised, sunken, ridget, groove)
#		of the widget's 3-D border.
#
#	Name:			width
#	Class:			Width
#	Command-line Switch:	-width
#	Default:		0
#		Requested width of the widget.
#
# Widget commands:
#c	pathName configure
#
#c	pathName configure -flag
#
#c	pathName configure ?-flag value?...
#
#		The zero-argument form returns a list of the widget's
#		configuration parameters, exactly as with the Tk `configure'
#		widget command.  Similarly, the one argument form
#		returns a list describing a single configuration parameter,
#		and the multi-argument form configures the widget.

composite_define Composite {
	{-bd borderWidth}
	{-borderwidth borderWidth BorderWidth 0 composite_configLocal}
	{-geometry geometry Geometry {} composite_configLocal}
	{-height height Height 0 composite_configLocal}
	{-relief relief Relief flat composite_configLocal}
	{-width width Width 0 composite_configLocal}
} composite:initProc

composite_subcommand Composite configure {w args} {
	set class [winfo class $w]
	upvar #0 composite_config$w config
	upvar #0 ${class}_configResource resource
	upvar #0 ${class}_configResourceClass resourceClass
	upvar #0 ${class}_configDefault configDefault
	upvar #0 ${class}_configSwitch switch
	case [llength $args] in {
		0 {
			set list {}
			foreach f [lsort [array names resource]] {
				set r $resource($f)
				set item [list $f $r]
				if {$f == $switch($r)} {
					lappend item $resourceClass($r) \
						$configDefault($r) \
						$config($r)
				}
				lappend list $item
			}
			return $list
		}
		1 {
			set s [lindex $args 0]
			set r [composite:findConfigFlag $w $s]
			return [list $switch($r) \
				 $r \
				 $resourceClass($r) \
				$configDefault($r) \
				$config($r)]
		}
		default {
			eval [list composite:applyConfig $w] $args
		}
	}
}

# Procedure:	composite_initProc
#
# Synopsis:
#	Initialization procedure for widgets that may not be instantiated.
#
# Usage:
#c	composite_initProc pathName
#
# Parameters:
#c	pathName
#		Path name of the widget that is being instantiated
#		erroneously.
#
# Return value:
#	None -- the procedure always throws an error.

proc composite_initProc w {
	error "can't create a raw [winfo class $w]"
}

# Widget:	colors
#
# Synopsis:
#	Mix-in widget class for widgets whose subtrees inherit the
#	colors of the parent widget.
#
# Usage:
#	The `colors' widget may not be instantiated directly; it defines
#	behavior that is inherited by other widget classes.
#
# Options:
#	Name:			activeBackground
#	Class:			Foreground
#	Command-line Switch:	-activebackground
#	Default:		#eed5b7
#		Background color to be applied to the active widget.
#
#	Name:			activeForeground
#	Class:			Background
#	Command-line Switch:	-activeforeground
#	Default:		Black
#		Foreground color to be applied to the active widget.
#
#	Name:			background
#	Class:			Background
#	Command-line Switch:	-background, -bg
#	Default:		#ffe4c4
#		Background color.
#
#	Name:			cursorBackground
#	Class:			Foreground
#	Command-line Switch:	-cursorbackground
#	Default:		Black
#		Background color of the insertion cursor.  This option
#		is deprecated and being replaced with `-insertbackground'.
#
#	Name:			disabledForeground
#	Class:			DisabledForeground
#	Command-line Switch:	-disabledforeground
#	Default:		#b0b0b0
#		Foreground color of disabled `greyed-out' widgets.
#
#	Name:			foreground
#	Class:			Foreground
#	Command-line Switch:	-foreground, -fg
#	Default:		Black
#		Foreground color.
#
#	Name:			insertBackground
#	Class:			Foreground
#	Command-line Switch:	-insertbackground
#	Default:		Black
#		Background color of the insertion cursor.
#
#	Name:			selectBackground
#	Class:			Foreground
#	Command-line Switch:	-selectbackground
#	Default:		#b2dfee
#		Background color of selected items within the widget.
#
#	Name:			selectForeground
#	Class:			Background
#	Command-line Switch:	-selectforeground
#	Default:		Black
#		Foreground color of selected items within the widget.
#
# Widget commands:
#
#c	pathName configure
#		See the `composite' widget for this command.

composite_define Colors {
	{-activebackground activeBackground Foreground	#eed5b7}
	{-activeforeground activeForeground Background	Black}
	{-background	background	Background	#ffe4c4}
	{-bg		background}
	{-cursorbackground cursorBackground Foreground	Black}
	{-disabledforeground disabledForeground DisabledForeground #b0b0b0}
	{-fg		foreground}
	{-foreground	foreground	Foreground	Black}
	{-insertbackground insertBackground Foreground	Black}
	{-selectbackground selectBackground Foreground	#b2dfee}
	{-selectforeground selectForeground Background	Black}
} composite:initProc

# Widget:	lookandfeel
#
# Synopsis:
#	Mix-in widget class for widgets whose subtrees inherit certain
#	look-and-feel qualities of the parent widget.
#
# Usage:
#	The `lookandfeel' widget may not be instantiated directly; it defines
#	behavior that is inherited by other widget classes.
#
# Options:
#	Name:			cursorBorderWidth
#	Class:			BorderWidth
#	Command-line Switch:	-cursorborderwidth
#	Default:		1
#		Border width of the insertion cursor.  This option is
#		deprecated, and being replaced with `-insertborderwidth'.
#
#	Name:			cursorOffTime
#	Class:			OffTime
#	Command-line Switch:	-cursorofftime
#	Default:		300
#		Off time of the flashing insertion cursor, in milliseconds.
#		This option is deprecated, and being replaced with
#		`-insertofftime'.
#
#	Name:			cursorOnTime
#	Class:			OnTime
#	Command-line Switch:	-cursorontime
#	Default:		600
#		On time of the flashing insertion cursor, in milliseconds.
#		This option is deprecated, and being replaced with
#		`-insertontime'.
#
#	Name:			cursorWidth
#	Class:			CursorWidth
#	Command-line Switch:	-cursorwidth
#	Default:		2
#		Width of the insertion cursor.  This option is deprecated,
#		and being replaced with `-insertwidth'.
#
#	Name:			exportSelection
#	Class:			ExportSelection
#	Command-line Switch:	-exportselection
#	Default:		1
#		Boolean value specifying whether the widget's selection
#		should be exported to the X selection.
#
#	Name:			insertBorderWidth
#	Class:			BorderWidth
#	Command-line Switch:	-insertborderwidth
#	Default:		1
#		Border width of the insertion cursor.
#
#	Name:			insertOffTime
#	Class:			OffTime
#	Command-line Switch:	-insertofftime
#	Default:		300
#		Off time of the flashing insertion cursor, in milliseconds.
#
#	Name:			insertOnTime
#	Class:			OnTime
#	Command-line Switch:	-insertOnTime
#	Default:		600
#		On time of the flashing insertion cursor, in milliseconds.
#
#	Name:			insertWidth
#	Class:			InsertWidth
#	Command-line Switch:	-insertwidth
#	Default:		2
#		Width of the insertion cursor.
#
#	Name:			repeatdelay
#	Class:			RepeatDelay
#	Command-line Switch:	-repeatdelay
#	Default:		300
#		Time, in milliseconds, before a keypress begins repeating.
#
#	Name:			repeatinterval
#	Class:			RepeatInterval
#	Command-line Switch:	-repeatinterval
#	Default:		100
#		Time, in milliseconds, between successive repeats of a
#		repeating keypress.
#
#	Name:			selectBorderWidth
#	Class:			BorderWidth
#	Command-line Switch:	-selectborderwidth
#	Default:		1
#		Border width of the selected component of the widget.
#
# Widget commands:
#
#c	pathName configure
#		See the `composite' widget for this command.

composite_define LookAndFeel {
	{-cursorborderwidth cursorBorderWidth BorderWidth 1}
	{-cursorofftime cursorOffTime	OffTime		300}
	{-cursorontime	cursorOnTime	OnTime		600}
	{-cursorwidth	cursorWidth	CursorWidth	2}
	{-exportselection exportSelection ExportSelection 1}
	{-insertborderwidth insertBorderWidth BorderWidth 1}
	{-insertofftime	insertOffTime	OffTime		300}
	{-insertontime	insertOnTime	OnTime		600}
	{-insertwidth	insertWidth	InsertWidth	2}
	{-repeatdelay	repeatDelay	RepeatDelay	300}
	{-repeatinterval repeatInterval	RepeatInterval	100}
	{-selectborderwidth selectBorderWidth BorderWidth 1}
} composite:initProc
