/*
 * tcl_snmp.c
 *
 * Extend a Tcl-Interpreter about the ability to speak SNMP (Version 1
 * as well as Version 2).
 *
 * Copyright (c) 1994
 *
 * Sven Schmidt, J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <tcl.h>

#ifdef DBMALLOC
#include <dbmalloc.h>
#endif

#include "snmp.h"
#include "xmalloc.h"
#include "memory.h"
#include "udp.h"
#include "agent.h"
#include "misc.h"
#include "mib.h"

/*
 * global variables
 */

struct session *session_list = NULL;

int hexdump = 0;

/*
 * Forward declarations for procedures defined later in this file:
 */


static int
configured	_ANSI_ARGS_((Tcl_Interp *interp, struct session *session));

static void
show_party	_ANSI_ARGS_((Tcl_DString *dst, struct party *party, 
			     char *pfx));
static void
show_session	_ANSI_ARGS_((Tcl_Interp	*interp,struct session *sess));

static int
conf_party	_ANSI_ARGS_((Tcl_Interp *interp, struct party *party,
			     char *args));
static int
conf_auth	_ANSI_ARGS_((Tcl_Interp *interp, struct party *party,
			     char *args));
static int
conf_priv	_ANSI_ARGS_((Tcl_Interp *interp, struct party *party,
			     char *args));
static int
conf_session	_ANSI_ARGS_((Tcl_Interp *interp, struct session *sess,
			       int argc, char **argv));

static int	
mib_cmd		_ANSI_ARGS_((ClientData	clientData, Tcl_Interp *interp,
			     int argc, char **argv));
static int
snmp_cmd	_ANSI_ARGS_((ClientData	clientData, Tcl_Interp *interp,
			     int argc, char **argv));
static int
session_cmd	_ANSI_ARGS_((ClientData	clientData, Tcl_Interp *interp,
			     int argc, char **argv));

static void
destroy		_ANSI_ARGS_((ClientData clientdata));

static int
request		_ANSI_ARGS_((Tcl_Interp *interp, struct session *sess,
			     int pdu_type, int argc, char **argv));
static int
walk		_ANSI_ARGS_((Tcl_Interp *interp, struct session *sess,
			     int argc, char **argv));


/*
 * configured() takes a session and checks wether it's parameters for
 * the communication with the desired agent are set. If communication
 * is possible, the function returns 1. If any parameter is missing,
 * so that the communication will fail (e.g. no community string is
 * given), 0 is returned and the missing parameter is reported in
 * interp->result.
 */

static int
configured (interp, session)
     Tcl_Interp *interp;
     struct session *session;
{
    if (! session->version) {
        Tcl_AppendResult (interp, "session \"", session->name, 
			  "\" not configured", (char *) NULL);
	return TCL_ERROR;
    }

    if (session->version == SNMPv1) {
	if (! session->dstParty.TAddress) {
	    Tcl_AppendResult (interp, "session \"", session->name,
			      "\" has no agent address", (char *) NULL);
	    return TCL_ERROR;
	}
	if (! session->community) {
	    Tcl_AppendResult (interp, "session \"", session->name,
			      "\" has no community string", (char *) NULL);
	    return TCL_ERROR;
	}
    }

    if (session->version == SNMPv2) {
	if (! *session->dstParty.Identity) {
	    Tcl_AppendResult (interp, "session \"", session->name,
			      "\" has no destination party", (char *) NULL);
	    return TCL_ERROR;
	}
	if (! *session->srcParty.Identity) {
	    Tcl_AppendResult (interp, "session \"", session->name,
			      "\" has no source party", (char *) NULL);
	    return TCL_ERROR;
	}
	if (! *session->context.Identity) {
	    Tcl_AppendResult (interp, "session \"", session->name,
			      "\" has no context", (char *) NULL);
	    return TCL_ERROR;
	}
    }
    
    return TCL_OK;
}

/*
 * show_party() writes the configuration of the given party into
 * the dynamic string dst.
 */

static void
show_party (dst, party, pfx)
     Tcl_DString *dst;
     struct party *party;
     char *pfx;
{
    char buf[120];
    char *tmp = SNMP_Oid2Str (party->Identity, party->IdentityLen);

    Tcl_DStringAppendElement (dst, "-");
    Tcl_DStringAppend (dst, pfx, -1);
    Tcl_DStringAppend (dst, "party", 5);
    Tcl_DStringStartSublist (dst);
    Tcl_DStringAppendElement (dst, tmp);
    Tcl_DStringAppendElement (dst, party->TDomain);
    Tcl_DStringAppendElement (dst, party->TAddress);
    sprintf (buf, "%d", party->TPort);
    Tcl_DStringAppendElement (dst, buf);
    sprintf (buf, "%d", party->MaxMessageSize);
    Tcl_DStringAppendElement (dst, buf);
    Tcl_DStringEndSublist (dst);
    free (tmp);
    if (party->AuthProtocol != NO_AUTH) {
	Tcl_DStringAppendElement (dst, "-");
	Tcl_DStringAppend (dst, pfx, -1);
	Tcl_DStringAppend (dst, "partyauth", 9);
	Tcl_DStringStartSublist (dst);
	sprintf (buf, "%d", party->AuthClock);
	Tcl_DStringAppendElement (dst, buf);
	sprintf (buf, "%d", party->AuthLifetime);
	Tcl_DStringAppendElement (dst, buf);
	bin_to_hex (party->AuthPrivate, 16, buf);
	Tcl_DStringAppendElement (dst, buf);
	Tcl_DStringEndSublist (dst);
    }
}

/*
 * show_session() writes the configuration settings into the interp.
 */

static void
show_session (interp, sess)
     Tcl_Interp	*interp;
     struct session *sess;
{
    char buf[120];
    Tcl_DString	result;
    
    Tcl_DStringInit (&result);

    if (sess->version == SNMPv1) {
	if (sess->dstParty.TAddress != NULL) {
	    Tcl_DStringAppendElement (&result, "-address");
	    Tcl_DStringAppendElement (&result, sess->dstParty.TAddress);
	}
	Tcl_DStringAppendElement (&result, "-port");
	sprintf (buf, "%d", sess->dstParty.TPort);
	Tcl_DStringAppendElement (&result, buf);
	if (sess->community != NULL) {
	    Tcl_DStringAppendElement (&result, "-community");
	    Tcl_DStringAppendElement (&result, sess->community);
	}
    }

    if (sess->version == SNMPv2) {
	if (sess->dstParty.Identity != NULL) {
	    show_party (&result, &sess->dstParty, "dst");
	}
	if (sess->srcParty.Identity != NULL) {
	    show_party (&result, &sess->srcParty, "src");
	}
	if (sess->context.Identity != NULL) {
	    char *tmp = SNMP_Oid2Str (sess->context.Identity,
				      sess->context.IdentityLen);
	    Tcl_DStringAppendElement (&result, "-context");
	    Tcl_DStringAppendElement (&result, tmp);
	    free (tmp);
	}
    }

    Tcl_DStringAppendElement (&result, "-timeout");
    sprintf (buf, "%d", sess->timeout);
    Tcl_DStringAppendElement (&result, buf);

    Tcl_DStringAppendElement (&result, "-retries");
    sprintf (buf, "%d", sess->retries);
    Tcl_DStringAppendElement (&result, buf);
    
    Tcl_DStringResult (interp, &result);
    Tcl_DStringFree (&result);
}

/*
 * conf_party() configures a party given the parameters in args.
 */

static int
conf_party (interp, party, args)
     Tcl_Interp *interp;
     struct party *party;
     char *args;
{
    int i1;
    u_int i2;
    char buf1[128], buf2[128], buf3[128];
    
    party->AuthProtocol = NO_AUTH;
    party->PrivProtocol = NO_PRIV;

    if (! args || *args == '-') {
	Tcl_SetResult (interp, "party parameters missing", TCL_STATIC);
	return TCL_ERROR;
    }
    
    if (sscanf (args, "%s %s %s %d %d", buf1, buf2, buf3, &i2, &i1) != 5) {
	Tcl_SetResult (interp, "bogus party parameter format", TCL_STATIC);
	return TCL_ERROR;
    }
    
    free (party->Identity);
    free (party->TDomain);
    free (party->TAddress);

    party->Identity = SNMP_Str2Oid (buf1, &party->IdentityLen);
    party->Identity = SNMP_OidDup (&party->IdentityLen, party->Identity, 
				   party->IdentityLen);
    party->TDomain  = xstrdup (buf2);
    party->TAddress = xstrdup (buf3);
    party->TPort = i2;
    party->MaxMessageSize = i1;

    return TCL_OK;
}

/*
 * conf_auth() configures the authentication parameter of a party.
 */

static int
conf_auth (interp, party, args)
     Tcl_Interp *interp;
     struct party *party;
     char *args;
{
    char buf[128];
    u_int clock, lifetime;
    int len;

    if (! args || *args == '-') {
	Tcl_SetResult (interp, "party authentication parameters missing",
		       TCL_STATIC);
	return TCL_ERROR;
    }
    
    if (sscanf (args, "%d %d %s", &clock, &lifetime, buf) != 3) {
	Tcl_SetResult (interp, "bogus party authentication parameter format",
		       TCL_STATIC);
	return TCL_ERROR;
    }

    /* check and asssign AuthPrivate */

    if ((len = strlen (buf)) == 16) {
	memcpy (party->AuthPrivate, buf, 16);
    } else if (len == 3*16-1) {
	hex_to_bin (buf, party->AuthPrivate, &len);
    } else {
	Tcl_SetResult (interp, "party auth key should be 16 octets", 
		       TCL_STATIC);
	return TCL_ERROR;
    }

    party->AuthProtocol = MD5_AUTH;
    party->AuthClock = clock;
    party->AuthLifetime = lifetime;

    return TCL_OK;
}

/*
 * conf_priv() configures the privacy parameter of a party.
 */

static int
conf_priv (interp, party, args)
     Tcl_Interp *interp;
     struct party *party;
     char *args;
{
    int len;

    if (! args || *args == '-') {
	Tcl_SetResult (interp, "party privacy parameters missing",
                       TCL_STATIC);
        return TCL_ERROR;
    }

    /* check and asssign PrivPrivate */
    
    if ((len = strlen (args)) == 16) {
        memcpy (party->PrivPrivate, args, 16);
    } else if (len == 3*16-1) {
        hex_to_bin (args, party->PrivPrivate, &len);
    } else {
	Tcl_SetResult (interp, "party privacy key should be 16 octets",
		       TCL_STATIC);
	return TCL_ERROR;
    }

    party->PrivProtocol = DES_PRIV;    

    return TCL_OK;
}

/*
 * conf_session() configures a session handle.
 */

static int
conf_session (interp, sess, argc, argv)
     Tcl_Interp *interp;
     struct session *sess;
     int argc;
     char **argv;
{
    while (--argc && **++argv == '-') {
	
	/*
	 * SNMPv1: address
	 */
	
	if (!strncmp (*argv, "-address", strlen (*argv))) {
	    sess->version = SNMPv1;
	    if (--argc <= 0 || **++argv == '-') {
		Tcl_SetResult (interp, "missing IP address", TCL_STATIC);
		return TCL_ERROR;
	    }
	    if (sess->dstParty.TAddress) free (sess->dstParty.TAddress);
	    sess->dstParty.TAddress = xstrdup (*argv);
	    continue;
	}
	
	/*
	 * SNMPv1: port
	 */
	
	if (!strncmp (*argv, "-port", strlen (*argv))) {
	    int num;
	    sess->version = SNMPv1;
	    if (--argc <= 0 || **++argv == '-') {
		Tcl_SetResult (interp, "missing UDP port number", TCL_STATIC);
		return TCL_ERROR;
	    }
	    if (Tcl_GetInt (interp, *argv, &num) != TCL_OK) return TCL_ERROR;
	    if (num <= 0) {
		Tcl_SetResult (interp, "port number must be positive", 
			       TCL_STATIC);
		return TCL_ERROR;
	    }
	    sess->dstParty.TPort = num;
	    continue;
	}
	
	/*
	 * SNMPv1: community string
	 */
	
	if (!strncmp (*argv, "-community", strlen (*argv))) {
	    sess->version = SNMPv1;
	    if (--argc <= 0 || **++argv == '-') {
		Tcl_SetResult (interp, "missing community string", TCL_STATIC);
		return TCL_ERROR;
	    }
	    free (sess->community);
	    sess->community = xstrdup (*argv);
	    continue;
	}
	
	/*
         * SNMPv2: basic parameters for source party
	 */

	if (!strncmp (*argv, "-srcparty", strlen (*argv))) {
	    sess->version = SNMPv2;
	    argc--;
	    if (conf_party (interp, &sess->srcParty, *++argv) != TCL_OK) {
		return TCL_ERROR;
	    }
	    continue;
	}

	/*
         * SNMPv2: authentication protocol parameters for source party
	 *     ==> AuthClock, AuthLifetime, AuthPrivate, AuthPublic
	 */

	if (!strncmp (*argv, "-srcpartyauth", strlen (*argv))) {
	    argc--;
	    if (conf_auth (interp, &sess->srcParty, *++argv) != TCL_OK) {
                return TCL_ERROR;
            }
            continue;
        }
	
	/*
         * SNMPv2: privacy protocol parameters for source party
         *     ==> PrivPrivate, PrivPublic
	 */

	if (!strncmp (*argv, "-srcpartypriv", strlen (*argv))) {
	    argc--;
	    if (conf_priv (interp, &sess->srcParty, *++argv) != TCL_OK) {
                return TCL_ERROR;
            }
            continue;
        }
	
	/*
	 * SNMPv2: basic parameters for destination party
	 */
	
	if (!strncmp (*argv, "-dstparty", strlen (*argv))) {
	    sess->version = SNMPv2;
	    argc--;
 	    if (conf_party (interp, &sess->dstParty, *++argv) != TCL_OK) {
		return TCL_ERROR;
	    }
	    continue;
	}
	
	/*
         * SNMPv2: authentication protocol parameters for destination party
         *     ==> AuthClock, AuthLifetime, AuthPrivate, AuthPublic
	 */
	
	if (!strncmp (*argv, "-dstpartyauth", strlen (*argv))) {
	    argc--;
	    if (conf_auth (interp, &sess->dstParty, *++argv) != TCL_OK) {
                return TCL_ERROR;
            }
            continue;
        }

	/*
	 * SNMPv2: privacy protocol parameters for destination party
         *     ==> PrivPrivate, PrivPublic
	 */

	if (!strncmp (*argv, "-dstpartypriv", strlen (*argv))) {
	    argc--;
	    if (conf_priv (interp, &sess->dstParty, *++argv) != TCL_OK) {
                return TCL_ERROR;
            }
            continue;
        }
	
	/*
	 * SNMPv2: context for SNMP operations
	 */
	
	if (! strncmp (*argv, "-context", strlen (*argv))) {
	    if (--argc <= 0 || **++argv == '-') {
		interp->result = "context identifier missing";
		return TCL_ERROR;
	    }
	    free (sess->context.Identity);
	    sess->context.Identity = SNMP_Str2Oid (*argv,
						   &sess->context.IdentityLen);
	    sess->context.Identity = SNMP_OidDup (&sess->context.IdentityLen,
						  sess->context.Identity, 
						  sess->context.IdentityLen);
	    continue;
	}
	
	/*
	 * general: retries
	 */
	
	if (! strncmp (*argv, "-retries", strlen (*argv))) {
	    int num;
	    if (--argc <= 0 || **++argv == '-') {
		Tcl_SetResult (interp, "missing retry parameter", TCL_STATIC);
		return TCL_ERROR;
	    }
	    if (Tcl_GetInt (interp, *argv, &num) != TCL_OK) return TCL_ERROR;
	    sess->retries = num;
	    continue;
	}
	
	/*
	 * general: timeout
	 */
	
	if (! strncmp (*argv, "-timeout", strlen (*argv))) {
	    int num;
	    if (--argc <= 0 || **++argv == '-') {
		Tcl_SetResult (interp, "missing timeout parameter", 
			       TCL_STATIC);
		return TCL_ERROR;
	    }
	    if (Tcl_GetInt (interp, *argv, &num) != TCL_OK) return TCL_ERROR;
	    sess->timeout = num;
	    continue;
	}
	
	Tcl_AppendResult (interp, "unknown configuration option \"", *argv,
			  "\"", (char *) NULL);
	return TCL_ERROR;
    }
    
    /*
     * invalid config option given ?
     */
    
    if (argc) {
	Tcl_AppendResult (interp, "bad configuration option \"", *argv,
			  "\"", (char *) NULL);
	return TCL_ERROR;
    }
    
    /*
     * get the agents network address, if it's yet unknown
     */
    
    if (SNMP_Host2Addr (interp, sess->dstParty.TAddress, 
			sess->dstParty.TPort, &sess->addr) == 0) {
	interp->result = "couldn't get agent's network address";
	return TCL_ERROR;
    }
    
    /*
     * return the current settings as a result
     */
    
    show_session (interp, sess);
    return TCL_OK;
}


/*
 * SNMP_Init() initializes the SNMP extension. Note, it does not initialize
 * the `mib' command. It simply registers the new command.
 */

int
SNMP_Init (interp)
     Tcl_Interp *interp;
{
    SNMP_SysUpTime();
    memset ((char *) &snmp_stats, '\0', sizeof(struct snmp_stats));

    Tcl_CreateCommand (interp, "snmp", snmp_cmd, (ClientData *) NULL,
		       (Tcl_CmdDeleteProc *) NULL);

    Tcl_CreateCommand (interp, "mib", mib_cmd, (ClientData *) NULL, 
		       (Tcl_CmdDeleteProc *) NULL);

    srand(time(NULL)*getpid());

    return TCL_OK;
}

/*
 * mib_cmd() implements the mib tcl command. See the documentation
 * for details about its operation.
 */

static int
mib_cmd (clientData, interp, argc, argv)
     ClientData	clientData;
     Tcl_Interp	*interp;
     int	argc;
     char	**argv;
{
    int exact = 0;
    int len = 0;
    char *cmd, *name, *arg;
    char *result = NULL;

    if (argc > 1 && strcmp (argv[1], "-exact") == 0) {
	exact = 1;
	argc--;
	argv++;
    }

    cmd  = argv[1];
    name = argv[2];
    arg  = argv[3];

    len = cmd ? strlen (cmd) : 0;
    
    if (argc == 4) {
	if (strncmp (cmd, "format", len) == 0) {
	    result = MIB_Format (name, exact, arg);
	} else if (strncmp (cmd, "scan", len) == 0) {
	    result = MIB_Scan (name, exact, arg);
	}
	if (! result) goto notFound;
	Tcl_SetResult (interp, result, TCL_VOLATILE);
	return TCL_OK;
    }
    
    if (argc != 3) {
	Tcl_AppendResult (interp, "wrong # args: should be \"mib",
			  " ?-exact? option arg\"", (char *) NULL);
	return TCL_ERROR;
    }
    
    if (strncmp (cmd, "load", len) == 0) {
	return MIB_Load (interp, name);

    } else if (strncmp (cmd, "oid", len) == 0) {
	result = MIB_Oid (name, exact);

    } else if (strncmp (cmd, "name", len) == 0) {
	result = MIB_Name (name, exact);

    } else if (strncmp (cmd, "syntax", len) == 0) {
	result = MIB_Syntax (name, exact);

    } else if (strncmp (cmd, "description", len) == 0) {
	result = MIB_Description (name, exact);

    } else if (strncmp (cmd, "access", len) == 0) {
	result = MIB_Access (name, exact);

    } else if (strncmp (cmd, "successor", len) == 0) {
	result = MIB_Succ (name);

    } else {
	
	Tcl_AppendResult (interp, "bad option \"", cmd,
			  "\": should be name, oid, syntax, descr, access, ",
			  "succ, format, or scan", (char *) NULL);
	return TCL_ERROR;
    }

    if (! result) goto notFound;

    Tcl_SetResult (interp, result, TCL_VOLATILE);
    return TCL_OK;

  notFound:
    Tcl_AppendResult (interp, "no object \"", name, "\"",
		      (char *) NULL);
    return TCL_ERROR;
}

/*
 * snmp_cmd() is called from the TCL interpreter to evaluate the snmp
 * command.
 */

static int
snmp_cmd (clientData, interp, argc, argv)
     ClientData	clientData;
     Tcl_Interp	*interp;
     int argc;
     char **argv;
{
    static int id = 0;
    int len = argc > 1 ? strlen (argv[1]) : 0;
    struct session *last, *ss, *sessions;

    if (argc < 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", argv[0],
			  " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }
    
#if 0
    if (strncmp (argv[1], "dump", len) == 0) {
	agent_dump ();
	return TCL_OK;
    }
#endif

    /*
     * snmp subcommand "instance"
     */

    if (strncmp (argv[1], "instance", len) == 0) {
        if (argc < 4) {
	    Tcl_SetResult (interp, "snmp instance oid varname ?defval?",
			   TCL_STATIC);
	    return TCL_ERROR;
	}
	
	return SNMP_CreateInstance (interp, argv[2], argv[3],
				    (argc > 4) ? argv[4] : "");
    }

    /*
     * snmp subcommand "session"
     */

    if (! strncmp (argv[1], "session", len)) {
	
	/* create socket if neccessary (this installs the event handler) */
	
	if (SNMP_ManagerSocket (interp) != TCL_OK) return TCL_ERROR;

	/* allocate memory for session */
	
	ss = SNMP_MallocSession (&id);
	if (SNMP_Host2Addr (interp, ss->dstParty.TAddress, 
			    ss->dstParty.TPort, &ss->addr) == 0) {
	    SNMP_FreeSession (ss);
	    Tcl_SetResult (interp, "couldn't get agent's network address",
			   TCL_STATIC);
	    return TCL_ERROR;
	}
	
	/* configure the session */

	if (conf_session (interp, ss, argc - 1, argv + 1) != TCL_OK) {
	    SNMP_FreeSession (ss);
	    return TCL_ERROR;
	}

	/* link the session to the list of sessions */
	
	if (session_list == NULL) {
	    session_list = ss;
	} else {
	    last = session_list;
	    while (last->next != NULL) last = last->next;
	    last->next = ss;
	    ss->prev = last;
	}

	/* 
	 * call an arbitrary mib command to trigger the autoload
	 * mechanism -- ugly but correct in most cases 
	 */

	Tcl_Eval (interp, "mib oid iso");
	Tcl_ResetResult (interp);
	
	/* create Tcl command for this session */
	
	sprintf (interp->result, "s%d", id++);
	Tcl_CreateCommand (interp, interp->result, session_cmd,
			   (ClientData *) ss, destroy);
	return TCL_OK;
    }

    /*
     * snmp subcommand "info"
     */
    
    if (! strncmp (argv[1], "info", len)) {
	for (sessions = session_list;
	     sessions != NULL;
	     sessions = sessions->next) {
	    Tcl_AppendElement (interp, sessions->name);
	}
	return TCL_OK;
    }
    
    /*
     * snmp subcommand "watch"
     */
    
    if (! strncmp (argv[1], "watch", len)) {
	if (argc < 3) {
	    Tcl_SetResult (interp, hexdump ? "on" : "off", TCL_STATIC);
	    return TCL_OK;
	}
	return (Tcl_GetBoolean (interp, argv[2], &hexdump));
    }
    
    /*
     * invalid "snmp" subcommand
     */

    Tcl_AppendResult (interp, "bad option \"", argv[1], 
		      "\": should be session, watch, or info",
		      (char *) NULL);
    return TCL_ERROR;
}

/*
 * session_cmd() is called from the TCL interpreter to evaluate a
 * command on a specific session handle.
 */

static int
session_cmd (clientData, interp, argc, argv)
     ClientData	clientData;
     Tcl_Interp	*interp;
     int argc;
     char **argv;
{
    int len = argc > 1 ? strlen (argv[1]) : 0;
    struct session *session = (struct session *) clientData;

    if (argc < 2) {
	Tcl_AppendResult (interp, "wrong # args: should be \"",
			  argv[0], " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (strncmp (argv[1], "configure", len) == 0) {
	if (argc == 2) {
	    show_session (interp, session);
	    return TCL_OK;
	}
	return (conf_session (interp, session, argc - 1, argv + 1));
    } else if (strncmp (argv[1], "destroy", len) == 0) {
	Tcl_DeleteCommand (interp, argv[0]);
	return (TCL_OK);
    }

    /*
     * all commands below need a well configured session
     */

    if (configured (interp, session) != TCL_OK) return TCL_ERROR;

    if (! strcmp (argv[1], "get")) {
	return request (interp, session, GET_REQUEST, argc-1, argv+1);

    } else if (! strcmp (argv[1], "getnext")) {
	return request (interp, session, GET_NEXT_REQUEST, argc-1, argv+1);

    } else if (! strcmp (argv[1], "getbulk")) {
	return request (interp, session, GET_BULK_REQUEST, argc-1, argv+1);

    } else if (! strcmp (argv[1], "inform")) {
	return request (interp, session, INFORM_REQUEST, argc-1, argv+1);

    } else if (! strcmp (argv[1], "set")) {
	return request (interp, session, SET_REQUEST, argc-1, argv+1);
	
    } else if (! strcmp (argv[1], "walk")) {
	return walk (interp, session, argc-1, argv+1);

    } else if (! strcmp (argv[1], "trap")) {
	if (session->version == SNMPv1) {
	    return request (interp, session, SNMPv1_TRAP, argc-1, argv+1);
	} else if (session->version == SNMPv2) {
	    return request (interp, session, SNMPv2_TRAP, argc-1, argv+1);
	} else {
	    Tcl_SetResult (interp, "unknown SNMP version", TCL_STATIC);
            return TCL_ERROR;
        }

    } else if (! strcmp (argv[1], "trapsink")) {
	if (argc < 3) {
	    Tcl_SetResult (interp, "callback missing", TCL_STATIC);
	    return TCL_ERROR;
	}
	if (SNMP_TrapSocket (interp) != TCL_OK) return TCL_ERROR;
	if (session->traceCallBack) free (session->traceCallBack);
	session->traceCallBack = xstrdup (argv[2]);
	return TCL_OK;

    } else if (! strcmp (argv[1], "agent")) {
	if (argc == 3) {
	    if (Tcl_GetInt (interp, argv[2], &session->srcParty.TPort) 
		!= TCL_OK) return TCL_ERROR;
	} else {
	    session->srcParty.TPort = 1701;
	}
	SNMP_AgentInit (interp);
	return SNMP_AgentSocket (interp, session);
    }

    Tcl_AppendResult (interp, "bad option \"", argv[1], "\": should be ",
		      "configure, destroy, get, getnext, getbulk, ",
		      "set, walk, trap, trapsink, or agent", (char *) NULL);
    return TCL_ERROR;
}

/*
 * destroy() is invoked when a session handle is deleted. It
 * frees the associated session structure.
 */

static void
destroy (clientData)
     ClientData clientData;
{
    struct session *s;
    
    for (s = session_list; s != (struct session *) clientData; s = s->next) ;
    
    if (s->prev == NULL) {			/* first element in list */
	if (s->next == NULL) {			/* single element */	
	    session_list = NULL;
	} else {
	    session_list = s->next;
	    session_list->prev = NULL;
	}
    } else {
	if (s->next == NULL) {			/* last element in list */
	    s->prev->next = NULL;
	} else {
	    s->prev->next = s->next;		/* somewhere in the middle */
	    s->next->prev = s->prev;
	}
    }
    SNMP_FreeSession (s);

    /*
     * close the listening socket if it's the last session
     */

    if (session_list == NULL) {
	SNMP_ManagerClose();
    }
}


/*
 * request() creates a pdu structure an calls SNMP_Encode to send the
 * packet to the destination.
 */

static int
request (interp, sess, pdu_type, argc, argv)
     Tcl_Interp	*interp;
     struct session *sess;
     int pdu_type;
     int argc;
     char **argv;
{
    char *callback = NULL, *error_callback = NULL;
    struct snmp_pdu _pdu;
    struct snmp_pdu *pdu = &_pdu;
    
    /* initialize the PDU */

    pdu->type         = pdu_type;
    pdu->request_id   = ++sess->reqid;
    pdu->error_status = E_NOERROR;
    pdu->error_index  = 0;    
    pdu->trapOID      = NULL;
    Tcl_DStringInit (&pdu->varbind);

    /*
     * check # of arguments
     */
    
    if ((pdu->type == GET_BULK_REQUEST && argc < 4) 
	|| (pdu->type == SNMPv1_TRAP && argc < 3)
	|| (pdu->type == SNMPv2_TRAP && argc < 3) || (argc < 2)) {
	goto usage;
    }
    
    /*
     * read NonRepeaters and MaxRepetitions for GetBulkRequest
     */
    
    if (pdu->type == GET_BULK_REQUEST) {
	int num;
	if (--argc) {
	    if (Tcl_GetInt (interp, *++argv, &num) != TCL_OK) goto errorExit;
	    pdu->error_status = (num < 0) ? 0 : num;
	}
	if (--argc) {
	    if (Tcl_GetInt (interp, *++argv, &num) != TCL_OK) goto errorExit;
	    pdu->error_index  = (num < 0) ? 0 : num;
	}
    } else if (pdu->type == SNMPv1_TRAP || pdu->type == SNMPv2_TRAP) {
	argc--;
	if (SNMP_IsOid (*++argv)) {
	    pdu->trapOID = xstrdup(*argv);
	} else {
	    char *tmp = MIB_Oid (*argv, 0);
	    if (! tmp) {
		Tcl_AppendResult (interp,  "no object \"", *argv, "\"",
				  (char *) NULL);
		goto errorExit;	
	    }    
	    pdu->trapOID = xstrdup (tmp);
	}
    } else {
	pdu->error_status = E_NOERROR;
	pdu->error_index  = 0;
    }
    
    /*
     * check for availability of VB-LIST and split it into args
     */
    
    if (!argc) goto usage;    
    Tcl_DStringAppend (&pdu->varbind, *++argv, -1);

    /*
     * check for availabilty of callback functions
     */
    
    if (--argc && *++argv != NULL) {
	callback = *argv;
	if (--argc && *++argv != NULL) {
	    error_callback = *argv;
	}
    }

    if (SNMP_Encode (interp, sess, pdu, callback, error_callback) != TCL_OK) {
	goto errorExit;
    }

    if (pdu->trapOID) free (pdu->trapOID);
    Tcl_DStringFree (&pdu->varbind);
    return TCL_OK;
    
  usage:
    if (pdu->type == GET_BULK_REQUEST) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", sess->name,
			  " getbulk non-repeaters max-repetitions ",
			  "list ?callback? ?error-callback?\"", (char *) NULL);
    } else if (pdu->type == SNMPv1_TRAP || pdu->type == SNMPv2_TRAP) {
	Tcl_AppendResult (interp, "wrong # args: should be \"", sess->name,
			  " trap trapID list", (char *) NULL);
    } else {
	Tcl_AppendResult (interp, "wrong # args: should be \"", sess->name,
			  " ", *argv, " ",
			  "list ?callback? ?error-callback?\"", 
			  (char *) NULL);
    }
    
  errorExit:
    if (pdu->trapOID) free (pdu->trapOID);
    Tcl_DStringFree (&pdu->varbind);
    return TCL_ERROR;
}


/*
 * walk() walks a MIB tree. It evaluates the given tcl command foreach
 * varbind retrieved using getbulk requests. First, all variables
 * contained in the list argument are converted to their OIDs. Then we
 * loop using gebulk requests until we get an error or until one returned
 * variable starts with an OID not being a valid prefix.
 */

static int
walk (interp, sess, argc, argv)
     Tcl_Interp		*interp;
     struct session		*sess;
     int			argc;
     char			**argv;
{
    int i, j, k, result;
    int oidc, respc;
    char **oidv = NULL, **respv = NULL;

    struct snmp_pdu _pdu;
    struct snmp_pdu *pdu = &_pdu;
    
    if (argc != 4) {
	Tcl_AppendResult (interp, "wrong # args: should be \"",
			  sess->name, " walk varName list command\"",
			  (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * initialize the PDU 
     */

    pdu->type         = GET_BULK_REQUEST;
    pdu->request_id   = ++sess->reqid;
    pdu->error_status = E_NOERROR;
    pdu->error_index  = 0;    
    pdu->trapOID      = NULL;
    Tcl_DStringInit (&pdu->varbind);

    /*
     * save the oid prefix contained in list in oidv and oidc
     */
    
    result = Tcl_SplitList(interp, argv[2], &oidc, &oidv);
    if (result != TCL_OK) {
	return result;
    }
    
    for (i = 0; i < oidc; i++) {
	char *tmp = MIB_Oid (oidv[i], 0);
	if (!tmp) {
	    free ((char *) oidv);
	    Tcl_DStringFree (&pdu->varbind);
	    Tcl_AppendResult (interp,  "no object \"", oidv[i], "\"",
			      (char *) NULL);
            return TCL_ERROR;
	}
	oidv[i] = xmalloc (strlen (tmp) + 2);
	strcpy (oidv[i], tmp);
	strcat (oidv[i], ".");
	Tcl_DStringAppendElement (&pdu->varbind, tmp);
    }

    while (1) {

	pdu->error_status = 0;		/* non-repeaters */
	pdu->error_index  = 16 / oidc;	/* man-repetitions -- why this ?? */
	pdu->request_id   = ++sess->reqid;

	result = SNMP_Encode (interp, sess, pdu, NULL, NULL);
	if (result == TCL_ERROR 
	    && (strcmp (interp->result, "noSuchName") == 0)) {
	    result = TCL_OK;
	    goto loopDone;
	}
	if (result != TCL_OK) {
            break;
        }
	
	if (respv) free ((char *) respv);
	result = Tcl_SplitList (interp, interp->result, &respc, &respv);
	if (result != TCL_OK) {
	    goto loopDone;
	}

	if (respc % oidc) {
	    Tcl_SetResult (interp, 
			   "received response with wrong # of varbinds",
			   TCL_STATIC);
	    result = TCL_ERROR;
	    goto loopDone;
	}

	for (j = 0; j < respc / oidc; j++) {

	    for (i = 0; i < oidc; i++) {
		if (strncmp (oidv[i], respv[j * oidc + i], 
			     strlen(oidv[i])) != 0) {
		    result = TCL_OK;
		    goto loopDone;
		}
	    }

	    Tcl_DStringFree (&pdu->varbind);
	    for (k = j * oidc; k < (j+1) * oidc; k++) {
		Tcl_DStringAppendElement (&pdu->varbind, respv[k]);
	    }

	    if (Tcl_SetVar (interp, argv[1], Tcl_DStringValue(&pdu->varbind),
			    TCL_LEAVE_ERR_MSG) == NULL) {
		result = TCL_ERROR;
		goto loopDone;
	    }

	    result = Tcl_Eval (interp, argv[3]);
	    if (result != TCL_OK) {
		if (result == TCL_CONTINUE) {
		    result = TCL_OK;
		} else if (result == TCL_BREAK) {
		    result = TCL_OK;
		    goto loopDone;
		} else if (result == TCL_ERROR) {
		    char msg[100];
		    sprintf(msg, "\n    (\"%s walk\" body line %d)",
			    sess->name, interp->errorLine);
		    Tcl_AddErrorInfo(interp, msg);
		    goto loopDone;
		} else {
		    goto loopDone;
		}
	    }
	}
    }

  loopDone:
    for (i = 0; i < oidc; i++) {
	free (oidv[i]);
    }
    free ((char *) oidv);
    if (respv)	free ((char *) respv);
    Tcl_DStringFree (&pdu->varbind);
    if (result == TCL_OK) {
	Tcl_ResetResult(interp);
    }
    return result;
}
