/*
 * snmp.h
 *
 * Copyright (c) 1994
 *
 * Sven Schmidt, J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <sys/time.h>
#include <sys/types.h>
#include <netinet/in.h>

#include "oid.h"
#include "event.h"

#define NO_AUTH			0
#define MD5_AUTH		1

#define NO_PRIV			0
#define DES_PRIV		1

#define	SNMP_PORT		161

#define SNMPv1			101		/* for safety reasons */
#define SNMPv2			102		/* for safety reasons */

#define GET_REQUEST		0
#define GET_NEXT_REQUEST	1
#define RESPONSE		2
#define SNMPv1_TRAP		4
#define	SET_REQUEST		3
#define GET_BULK_REQUEST	5
#define INFORM_REQUEST		6
#define SNMPv2_TRAP		7

#define DEFAULT_RETRIES		3
#define DEFAULT_TIMEOUT		5

/*
 * Error status definitions as of RFC 1448 and RFC 1157. Note, that
 * RFC 1157 only defines the errors NOERROR - GENERR. Some of the
 * version 1 error codes are only for V1/V2 proxy compatibility.
 */

#define E_NOERROR		0
#define E_TOOBIG		1
#define E_NOSUCHNAME		2	/* for proxy compatibility */
#define E_BADVALUE		3	/* for proxy compatibility */
#define E_READONLY		4	/* for proxy compatibility */
#define E_GENERR		5
#define E_NOACCESS		6
#define E_WRONGTYPE		7
#define E_WRONGLENGTH		8
#define E_WRONGENCODING		9
#define E_WRONGVALUE		10
#define E_NOCREATION		11
#define E_INCONSISTENTVALUE	12
#define E_RESOURCEUNAVAILABLE	13
#define E_COMMITFAILED		14
#define E_UNDOFAILED		15
#define E_AUTHORIZATIONERROR	16
#define E_NOTWRITABLE		17
#define E_INCONSISTENTNAME	18

/*
 * Party based on RFC 1445 and RFC 1446.
 */

struct party {
   u_char		*Name;
   oid			*Identity;
   int			IdentityLen;
   char			*TDomain;
   char			*TAddress;
   int			TPort;
   int			MaxMessageSize;
   int			AuthProtocol;
   u_int		AuthClock;
   u_char		AuthPrivate[17];
   u_char		AuthPublic[33];
   u_int		AuthLifetime;
   int			PrivProtocol;
   u_char		PrivPrivate[17];
   u_char		PrivPublic[33];
};

/*
 * Context based on RFC 1445
 */

struct context {
   oid			*Identity;
   int			IdentityLen;
};

/*
 * A session structure contains all infomation needed for agent < - >
 * manager communication.  
 */

struct session {
   char			name[6];        /* session name (really ?)          */
   struct sockaddr_in	addr;		/* the agents communication endpoint*/
   int			version;	/* SNMP version 1 or 2		    */
   char			*community;	/* community string (SNMPv1)	    */
   struct party		dstParty;	/* party that agent acts at	    */
   struct party		srcParty;	/* party that manager acts at	    */
   struct context	context;	/* chosen context for session	    */
   int			retries;	/* Number of retries		    */
   u_int		timeout;	/* Length to wait for timeout	    */
   u_int		reqid;		/* next request to use		    */
   char			*traceCallBack;	/* command to eval on a trap msg    */
   struct request_list	*request_list; 	/* outstanding requests		    */
   int			agentSocket;	/* socket used if we are an agent   */
   struct session	*prev;		/* pointer to previous session	    */
   struct session	*next;		/* pointer to next session	    */
};

/*
 * global variables
 */

extern struct session *session_list;

/*
 * structure for the SNMP PDU format as defined in RFC 1445
 */

struct snmp_pdu {
   int			type;
   int			request_id;
   int			error_status;
   int			error_index;
   char			*trapOID;
   Tcl_DString		varbind;
};

/*
 * The timeout structure is passed to the event loop.
 */

struct request_timeout {
    Tcl_Interp *interp;
    int	request_id;
};

/*
 * The timeout structure is passed to the event loop.
 */

struct agentToken {
    Tcl_Interp *interp;
    struct session *session;
};

/*
 * Structures required for management of asynchronous requests.
 */

struct request_list {
   int			request_id;	/* Request id			    */
   int			retr_nr;        /* Number of retransmissions        */
   u_char		*packet;
   int			packetlen;
   Tk_TimerToken	timeout_token;	/* Token for Timer Handler	    */
   struct request_timeout *timeo_data;	/* client data for Timer Handler    */
   char			*callback_func;	/* function to proc. response	    */
   char			*error_func;    /* function to handle errors	    */
   struct request_list	*prev;		/* preivious pending request	    */
   struct request_list	*next;		/* next pending request		    */
};

/*
 * Encode a SNMP request and do all required actions.
 */

int
SNMP_Encode		_ANSI_ARGS_((Tcl_Interp *interp, struct session	*sess,
				     struct snmp_pdu *pdu, char *callback,
				     char *error_callback));

/*
 * Decode a received SNMP packet and do all required actions.
 */

int
SNMP_Decode		_ANSI_ARGS_((Tcl_Interp *interp, int *reqid,
				     u_char *packet, int packetlen));

/*
 * Evaluate a callback function.
 */

int
SNMP_EvalCallback	_ANSI_ARGS_((Tcl_Interp *interp,
				     struct session *session,
				     int request_id,
				     int error_status,
				     int error_index,
				     char *callback,
				     Tcl_DString *varbind));


/*
 * Structures to keep snmp related statistics.
 */

struct snmp_stats {
    /* RFC 1213 */
    unsigned int snmpInPkts;
    unsigned int snmpOutPkts;
    unsigned int snmpInBadVersions;
    unsigned int snmpInBadCommunityNames;
    unsigned int snmpInBadCommunityUses;
    unsigned int snmpInASNParseErrs;
    unsigned int snmpInTooBigs;
    unsigned int snmpInNoSuchNames;
    unsigned int snmpInBadValues;
    unsigned int snmpInReadOnlys;
    unsigned int snmpInGenErrs;
    unsigned int snmpInTotalReqVars;
    unsigned int snmpInTotalSetVars;
    unsigned int snmpInGetRequests;
    unsigned int snmpInGetNexts;
    unsigned int snmpInSetRequests;
    unsigned int snmpInGetResponses;
    unsigned int snmpInTraps;
    unsigned int snmpOutTooBigs;
    unsigned int snmpOutNoSuchNames;
    unsigned int snmpOutBadValues;
    unsigned int snmpOutGenErrs;
    unsigned int snmpOutGetRequests;
    unsigned int snmpOutGetNexts;
    unsigned int snmpOutSetRequests;
    unsigned int snmpOutGetResponses;
    unsigned int snmpOutTraps;
    unsigned int snmpEnableAuthenTraps;
    /* RFC 1450 XXX not yet implemented */
    unsigned int snmpStatsPackets;
    unsigned int snmpStats30Something;
    unsigned int snmpStatsUnknownDstParties;
    unsigned int snmpStatsDstPartyMismatches;
    unsigned int snmpStatsUnknownSrcParties;
    unsigned int snmpStatsBadAuths;
    unsigned int snmpStatsNotInLifetimes;
    unsigned int snmpStatsWrongDigestValues;
    unsigned int snmpStatsUnknownContexts;
    unsigned int snmpStatsBadOperations;
    unsigned int snmpStatsSilentDrops;
    /* snmpV1BadCommunityNames is the same as snmpInBadCommunityNames */
    /* snmpV1BadCommunityUses  is the same as snmpInBadCommunityUses  */
};

extern struct snmp_stats snmp_stats;

/*
 * And finally the function that initializes the snmp tcl extension.
 */

int
SNMP_Init		_ANSI_ARGS_((Tcl_Interp *interp));
