/*
 * response.c
 *
 * This file contains all functions that decode a received SNMP packet
 * and do the appropriate actions. 
 *
 * Copyright (c) 1994
 *
 * Sven Schmidt, J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <tcl.h>

#ifdef DBMALLOC
# include <dbmalloc.h>
#endif

#include "snmp.h"
#include "asn1.h"
#include "xmalloc.h"
#include "memory.h"
#include "udp.h"
#include "misc.h"
#include "mib.h"
#include "agent.h"

/*
 * local variables (see rfc 1448)
 */

static char *snmp_error[] = {
    "noError",
    "tooBig",
    "noSuchName",
    "badValue",
    "readOnly",
    "genErr",
    "noAccess",
    "wrongType",
    "wrongLength",
    "wrongEncoding",
    "wrongValue",
    "noCreation",
    "inconsistentValue",
    "resourceUnavailable",
    "commitFailed",
    "undoFailed",
    "authorizationError",
    "notWritable",
    "inconsistentName"
};

/*
 * A damned global variable that stores the agent address of an SNMPv1
 * trap PDU. This is bad code, but I really have no idea where to put 
 * it in cleanly. So we make this silly variable :-(
 */

struct snmp_header {
   char	*community;
   oid	srcParty[OID_MAXLEN];
   int  srcPartyLen;
   oid	dstParty[OID_MAXLEN];
   int  dstPartyLen;
   oid	context[OID_MAXLEN];
   int  contextLen;
};

static char trapAgentAddress[20];

/*
 * Forward declarations for procedures defined later in this file:
 */

static int
decodeV1Message	_ANSI_ARGS_((Tcl_Interp	*interp, 
			     struct snmp_header *msg, struct snmp_pdu *pdu,
			     u_char *packet, int packetlen));

static int
decodeV2Message	_ANSI_ARGS_((Tcl_Interp *interp,
			     struct snmp_header *msg, struct snmp_pdu *pdu,
			     u_char *packet, int packetlen));

static int
decodePDU	_ANSI_ARGS_((Tcl_Interp *interp, struct snmp_pdu *pdu,
			     u_char *packet, int *packetlen));


/*
 * SNMP_Decode() decode a complete SNMP packet and does all required
 * actions (mostly executing callbacks or doing gets/sets on the
 * agent module).
 */

int
SNMP_Decode (interp, reqid, packet, packetlen)
     Tcl_Interp	*interp;
     int	*reqid;
     u_char	*packet;
     int	packetlen;
{
    int	version = 0;

    struct request_list	*request = NULL;
    struct session *session = NULL;

    struct snmp_pdu _pdu;
    struct snmp_pdu *pdu = &_pdu;
    struct snmp_header msg;

    *reqid = 0;
    msg.srcPartyLen = 0;
    msg.dstPartyLen = 0;
    msg.contextLen = 0;
    Tcl_DStringInit (&pdu->varbind);

    /*
     * read the version from the received packet.
     */

    if (*packet == (UNIVERSAL | CONSTRUCTED | ASN1_SEQUENCE)) {
	version = SNMPv1;
    } else if (*packet == (CONTEXT_SPECIFIC | CONSTRUCTED | TAG_1)) {
	version = SNMPv2;
    } else {
	snmp_stats.snmpInBadVersions++;
	Tcl_SetResult (interp, "received packet with unknown SNMP version",
		       TCL_STATIC);
	return TCL_CONTINUE;
    }

    /*
     * build response message structure for the packet and decode message
     */

    if (version == SNMPv1) {
	if (decodeV1Message (interp, &msg, pdu, packet, packetlen) 
	    == TCL_ERROR) {
	    return TCL_CONTINUE;
	}
    } else {
	if (decodeV2Message (interp, &msg, pdu, packet, packetlen) 
	    == TCL_ERROR) {
	    return TCL_CONTINUE;
	}
    }

    *reqid = pdu->request_id;

    /*
     * Sort out the pdu type and take appropriate action
     */

    if (pdu->type == SNMPv1_TRAP) {
	int ok = 0;
	for (session = session_list; session; session = session->next) {
	    if (session->version != SNMPv1) continue;
	    if (session->traceCallBack && msg.community
		&& (strcmp (session->community, msg.community) == 0)) {
		/* put the trapAgentAddress into the dstParty structure */
		char *tmp = session->dstParty.TAddress;
		session->dstParty.TAddress = trapAgentAddress;
		SNMP_EvalCallback (interp, session, 0, 0, 0, 
				   session->traceCallBack, &pdu->varbind);
		session->dstParty.TAddress = tmp;
		ok++;
	    }
	}
	if (!ok) { 
	    snmp_stats.snmpInBadCommunityNames++;
	} else {
	    snmp_stats.snmpInTraps++;
	}
	if (msg.community) free (msg.community);
        return TCL_OK;
    }

    if (pdu->type == SNMPv2_TRAP) {
	for (session = session_list; session; session = session->next) {
	    if (session->version != SNMPv2) continue;
	    if (session->traceCallBack) {
		SNMP_EvalCallback (interp, session, 0, 0, 0, 
				   session->traceCallBack, &pdu->varbind);
	    }
	}
	if (msg.community) free (msg.community);
        return TCL_OK;
    }

    if (pdu->type != RESPONSE) {
	if (version == SNMPv1) {
	    int ok = 0;
	    for (session = session_list; session; session = session->next) {
		if (session->version != SNMPv1) continue;
		if (msg.community
		    && (strcmp (session->community, msg.community) == 0)) {
		    SNMP_AgentRequest (interp, session, pdu);
		    ok++;
		}
	    }
	    if (!ok) snmp_stats.snmpInBadCommunityNames++;
	    if (msg.community) free (msg.community);
	    return TCL_OK;
	} else {
	    for (session = session_list; session; session = session->next) {
                if (session->version != SNMPv2) continue;
		if (msg.contextLen == session->context.IdentityLen
		    && msg.dstPartyLen == session->srcParty.IdentityLen
		    && msg.srcPartyLen == session->dstParty.IdentityLen) {
		    SNMP_AgentRequest (interp, session, pdu);
		}
	    }
	    if (msg.community) free (msg.community);
	    return TCL_CONTINUE;
	}
    }

    snmp_stats.snmpInGetResponses++;
    
    /*
     * lookup the request for this response -- return the result if we
     * can not find an async request (assuming synchronous request)
     */
    
    request = SNMP_LookupRequest (pdu->request_id, &session);
    if (request == NULL) {
	if (pdu->error_status) {
	    char buf[20];
	    Tcl_ResetResult (interp);
	    Tcl_AppendResult (interp, snmp_error[pdu->error_status], " ",
			      (char *) NULL);
	    sprintf (buf, "%d ", pdu->error_index);
	    Tcl_AppendResult (interp, buf, Tcl_DStringValue (&pdu->varbind),
			      (char *) NULL);
	    return TCL_ERROR;
	}
	if (msg.community) free (msg.community);
	Tcl_ResetResult (interp);
	Tcl_DStringResult (interp, &pdu->varbind);
	return TCL_OK;
    }

    /*
     * Check if the response packet is a good one. Currently only
     * version SNMPv1 community.
     */

    if ((version == SNMPv1) 
	&& ((msg.community == NULL) 
	    || (strcmp (session->community, msg.community) != 0))) {
	Tcl_SetResult (interp, "authentication failure", TCL_STATIC);
	if (msg.community) free (msg.community);
	return TCL_CONTINUE;
    }

    /*
     * if "error_status" is not 0, evaluate the error callback,
     * otherwise evalute the normal callback.
     */

    if (pdu->error_status && request->error_func)  {
	SNMP_EvalCallback (interp, session, pdu->request_id,
			   pdu->error_status, pdu->error_index,
			   request->error_func, &pdu->varbind);
    } else {
	if (request->callback_func) {
	    SNMP_EvalCallback (interp, session, pdu->request_id,
			       pdu->error_status, pdu->error_index,
			       request->callback_func, &pdu->varbind);
	}
    }

    /*
     * free response message structure and delete request from
     * request_list (this free's allocated memory)
     */
    
    SNMP_DeleteRequest (session, request);
    if (msg.community) free (msg.community);
    return TCL_OK;
}

/*
 * decodeV1Message() takes a serialized packet and decodes the SNMPv1
 * message header.
 */

static int
decodeV1Message (interp, msg, pdu, packet, packetlen)
     Tcl_Interp *interp;
     struct snmp_header *msg;
     struct snmp_pdu *pdu;
     u_char *packet;
     int packetlen;
{
    int version;
    int buflen = 0;
    u_int msglen = 0;
    int len = 0;
    
    /*
     * decode "Packet Header" header ( SEQUENCE 0x30 msglen )
     */
    
    if (*packet++ != (UNIVERSAL | CONSTRUCTED | ASN1_SEQUENCE)) {
	sprintf (interp->result,
		 "Message header: invalid value 0x%.2x; expecting 0x%.2x",
		 *--packet, (UNIVERSAL | CONSTRUCTED | ASN1_SEQUENCE));
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }
    buflen += 1;
    
    packet = ASN1_DecodeLength (packet, &buflen, &msglen);
    if (packet == NULL) goto asn1Error;
    
    if ((buflen + msglen) != packetlen) {
	interp->result = "invalid length field in message header";
	return TCL_ERROR;
    }
    
    buflen = 0;		/* buflen from here must be the same as msglen */

   /*
    * decode Version field of this message ( must be 0 )
    */
    
    packet = ASN1_DecodeInt (packet, &buflen, ASN1_INTEGER, &version);
    if (packet == NULL) goto asn1Error;
    
    /*
     * decode "community" string
     */

    if (*packet != ASN1_OCTET_STRING) {
        sprintf (interp->result,
		 "Community string: invalid value 0x%.2x; expecting 0x%.2x",
		 *packet, ASN1_OCTET_STRING);
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }
    packet = ASN1_DecodeOctetString (packet, &buflen, ASN1_OCTET_STRING,
				     (u_char **) &msg->community, &len);
    if (packet == NULL) goto asn1Error;

    /*
     * decode PDU and validate total message length
     */

    if (decodePDU (interp, pdu, packet, &buflen) != TCL_OK) {
	return TCL_ERROR;
    }

    if (buflen != msglen) {
	sprintf (interp->result,
		 "Message sequence length (%d) differs from real length (%d).",
		 buflen, (int) msglen);
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }

    return TCL_OK;

  asn1Error:
    Tcl_SetResult (interp, ASN1_ErrorString(), TCL_STATIC);
    snmp_stats.snmpInASNParseErrs++;
    return TCL_ERROR;
}

/*
 * decodeV2Message() takes a serialized packet and decodes the SNMPv2
 * message header. This is more complicated since we must handle the
 * authentication information.
 */

static int
decodeV2Message (interp, msg, pdu, packet, packetlen)
     Tcl_Interp *interp;
     struct snmp_header *msg;
     struct snmp_pdu *pdu;
     u_char *packet;
     int packetlen;
{
    int oidlen = 0, buflen = 0;
    
    u_int asnlen = 0, msglen = 0;
    
    oid	oidbuf[OID_MAXLEN];
    
    /*
     * decode tag & length field of this message
     */
    
    if (*packet++ != (CONTEXT_SPECIFIC | CONSTRUCTED | TAG_1)) {
	sprintf (interp->result,
		 "SnmpPrivMsg: invalid tag 0x%.2x; expecting 0x%.2x",
		 *--packet, (CONTEXT_SPECIFIC | CONSTRUCTED | TAG_1));
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }
    buflen += 1;
    
    packet = ASN1_DecodeLength (packet, &buflen, &msglen);
    if (packet == NULL) goto asn1Error;
    
    if ((buflen + msglen) != packetlen) {
	interp->result = "invalid length field in message header";
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }
    
    buflen = 0;		/* buflen from here must be the same as msglen */

    /*
     * decode "privDst" field and set buflen.
     */
    
    packet = ASN1_DecodeOID (packet, &buflen, oidbuf, &oidlen);
    if (packet == NULL) goto asn1Error;
    
    /*
     * decode "privData" header
     */
    
    if (*packet++ != (CONTEXT_SPECIFIC | PRIMITIVE | TAG_1)) {
	sprintf (interp->result,
		 "privData: invalid tag 0x%.2x; expecting 0x%.2x",
		 *--packet, (CONTEXT_SPECIFIC | PRIMITIVE | TAG_1));
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }
    buflen += 1;

    packet = ASN1_DecodeLength (packet, &buflen, &asnlen);
    if (packet == NULL) goto asn1Error;

    if ((asnlen + buflen) != msglen) {
	interp->result = "privData: invalid length field";
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }

    /*
     * decode "SnmpAuthMsg" header
     */

    if (*packet++ != (CONTEXT_SPECIFIC | CONSTRUCTED | TAG_1)) {
	sprintf (interp->result,
		 "SnmpAuthMsg: invalid tag 0x%.2x; expecting 0x%.2x",
		 *--packet, (CONTEXT_SPECIFIC | CONSTRUCTED | TAG_1));
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }
    buflen += 1;
    
    packet = ASN1_DecodeLength (packet, &buflen, &asnlen);
    if (packet == NULL) goto asn1Error;

    if ((asnlen + buflen) != msglen) {
	interp->result = "SnmpAuthMsg: invalid length field";
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }

    /*
     * decode "authInformation" field
     */
    
    if (*packet == ASN1_OCTET_STRING) {
	packet = ASN1_DecodeOctetString (packet, &buflen, ASN1_OCTET_STRING, 
					 NULL, 0);
	if (packet == NULL) goto asn1Error;
    }

    /*
     * decode "SnmpMgmtCom" header
     */

    pdu->type = *packet++;

    if (pdu->type != (CONTEXT_SPECIFIC | CONSTRUCTED | RESPONSE)) {
	sprintf (interp->result,
		 "SnmpMgmtCom: invalid tag 0x%.2x; expecting 0x%.2x",
		 pdu->type,
		 (CONTEXT_SPECIFIC | CONSTRUCTED | RESPONSE));
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }
    buflen += 1;
    
    packet = ASN1_DecodeLength (packet, &buflen, &asnlen);
    if (packet == NULL) goto asn1Error;
    
    if ((asnlen + buflen) != msglen) {
        interp->result = "SnmpMgmtCom: invalid length field";
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }
    
    /*
     * decode "dstParty", "srcParty" and "context" field
     */    
    
    packet = ASN1_DecodeOID (packet, &buflen, 
			     msg->dstParty, &msg->dstPartyLen);
    if (packet == NULL) goto asn1Error;
    
    packet = ASN1_DecodeOID (packet, &buflen, 
			     msg->srcParty, &msg->srcPartyLen);
    if (packet == NULL) goto asn1Error;

    packet = ASN1_DecodeOID (packet, &buflen, 
			     msg->context, &msg->contextLen);
    if (packet == NULL) goto asn1Error;
    
    /*
     * decode PDU and validate total message length
     */

    if (decodePDU (interp, pdu, packet, &buflen) != TCL_OK) {
	return TCL_ERROR;
    }

    if (buflen != msglen) {
	sprintf (interp->result,
		 "Message sequence length (%d) differs from real length (%d).",
		 buflen, (int) msglen);
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }
    
    return TCL_OK;

  asn1Error:
    Tcl_SetResult (interp, ASN1_ErrorString(), TCL_STATIC);
    snmp_stats.snmpInASNParseErrs++;
    return TCL_ERROR;
}


/*
 * decodePDU() takes a serialized packet and decodes the PDU. The result
 * is written to the pdu structure and varbind list is converted to a
 * TCL list contained in pdu->varbind.
 */

static int
decodePDU (interp, pdu, packet, packetlen)
     Tcl_Interp		*interp;
     struct snmp_pdu	*pdu;
     u_char		*packet;
     int		*packetlen;
{
    int		oidlen	= 0,		/* */
		pdulen	= 0;		/* # of bytes read parsing the PDU  */
    
    u_int	asnlen	= 0,		/* asn sequence length		    */
		deflen	= 0;		/* pdu total length iwithout header */
    
    oid		oidbuf[OID_MAXLEN];

    int		int_val;

    char buf[20];
    char *freeme;
    char *vboid = NULL;

    Tcl_DStringInit (&pdu->varbind);

    /*
     * first decode PDU tag -- let's see what we have to deal with
     */
    
    switch (*packet++) {
      case CONTEXT_SPECIFIC | CONSTRUCTED | RESPONSE:
	pdu->type = RESPONSE;
	break;
      case CONTEXT_SPECIFIC | CONSTRUCTED | GET_REQUEST:
	pdu->type = GET_REQUEST;
	break;
      case CONTEXT_SPECIFIC | CONSTRUCTED | GET_NEXT_REQUEST:
	pdu->type = GET_NEXT_REQUEST;
	break;
      case CONTEXT_SPECIFIC | CONSTRUCTED | GET_BULK_REQUEST:
	pdu->type = GET_BULK_REQUEST;
	break;
      case CONTEXT_SPECIFIC | CONSTRUCTED | SET_REQUEST:
	pdu->type = SET_REQUEST;
	break;
      case CONTEXT_SPECIFIC | CONSTRUCTED | INFORM_REQUEST:
	pdu->type = INFORM_REQUEST;
	break;
      case CONTEXT_SPECIFIC | CONSTRUCTED | SNMPv1_TRAP:
	pdu->type = SNMPv1_TRAP;
	break;
      case CONTEXT_SPECIFIC | CONSTRUCTED | SNMPv2_TRAP:
	pdu->type = SNMPv2_TRAP;
	break;
      default:
	sprintf (interp->result,
		 "Response-PDU: invalid tag 0x%.2x.", *--packet);
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }
    pdulen = 1;

    packet = ASN1_DecodeLength (packet, &pdulen, &deflen);
    if (packet == NULL) goto asn1Error;
    
    *packetlen += pdulen;
    pdulen = 0;

    if (pdu->type == SNMPv1_TRAP) {

	int generic, specific;
	char *toid = NULL;

	/*
	 * V1 trap PDUs require special attention - sad story
	 */

	packet = ASN1_DecodeOID (packet, &pdulen, oidbuf, &oidlen);
        if (packet == NULL) goto asn1Error;

	packet = ASN1_DecodeOctetString (packet, &pdulen, ASN1_IpAddress, 
					 (u_char **) &freeme, &int_val);
	if (packet == NULL) goto asn1Error;
	Tcl_DStringAppend (&pdu->varbind, " ", 1);
	{    unsigned int addr = *(int *) freeme;
	     sprintf (trapAgentAddress, "%u.%u.%u.%u",
		      (addr >> 24) & 0xff, (addr >> 16) & 0xff,
		      (addr >> 8) & 0xff, addr & 0xff);
	}
	free (freeme);

	packet = ASN1_DecodeInt (packet, &pdulen, ASN1_INTEGER, &generic);
        if (packet == NULL) goto asn1Error;

	packet = ASN1_DecodeInt (packet, &pdulen, ASN1_INTEGER, &specific);
        if (packet == NULL) goto asn1Error;

	/* ignore error here to accept bogus trap messages */

	packet = ASN1_DecodeInt (packet, &pdulen, ASN1_TimeTicks, &int_val);
	if (packet == NULL) goto asn1Error;
	{   int d, h, m, s, f;
	    d = int_val;
	    f = d % 100; d = d /100;
	    s = d % 60;  d = d / 60;
	    m = d % 60;  d = d / 60;
	    h = d % 24;  d = d / 24;
	    sprintf (buf, "%dd %d:%02d:%02d.%02d", d, h, m, s, f);
	    Tcl_DStringStartSublist (&pdu->varbind);
	    Tcl_DStringAppendElement (&pdu->varbind, "1.3.6.1.2.1.1.3.0");
	    Tcl_DStringAppendElement (&pdu->varbind, "TimeTicks");
	    Tcl_DStringAppendElement (&pdu->varbind, buf);
	    Tcl_DStringEndSublist (&pdu->varbind);
	}

	switch (generic) {
	  case 0:				/* coldStart*/
	    toid = "1.3.6.1.6.3.1.1.5.1";
	    break;
	  case 1:				/* warmStart */
	    toid = "1.3.6.1.6.3.1.1.5.2";
	    break;
	  case 2:				/* linkDown */
	    toid = "1.3.6.1.6.3.1.1.5.3";
	    break;
	  case 3:				/* linkUp */
	    toid = "1.3.6.1.6.3.1.1.5.4";
	    break;
	  case 4:				/* authenticationFailure */
	    toid = "1.3.6.1.6.3.1.1.5.5";
	    break;
	  case 5:				/* egpNeighborLoss */
	    toid = "1.3.6.1.6.3.1.1.5.6";
	    break;
	  default:
	    oidbuf[oidlen++] = 0;
	    oidbuf[oidlen++] = specific;
	    toid = SNMP_Oid2Str (oidbuf, oidlen); /* enterpriseSpecific */
	    break;
	}

	Tcl_DStringStartSublist (&pdu->varbind);
	Tcl_DStringAppendElement (&pdu->varbind, "1.3.6.1.6.3.1.1.4.1.0");
	Tcl_DStringAppendElement (&pdu->varbind, "OBJECT IDENTIFIER");
	{
	    char *tmp = MIB_Name (toid, 0);
	    if (tmp) {
		Tcl_DStringAppendElement (&pdu->varbind, tmp);
	    } else {
		Tcl_DStringAppendElement (&pdu->varbind, toid);
	    }
	}

	if (((generic < 0) || (generic > 5)) && toid) {
	    free (toid);
	    toid = NULL;
	}
	Tcl_DStringEndSublist (&pdu->varbind);

	if (packet == NULL) {
	    fprintf (stderr, "** received trap with missing timestamp: %s\n",
		     Tcl_DStringValue (&pdu->varbind));
	    goto trapError;
	}

    } else {

	/*
	 * decode "request-id", "error-status", & "error index" field
	 */
	
	packet = ASN1_DecodeInt (packet, &pdulen, ASN1_INTEGER, 
				 &pdu->request_id);
	if (packet == NULL) goto asn1Error;
	
	packet = ASN1_DecodeInt (packet, &pdulen, 
				 ASN1_INTEGER, &pdu->error_status);
	if (packet == NULL) goto asn1Error;

	switch (pdu->error_status) {
	  case E_TOOBIG:	snmp_stats.snmpInTooBigs++; break;
	  case E_NOSUCHNAME:	snmp_stats.snmpInNoSuchNames++; break;
	  case E_BADVALUE:	snmp_stats.snmpInBadValues++; break;
	  case E_READONLY:	snmp_stats.snmpInReadOnlys++; break;
	  case E_GENERR:	snmp_stats.snmpInGenErrs++; break;
	}
	
	packet = ASN1_DecodeInt (packet, &pdulen, ASN1_INTEGER, 
				 &pdu->error_index);
	if (packet == NULL) goto asn1Error;
	
    }
    
    /*
     * decode "VarBindList" header ( 0x30 asnlen )
     */
    
    if (*packet++ != (UNIVERSAL | CONSTRUCTED | ASN1_SEQUENCE)) {
	if (pdu->type == SNMPv1_TRAP) {
	    fprintf (stderr, "** received trap with missing varbind: %s\n",
		     Tcl_DStringValue (&pdu->varbind));
	    goto trapError;
	}
	sprintf (interp->result,
		 "VarBindList: invalid tag 0x%.2x; expecting 0x%.2x",
		 *--packet, (UNIVERSAL | CONSTRUCTED | ASN1_SEQUENCE));
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }
    pdulen += 1;
    
    packet = ASN1_DecodeLength (packet, &pdulen, &asnlen);
    if (packet == NULL) goto asn1Error;
	
    if ((pdulen + asnlen) != deflen) {
	interp->result = "VarBindList: invalid length field";
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }

    /* 
     * decode each "VarBind" 
     */

    while (*packet && pdulen < deflen) {
	
	/*
	 * decode "VarBind" header ( 0x30 asnlen )
	 */
	
	if (*packet++ !=  (UNIVERSAL | CONSTRUCTED | ASN1_SEQUENCE)) {
	    sprintf (interp->result,
		     "VarBind: invalid tag 0x%.2x; expecting 0x%.2x",
		     *--packet, (UNIVERSAL | CONSTRUCTED | ASN1_SEQUENCE));
	    snmp_stats.snmpInASNParseErrs++;
	    return TCL_ERROR;
	}
	pdulen += 1;
	
	packet = ASN1_DecodeLength (packet, &pdulen, &asnlen);
	if (packet == NULL) goto asn1Error;
	
	/* Allocate new element for VarBindList */

	Tcl_DStringStartSublist (&pdu->varbind);
	
	/*
	 * decode OBJECT-IDENTIFIER
	 */
	
	packet = ASN1_DecodeOID (packet, &pdulen, oidbuf, &oidlen);
	if (packet == NULL) goto asn1Error;

	vboid = SNMP_Oid2Str (oidbuf, oidlen);
	Tcl_DStringAppendElement (&pdu->varbind, vboid);

	/*
	 * decode the ASN.1 type
	 */

	Tcl_DStringAppendElement (&pdu->varbind, ASN1_Sntx2Str (*packet));

	/*
	 * decode value for object
	 */

	switch (*packet) {
	  case ASN1_Counter:
            packet = ASN1_DecodeInt (packet, &pdulen, 
				     ASN1_Counter, &int_val);
            if (packet == NULL) goto asn1Error;
	    sprintf (buf, "%u", int_val);
            Tcl_DStringAppendElement (&pdu->varbind, buf);
            break;
	  case ASN1_Gauge:
            packet = ASN1_DecodeInt (packet, &pdulen, 
				     ASN1_Gauge, &int_val);
            if (packet == NULL) goto asn1Error;
	    sprintf (buf, "%u", int_val);
            Tcl_DStringAppendElement (&pdu->varbind, buf);
            break;
	  case ASN1_UInteger32:
            packet = ASN1_DecodeInt (packet, &pdulen, 
				     ASN1_UInteger32, &int_val);
            if (packet == NULL) goto asn1Error;
	    sprintf (buf, "%u", int_val);
            Tcl_DStringAppendElement (&pdu->varbind, buf);
            break;
	  case ASN1_INTEGER:
            packet = ASN1_DecodeInt (packet, &pdulen, 
				     ASN1_INTEGER, &int_val);
            if (packet == NULL) goto asn1Error;
	    {   char *tmp;
		sprintf (buf, "%d", int_val);
		tmp = MIB_Format (vboid, 0, buf);
		if (tmp) {
		    Tcl_DStringAppendElement (&pdu->varbind, tmp);
		} else {
		    Tcl_DStringAppendElement (&pdu->varbind, buf);
		}
	    }
            break;
	  case ASN1_TimeTicks:
            packet = ASN1_DecodeInt (packet, &pdulen, 
				     ASN1_TimeTicks, &int_val);
            if (packet == NULL) goto asn1Error;
	    {   int d, h, m, s, f;
                d = int_val;
                f = d % 100; d = d /100;
                s = d % 60;  d = d / 60;
                m = d % 60;  d = d / 60;
                h = d % 24;  d = d / 24;
                sprintf (buf, "%dd %d:%02d:%02d.%02d", d, h, m, s, f);
                Tcl_DStringAppendElement (&pdu->varbind, buf);
            }
            break;
	  case ASN1_NULL:
            packet += 2;
            pdulen += 2;
	    Tcl_DStringAppendElement (&pdu->varbind, "");
            break;
	  case ASN1_OBJECT_IDENTIFIER:
            packet = ASN1_DecodeOID (packet, &pdulen, oidbuf, &oidlen);
            if (packet == NULL) goto asn1Error;
	    {   char *tmp;
		freeme = SNMP_Oid2Str (oidbuf, oidlen);
		tmp = MIB_Name (freeme, 0);
		if (tmp) {
		    Tcl_DStringAppendElement (&pdu->varbind, tmp);
		} else {
		    Tcl_DStringAppendElement (&pdu->varbind, freeme);
		}
                free (freeme);
            }
            break;
	  case ASN1_IpAddress:
            packet = ASN1_DecodeOctetString (packet, &pdulen, ASN1_IpAddress, 
					     (u_char **) &freeme, &int_val);
            if (packet == NULL) goto asn1Error;
            Tcl_DStringAppend (&pdu->varbind, " ", 1);
            {    unsigned int addr = ntohl(*(int *) freeme);
                 sprintf (buf, "%u.%u.%u.%u",
			  (addr >> 24) & 0xff, (addr >> 16) & 0xff,
			  (addr >> 8) & 0xff, addr & 0xff);
	     }
            Tcl_DStringAppendElement (&pdu->varbind, buf);
	    free (freeme);
            break;
	  case ASN1_OCTET_STRING:
            packet = ASN1_DecodeOctetString (packet, &pdulen, 
					     ASN1_OCTET_STRING, 
					     (u_char **) &freeme, &int_val);
            if (packet == NULL) goto asn1Error;
	    {   char *foo;
		char *tmp = xmalloc (int_val * 3 + 1);
		bin_to_hex (freeme, int_val, tmp);
		foo = MIB_Format (vboid, 0, tmp);
		if (foo) {
		    Tcl_DStringAppendElement (&pdu->varbind, foo);
		} else {
		    Tcl_DStringAppendElement (&pdu->varbind, tmp);
		}
		free (tmp);
	    }
	    free (freeme);
            break;
	  default:
	    free (vboid);
            sprintf (interp->result, "unknown ObjectSyntax: %.2x", *packet);
	    snmp_stats.snmpInASNParseErrs++;
            return TCL_ERROR;
	}
	
	Tcl_DStringEndSublist (&pdu->varbind);
	free (vboid);
    }
    
    *packetlen += pdulen;
    
    if (pdulen != deflen) {
	sprintf (interp->result,
		 "PDU sequence length (%d) differs from real length (%d).",
		 pdulen, (int) deflen);
	snmp_stats.snmpInASNParseErrs++;
	return TCL_ERROR;
    }

    return TCL_OK;
    
  asn1Error:
    if (vboid) free (vboid);
    Tcl_SetResult (interp, ASN1_ErrorString(), TCL_STATIC);
	snmp_stats.snmpInASNParseErrs++;
    return TCL_ERROR;

  trapError:
    return TCL_OK;
}

/*
 * SNMP_EvalCallback() evaluates a callback string. The command is
 * modified according to the known % escapes before evaluation. The
 * list of supported escapes is %R = request id, %S = session name,
 * %E = error status, %I = error index and %A the agent address.
 */

int
SNMP_EvalCallback (interp, session, request_id, 
		   error_status, error_index, callback, result)
     Tcl_Interp		*interp;
     struct session	*session;
     int		request_id;
     int		error_status;
     int		error_index;
     char		*callback;
     Tcl_DString	*result;
{
    char buf[20];
    int	code;
    Tcl_DString dstr_Tcl_Code;
    char *startPtr, *scanPtr;

    /* substitue %-escapes in argument list for callback command */

    Tcl_DStringInit (&dstr_Tcl_Code);
    startPtr = callback;
    for (scanPtr = startPtr; *scanPtr != '\0'; scanPtr++) {
	if (*scanPtr != '%')
	  continue;
	Tcl_DStringAppend (&dstr_Tcl_Code, startPtr, scanPtr - startPtr);
	scanPtr++;
	startPtr = scanPtr + 1;
	switch (*scanPtr) {
	  case 'R':  
	    sprintf (buf, "%d", request_id);
	    Tcl_DStringAppend (&dstr_Tcl_Code, buf, -1);
	    break;
	  case 'S':
	    if (session) {
		Tcl_DStringAppend (&dstr_Tcl_Code, session->name, -1);
	    }
	    break;
	  case 'V':
	    Tcl_DStringAppend (&dstr_Tcl_Code, Tcl_DStringValue(result), -1);
	    break;
	  case 'E':
	    if (error_status < 0) {
		Tcl_DStringAppend (&dstr_Tcl_Code, "noResponse", 10);
	    } else {
		Tcl_DStringAppend (&dstr_Tcl_Code, snmp_error[error_status], 
				   -1);
	    }
	    break;
	  case 'I':
	    sprintf (buf, "%d", error_index);
	    Tcl_DStringAppend (&dstr_Tcl_Code, buf, -1);
	    break;
	  case 'A':
	    Tcl_DStringAppend (&dstr_Tcl_Code, session->dstParty.TAddress, -1);
	    break;
	  case 'P':
	    buf[0] = '\0';
	    sprintf (buf, "%u", session->dstParty.TPort);
	    Tcl_DStringAppend (&dstr_Tcl_Code, buf, -1);
	    break;
	  case '%':
	    Tcl_DStringAppend (&dstr_Tcl_Code, "%", -1);
	    break;
	  default:
	    sprintf (buf, "%%%c", *scanPtr);
	    Tcl_DStringAppend (&dstr_Tcl_Code, buf, -1);
	}
    }
    Tcl_DStringAppend (&dstr_Tcl_Code, startPtr, scanPtr - startPtr);
    
    /* evaluate the callback function */
    
    code = Tcl_Eval (interp, Tcl_DStringValue (&dstr_Tcl_Code));
    Tcl_DStringFree (&dstr_Tcl_Code);
    
    return code;
}

