/*
 * parse.c
 *
 * Functions to parse MIB files for creating a MIB tree. The structure
 * of MIB information must meet the definitions specified in RFC 1155.
 * This code is partly derived from the MIB parser of the CMU SNMP
 * implementation, but it now looks quite different as we have made 
 * many changes and rewritten large parts of it.
 *
 * Copyright (c) 1994
 *
 * Sven Schmidt, J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <tcl.h>

#ifdef DBMALLOC
#include <dbmalloc.h>
#endif

#include "snmp.h"
#include "asn1.h"
#include "mib.h"
#include "xmalloc.h"

/*
 * DEFINES
 */

#define ERROR		-2

#define DEFINITIONS	51
#define EQUALS		52
#define BEGIN		53
#define IMPORTS		54
#define EXPORTS		55
#define END		57
#define COMMENT		58
#define LABEL		59
#define	CONTINUE	60

#define SYNTAX		70

#define LEFTBRACKET	80
#define RIGHTBRACKET	81
#define	LEFTPAREN	82
#define RIGHTPAREN	83
#define COMMA		84
#define SEMICOLON	85

#define ACCESS		90
#define READONLY	91
#define READWRITE	92
#define READCREATE	93
#define	WRITEONLY	94
#define NOACCESS	95

#define STATUS		100
#define MANDATORY	101
#define OPTIONAL	102
#define OBSOLETE	103
#define CURRENT		104
#define DEPRECATED	105
#define RECOMMENDED	106

#define OBJGROUP	110
#define OBJECTS		111
#define OBJECTIDENTITY	112

#define MODULECOMP	120
#define REFERENCE	121

#define MODULEIDENTITY	130
#define LASTUPDATED	153
#define ORGANIZATION	154
#define CONTACTINFO	155

#define NOTIFYTYPE	156

#define TRAPTYPE	157

#define	CAPABILITIES	158

#define TREE		100
#define SUBTREE		101

#define TEXTUALCONV	110
#define DISPLAYHINT	1101		/* XXX: added */

#define OBJTYPE		186
#define PUNCT		127
#define NUMBER		129
#define DESCRIPTION	135
#define QUOTESTRING	136
#define INDEX		137
#define DEFVAL		138
#define SIZE		140
#define AUGMENTS	146
#define UNITS		149
#define NUM_ENTRIES	151

#define	HASHTAB_SIZE	17			/* size of hash table */
#define	HASH(x)		(x % HASHTAB_SIZE)	/* Hash function      */

#define NHASHSIZE	127			/* hashsize for nodes */
#define NODEHASH(x)	(x % NHASHSIZE)		/* Hash function      */

#define MAXTC		128

#define IN_MIB		1
#define OUT_OF_MIB	2

/*
 * structure definition to hold the keyword table
 */

struct keyword {
   char			*name;		/* keyword name			    */
   int			key;		/* value			    */
   int			hash;		/* hash of name			    */
   struct keyword	*next;		/* pointer to next in hash table    */
};

struct keyword keywords[] =
{
   { "DEFINITIONS",		DEFINITIONS,		0 },
   { "BEGIN",			BEGIN,			0 },
   { "END",			END,			0 },
   { "IMPORTS",			IMPORTS,		0 },
   { "EXPORTS",			EXPORTS,		0 },
   { "::=",			EQUALS,			0 },

   { "{",			LEFTBRACKET,		0 },
   { "}",			RIGHTBRACKET,		0 },
   { "(",			LEFTPAREN,		0 },
   { ")",			RIGHTPAREN,		0 },
   { ",",			COMMA,			0 },
   { ";",			SEMICOLON,		0 },

   { "SYNTAX",			SYNTAX,			0 },

   { "INTEGER",			ASN1_INTEGER,		0 },
   { "OCTETSTRING",		ASN1_OCTET_STRING,	0 },
   { "OCTET",			CONTINUE,		0 },
   { "OBJECTIDENTIFIER",	ASN1_OBJECT_IDENTIFIER,	0 },
   { "OBJECT",			CONTINUE,		0 },
   { "BITSTRING",		ASN1_BIT_STRING,	0 },
   { "BIT",			CONTINUE,		0 },
   { "NULL",			ASN1_NULL,		0 },
   { "SEQUENCE",		ASN1_SEQUENCE,		0 },
   { "SEQUENCEOF",		SEQUENCE_OF,		0 },
   { "Integer32",		Integer32,		0 },
   { "IpAddress",		ASN1_IpAddress,		0 },
   { "NetworkAddress",		NetworkAddress,		0 },
   { "Counter",			ASN1_Counter,		0 },
   { "Counter32",		Counter32,		0 },
   { "Gauge",			ASN1_Gauge,		0 },
   { "Gauge32",			Gauge32,		0 },
   { "TimeTicks",		ASN1_TimeTicks,		0 },
   { "Opaque",			ASN1_Opaque,		0 },
   { "NsapAddress",		ASN1_NsapAddress,	0 },
   { "Counter64",		ASN1_Counter64,		0 },
   { "UInteger32",		ASN1_UInteger32,	0 },

   { "ACCESS",			ACCESS,			0 },
   { "read-only",		READONLY,		0 },
   { "read-write",		READWRITE,		0 },
   { "read-create",		READCREATE,		0 },
   { "write-only",		WRITEONLY,		0 },
   { "not-accessible",		NOACCESS,		0 },

   { "STATUS",			STATUS,			0 },
   { "mandatory",		MANDATORY,		0 },
   { "optional",		OPTIONAL,		0 },
   { "obsolete",		OBSOLETE,		0 },
   { "current",			CURRENT,		0 },
   { "deprecated",		DEPRECATED,		0 },
   { "recommended",		RECOMMENDED,		0 },

   { "DESCRIPTION",		DESCRIPTION,		0 },

   { "OBJECT-GROUP",		OBJGROUP,		0 },
   { "OBJECTS",			OBJECTS,		0 },
   { "OBJECT-TYPE",		OBJTYPE,		0 },

   { "OBJECT-IDENTITY",		OBJECTIDENTITY,		0 },

   { "MODULE-COMPLIANCE",	MODULECOMP,		0 },
   { "REFERENCE",		REFERENCE,		0 },

   { "MODULE-IDENTITY",		MODULEIDENTITY,		0 },
   { "LAST-UPDATED",		LASTUPDATED,		0 },
   { "ORGANIZATION",		ORGANIZATION,		0 },
   { "CONTACT-INFO",		CONTACTINFO,		0 },

   { "NOTIFICATION-TYPE",	NOTIFYTYPE,		0 },

   { "TRAP-TYPE",		TRAPTYPE,		0 },

   { "AGENT-CAPABILITIES",	CAPABILITIES,		0 },

   { "TEXTUAL-CONVENTION",	TEXTUALCONV,		0 },
   { "DISPLAY-HINT",		DISPLAYHINT,		  },

   { "AUGMENTS",		AUGMENTS,		0 },
   { "UNITS",			UNITS,			0 },
   { "NUM-ENTRIES",		NUM_ENTRIES,		0 },
   { "INDEX",			INDEX,			0 },
   { "DEFVAL",			DEFVAL,			0 },
   { "SIZE",			SIZE,			0 },
   { "MAX-ACCESS",		ACCESS,			0 },
   { NULL }
};

struct subid {			/* Element of an object identifier          */
   char		*parent_name;	/* with an integer subidentifier or         */
   char		*label;		/* a textual string label, or both.         */
   int		subid;
   struct subid	*next;
};

/* list of all seen textual_type conventions: */
struct textual_type *all_tt_conv = 0;

/* simple hack: mark to first already saved elem: */
struct textual_type *tt_save_mark = 0;


/*
 * some useful global variables
 */

static int lastchar = ' ';
static int line     = 0;

static struct tree	*nodehashtab[NHASHSIZE];
static struct tree	*root = NULL;
static struct keyword	*hashtab[HASHTAB_SIZE];

static FILE *fp;
static char *filename;

/*
 * Forward declarations for procedures defined later in this file:
 */

static struct tree*
malloc_tree		_ANSI_ARGS_((char *name));

static void 
write_conv_file _ANSI_ARGS_((FILE *fp, struct textual_type *tt, 
			     struct textual_type *mark));

static void
write_idxfile		_ANSI_ARGS_((FILE *fp, struct tree *nodelist));

static struct tree*
read_idxfile		_ANSI_ARGS_((char *file));

static struct tree*
parse_mibfile		_ANSI_ARGS_((FILE *fp));

static void
hash_init		_ANSI_ARGS_((void));

static int
parse_mib_header	_ANSI_ARGS_((FILE *fp, char *keyword));

static int
parse_asn_type		_ANSI_ARGS_((FILE *fp, char *keyword));

static struct tree*
parse_module_compliance _ANSI_ARGS_((FILE *fp, char *name));

static struct tree*
parse_module_identity	_ANSI_ARGS_((FILE *fp, char *name));

static struct tree*
parse_notification_type _ANSI_ARGS_((FILE *fp, char *name));

static int
parse_capabilities_type _ANSI_ARGS_((FILE *fp));

static int
parse_trap_type		_ANSI_ARGS_((FILE *fp, char *name));

static struct tree*
parse_object_group	_ANSI_ARGS_((FILE *fp, char *name));

static struct tree*
parse_object_id		_ANSI_ARGS_((FILE *fp, char *name));

static struct tree*
parse_object_type	_ANSI_ARGS_((FILE *fp, char *name));

static struct tree*
parse_object_identity	_ANSI_ARGS_((FILE *fp, char *name));

static int
read_keyword		_ANSI_ARGS_((FILE *fp, char *keyword));

static struct subid *
read_subid		_ANSI_ARGS_((FILE *fp));

static struct tree*
build_tree		_ANSI_ARGS_((struct tree *nlist));

static void
node_hash_init		_ANSI_ARGS_((struct tree *nlist));

static void
addto_tree		_ANSI_ARGS_((struct tree *root,
				      struct tree *nlist));
static void
build_subtree		_ANSI_ARGS_((struct tree *root));

static struct idesc*
tree_scan_intdesc	_ANSI_ARGS_((char *str));

static int
scan_enum_descr		_ANSI_ARGS_((Tcl_DString *stats));

static struct tree*
lookup_node		_ANSI_ARGS_((struct tree *root, char *name));

/*
 * malloc_tree() allocates a new tree element and does some basic
 * initialization.
 */

static struct tree*
malloc_tree (name)
     char *name;
{
    struct tree *np = (struct tree *) xmalloc (sizeof (struct tree));
    memset ((char *) np, '\0', sizeof (struct tree));
    np->label = xstrdup (name);
    return np;
}


/*
 * MIB_Parse() opens a MIB file specified by *file and returns a tree
 * of objects in that MIB. The function returns a pointer to the
 * "root" of the MIB, or NULL if an error occurred.
 */

struct tree *
MIB_Parse (file, root)
     char *file;
     struct tree *root;
{
    struct tree	*node = NULL;
    struct stat stbuf;
    char *frozen = xmalloc (strlen (file) + 5);
    time_t mib_mtime = 0, frozen_mtime = 0;

    if ((fp = fopen (file, "r")) == NULL) return NULL;

    filename = xstrdup (file);
    if (stat (file, &stbuf) == 0) {
	mib_mtime = stbuf.st_mtime;
    }

    strcpy (frozen, filename);
    strcat (frozen, ".idx");
    if (stat (frozen, &stbuf) == 0) {
	frozen_mtime = stbuf.st_mtime;
    }

    if (mib_mtime == 0 || frozen_mtime == 0 || frozen_mtime < mib_mtime) {
	/* save pointer to still known tt's: */
	tt_save_mark = all_tt_conv;
	node = parse_mibfile (fp);
	fclose (fp);
	if (node != NULL || all_tt_conv != tt_save_mark) {
	    fp = fopen (frozen, "w");
	    if (fp != NULL) {
		write_conv_file (fp, all_tt_conv, tt_save_mark);
		write_idxfile (fp, node);
		fclose (fp);
	    }
	}
    } else {
	fclose (fp);
	node = read_idxfile (frozen);
    }

    if (! root)
      root = build_tree (node);
    else
      addto_tree (root, node);
    
    free (frozen);

    return root;
}




/*
 * string contains pairs of a description string and a ninteger value.
 * scan them into a chained list of idesc's
 * NOTE: this puts \0 into the passed string.  */

static struct idesc *
tree_scan_intdesc (str)
     char *str;
{
    int done = 0;
    struct idesc *ret = 0;

    if (! str)
      return ret;

    if (! strncmp (str, "D ", 2))
      str += 2;
#if 0
    /* be silent... */
    else {
	fprintf (stderr, "%s: Warning: Don't know to interpret ``%s''\n",
		 filename, str);
	return ret;
    }
#endif

    while (*str)
      {
	  struct idesc *nnew;
	  char *val, *desc = str;
	  while (*str && isspace (*str))
	    str++;
	  while (*str && ! isspace (*str))
	    str++;
	  if (! *str)
	    break;
	  *str++ = 0;
	  val = str;
	  while (*str && ! isspace (*str))
            str++;
	  if (! *str)
	    done = 1;
	  else
	    *str++ = 0;
	  nnew = (struct idesc *) xmalloc (sizeof (struct idesc));
	  nnew->val = val;
	  nnew->desc = desc;
	  nnew->next = ret;
	  ret = nnew;
	  if (done)
	    break;
      }

    return ret;
}



/*
 * save all textual conventions up to the tt_save_mark to
 * the passed fp file.
 * format is: __<name>\t<syntax>\t[TD <value>...]
 */

static void
write_conv_file (fp, tt, mark)
     FILE *fp;
     struct textual_type *tt, *mark;
{
    while (tt && tt != mark)
      {
	  fprintf (fp, "__%s\t%d\t", tt->name, tt->syntax);
	  if (tt->textconv)
	    fprintf (fp, "T %s", tt->textconv);
	  else if (tt->desc)
	    {
		struct idesc *d;
		fprintf (fp, "D");
		for  (d = tt->desc; d; d = d->next)
		  fprintf (fp, " %s %s", d->desc, d->val);
	    }
	  fputs ("\n", fp);

	  tt = tt->next;
      }
}


/*
 * write_idxfile() writes the node list as an index file to fp. Done
 * recursively to get the ordering right.
 */

static void
write_idxfile (fp, node)
     FILE *fp;
     struct tree *node;
{
    if (! node) return;

    write_idxfile (fp, node->next);
    fprintf (fp, "%s\t%s\t%d\t%d\t%d\t%d", node->label, node->parent_name, 
	     node->subid, node->syntax, node->description,
	     node->access);
    /* 
     * for textual-conventions always append a tab and the name.
     */
    if (node->tt)
      fprintf (fp, "\t%s", node->tt->name);
    
    fputs ("\n", fp);
}

/*
 * read_idxfile() reads an index file that was written by write_idxfile().
 */

static struct tree *
read_idxfile (file)
     char	*file;
{
    struct tree *root = NULL;
    struct tree *new;
    FILE *f;
    char lline[2048];
    char ch, *p, *s;

    if ((f = fopen (file, "r")) == NULL) return NULL;
    
    while (fgets (lline, sizeof (lline), f) != NULL) {

	if (*lline == '_' && lline[1] == '_')
	  {
	      /* scan a textual convention: */
	      struct textual_type *nnew;

	      /* clear eol: */
	      p = lline + strlen (lline) - 1;
	      if (*p != '\n')
		continue;
	      *p = 0;

	      nnew = (struct textual_type *)
		xmalloc (sizeof (struct textual_type));
	      memset ((char *) nnew, '\0', sizeof (struct textual_type));

	      /* scan name: */
	      for (s = p = lline + 2; *p && *p != '\t'; p++);
	      if (*p != '\t') { free (nnew); continue; }
	      *p++ = 0;
	      nnew->name = xstrdup (s);
	      /* scan syntax: */
	      for (s = p; *p && *p != '\t'; p++);
	      if (*p != '\t') { free (nnew); continue; }
	      *p++ = 0;
	      nnew->syntax = atoi (s);

	      /* scan arg: */
	      s = p;
	      if (*p == 'T' && p[1] == ' ')
		    nnew->textconv = xstrdup (p + 2);
	      else if (*p == 'D' && p[1] == ' ')
		{
#if 0
		    nnew->int_desc = xstrdup (p + 2);
#endif
		    nnew->desc = tree_scan_intdesc (xstrdup (p));
		}

	      nnew->next = all_tt_conv;
	      tt_save_mark = all_tt_conv = nnew;
	      
	      /* now go on with the next lline: */
	      continue;
	  }

	/* scan a node: */

      	new = (struct tree *) xmalloc (sizeof (struct tree));
	memset ((char *) new, '\0', sizeof (struct tree));
	s = p = lline;
	while (*p && *p != '\t') p++;
	if (! *p) {
	    free (new);
	    continue;
	}
	*p++ = '\0';
	new->label = xstrdup (s);
	s = p;
	while (*p && *p != '\t') p++;
	if (! *p) {
	    free (new->parent_name);
	    free (new);
	    continue;
	}
	*p++ = '\0';
	new->parent_name  = xstrdup (s);
	s = p;
	while (*p && *p != '\t') p++;
	if (! *p) {
	    free (new->label);
	    free (new->parent_name);
	    free (new);
            continue;
        }
	*p++ = '\0';
	new->subid = atoi (s);
	s = p;
	while (*p && *p != '\t') p++;
	if (! *p) {
	    free (new->label);
            free (new->parent_name);
            free (new);
            continue;
        }
	*p++ = '\0';
	new->syntax = atoi (s);
	s = p;
        while (*p && *p != '\t') p++;
        if (! *p) {
	    free (new->label);
            free (new->parent_name);
            free (new);
            continue;
	}
        *p++ = '\0';
	new->description = atoi (s);
	/*
	 * at p now the access field follows:
	 */
	s = p;
	while (*p && *p != '\t') p++;
	if (s == p) {
	    /* nothing found: */
	    free (new->label);
	    free (new->parent_name);
	    free (new);
	    continue;
	}
	ch = *p;			/* save this one */
	*p++ = '\0';
	new->access = atoi (s);
	/*
	 * if ch was a tab, there follows a textual-conv,
	 * if its a 0 we are at the end:
	 */
	if (ch == '\t')
	  {
	      if (p [strlen (p) - 1] == '\n')
		p [strlen (p) - 1] = 0;
	      for (new->tt = all_tt_conv; new->tt; new->tt = new->tt->next)
		if (! strcmp (p, new->tt->name))
		  break;
	      if (! new->tt)
		fprintf (stderr, 
	"%s:%d: Warning: textual-convention ``%s'' not found for ``%s''\n",
			 filename, line, p, new->label);
	}
        new->next = root;
        root = new;
    }

    fclose (f);

    return root;
}



/*
 * scan a enum description, like ``INTEGER { foo(1), bar(2) }'', with
 * the next token to be read the one after the ``{''; upon return the 
 * closing ``}'' is read.
 * the scanned datas are return in the passed dstring, in the form:
 * ``D foo 3 bar 4''
 */

static int
scan_enum_descr (stats)
     Tcl_DString *stats;
{
    int syntax;
    int fail = 0;		/* parser thinks anything as alright */

    Tcl_DStringInit (stats);
    
    /* mark we append Descriptions: */
    Tcl_DStringAppend (stats, "D", 1);
    
    /* got LEFTBRACKET:  ``{'' */
    do {
	char str [SYMBOL_MAXLEN];
	char num [SYMBOL_MAXLEN];
	char keyword [SYMBOL_MAXLEN];
	
	while ((syntax = read_keyword (fp, str)) == COMMENT);
#if 0
/** XXX: dont check - all we need is a string */
	if (syntax != LABEL) { fail = 1; break; }
#endif
	/* got the string:  ``{ foo'' */
	while ((syntax = read_keyword (fp, keyword)) == COMMENT);
	if (syntax != LEFTPAREN) { fail = 1; break; }
	/* ``{ foo ('' */
	while ((syntax = read_keyword (fp, num)) == COMMENT);
	if (syntax != NUMBER) { fail = 1; break; }
	/* append to the collecting string: */
	Tcl_DStringAppend (stats, " ", 1);
	Tcl_DStringAppend (stats, str, -1);
	Tcl_DStringAppend (stats, " ", 1);
	Tcl_DStringAppend (stats, num, -1);
	/* ``{ foo (99'' */
	while ((syntax = read_keyword (fp, keyword)) == COMMENT);
	if (syntax != RIGHTPAREN) { fail = 1; break; }
	/* ``{ foo (99)'' */
	/* now there must follow either a ``,'' or a ``}'' */
	while ((syntax = read_keyword (fp, keyword)) == COMMENT);
    } while (syntax == COMMA);
    
    /* the list is scanned, now there must be a ``}'' */
    
    if (fail || syntax != RIGHTBRACKET)
      {
	  fprintf (stderr,
		   "%s: %d: Warning: cannot scan enums - ignored\n",
		   filename, line);
      }

    return syntax;
}



/*
 * parse_mibfile() opens a MIB file specified by *fp and return a
 * linked list of objects in that MIB.
 */

static struct tree *
parse_mibfile (fp)
     FILE *fp;
{
    u_char name[SYMBOL_MAXLEN];
    u_char keyword[SYMBOL_MAXLEN];

    int	state  = OUT_OF_MIB;
    int	syntax = 0;

    struct tree *new = NULL;

    /* 
     * this is really a klude: try to fetch type-definitions
     * like: ``Boolean ::= INTEGER { true(1), false(2) }''
     * and: ``KBytes ::= INTEGER''.
     */
    char tt_name[SYMBOL_MAXLEN];
    static struct textual_type tt = { 0, };
    struct textual_type *nnew = NULL, *last_tt;

    /*
     * init hashtable and parse mib 
     */

    hash_init();
    root = NULL;
    line = 0;

    while ((syntax = read_keyword (fp, keyword)) != EOF) {
	
	if (state == OUT_OF_MIB) {

	    /*
	     * we are at the beginning of a mib
	     */
	
	    switch (syntax) {
	      case COMMENT:
		break;
	      case DEFINITIONS:
		state = IN_MIB;
		if ((syntax = parse_mib_header (fp, name)) == EOF) {
		    fprintf (stderr,
			     "%s:%d: bad format in MIB header\n",
			     filename, line);
		    return NULL;
		}
		break;
	      case END:
		fprintf (stderr, "%s: end before start of MIB.\n", filename);
		return NULL;
	      case ERROR:
		fprintf (stderr, "%s:%d: error in MIB\n", filename, line);
		return NULL;
	      case LABEL:
		strncpy (name, keyword, SYMBOL_MAXLEN);
		break;
	      default:
		fprintf (stderr, "%s:%d: %s is a reserved word\n", 
			 filename, line, keyword);
		return NULL;
	    }

	} else {

	    /*
	     * we are in a mib
	     */
	
	    switch (syntax) {
	      case COMMENT:
		break;
	      case DEFINITIONS:
		fprintf (stderr, "%s: Fatal: nested MIBS\n", filename);
		return NULL;
		break;
	      case END:
		state = OUT_OF_MIB;
		break;
	      case EQUALS:
		syntax = parse_asn_type (fp, name);
		if (syntax == END) 
                  state = OUT_OF_MIB;
	
		if (syntax == ASN1_INTEGER || syntax == ASN1_BIT_STRING
		    || syntax == ASN1_OCTET_STRING) 
		  {
		    /* sure a type; if one with the same name already
		     * exists (eg. DisplayString in rfc1443.tc
		     * (wanted) and then in rfc1213.mib (unwanted))
		     * ignore and use the existing one (but this may
		     * hurt -- you have benn warned) */
		      
		      struct textual_type *t = all_tt_conv;
		      while (t && strcmp (t->name, tt_name))
			t = t->next;
		      if (! t)
			{
			    tt.syntax = syntax;
			    tt.name = tt_name;
			    
			    nnew = (struct textual_type *)
			      xmalloc (sizeof (struct textual_type));
			    memset ((char *) nnew, '\0', 
				    sizeof (struct textual_type));
			    nnew->name = xstrdup (tt.name);
			    nnew->syntax = tt.syntax;
			    nnew->desc = 0;		/* maybe later */
			    /* chain in - XXX alert: these local conventions
			     * are now used by the following mibs too ! */
			    nnew->next = all_tt_conv;
			    last_tt = all_tt_conv = nnew;
			}
		  }
		break;
	      case ERROR:
		fprintf (stderr, "%s:%d: error in MIB\n", filename, line);
		return NULL;
	      case LABEL:
		strncpy (name, keyword, SYMBOL_MAXLEN);
		/* maybe a type: */
		strncpy (tt_name, keyword, SYMBOL_MAXLEN);
		break;
	      case MODULECOMP:
		if ((new = parse_module_compliance (fp, name)) == NULL) {
		    fprintf (stderr,
			     "%s:%d: bad format in MODULE-COMPLIANCE\n",
			     filename, line);
		    return NULL;
		}
		new->next = root;
		root = new;
		break;
	      case MODULEIDENTITY:
		if ((new = parse_module_identity (fp, name)) == NULL) {
		    fprintf (stderr,
			     "%s:%d: bad format in MODULE-IDENTIY\n",
			     filename, line);
		    return NULL;
		}
		new->next = root;
		root = new;
		break;
	      case NOTIFYTYPE:
		if ((new = parse_notification_type (fp, name)) == NULL) {
		    fprintf (stderr,
			     "%s:%d: bad format in NOTIFICATION-TYPE\n",
			     filename, line);
		    return NULL;
		}
		new->next = root;
		root = new;
		break;
	      case CAPABILITIES:
		if (parse_capabilities_type (fp) == 0) {
		    fprintf (stderr, 
			     "%s:%d: bad format in AGENT-CAPABILITIES\n",
			     filename, line);
                    return NULL;
		}
		break;
	      case TRAPTYPE:
		if (parse_trap_type (fp, name) < 0) {
		    fprintf (stderr,
			     "%s:%d: bad format in TRAP-TYPE\n",
			     filename, line);
		    return NULL;
		}
		break;
	      case OBJGROUP:
		if ((new = parse_object_group (fp, name)) == NULL) {
		    fprintf (stderr,
			     "%s:%d: bad format in OBJECT-GROUP\n",
			     filename, line);
		    return NULL;
		}
		new->next = root;
		root = new;
		break;
	      case OBJECTIDENTITY:
		if ((new = parse_object_identity (fp, name)) == NULL) {
		    fprintf (stderr,
			     "%s:%d: bad format in OBJECT-IDENTITY\n",
			     filename, line);
		    return NULL;
		}
		new->next = root;
		root = new;
		break;
	      case ASN1_OBJECT_IDENTIFIER:
		if ((new = parse_object_id (fp, name)) == NULL) {
		    fprintf (stderr,
			     "%s:%d: bad format in OBJECT-IDENTIFIER\n",
			     filename, line);
		    return NULL;
		}
		new->next = root;
		root = new;
		break;
	      case OBJTYPE:
               if ((new = parse_object_type (fp, name)) == NULL) {
		   fprintf(stderr,
			   "%s:%d: bad format in OBJECT-TYPE\n",
			   filename, line);
                  return NULL;
               }
		new->next = root;
		root = new;
		break;
		
		/*
		 *  some textual conventions
		 */
		
	      case DisplayString:
	      case EntryStatus:
	      case OwnerString:
	      case Party:
	      case PhysAddress:
	      case ASN1_TAddress:
		strncpy (name, keyword, SYMBOL_MAXLEN);
		/** XXX: fprintf (stderr, "XXX: what to do with ``%s''\n",
			keyword); **/
		break;

		/* 
		 * expect a leftparen; eg. for ``MacAddress ::= OCTET
		 * STRING (SIZE (6))''; action is to skip over. 
		 */
	      case LEFTPAREN:
		{ int cnt = 1;
		  for (;;) {
		      char buf [SYMBOL_MAXLEN];
		      int syntax;
		      if ((syntax = read_keyword (fp, buf)) == LEFTPAREN)
			cnt ++;
		      else if (syntax == RIGHTPAREN)
			cnt--;
		      if (! cnt)
			break;
		  }
	        }
		break;

	      case LEFTBRACKET:
		{ Tcl_DString stats;
		  if (scan_enum_descr (&stats) != RIGHTBRACKET)
		    fprintf (stderr, "%s:%d: bad mib format\n",
			     filename, line);
		  else
		      nnew->desc = tree_scan_intdesc (
				xstrdup (Tcl_DStringValue (&stats)));
	        }
		break;

	      default:
		fprintf (stderr, "%s:%d: bad mib format\n", 
			 filename, line);
		return NULL;
		break;
	    }
	}
    }
    
    /* did we finish correctly */

    if (state == OUT_OF_MIB) return root;

    fprintf (stderr, "%s: Fatal: incomplete MIB\n", filename);
    return NULL;
}

/*
 * hash_init() builds up a hash table, including the defined keywords
 * as records. This enables the other functions to directly reference
 * records by doing arithmetic transformations on the keywords name.
 */

static void
hash_init()
{
    char *cp = NULL;
    int	hash_val = 0;
    int	hash_index = 0;
    struct keyword *tp = NULL;

    memset ((char *) hashtab, '\0', sizeof (hashtab));
    
    for (tp = keywords; tp->name; tp++) {
	hash_val = 0;
	for (cp = tp->name; *cp; cp++) 
	  hash_val += *cp;
	hash_index = HASH (hash_val);
	tp->hash   = hash_val;

	if (hashtab[hash_index] != NULL)
	  tp->next = hashtab[hash_index];
	hashtab[hash_index] = tp;
    }
}

/*
 * parse_mib_header() parses a MIB header and places the LABEL that
 * follows in the string pointed to by keyword. Returns EOF if no end
 * of MIB header found, ERROR if no LABEL found and LABEL if MIB
 * header is passed and a label placed into keyword.
 */

static int
parse_mib_header (fp, keyword)
     FILE *fp;
     char *keyword;
{
    int syntax;
   
    if ((syntax = read_keyword (fp, keyword)) != EQUALS)
      return ERROR;

    if ((syntax = read_keyword (fp, keyword)) != BEGIN)
      return ERROR;

    while ((syntax = read_keyword (fp, keyword)) == COMMENT) ;

    /*
     * if it's EXPORTS clause, read the next keyword after SEMICOLON
     */

    if (syntax == EXPORTS) {
	while ((syntax = read_keyword (fp, keyword)) != SEMICOLON)
	  if (syntax == EOF) return EOF;
	syntax = read_keyword (fp, keyword);
    }

    /*
     * if it's IMPORTS clause, read the next keyword after SEMICOLON
     */

    if (syntax == IMPORTS) {
	while ((syntax = read_keyword (fp, keyword)) != SEMICOLON)
	  if (syntax == EOF) return EOF;
	syntax = read_keyword (fp, keyword);
    }
    
    /*
     * skip all comments
     */

    while (syntax == COMMENT)
      syntax = read_keyword (fp, keyword);
    
    /*
     * return syntax (should be label or defined label (e.g. DisplayString))
     */

    return syntax;
}



/*
 * Parses an ASN syntax and places the LABEL that follows in the
 * string pointed to by keyword.
 *
 * Returns 0 on error, LABEL on success.  
 */

static int
parse_asn_type (fp, keyword)
     FILE *fp;
     char *keyword;
{
    int level = 0, syntax = 0, merk, osyntax;
    char name [SYMBOL_MAXLEN];
    char otype [SYMBOL_MAXLEN];
    char convention [1024];
    Tcl_DString stats;
    char *ds_val;
    
    /* save passed name: */
    strcpy (name, keyword);
    
    syntax = read_keyword (fp, keyword);
    switch (syntax) {
      case ASN1_INTEGER:
#if 0
	while ((syntax = read_keyword (fp, keyword)) != RIGHTBRACKET)
	  if (syntax == EOF) return 0;
#endif
	break;
      case ASN1_OCTET_STRING:
	break;
      case ASN1_OBJECT_IDENTIFIER:
	break;
      case ASN1_SEQUENCE:
	while ((syntax = read_keyword (fp, keyword)) != RIGHTBRACKET)
	  if (syntax == EOF) return 0;
	break;
      case TEXTUALCONV:

	 /* default: no convention/enums seen: */
	 convention [0] = 0;
	 Tcl_DStringInit (&stats);

         while ((syntax = read_keyword (fp, keyword)) != SYNTAX
		&& syntax != DISPLAYHINT)
	   if (syntax == EOF)
	     return 0;

         /*
	  * read the keyword following SYNTAX or DISPLAYHINT
	  */

	 merk = read_keyword (fp, keyword);
	 if (syntax == SYNTAX && merk == LABEL)
	   return 0;

	 if (syntax == DISPLAYHINT)
	   {
	       /* save convention */
	       strcpy (convention, keyword);

	       /* skip to SYNTAX: */
	       while ((syntax = read_keyword (fp, keyword)) != SYNTAX)
		 if (syntax == EOF)
		   return 0;
	       
	       if ((merk = read_keyword (fp, keyword)) == LABEL)
		 return 0;
	   }
	
	/* XXX */
	/* save object type: */
	strcpy (otype, keyword);
	osyntax = merk;
	
	/*
	 * if next keyword is a bracket, we have to continue
	 */
	
	if ((syntax = read_keyword (fp, keyword)) == LEFTPAREN) {
	    level = 1;
	    while (level != 0) {
		if ((syntax = read_keyword (fp, keyword)) == EOF)
		  return 0;
		if (syntax == LEFTPAREN)
		  ++level;
		if (syntax == RIGHTPAREN)
		  --level;
	    }
	    syntax = read_keyword (fp, keyword);
	}
	
	if (syntax == LEFTBRACKET) {
	    syntax = scan_enum_descr (&stats);
	    ds_val = Tcl_DStringValue (&stats);
	}
	
	 /* found textconv: */
	
	{   struct textual_type *nnew = (struct textual_type *) 
	      xmalloc (sizeof (struct textual_type));
	    char *res;
	    memset ((char *) nnew, '\0', sizeof (struct textual_type));
	    nnew->name = xstrdup (name);
	    nnew->syntax = osyntax;
	    if (convention && *convention)
	      nnew->textconv = xstrdup (convention);
	    if ((res = Tcl_DStringValue (&stats)) && *res)
	      {
#if 0
		   nnew->int_desc = xstrdup (res);
#endif
		   nnew->desc = tree_scan_intdesc (xstrdup (res));
	       }
	    nnew->next = all_tt_conv;
	    all_tt_conv = nnew;
	}
	
	break;
      default:
	return 0;
    }
    
    /* skip all comments */
    
    while (syntax == COMMENT)
      syntax = read_keyword (fp, keyword);
    
    return syntax;
}




/*
 * parse_module_compliance() parses a MODULE-COMPLIANCE macro.
 * Returns NULL if an error was detected.
 */

static struct tree *
parse_module_compliance (fp, name)
     FILE *fp;
     char *name;
{
    char keyword[SYMBOL_MAXLEN];
    int syntax;
    struct tree *new, *np;
    struct subid	*nodelist, *freeptr;
    
    np = malloc_tree (name);
    
    /*
     * read keywords until syntax EQUALS is found
     */
    
    while ((syntax = read_keyword (fp, keyword)) != EQUALS) {
	switch (syntax) {
	  case DESCRIPTION:
	    np->description = ftell (fp);
	    if ((syntax = read_keyword (fp, keyword)) != QUOTESTRING) {
		fprintf (stderr, "%d --> %s\n", syntax, keyword);
		return NULL;
	    }
	    break;
	  case EOF:
	    return NULL;
	  default:
	    break;
	}
    }
    
    /*
     * parse list of the form { parent [parent list [...]] nr }
     */
    
    if ((nodelist = read_subid (fp)) == NULL)
      return NULL;
    
    /*
     * create new nodes, if neccessary and free nodelist
     */
    
    while (nodelist != NULL && nodelist->subid != -1) {
	
	/*
	 * check if it's node for this OBJECT-IDENTIFIER
	 */
	
	if (nodelist->label == NULL) {
	    np->parent_name = xstrdup (nodelist->parent_name);
	    np->subid = nodelist->subid;
	    
	} else {
	    
	    /*
	     * a new node must be created and linked into the list
	     */
	    
	    new = malloc_tree (nodelist->label);
	    new->parent_name = xstrdup (nodelist->parent_name);
	    new->syntax = SUBTREE;
	    new->subid  = nodelist->subid;
	    new->next   = root;
	    root        = new;
	}
	
	/*
	 * free this entry
	 */
       
	freeptr  = nodelist;
	nodelist = nodelist->next;
	free (freeptr);
   }
    
    return np;
}




/*
 * parse_module_identity() parses a MODULE-IDENTITY macro.
 */

static struct tree *
parse_module_identity (fp, name)
     FILE *fp;
     char *name;
{
    char keyword[SYMBOL_MAXLEN];
    int	syntax;
    struct tree	*new, *np;
    struct subid *nodelist, *freeptr;
    
    np = malloc_tree (name);

    /*
     * read keywords until syntax EQUALS is found
     */

    while ((syntax = read_keyword (fp, keyword)) != EQUALS) {
	switch (syntax) {
	  case DESCRIPTION:
            np->description = ftell (fp);
            if ((syntax = read_keyword (fp, keyword)) != QUOTESTRING) {
		fprintf (stderr, "%d --> %s\n", syntax, keyword);
		return NULL;
            }
            break;
	  case EOF:
            return NULL;
	  default:
            break;
	}
    }

    /*
     * parse list of the form { parent [parent list [...]] nr }
     */

    if ((nodelist = read_subid (fp)) == NULL)
      return NULL;
    
    /*
     * create new nodes, if neccessary and free nodelist
     */

    while (nodelist != NULL && nodelist->subid != -1) {

	/*
         * check if it's node for this OBJECT-IDENTIFIER
	 */

	if (nodelist->label == NULL) {
	    np->parent_name = xstrdup (nodelist->parent_name);
	    np->subid = nodelist->subid;

	} else {

	    /*
	     * a new node must be created and linked into the list
	     */

	    new = malloc_tree (nodelist->label);
	    new->parent_name = xstrdup (nodelist->parent_name);
	    new->syntax = SUBTREE;
	    new->subid  = nodelist->subid;
	    new->next   = root;
	    root        = new;
	}
	
	/*
	 * free this entry
	 */
	
	freeptr = nodelist;
	nodelist = nodelist->next;
	free (freeptr);
    }

    return np;
}




/*
 * parse_notification_type() parses a NOTIFICATION-TYPE macro.
 */

static struct tree *
parse_notification_type (fp, name)
     FILE *fp;
     char *name;
{
    char keyword[SYMBOL_MAXLEN];
    int	syntax;
    struct tree	*new, *np;
    struct subid *nodelist, *freeptr;

    np = malloc_tree (name);

    /*
     * read keywords until syntax EQUALS is found
     */

    while ((syntax = read_keyword (fp, keyword)) != EQUALS) {
	switch (syntax) {
	  case DESCRIPTION:
            np->description = ftell (fp);
            if ((syntax = read_keyword (fp, keyword)) != QUOTESTRING) {
		fprintf (stderr, "%d --> %s\n", syntax, keyword);
		return NULL;
            }
            break;
	  case EOF:
            return NULL;
	  default:
            break;
	}
    }
    
    /*
     * parse list of the form { parent [parent list [...]] nr }
     */
    
    if ((nodelist = read_subid (fp)) == NULL)
      return NULL;

    /*
     * create new nodes, if neccessary and free nodelist
     */

    while (nodelist != NULL && nodelist->subid != -1) {

	/*
         * check if it's node for this OBJECT-IDENTIFIER
	 */

	if (nodelist->label == NULL) {
	    np->parent_name = xstrdup (nodelist->parent_name);
	    np->subid = nodelist->subid;

	} else {

	    /*
	     * a new node must be created and linked into the list
	     */

	    new = malloc_tree (nodelist->label);
	    new->parent_name = xstrdup (nodelist->parent_name);
	    new->syntax = SUBTREE;
	    new->subid  = nodelist->subid;
	    new->next   = root;
	    root        = new;
	}

	/*
         * free this entry
	 */
	
	freeptr = nodelist;
	nodelist = nodelist->next;
	free (freeptr);
    }
    
    return np;
}


/*
 * parse_capabilities_type() parses or better ignores agent 
 * capabilities macros.
 */

static int
parse_capabilities_type (fp)
     FILE *fp;
{
    char keyword[SYMBOL_MAXLEN];
    int	syntax;

    while ((syntax = read_keyword (fp, keyword)) != EQUALS) {
	switch (syntax) {
	  case EOF:
            return 0;
	  default:
            break;
	}
    }

    while ((syntax = read_keyword (fp, keyword)) != RIGHTBRACKET) {
	switch (syntax) {
          case EOF:
            return 0;
          default:
            break;
        }
    }

    return 1;
}

/*
 * parse_trap_type() parses a TRAP-TYPE macro.
 */

static int
parse_trap_type (fp, name)
     FILE *fp;
     char *name;
{
    char keyword[SYMBOL_MAXLEN];
    int  syntax;

    /*
     * read keywords until syntax EQUALS is found
     */

    while ((syntax = read_keyword (fp, keyword)) != EQUALS) {
	if (syntax == EOF) return -1;
    }

    /*
     * parse a number defining the trap number
     */

    syntax = read_keyword (fp, keyword);
    if (syntax != NUMBER) 
      return -1;

    /*
     * TRAP-TYPE macro is complete
     */
    
    return (atoi (keyword));
}




/*
 * parse_object_group() parses an OBJECT-GROUP macro.
 */

static struct tree *
parse_object_group (fp, name)
     FILE *fp;
     char *name;
{
    char keyword[SYMBOL_MAXLEN];
    int	syntax;
    struct tree	*new, *np;
    struct subid *nodelist, *freeptr;

    /*
     * next keyword must be OBJECTS
     */
    
    if ((syntax = read_keyword (fp, keyword)) != OBJECTS)
      return NULL;
    
    np = malloc_tree (name);
    
    /*
     * read keywords until EQUALS are found
     */
    
    while ((syntax = read_keyword (fp, keyword)) != EQUALS) {
	switch (syntax) {
	  case STATUS:
	    syntax = read_keyword (fp, keyword);
	    if (syntax != CURRENT && syntax != OBSOLETE) {
		fprintf (stderr, "%d --> %s\n", syntax, keyword);
		return NULL;
	    }
	    break;
	  case DESCRIPTION:
	    np->description = ftell (fp);
	    if ((syntax = read_keyword (fp, keyword)) != QUOTESTRING) {
		fprintf (stderr, "%d --> %s\n", syntax, keyword);
		return NULL;
	    }
	    break;
	  case EOF:
	    return NULL;
	  default:
	    break;
	}
    }
    
    /*
     * parse list of the form { parent [parent list [...]] nr }
     */

    if ((nodelist = read_subid (fp)) == NULL)
      return NULL;
    
    /*
     * create new nodes, if neccessary and free nodelist
     */

    while (nodelist != NULL && nodelist->subid != -1) {

	/*
         * check if it's node for this OBJECT-IDENTIFIER
	 */

	if (nodelist->label == NULL) {
	    np->parent_name = xstrdup (nodelist->parent_name);
	    np->subid = nodelist->subid;

	} else {

	    /*
	     * a new node must be created and linked into the list
	     */

	    new = malloc_tree (nodelist->label);
	    new->parent_name = xstrdup (nodelist->parent_name);
	    new->syntax = SUBTREE;
	    new->subid  = nodelist->subid;
	    new->next   = root;
	    root        = new;
	}
	
	/*
	 * free this entry
	 */

	freeptr = nodelist;
	nodelist = nodelist->next;
	free (freeptr);
    }
    
    return np;
}




/*
 * parse_object_identity() parses an OBJECT-IDENTITY macro.
 */

static struct tree *
parse_object_identity (fp, name)
     FILE *fp;
     char *name;
{
    char keyword[SYMBOL_MAXLEN];
    int	syntax;
    struct tree	*new, *np;
    struct subid *nodelist, *freeptr;
    
    np = malloc_tree (name);

    /*
     * read keywords until EQUALS are found
     */

    while ((syntax = read_keyword (fp, keyword)) != EQUALS) {
	switch (syntax) {
	  case STATUS:
            syntax = read_keyword (fp, keyword);
            if (syntax != CURRENT && syntax != OBSOLETE) {
		fprintf (stderr, "%d --> %s\n", syntax, keyword);
		return NULL;
            }
            break;
          case DESCRIPTION:
            np->description = ftell (fp);
            if ((syntax = read_keyword (fp, keyword)) != QUOTESTRING) {
		fprintf (stderr, "%d --> %s\n", syntax, keyword);
		return NULL;
            }
            break;
	  case EOF:
            return NULL;
	  default:
            break;
	}
    }
    
    /*
     * parse list of the form { parent [parent list [...]] nr }
     */

    if ((nodelist = read_subid (fp)) == NULL)
      return NULL;

    /*
     * create new nodes, if neccessary and free nodelist
     */

    while (nodelist != NULL && nodelist->subid != -1) {

	/*
         * check if it's node for this OBJECT-IDENTIFIER
	 */

	if (nodelist->label == NULL) {
	    np->parent_name = xstrdup (nodelist->parent_name);
	    np->subid = nodelist->subid;
	    
	} else {
	
	    /*
	     * a new node must be created and linked into the list
	     */

	    new = malloc_tree (nodelist->label);
	    new->parent_name = xstrdup (nodelist->parent_name);
	    new->syntax = SUBTREE;
	    new->subid  = nodelist->subid;
	    new->next   = root;
	    root        = new;
	}

	/*
	 * free this entry
	 */

	freeptr = nodelist;
	nodelist = nodelist->next;
	free (freeptr);
    }

    return np;
}




/*
 * parse_object_id() parses an OBJECT-IDENTIFIER entry.
 */

static struct tree *
parse_object_id (fp, name)
     FILE *fp;
     char *name;
{
    char keyword[SYMBOL_MAXLEN];
    int	syntax;
    struct tree	*new, *np;
    struct subid *nodelist, *freeptr;

    /*
     * next keyword must be EQUALS
     */

    if ((syntax = read_keyword (fp, keyword)) != EQUALS)
      return NULL;
    
    np = malloc_tree (name);
    np->syntax = ASN1_OBJECT_IDENTIFIER;
    
    /*
     * parse list of the form { parent [parent list [...]] nr }
     */

    if ((nodelist = read_subid (fp)) == NULL)
      return NULL;
    
    /*
     * create new nodes, if neccessary
     */

    while (nodelist != NULL && nodelist->subid != -1) {

	/*
         * check if it's node for this OBJECT-IDENTIFIER
	 */

	if (nodelist->label == NULL) {
	    np->parent_name = xstrdup (nodelist->parent_name);
	    np->subid  = nodelist->subid;
	    
	} else {
	
	    /*
	     * a new node must be created and linked into the list
	     */

	    new = malloc_tree (nodelist->label);
	    new->parent_name = xstrdup (nodelist->parent_name);
	    new->syntax = SUBTREE;
	    new->subid  = nodelist->subid;
	    new->next   = root;
	    root        = new;
	}
	
	/*
	 * free this entry
	 */

	freeptr = nodelist;
	nodelist = nodelist->next;
	free (freeptr);
    }
    
    return (np);
}



/*
 * parse_object_type() parses an OBJECT-TYPE macro.
 */

static struct tree *
parse_object_type (fp, name)
     FILE *fp;
     char *name;
{
    char keyword[SYMBOL_MAXLEN];
    int	syntax;
    struct tree	*new, *np;
    struct subid *nodelist, *freeptr;

    /*
     * next keyword must be SYNTAX
     */

    if ((syntax = read_keyword (fp, keyword)) != SYNTAX)
      return NULL;
    
    np = malloc_tree (name);
    np->description = -1;

    /*
     * next keyword defines OBECT-TYPE syntax
     */

    if ((syntax = read_keyword (fp, keyword)) == ACCESS)
      return NULL;
    
    np->syntax = syntax;
    
    /*
     * no read keywords until ACCESS is found
     */

    /*
     * if the syntax found is LABEL, we should lookup the typedef:
     */
    
    if (syntax == LABEL) {
	struct textual_type *t = all_tt_conv;
	
	while (t)
	  if (! strcmp (t->name, keyword))
	    break;
	  else
	    t = t->next;
	
	if (t)
	  /* assign text-conv to the node: */
	  np->tt = t;
	
	/* 
	 * old eat-it-up code: skip anything to the ACCESS keyword: 
	 */
	
	while ((syntax = read_keyword (fp, keyword)) != ACCESS)
	  if (syntax == EOF)
	    return NULL;

    } else if (syntax == ASN1_INTEGER) {
    
	/*
	 * if the syntax found is INTEGER, there may follow the enums for a
	 * textual description of the integer value; extract - don't skip.
	 */

	char *ds_val;
	Tcl_DString stats;			/* collect name/val pairs */
	Tcl_DStringInit (&stats);
	
	/* 
	 * We may get something like ``{ foo ( 99), ... }'' or
	 * ``(0..99)'' or nothing. The second case is ignored.
	 */
	
	while ((syntax = read_keyword (fp, keyword)) == COMMENT) ;
	if (syntax == LEFTBRACKET) {
	    syntax = scan_enum_descr (&stats);
	    
	    /*
	     * old eat-it-up code: skip anything to the ACCESS keyword:
	     */
	    
	    while ((syntax = read_keyword (fp, keyword)) != ACCESS)
	      if (syntax == EOF)
		return NULL;

	} else if (syntax == LEFTPAREN) {

	    /* got LEFTPAREN: ``('' */
	    /* XXX: fetch here ranges... -- we simply skip */
	    int level = 1;
	    
	    while ((syntax = read_keyword (fp, keyword)) != RIGHTPAREN
		   && level > 0)
	      {
		  if (syntax == EOF)
		    return NULL;
		  else if (syntax == RIGHTPAREN)
		    level--;
	      }
	}
	
	/* 
	 * We simply skip up to the ACCESS key, like the old version 
	 * did: wrong, but effective... 
	 */
	
	while (syntax != ACCESS) {
	    syntax = read_keyword (fp, keyword);
	    if (syntax == EOF)
	      return NULL;
	}

	ds_val = Tcl_DStringValue (&stats);
	if (ds_val && *ds_val) {
	    np->tt = (struct textual_type *) 
	      xmalloc (sizeof (struct textual_type));
	    memset ((char *) np->tt, '\0', sizeof (struct textual_type));
	    
	    /* we fake a name, to allow saving to the frozen-index: */
	    np->tt->name = xmalloc (strlen (np->label) + 10);
	    sprintf (np->tt->name, "_%s", np->label);
	    np->tt->syntax = ASN1_INTEGER;
#if 0
	    np->tt->int_desc = xstrdup (ds_val);
#endif
	    np->tt->desc = tree_scan_intdesc (xstrdup (ds_val));
	    
	    /* because we now have a fakename, we will save it: */
	    np->tt->next = all_tt_conv;
	    all_tt_conv = np->tt;
	}
	
	/* we are fine. now continue below with the ACCESS mode: */ 
    } else {
	
	/* 
	* old eat-it-up code: skip anything to the ACCESS keyword: 
	*/
	
	while ((syntax = read_keyword (fp, keyword)) != ACCESS)
	  if (syntax == EOF)
	    return NULL;
    }
    
    /*
     * next keyword defines ACCESS mode for object
     */

    syntax = read_keyword (fp, keyword);
    if (syntax < READONLY || syntax > NOACCESS) {
	fprintf (stderr, "%s:%d: scan error near `%s'\n", 
		 filename, line, keyword);
	return NULL;
    }

    switch (syntax) {
      case READONLY:   np->access = M_READONLY; break;
      case READCREATE: np->access = M_READCREATE; break;
      case READWRITE:  np->access = M_READWRITE; break;
      case WRITEONLY:  np->access = M_WRITEONLY; break;
      default:         np->access = M_NOACCESS;
    }
    
    /*
     * next keyword must be STATUS
     */

    if ((syntax = read_keyword (fp, keyword)) != STATUS)
      return NULL;
    
    /*
     * next keyword defines status of object
     */

    syntax = read_keyword (fp, keyword);
    if (syntax < MANDATORY || syntax > DEPRECATED) {
	fprintf (stderr, "%s:%d: scan error near `%s'\n", 
		 filename, line, keyword);
	return NULL;
    }

    /*
     * now determine optional parts of OBJECT-TYPE macro
     */

    while ((syntax = read_keyword (fp, keyword)) != EQUALS) {
	switch (syntax) {
	  case DESCRIPTION:
            np->description = ftell (fp);
            if ((syntax = read_keyword (fp, keyword)) != QUOTESTRING) {
		fprintf (stderr, "%d --> %s\n", syntax, keyword);
		return NULL;
            }
            break;
	  case EOF:
            return NULL;
	  default:
            break;
	}
    }
    
    /*
     * parse list of the form { parent [parent list [...]] nr }
     */
    
    if ((nodelist = read_subid (fp)) == NULL)
      return NULL;

    /*
     * create new nodes, if neccessary
     */

    while (nodelist != NULL && nodelist->subid != -1) {
    
	/*
	 * check if it's node for this OBJECT-IDENTIFIER
	 */

	if (nodelist->label == NULL) {
	    np->parent_name = xstrdup (nodelist->parent_name);
	    np->subid = nodelist->subid;

	} else {

	    /*
	     * a new node must be created and linked into the list
	     */

	    new = malloc_tree (nodelist->label);
	    new->parent_name = xstrdup (nodelist->parent_name);
	    new->syntax = SUBTREE;
	    new->subid  = nodelist->subid;
	    new->next   = root;
	    root        = new;
	}
	
	/*
         * free this entry
	 */

	freeptr  = nodelist;
	nodelist = nodelist->next;
	free (freeptr);
    }
    
    return np;
}





/*
 * read_keyword() parses a keyword from the MIB file and places it in
 * the string pointed to by keyword. Returns the syntax of keyword or
 * EOF if any error.
 */

static int
read_keyword (fp, keyword)
     FILE *fp;
     char *keyword;

{
    char *cp = keyword;
    int	ch = lastchar;
    int	hash_val = 0;

    struct keyword *tp;
    
    *keyword = '\0';

    /*
     * skip spaces
     */

    while (isspace (ch) && ch != EOF) {
	if (ch == '\n') line++;
	ch = getc (fp);
    }

    if (ch == EOF) return EOF;

    /*
     * skip textual descriptions enclosed in " characters
     */

    if (ch == '"') {
	int len = 0;
	*keyword = '\0';
	while ((ch = getc (fp)) != EOF) {
	    if (ch == '\n') {
		line++;
	    } else if (ch == '"') {
		lastchar = ' ';
		return QUOTESTRING;
	    } else if (len < SYMBOL_MAXLEN - 2) {
		keyword [len++] = ch, keyword [len] = 0;
	    }
	}
	return EOF;
    }

    /*
     * skip comments
     */

    if (ch == '-') {
	hash_val += ch;
	*cp++ = ch;
	
	if ((ch = getc (fp)) == '-') {
	    *keyword = '\0';
	    while ((ch = getc (fp)) != EOF) {
		if (ch == '\n') {
		    line++;
		    break;
		}
	    }
	    if (ch == EOF) return EOF;
	    
	    lastchar = ' ';
	    return COMMENT;
	}
    }
   
    /*
     * Read characters until end of keyword is found
     */

    do {
	if (ch == '\n') line++;
	
	if (isspace (ch) || ch == '(' || ch == ')' || ch =='{' ||
	    ch == '}' || ch == ',' || ch == ';') {

	    /*
             * check for keyword of length 1
	     */

	    if (!isspace (ch) && *keyword == '\0') {
		hash_val += ch;
		*cp++ = ch;
		lastchar = ' ';
	    } else if (ch == '\n') {
		lastchar = ' ';
	    } else {
		lastchar = ch;
	    }
	       
	    *cp = '\0';

	    /*
	     * is this a defined keyword ?
	     */

	    for (tp = hashtab[HASH(hash_val)]; tp != NULL; tp = tp->next) {
		if ((tp->hash == hash_val) && !strcmp (tp->name, keyword))
		  break;
	    }
	    
	    if (tp != NULL) {

		/*
		 * if keyword is not complete, continue; otherwise return
		 */
		
		if (tp->key == CONTINUE)
		  continue;
		return tp->key;
	    }
	    
	    /*
	     * is it a LABEL ?
	     */
	    
	    for (cp = keyword; *cp; cp++) {
		if (! isdigit (*cp)) return LABEL;
	    }
	    
	    /*
	     * keywords consists of digits only
	     */
	    
	    return NUMBER;

	} else {
	
	    /*
	     * build keyword and hashvalue
	     */

	    hash_val += ch;
	    *cp++ = ch;
	}
    } while ((ch = getc (fp)) != EOF);
    
    return EOF;
}




/*
 * read_subid() parses a list of the form { iso org(3) dod(6) 1 }
 * and creates a parent-child entry for each node. Returns NULL
 * if we found an error.
 */

static struct subid*
read_subid (fp)
     FILE *fp;
{
   char	name[SYMBOL_MAXLEN]; 
   char	keyword[SYMBOL_MAXLEN]; 
   int is_child	= 0;
   int syntax = 0;
   struct subid	*np = NULL;
   struct subid *nodelist = NULL;

   /*
    * EQUALS are passed, so first keyword must be LEFTBRACKET
    */

   if ((syntax = read_keyword (fp, keyword)) != LEFTBRACKET) return NULL;

   /*
    * now read keywords until RIGHTBRACKET is passed
    */

   while ((syntax = read_keyword (fp, keyword)) != RIGHTBRACKET) {
       switch (syntax) {
	 case EOF:
	   return NULL;
         case LABEL:
	   
	   /* allocate memory for new element */
	   
	   np = (struct subid *) xmalloc (sizeof(struct subid));
	   memset ((char *) np, '\0', sizeof (struct subid));
	   np->subid  = -1;
	   
	   /* if this label followed another one, it's the child */
	   
	   if (is_child) {
	       np->parent_name = xstrdup (name);
               np->label  = xstrdup (keyword);
	   } else {
	       np->parent_name = xstrdup (keyword);
               is_child = 1;		/* next labels are children   */
	   }
	   
	   np->next = nodelist;
	   nodelist = np;
	   strcpy (name, keyword);
	   break;
         case LEFTPAREN:
	   if ((syntax = read_keyword (fp, keyword)) != NUMBER) return NULL;
	   np->subid = atoi (keyword);
	   if ((syntax = read_keyword (fp, keyword)) != RIGHTPAREN)
	     return NULL;
	   break;            
         case NUMBER:
            if (np->subid != -1) {
		np = (struct subid *) xmalloc (sizeof (struct subid));
		memset ((char *) np, '\0', sizeof (struct subid));
		np->parent_name = xstrdup (name);
		np->subid = -1;
		np->next = nodelist;
		nodelist = np;
	    }
	   np->subid = atoi (keyword);
	   break;
         default:
	   return NULL;
       }
   }
   return nodelist;
}


/*
 * lookup_node() finds the node given by name. Returns the pointer to
 * the the tree node or NULL if not found.
 */

static struct tree *
lookup_node (root, name)
     struct tree *root;
     char *name;
{
    struct tree *brother, *new_root;
    
    /* examine all nodes of this tree level */

    for (brother = root; brother != NULL; brother = brother->next) {
	if (!strcmp (name, brother->label)) return brother;
	
	/* for each node parse the subtree */

	if (brother->child != NULL) {
	    new_root = lookup_node (brother->child, name);
	    if (new_root != NULL) return new_root;
	}
    }
    
    return NULL;
}


/*
 * build_tree() builds a MIB tree. Returns pointer to root if
 * successful, NULL otherwise.
 */

static struct tree*
build_tree (nlist)
     struct tree *nlist;
{
    struct tree	*ccitt, *iso, *joint;
    
    /* init the node hashtable */

    node_hash_init (nlist);

    /* build the "ccitt" node */

    ccitt = malloc_tree ("ccitt");
    ccitt->parent_name = xstrdup ("(unknown)");
    ccitt->syntax      = TREE;
    
    /* build the "iso" node */

    iso = malloc_tree ("iso");
    iso->parent_name = xstrdup ("(unknown)");
    iso->subid       = 1;
    iso->syntax      = TREE;
    ccitt->next      = iso;

    /* build the "joint-iso-ccitt" node */
    
    joint = malloc_tree ("joint-iso-ccitt");
    joint->parent_name = xstrdup ("(unknown)");
    joint->subid       = 2;
    joint->syntax      = TREE;
    iso->next          = joint;
    
    /* build the subtrees */

    build_subtree (ccitt);
    build_subtree (iso);
    build_subtree (joint);

    /*
     * Return the first node in the tree (perhaps "iso" would be
     * better, but this way is consistent)
     */
    
    return ccitt;
}


/*
 * node_hash_init() reads the input nodelist and puts them into the
 * node hashtable, where the hash function is build by the parent
 * names. This means, that all nodes with the same parent are in one
 * list.
 */

static void
node_hash_init (nlist)
     struct tree *nlist;
{
    char *cp;
    int	hash_val;
    int	hash_index;
    struct tree	*np, *nextp;

    memset ((char *) nodehashtab,'\0', sizeof (nodehashtab));

    for (np = nlist; np != NULL; ) {
	if (! np->parent_name) {
	    fprintf (stderr, "%s: %s has no parent in the MIB tree!\n",
		     filename, np->label);
	    return;
	}
	for (hash_val = 0, cp = np->parent_name; *cp; cp++) hash_val += *cp;
	hash_index = NODEHASH (hash_val);	
	nextp = np->next;
	np->next = nodehashtab[hash_index];
	nodehashtab[hash_index] = np;
	np = nextp;
    }
}


/*
 * addto_tree()
 */

static void
addto_tree (root, nlist)
     struct tree *root;
     struct tree *nlist;
{
    struct tree	*np;
    struct tree *tree = NULL;

   /*
    * Test if the parent of the first node exists. We expect that we
    * load individual subtrees where the parent of the first node
    * defines the anchor. This must already exist.
    */

    if (! nlist) return;

    for (np = nlist; np->next; np = np->next) ;
    tree = lookup_node (root, np->parent_name);
    if (tree == NULL) {
	fprintf (stderr, "%s: can not link subtree %s to parent %s\n", 
		 filename, np->label, np->parent_name);
	return;
    }
    
   /*
    * Init the node hashtable and build the subtree.
    */

    node_hash_init (nlist);
    build_subtree (tree);
}


/*
 * build_subtree() finds all the children of root in the list of
 * nodes, link them in the tree and out of the node list. Returns
 * pointer to root if successful, NULL otherwise.
 */

static void
build_subtree (root)
     struct tree *root;
{
    char *cp;
    int	hash_val;
    int	hash_index;
    struct tree	**np, *this_node;
    struct tree	*prev = NULL;

    /* get hash_index to look up the children */

    for (hash_val = 0, cp = root->label; *cp; cp++)
      hash_val += *cp;
    hash_index = NODEHASH (hash_val);

    /*
     * Search the nodes whose parent is root
     */

    for (np = &nodehashtab[hash_index]; *np != NULL; ) {
	
	if (!strcmp (root->label, (*np)->parent_name)) {

	    this_node = *np;
	    *np = (*np)->next;

	    this_node->filename = filename;
	    if (this_node->parent_name) {
		free (this_node->parent_name);
		this_node->parent_name = NULL;
	    }
	    this_node->parent   = root;
	    this_node->child    = NULL;
	    this_node->next     = NULL;
	    
	    /* link node in the tree */
	    if (root->child == NULL)
	      root->child = this_node;
	    else if (prev != NULL)
	      prev->next = this_node;
	    else {
		for (prev = root->child; prev->next; prev = prev->next) ;
		prev->next = this_node;
	    }

	    prev = this_node;

	    build_subtree (this_node);		/* recurse on child */	    
	}
	else
	  np = &(*np)->next;
    }
}
