/*
 * misc.c
 *
 * This file contains some functions to manipulate request lists
 * that keep track of outstanding requests. It also contains a
 * wrapper around the MD5 digest algorithm.
 *
 * Copyright (c) 1994
 *
 * Sven Schmidt, J. Schoenwaelder
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <tcl.h>

#ifdef DBMALLOC
# include <dbmalloc.h>
#endif

#include "snmp.h"
#include "md5.h"
#include "memory.h"
#include "udp.h"
#include "misc.h"
#include "xmalloc.h"

/*
 * SNMP_MD5_digest() computes the message digest of the given packet.
 */

void
SNMP_MD5_digest (packet, length, digest)
     u_char *packet;
     int length;
     u_char *digest;
{
    MDstruct MD;
    int i, j;
    u_char buf[BUFSIZE], *cp;

    /*
     * as the computation has side effects on machines with LOWBYTEFIRST,
     * do the computation in a static array
     */

    memcpy ((char *)buf, (char *) packet, length);
    cp = buf;

    MDbegin (&MD);
    while (length > 64) {
        MDupdate (&MD, cp, 512);
	cp += 64;
	length -= 64;
    }
    MDupdate (&MD, cp, length * 8);

    cp = digest;
    for (i = 0; i < 4; i++) {
        for (j = 0; j < 32; j+=8)
	  *cp++ = (MD.buffer[i] >> j) & 0xFF;
    }
    *cp = '\0';
}

/*
 * SNMP_RecordRequest() saves this message in the request list for the
 * given session, together with it's callback function, which is
 * executed when the response request is received from the agent.
 */

struct request_list*
SNMP_RecordRequest (interp, sess, reqid, packet, packetlen,
		    callback, error_callback)
     Tcl_Interp *interp;
     struct session *sess;
     int reqid;
     u_char *packet;
     int packetlen;
     char *callback;
     char *error_callback;
{
    struct request_list *req, *eol;
    struct request_timeout *timeout;

    /*
     * allocate and initialize a request_timeout structure
     */

    timeout = (struct request_timeout *) 
			xmalloc (sizeof (struct request_timeout));
    timeout->interp = interp;
    timeout->request_id = reqid;
    
    /*
     * setup the information for this request and install TimeoutHandler
     * for ansychronous requests
     */
    
    req = SNMP_MallocRequest();
    req->request_id = reqid;
    req->packet = (u_char *) xmalloc (packetlen);
    memcpy (req->packet, packet, packetlen);
    req->packetlen = packetlen;
    req->timeo_data = timeout;
    
    req->timeout_token = Tk_CreateTimerHandler((sess->timeout * 1000) 
					       / sess->retries,
					       SNMP_TimeoutEvent,
					       (ClientData *)timeout);
    if (callback) req->callback_func = xstrdup (callback);
    if (error_callback) req->error_func = xstrdup (error_callback);
    
    /*
     * append request at list of this session
     */

    if (sess->request_list == NULL) {
	sess->request_list = req;
    } else {
        eol = sess->request_list;
	while (eol->next != NULL) {
	    eol = eol->next;
	}
	req->prev = eol;
	eol->next = req;
    }
    
    return req;
}


/*
 * SNMP_LookupRequest() walks through the request lists of the open
 * sessions and tries to find the given request_id. Returns a pointer
 * to the request structure, if the request is found. The session that
 * invoked the request, is returnd in req_sess.  If there is no
 * corresponding request to this id, NULL is returned. The calling
 * function has to decide how to handle this situation.
 */

struct request_list*
SNMP_LookupRequest (reqid, req_sess)
     int reqid;
     struct session **req_sess;
{
    struct session *sess;
    struct request_list	*request;

    for (sess = session_list; sess != NULL; sess = sess->next) {
	request = sess->request_list;
	while (request != NULL) {
	    if (request->request_id == reqid) {
		*req_sess = sess;
		return request;
	    }
	    request = request->next;
	}
    }
    return NULL;
}

/*
 * SNMP_DeleteRequest() deletes the request pointed to by request from
 * the given session pointed to by session.
 */

void
SNMP_DeleteRequest (sess, request)
     struct session *sess;
     struct request_list *request;
{
    /* 
     * free present timeout client-data: 
     */
    
    if (request->timeo_data) free ((char *) request->timeo_data);
    request->timeo_data = 0;
    
    /*
     * first of all, lets delete the TimerHandler for this request  
     */
    
    Tk_DeleteTimerHandler (request->timeout_token);

    /*
     * if it is the first element in the request list, we have to set
     * the sess->request_list pointer
     */

    if (request->prev == NULL) {		/* first request in list */
	if (request->next == NULL) {		/* single request */
	    sess->request_list = NULL;
	} else {
	    sess->request_list = sess->request_list->next;
	    sess->request_list->prev = NULL;
	}
    } else {
	if (request->next == NULL) {		/* last element in list */
	    request->prev->next = NULL;
	} else {
	    request->prev->next = request->next;
	    request->next->prev = request->prev;
	}
    }
    
    SNMP_FreeRequest (request); 
}

/*
 * Return the uptime of this agent in hundreds of seconds. Should
 * be initialzed when registering the SNMP extension.
 */

int
SNMP_SysUpTime ()
{
    static time_t boottime = 0;

    if (!boottime) {
	boottime = time ((time_t *) NULL);
	return 0;
    } else {
	return (time ((time_t *) NULL) - boottime) * 100;
    }
}


/*
 * bin_to_hex() converts the binary buffer s with len n into readable
 * format (eg: 1A:2B:3D). The result is returned in d (with trailing
 * '\0') 
 */

void
bin_to_hex (s, n, d)
     char *s;
     int n;
     char *d;
{
    while (n-- > 0) {
	char c = *s++;
	int c1 = (c & 0xf0) >> 4;
	int c2 = c & 0x0f;
	if ((c1 += '0') > '9') c1 += 7;
	if ((c2 += '0') > '9') c2 += 7;
	*d++ = c1, *d++ = c2;
	if (n > 0)
	  *d++ = ':';
    }
    *d = 0;
}


/*
 * hex_to_bin() converts the readable hex buffer s to pure binary
 * octets into buffer d and return the length in n.
 */

void
hex_to_bin (s, d, n)
     char *s, *d;
     int *n;
{
    *n = 0;

    while (*s) {
	char c = *s++ & 0xff;
	int v = c >= 'a' ?  c - 87 : (c >= 'A' ? c - 55 : c - 48);
	c = *s++ & 0xff;
	v = (v << 4) + (c >= 'a' ?  c - 87 : (c >= 'A' ? c - 55 : c - 48));
	*d++ = v;
	(*n)++;
	if (*s == ':')
	  s++;
    }
}
