/*
 * memory.c
 *
 * Functions to allocate and free memory for the different
 * structures used by the Tcl API.
 *
 * Copyright (c) 1994
 *
 * Sven Schmidt
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdio.h>
#include <string.h>
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <tcl.h>

#ifdef DBMALLOC
#include <dbmalloc.h>
#endif

#include "snmp.h"
#include "asn1.h"
#include "xmalloc.h"
#include "memory.h"
#include "misc.h"
#include "udp.h"

/*
 * SNMP_MallocSession() allocates a session structure and initializes all
 * pointers to NULL. The functions returns a pointer to the session 
 * structure or NULL, if any error occures.
 */

struct session*
SNMP_MallocSession (sid)
     int *sid;
{
    struct session *s;
    
    s = (struct session *) xmalloc (sizeof (struct session));

    memset ((char *) s, '\0', sizeof (struct session));

    sprintf (s->name, "s%d", *sid);
    s->version		 = SNMPv1;
    s->community	 = xstrdup( "public" );
    s->dstParty.TAddress = xstrdup ("127.0.0.1");
    s->dstParty.TPort    = SNMP_PORT;
    s->srcParty.TAddress = xstrdup ("127.0.0.1");
    s->srcParty.TPort    = SNMP_PORT;
    s->reqid		 = rand();
    s->retries		 = DEFAULT_RETRIES;
    s->timeout		 = DEFAULT_TIMEOUT;

    return s;
}

/*
 * SNMP_MallocRequest() allocates a request_list structure and initializes
 *  all pointers to NULL. The functions returns a pointer to the
 *  structure or NULL if any error occures.
 */

struct request_list*
SNMP_MallocRequest()
{
    struct request_list *req;

    req = (struct request_list *) xmalloc (sizeof (struct request_list));
    
    req->request_id	= 0;
    req->retr_nr	= 0;
    req->packet		= NULL;
    req->packetlen	= 0;
    req->timeo_data	= NULL;
    req->callback_func	= NULL;
    req->error_func	= NULL;
    req->prev		= NULL;
    req->next		= NULL;
    
    return req;
}

/*
 * SNMP_FreeSession() frees the memory allocated by a session and all it's
 * associated structures.
 */

void
SNMP_FreeSession (session)
     struct session *session;
{
    struct request_list *req;
    
    if (session != NULL) {
	if (session->community) free (session->community);

	if (session->dstParty.Name) free (session->dstParty.Name);
	if (session->dstParty.Identity) free (session->dstParty.Identity);
	if (session->dstParty.TDomain) free (session->dstParty.TDomain);
	if (session->dstParty.TAddress) free (session->dstParty.TAddress);

	if (session->srcParty.Name) free (session->srcParty.Name);
	if (session->srcParty.Identity) free (session->srcParty.Identity);
	if (session->srcParty.TDomain) free (session->srcParty.TDomain);
	if (session->srcParty.TAddress) free (session->srcParty.TAddress);

	if (session->traceCallBack) free (session->traceCallBack);
	while (session->request_list != NULL) {
	    req = session->request_list;
	    session->request_list = session->request_list->next;
	    SNMP_FreeRequest (req);
	}
	if (session->agentSocket) SNMP_AgentClose (session);
	free (session);
    }
}

/*
 * SNMP_FreeRequest() frees the memory allocated by a request_list structure.
 */

void
SNMP_FreeRequest (request)
     struct request_list *request;
{
    if (request != NULL) {
	if (request->packet) free (request->packet);
	if (request->callback_func) free (request->callback_func);
	if (request->error_func) free (request->error_func);
	/* timeo_data is freed when removing the timeout-handler */
	free ((char *) request);
    }
}
