/*
 * create.c 
 *
 * Copyright (c) 1994
 *
 * M. Kernchen
 * TU Braunschweig, Germany
 * Institute for Operating Systems and Computer Networks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that this copyright
 * notice appears in all copies.  The University of Braunschweig
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include "cmip.h"

int
cmip_create(clientData, interp, argc, argv)
    ClientData	  clientData;	/* (char *) to the cmip handle */
    Tcl_Interp  *interp;
    int          argc;
    char       **argv;
{
    char	 *cmipHdl = (char *) clientData;

    char	 *strObjClass = NULLCP;
    char	 *strObjInst  = NULLCP;
    char	 *strRefInst  = NULLCP;

    char	**strAttrs    = NULL;
    int		  nattrs      = 0;
    char	**strAttrType = NULL;
    char	**strAttrValue= NULL;

    char	 *callback    = NULLCP;
    char	 *errorback   = NULLCP;

    int		  listArgc;
    char	**listArgv;

    int		  i, j, status;
 
    MIDentifier	  obj_class;
    MName	  obj_inst;
    MName	  ref_inst;
    int		  instType	= -1; /* see CMISCreateArg in mparm.h */
    CMISParam	 *attrs;

    MSAPIndication	mi;

    cmipHandle    *cmiph;
    requestHandle *rh;

    Tcl_HashEntry *ht_entry;
    int		   flag = 0;

    char	buf[256];

    obj_class.mid_type   = MID_GLOBAL;
    obj_class.mid_global = NULLOID;

    obj_inst.mn_type = ref_inst.mn_type = -1;	/* not given */
    obj_inst.mn_dn   = ref_inst.mn_dn = NULLDN;

    /* parse command line arguments */

    if (argc < 3)
	goto usage;

    if (argv[2][0] != '\0') strObjClass = argv[2];

    if (!strObjClass)
	goto usage;

    i = 3;
    while (i < argc) {
        if (i == argc-1) {
	    goto usage;
	}
        if ((argv[i][0] == 'i') && (strcmp(argv[i], "instance") == 0)) {
	    if (strObjInst) {
		if (instType == CA_SUPERIOR_INST) {
		    Tcl_AppendResult(interp, "instance and superior used:",
				      " try only one of them",
				     (char *) NULL);
		    return TCL_ERROR;
		} else {
		    Tcl_AppendResult(interp, "duplicate keyword \"", argv[i],
				      "\"", (char *) NULL);
		    return TCL_ERROR;
		}
	    }
            strObjInst = argv[++i];
	    instType = CA_OBJECT_INST;
        } else if ((argv[i][0] == 's') && (strcmp(argv[i], "superior") == 0)) {
	    if (strObjInst) {
		if (instType == CA_OBJECT_INST) {
		    Tcl_AppendResult(interp, "instance and superior used:",
				      " try only one of them",
				     (char *) NULL);
		    return TCL_ERROR;
		} else {
		    Tcl_AppendResult(interp, "duplicate keyword \"", argv[i],
				      "\"", (char *) NULL);
		    return TCL_ERROR;
		}
	    }
            strObjInst = argv[++i];
	    instType = CA_SUPERIOR_INST;
        } else if ((argv[i][0] == 'r') &&
		   (strcmp(argv[i], "reference") == 0)) {
	    if (strRefInst) {
		Tcl_AppendResult(interp, "duplicate keyword \"", argv[i],
				  "\"", (char *) NULL);
		return TCL_ERROR;
	    }
	    strRefInst = argv[++i];
        } else if ((argv[i][0] == 'a') &&
                   (strcmp(argv[i], "attributes") == 0)) {
	    if (nattrs > 0) {
		Tcl_AppendResult(interp, "duplicate keyword \"", argv[i],
				  "\"", (char *) NULL);
		return TCL_ERROR;
	    }
            if (Tcl_SplitList(interp, argv[++i], &nattrs, &strAttrs) != TCL_OK)
	    {
                return TCL_ERROR;
            }
	    if (nattrs <= 0) goto usage;
	    strAttrType = (char **) xmalloc(nattrs * sizeof(*strAttrType));
	    strAttrValue = (char **) xmalloc(nattrs * sizeof(*strAttrValue));
	    for (j=0; j < nattrs; j++) {
		if (Tcl_SplitList(interp, strAttrs[j],
				  &listArgc, &listArgv) != TCL_OK) {
		    return TCL_ERROR;
		}
		if (listArgc == 2) {
		    strAttrType[j]  = xstrdup(listArgv[0]);
		    strAttrValue[j] = xstrdup(listArgv[1]);
		} else {
		    free((char *) strAttrs);
		    free((char *) listArgv);
		    Tcl_AppendResult(interp, "malformed attribute-list:",
				      " should be a list of attribute-type",
				      " and -value pairs",
				     (char *)NULL);
		    return TCL_ERROR;
		}
		free((char *) listArgv);
	    }
            free((char *) strAttrs);
        } else if ((argv[i][0] == 'c') &&
		   (strcmp(argv[i], "callback") == 0)) {
	    if (callback) {
		Tcl_AppendResult(interp, "duplicate keyword \"", argv[i],
				  "\"", (char *) NULL);
		return TCL_ERROR;
	    }
            callback = argv[++i];
	    if (i < argc-2 && argv[i+1][0] == 'e' &&
		strcmp(argv[i+1], "errorback") == 0) {
		if (errorback) {
		    Tcl_AppendResult(interp, "duplicate keyword \"",
				      argv[i+1], "\"", (char *) NULL);
		    return TCL_ERROR;
		}
		i += 2;
		errorback = argv[i];
	    }
        } else {
	    Tcl_AppendResult(interp, "wrong keyword \"", argv[i], "\":",
			     "should be instance, superior,",
			     " reference, attributes",
			     " or callback",
			     (char *)NULL);
	    return TCL_ERROR;
	}
        i++;
    }
#ifdef DEBUG_CMIP
    fprintf(stderr, "Aufruf: \"%s %s", argv[0], argv[1]);
    if (strObjClass)
      fprintf(stderr, " %s", strObjClass);
    if (strObjInst)
      if (instType==CA_OBJECT_INST)
        fprintf(stderr, " instance %s", strObjInst);
      else
        fprintf(stderr, " superior %s", strObjInst);
    if (strRefInst)
      fprintf(stderr, " reference %s", strRefInst);
    if (nattrs>0) {
      int q;
      fprintf(stderr, " attributes {");
      for (q = 0; q < nattrs; q++) {
	fprintf(stderr, " {%s %s}", strAttrType[q], strAttrValue[q]);
      }
      fprintf(stderr, "}");
    }
    fprintf(stderr, "\"\n");
#endif

    ht_entry = Tcl_FindHashEntry(&cmip_table, cmipHdl);
    if (ht_entry == NULL) {
        Tcl_AppendResult(interp, "no such cmip handle \"", cmipHdl, "\"",
                         (char *) NULL);
        return TCL_ERROR;
    }
    cmiph = (cmipHandle *) Tcl_GetHashValue(ht_entry);

    /* convert the string arguments to CMIS parameters */

    if ((obj_class.mid_global = name2oid(strObjClass)) == NULLOID) {
	Tcl_AppendResult(interp, "no such object class \"", strObjClass, "\"",
                         (char *)NULL);
        return TCL_ERROR;
    }

    if (strObjInst) {
	if ((obj_inst.mn_dn = str2dn(strObjInst)) == NULLDN) {
	    Tcl_AppendResult(interp, "no such object instance \"", strObjInst,
			      "\"", (char *)NULL);
	    return TCL_ERROR;
	}
	/* in MCREATE: MN_LOCALDN aber in M_Create(3N): MN_DN */
	obj_inst.mn_type = MN_DN;
    }
    if (strRefInst) {
	if ((ref_inst.mn_dn = str2dn(strRefInst)) == NULLDN) {
	    Tcl_AppendResult(interp, "no such reference instance \"",
			      strRefInst, "\"",
			     (char *)NULL);
	    return TCL_ERROR;
	}
	/* ^ s. o. obj_inst ^ */
	ref_inst.mn_type = MN_DN;
    }
    attrs= (CMISParam *) xmalloc(nattrs*sizeof(CMISParam));
    for (i = 0; i < nattrs; i++) {
	Ava* aVa;
	if ((aVa = str2ava(strAttrType[i], strAttrValue[i])) == NULLAVA) {
	    Tcl_AppendResult(interp, "no such attribute type \"",
			      strAttrType[i], "\" or value \"",
			      strAttrValue[i], "\"",
			     (char *)NULL);
	    free((char *) attrs);
	    return TCL_ERROR;
	}
	attrs[i].mp_id.mid_type = MID_GLOBAL;
	attrs[i].mp_id.mid_global = aVa->ava_oid;
	attrs[i].mp_val = aVa->ava_value;
	free((char*) aVa);
    }

    /* perform the M-CREATE request */
    status = M_Create(cmiph->msd, cmiph->req_nr, &obj_class,
		      obj_inst.mn_type == -1 ? NULLMN : &obj_inst, instType,
		      ref_inst.mn_type == -1 ? NULLMN : &ref_inst,
		      NULLMACCESS, nattrs, attrs, &mi);

    if (status == NOTOK) {
        Tcl_AppendResult(interp, argv[0], " ", argv[1], ": ",
			  mi.mi_preject.mpr_data,
                         (char *)NULL);
        return TCL_ERROR;
    }

    sprintf(buf, "create%d", cmiph->req_nr); /* compose the request handle */
    rh = (requestHandle *) xmalloc(sizeof(requestHandle));
    rh->reqHdl	     = xstrdup(buf);
    rh->request_id   = cmiph->req_nr;
    rh->reqId	     = xstrdup("CREATE");
    rh->reqcallback  = callback ? xstrdup(callback) : NULLCP;
    rh->reqerrorback = errorback ? xstrdup(errorback) : NULLCP;
    Tcl_DStringInit(&rh->dStrResult);

    (cmiph->req_nr)++;

    ht_entry = Tcl_CreateHashEntry(cmiph->req_table, rh->reqHdl, &flag);
    if (flag != 1) {
        /* shouldn't really happen */
	Tcl_AppendResult(interp, "*** request handle already exist!? ***",
			 (char *)NULL);
        return TCL_ERROR;
    }
    Tcl_SetHashValue(ht_entry, (ClientData) rh);

    /* release memory of arguments */
    mid_free(&obj_class);
    if (obj_inst.mn_dn)
	mn_free(&obj_inst);
    if (ref_inst.mn_dn)
	mn_free(&ref_inst);
    for (i = 0; i < nattrs; i++)
	mparm_free(&attrs[i]);
    free((char*) attrs);
    free((char*) strAttrType);
    free((char*) strAttrValue);

    if (callback) { /* asynchron */
	status = TCL_OK;
    } else {
	/* synchron: do while there are outstanding responses */
	do {
	    status = wait(cmipHdl, interp, rh);
	    switch (status) {
	    case TCL_OK:
		Tcl_DStringResult(interp, &rh->dStrResult);
	    case TCL_ERROR:
	    case TCL_CONTINUE:
		break;
	    default:
		sprintf(buf, "\"$d\"", status);
		Tcl_AppendResult(interp, "cmip_create():", 
				 " unexpected returnvalue from wait() ",
				 buf, (char*)NULL);
		return TCL_ERROR;
	    }
	} while (status == TCL_CONTINUE);

	Tcl_DeleteHashEntry(ht_entry);
	rh_free(rh);
    }
    return status;

 usage:
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ", argv[1],
		     " <class> [instance <instance> |",
		     " superior <superiorInst>] [reference <referenceInst>]",
		     " [attributes {{<attributeType> <attributeValue>}...}]",
		     " [callback <callback> [errorback <errorback>]]\"",
		     (char *)NULL);
    return TCL_ERROR;
} /* cmip_create() */
